/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "TrapPatches.h"

#include "CPU_REG.h"			// Emulator::GetCycleCount
#include "MetaMemory.h"			// MetaMemory mark functions


/* ===========================================================================
	The following macros are used to help us generate patches to all the
	Memory Manager functions.  We patch the entire Memory Manager in order
	to track when we are "in" the Memory Manager. We want to know this so
	that we can extend special privileges to the Memory Manager when it
	accesses memory; the Memory Manager can touch more of RAM than any other
	executing code.
	
	Since our patches mostly just increment and decrement a counter, most of
	them are generated automatically with macros.  I call these "simple"
	head- and tailpatches.  However, we patch a few other the other Memory
	Manager functions for other reasons.  We can't let the macros generate
	simple patches for these functions and instead have to write them
	ourselves, making sure to perform the same incrementing and decrementing
	the simple functions would do.  I call these patches "complex" head- and
	tailpatches.
	
	Therefore, there are four "categories" of patchings: functions for which
	we want simple head- and tailpatches, functions for which we have a
	complex headpatch but need a simple tailpatch, functions for which we
	have a complex tailpatch but need a simple headpatch, and functions for
	which we already have complex head- and tailpatches and don't need
	simple patches.
	
	(Actually, there is a fifth category: Memory Manager functions that we
	don't want to treat as Memory Manager functions.  The functions in this
	category are MemSet, MemMove, and MemCmp.  We don't want to extend to
	these functions the same privileges that other Memory Manager functions
	get.)
	
	(And there's kind of a sixth category as well: non-Memory Manager
	functions that we want to treat as Memory Manager functions. For
	instance, DmWriteCheck grovels over Memory Manager data structures, so
	we have to give it Memory Manager function privileges. 
	PrvFindMemoryLeaks is also a function like this, but since it's not in
	the trap table, we can't patch it, and have to handle it differently. 
	See MetaMemory.cpp for this.)
	
	The macros below divide the set of Memory Manager functions into the
	four categories.  Any function on any of the four lists will
	automatically have generated for it declarations for head- and
	tailpatches.  Additionally, depending on which list it's on, it will
	optionally have simple head- and tailpatch functions generated for it.
	
	List of functions for which we want to automatically generate simple
	head- and tailpatches.
	
	DmWriteCheck is on this list because it grovels over memory, making sure
	everything is OK before it writes to the storage heap.  ScrInit is on
	this list because it calls MemMove to move the contents of the temporary
	LCD buffer to the real one after it's been allocated; the source of this
	move is a faux buffer out in the middle of an unallocated block.
=========================================================================== */

#define FOR_EACH_STUB_BOTHPATCH_FUNCTION(DO_TO_FUNCTION)	\
	DO_TO_FUNCTION(MemInit)					\
	DO_TO_FUNCTION(MemInitHeapTable)		\
	DO_TO_FUNCTION(MemNumCards)				\
	DO_TO_FUNCTION(MemCardFormat)			\
	DO_TO_FUNCTION(MemCardInfo)				\
	DO_TO_FUNCTION(MemStoreInfo)			\
	DO_TO_FUNCTION(MemStoreSetInfo)			\
	DO_TO_FUNCTION(MemNumHeaps)				\
	DO_TO_FUNCTION(MemNumRAMHeaps)			\
	DO_TO_FUNCTION(MemHeapID)				\
	DO_TO_FUNCTION(MemHeapDynamic)			\
	DO_TO_FUNCTION(MemHeapFreeBytes)		\
	DO_TO_FUNCTION(MemHeapSize)				\
	DO_TO_FUNCTION(MemHeapFlags)			\
	DO_TO_FUNCTION(MemHeapInit)				\
	DO_TO_FUNCTION(MemPtrRecoverHandle)		\
	DO_TO_FUNCTION(MemPtrFlags)				\
	DO_TO_FUNCTION(MemPtrSize)				\
	DO_TO_FUNCTION(MemPtrOwner)				\
	DO_TO_FUNCTION(MemPtrHeapID)			\
	DO_TO_FUNCTION(MemPtrDataStorage)		\
	DO_TO_FUNCTION(MemPtrCardNo)			\
	DO_TO_FUNCTION(MemPtrToLocalID)			\
	DO_TO_FUNCTION(MemPtrSetOwner)			\
	DO_TO_FUNCTION(MemHandleFlags)			\
	DO_TO_FUNCTION(MemHandleSize)			\
	DO_TO_FUNCTION(MemHandleOwner)			\
	DO_TO_FUNCTION(MemHandleLockCount)		\
	DO_TO_FUNCTION(MemHandleHeapID)			\
	DO_TO_FUNCTION(MemHandleDataStorage)	\
	DO_TO_FUNCTION(MemHandleCardNo)			\
	DO_TO_FUNCTION(MemHandleToLocalID)		\
	DO_TO_FUNCTION(MemHandleSetOwner)		\
	DO_TO_FUNCTION(MemLocalIDToGlobal)		\
	DO_TO_FUNCTION(MemLocalIDKind)			\
	DO_TO_FUNCTION(MemLocalIDToPtr)			\
	DO_TO_FUNCTION(MemDebugMode)			\
	DO_TO_FUNCTION(MemSetDebugMode)			\
	DO_TO_FUNCTION(MemHeapCheck)			\
	DO_TO_FUNCTION(MemHeapPtr)				\
	DO_TO_FUNCTION(MemStoreSearch)			\
	DO_TO_FUNCTION(MemStoreInit)			\
	DO_TO_FUNCTION(MemNVParams)				\
	DO_TO_FUNCTION(DmWriteCheck)			\
	DO_TO_FUNCTION(WinScreenInit)

// List of functions for which we want to automatically
// generate simple headpatches only.

#define FOR_EACH_STUB_HEADPATCH_FUNCTION(DO_TO_FUNCTION)	\
	DO_TO_FUNCTION(MemKernelInit)			\
	DO_TO_FUNCTION(MemHeapCompact)			\
	DO_TO_FUNCTION(MemHeapFreeByOwnerID)	\
	DO_TO_FUNCTION(MemChunkNew)				\
	DO_TO_FUNCTION(MemPtrNew)				\
	DO_TO_FUNCTION(MemPtrUnlock)			\
	DO_TO_FUNCTION(MemPtrResetLock)			\
	DO_TO_FUNCTION(MemHandleNew)			\
	DO_TO_FUNCTION(MemHandleLock)			\
	DO_TO_FUNCTION(MemHandleUnlock)			\
	DO_TO_FUNCTION(MemHandleResetLock)		\
	DO_TO_FUNCTION(MemLocalIDToLockedPtr)	\
	DO_TO_FUNCTION(MemHeapScramble)

// List of functions for which we want to automatically
// generate simple tailpatches only.

#define FOR_EACH_STUB_TAILPATCH_FUNCTION(DO_TO_FUNCTION)	\
	DO_TO_FUNCTION(MemChunkFree)			\
	DO_TO_FUNCTION(MemHandleFree)			\
	DO_TO_FUNCTION(MemSemaphoreReserve)		\
	DO_TO_FUNCTION(MemSemaphoreRelease)

// List of functions for which we have compex head- and tailpatches.

#define FOR_EACH_STUB_NOPATCH_FUNCTION(DO_TO_FUNCTION)	\
	DO_TO_FUNCTION(MemPtrResize)			\
	DO_TO_FUNCTION(MemHandleResize)

// This macro contains _all_ memory manager functions.

#define FOR_EACH_MEMMGR_FUNCTION(DO_TO_FUNCTION)		\
	FOR_EACH_STUB_BOTHPATCH_FUNCTION(DO_TO_FUNCTION)	\
	FOR_EACH_STUB_HEADPATCH_FUNCTION(DO_TO_FUNCTION)	\
	FOR_EACH_STUB_TAILPATCH_FUNCTION(DO_TO_FUNCTION)	\
	FOR_EACH_STUB_NOPATCH_FUNCTION(DO_TO_FUNCTION)


class MemMgrHeadpatch
{
	public:
		// Declare headpatches for all the memory manager functions.

		#define DECLARE_HEAD_PATCH(fn) static CallROMType fn (void);
		FOR_EACH_MEMMGR_FUNCTION(DECLARE_HEAD_PATCH)
};


class MemMgrTailpatch
{
	public:
		// Declare tailpatches for all the memory manager functions.

		#define DECLARE_TAIL_PATCH(fn) static void fn (void);
		FOR_EACH_MEMMGR_FUNCTION(DECLARE_TAIL_PATCH)
};


// ======================================================================
//	Proto patch table for the system functions.  This array will be used
//	to create a sparse array at runtime.
// ======================================================================

ProtoPatchTableEntry	gProtoMemMgrPatchTable[] =
{
	#undef INSTALL_PATCH
	#define INSTALL_PATCH(fn) {sysTrap##fn, MemMgrHeadpatch::fn, MemMgrTailpatch::fn},

	FOR_EACH_MEMMGR_FUNCTION(INSTALL_PATCH)

	{0,	NULL,	NULL}
};


const int			kChunkNewValue		= 0x31;
const int			kChunkResizeValue	= 0x33;
const int			kChunkFreeValue		= 0x35;
const int			kHandleNewValue		= 0x71;
const int			kHandleResizeValue	= 0x73;
const int			kHandleFreeValue	= 0x75;
const int			kPtrNewValue		= 0x91;
const int			kPtrResizeValue		= 0x93;
const int			kPtrFreeValue		= 0x95;

long				gMemMgrCount;
long				gMemSemaphoreCount;
unsigned long		gMemSemaphoreReserveTime;
ULong				gResizeOrigSize;
UInt				gHeapID;


// ========================================================================
// Macros for extracting parameters from the emulated stack
// Note that use of these has been superceded by the PARAM_FOO
// macros;  we should eventually move over to those macros.
// ========================================================================

#define PARAMETER_SIZE(x)	\
	(sizeof (((StackFrame*) 0)->x))

#define PARAMETER_OFFSET(x)	\
	(m68k_areg (regs, 7) + offsetof (StackFrame, x))

#define GET_PARAMETER(x)		\
	((PARAMETER_SIZE(x) == sizeof (char)) ? get_byte (PARAMETER_OFFSET(x)) :	\
	 (PARAMETER_SIZE(x) == sizeof (short)) ? get_word (PARAMETER_OFFSET(x)) :	\
											get_long (PARAMETER_OFFSET(x)))

#define SET_PARAMETER(x, v)		\
	((PARAMETER_SIZE(x) == sizeof (char)) ? put_byte (PARAMETER_OFFSET(x), v) :	\
	 (PARAMETER_SIZE(x) == sizeof (short)) ? put_word (PARAMETER_OFFSET(x), v) :	\
											put_long (PARAMETER_OFFSET(x), v))


// ========================================================================
// The following functions define a bunch of StackFrame structs.
// These structs need to mirror the format of parameters pushed
// onto the stack by the emulated code, and so need to be packed
// to 2-byte boundaries.
//
// The pragmas are reversed at the end of the file.
// ========================================================================

#include "PalmPack.h"

// Define a bunch of wrapper head- and tailpatches.
// These are defined via macros for those functions that don't have a
// complex implementation.

#define DEFINE_HEAD_PATCH(fn) CallROMType MemMgrHeadpatch::fn (void) { Patches::EnterMemMgr (#fn); return kExecuteROM; }
#define DEFINE_TAIL_PATCH(fn) void MemMgrTailpatch::fn (void) { Patches::ExitMemMgr (#fn); }

FOR_EACH_STUB_HEADPATCH_FUNCTION(DEFINE_HEAD_PATCH)
FOR_EACH_STUB_BOTHPATCH_FUNCTION(DEFINE_HEAD_PATCH)

FOR_EACH_STUB_TAILPATCH_FUNCTION(DEFINE_TAIL_PATCH)
FOR_EACH_STUB_BOTHPATCH_FUNCTION(DEFINE_TAIL_PATCH)



/***********************************************************************
 *
 * FUNCTION:	MemMgrHeadpatch::MemChunkFree
 *
 * DESCRIPTION:	If the user wants disposed blocks to be filled with a
 *				special value, handle that here.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType MemMgrHeadpatch::MemChunkFree (void)
{
	// Err MemChunkFree(VoidPtr chunkDataP)

	Patches::EnterMemMgr ("MemChunkFree");

	struct StackFrame
	{
		VoidPtr chunkDataP;
	};

	uaecptr	p = GET_PARAMETER (chunkDataP);

#ifdef FILL_BLOCKS

	gHeapID = ::MemPtrHeapID ((VoidPtr) p);

	if (p && gPrefs->FillDisposedBlocks ())
	{
		ULong	size = ::MemPtrSize ((VoidPtr) p);

		CEnableFullAccess	munge;
		uae_memset (p, kChunkFreeValue, size);
	}

#endif

	MetaMemory::MemChunkFree ((VoidPtr) p);

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrHeadpatch::MemHandleFree
 *
 * DESCRIPTION:	If the user wants disposed blocks to be filled with a
 *				special value, handle that here.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType MemMgrHeadpatch::MemHandleFree (void)
{
	// Err MemHandleFree(VoidHand h)

	Patches::EnterMemMgr ("MemHandleFree");

	struct StackFrame
	{
		VoidHand h;
	};

	uaecptr	h = GET_PARAMETER (h);

#ifdef FILL_BLOCKS

	gHeapID = ::MemHandleHeapID ((VoidHand) h);

	if (h && gPrefs->FillDisposedBlocks ())
	{
		ULong	size = ::MemHandleSize ((VoidHand) h);
		uaecptr	p = (uaecptr) ::MemHandleLock ((VoidHand) h);
		if (p)
		{
			{
				CEnableFullAccess	munge;
				uae_memset (p, kHandleFreeValue, size);
			}

			::MemHandleUnlock ((VoidHand) h);
		}
	}

#endif

	MetaMemory::MemHandleFree ((VoidHand) h);

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrHeadpatch::MemHandleResize
 *
 * DESCRIPTION:	If the user wants resized blocks to be filled with a
 *				special value, handle that here.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType MemMgrHeadpatch::MemHandleResize (void)
{
	// Err MemHandleResize(VoidHand h,  ULong newSize) 

	Patches::EnterMemMgr ("MemHandleResize");

	struct StackFrame
	{
		VoidHand h;
		ULong newSize;
	};

//	if (gPrefs->FillResizedBlocks ())
	{
		uaecptr	h = GET_PARAMETER (h);

		gResizeOrigSize = ::MemHandleSize ((VoidHand) h);
	}

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrHeadpatch::MemPtrResize
 *
 * DESCRIPTION:	If the user wants resized blocks to be filled with a
 *				special value, handle that here.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType MemMgrHeadpatch::MemPtrResize (void)
{
	// Err MemPtrResize(VoidPtr p, ULong newSize)

	Patches::EnterMemMgr ("MemPtrResize");

	struct StackFrame
	{
		VoidPtr p;
		ULong newSize;
	};

//	if (gPrefs->FillResizedBlocks ())
	{
		uaecptr	p = GET_PARAMETER (p);
		gResizeOrigSize = ::MemPtrSize ((VoidPtr) p);
	}

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrHeadpatch::MemSemaphoreRelease
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType MemMgrHeadpatch::MemSemaphoreRelease (void)
{
	// Err MemSemaphoreRelease (Boolean writeAccess)

	Patches::EnterMemMgr ("MemSemaphoreRelease");

	struct StackFrame
	{
		Boolean writeAccess;
	};

	Boolean	writeAccess = GET_PARAMETER (writeAccess);

	if (writeAccess && --gMemSemaphoreCount == 0 && Patches::HasWellBehavedMemSemaphoreUsage ())
	{
		const unsigned long	kMicrosecondsPerMillisecond = 1000;
		const unsigned long	kMillisecondsPerSecond = 1000;
		const unsigned long	kSecondsPerMinute = 60;

		const unsigned long	kCyclesPerMicrosecond = 16;
		const unsigned long	kCyclesPerMillisecond = kCyclesPerMicrosecond * kMicrosecondsPerMillisecond;

		const unsigned long	kMaxElapsedMilliseconds = 1 * kSecondsPerMinute * kMillisecondsPerSecond;

		unsigned long	curCount = Emulator::GetCycleCount ();
		unsigned long	elapsedCycles = curCount - gMemSemaphoreReserveTime;

		unsigned long	elapsedMilliseconds = elapsedCycles / kCyclesPerMillisecond;

		if (elapsedMilliseconds > kMaxElapsedMilliseconds)
		{
			int button = Errors::ReportLongMemSemaphore (elapsedMilliseconds);
			Emulator::HandleDlgButton (button, m68k_getpc ());
		}
	}

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrHeadpatch::MemSemaphoreReserve
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType MemMgrHeadpatch::MemSemaphoreReserve (void)
{
	// Err MemSemaphoreReserve (Boolean writeAccess)

	Patches::EnterMemMgr ("MemSemaphoreReserve");

	struct StackFrame
	{
		Boolean writeAccess;
	};

	Boolean	writeAccess = GET_PARAMETER (writeAccess);

	if (writeAccess && gMemSemaphoreCount++ == 0)
	{
		gMemSemaphoreReserveTime = Emulator::GetCycleCount ();
	}

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrTailpatch::MemChunkNew
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void MemMgrTailpatch::MemChunkNew (void)
{
	// VoidPtr MemChunkNew(UInt heapID, ULong size, UInt attr)

	struct StackFrame
	{
		UInt heapID;
		ULong size;
		UInt attr;
	};

//	uae_u32	size = GET_PARAMETER (size);
	uae_u16	attr = GET_PARAMETER (attr);
	uaecptr	c = m68k_areg (regs, 0);

#ifdef FILL_BLOCKS

	if (gPrefs->FillNewBlocks ())
	{
		if (c)
		{

			if (attr & memNewChunkFlagNonMovable)
			{
				CEnableFullAccess	munge;
				uae_memset (c, kChunkNewValue, size);
			}
			else
			{
				uaecptr	p = (uaecptr) ::MemHandleLock ((VoidHand) c);
				if (p)
				{
					{
						CEnableFullAccess	munge;
						uae_memset (p, kChunkNewValue, size);
					}

					::MemHandleUnlock ((VoidHand) c);
				}
			}
		}
	}

#endif

	MetaMemory::MemChunkNew ((VoidPtr) c, attr);

	Patches::ExitMemMgr ("MemChunkNew");
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrTailpatch::MemHandleNew
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void MemMgrTailpatch::MemHandleNew (void)
{
	// VoidHand MemHandleNew(ULong size)

	struct StackFrame
	{
		ULong size;
	};

//	uae_u32	size = GET_PARAMETER (size);
	uaecptr	h = m68k_areg (regs, 0);

#ifdef FILL_BLOCKS

	if (h)
	{
		if (size && gPrefs->FillNewBlocks ())
		{
			uaecptr	p = (uaecptr) ::MemHandleLock ((VoidHand) h);
			if (p)
			{
				{
					CEnableFullAccess	munge;
					uae_memset (p, kHandleNewValue, size);
				}

				::MemHandleUnlock ((VoidHand) h);
			}
		}
	}

#endif

	MetaMemory::MemHandleNew ((VoidHand) h);

	Patches::ExitMemMgr ("MemHandleNew");
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrTailpatch::MemHandleLock
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void MemMgrTailpatch::MemHandleLock (void)
{
	// Err MemHandleLock(VoidHand h) 

	struct StackFrame
	{
		VoidHand h;
	};

	VoidHand	h = (VoidHand) GET_PARAMETER (h);

	MetaMemory::MemHandleLock (h);

	Patches::ExitMemMgr ("MemHandleLock");
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrTailpatch::MemHandleResetLock
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void MemMgrTailpatch::MemHandleResetLock (void)
{
	// Err MemHandleResetLock(VoidHand h) 

	struct StackFrame
	{
		VoidHand h;
	};

	VoidHand	h = (VoidHand) GET_PARAMETER (h);

	MetaMemory::MemHandleResetLock (h);

	Patches::ExitMemMgr ("MemHandleResetLock");
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrTailpatch::MemHandleResize
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void MemMgrTailpatch::MemHandleResize (void)
{
	// Err MemHandleResize(VoidHand h,  ULong newSize) 

	struct StackFrame
	{
		VoidHand h;
		ULong newSize;
	};

	uaecptr	h = GET_PARAMETER (h);
//	uae_u32	newSize = GET_PARAMETER (newSize);

#ifdef FILL_BLOCKS

	if (h)
	{
		if (newSize > gResizeOrigSize && gPrefs->FillResizedBlocks ())
		{
			uaecptr	p = (uaecptr) ::MemHandleLock ((VoidHand) h);
			if (p)
			{
				{
					CEnableFullAccess	munge;
					uae_memset (p + gResizeOrigSize, kHandleResizeValue, newSize - gResizeOrigSize);
				}

				::MemHandleUnlock ((VoidHand) h);
			}
		}
	}

#endif

	MetaMemory::MemHandleResize ((VoidHand) h, gResizeOrigSize);

	Patches::ExitMemMgr ("MemHandleResize");
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrTailpatch::MemHandleUnlock
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void MemMgrTailpatch::MemHandleUnlock (void)
{
	// Err MemHandleUnlock(VoidHand h) 

	struct StackFrame
	{
		VoidHand h;
	};

	VoidHand	h = (VoidHand) GET_PARAMETER (h);

	MetaMemory::MemHandleUnlock (h);

	Patches::ExitMemMgr ("MemHandleUnlock");
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrTailpatch::MemHeapCompact
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void MemMgrTailpatch::MemHeapCompact (void)
{
	// Err MemHeapCompact(UInt heapID)

	struct StackFrame
	{
		UInt heapID;
	};

	UInt	heapID = GET_PARAMETER (heapID);

	assert (heapID < 0x1000);

	uaecptr	heapHdr = PalmHeap::GetHeapHeader (heapID);

	PalmHeap::HeapInfo	heapInfo;
	PalmHeap::GetHeapInfo (heapHdr, &heapInfo);

	MetaMemory::SyncHeap (heapInfo);

	Patches::ExitMemMgr ("MemHeapCompact");
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrTailpatch::MemHeapFreeByOwnerID
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

//	!!! This should probably walk the heap itself so that it can just mark
//	the chunks that are about to be freed.

void MemMgrTailpatch::MemHeapFreeByOwnerID (void)
{
	// Err MemHeapFreeByOwnerID(UInt heapID, UInt ownerID)

	struct StackFrame
	{
		UInt heapID;
		UInt ownerID;
	};

	UInt	heapID = GET_PARAMETER (heapID);

	uaecptr	heapHdr = PalmHeap::GetHeapHeader (heapID);

	PalmHeap::HeapInfo	heapInfo;
	PalmHeap::GetHeapInfo (heapHdr, &heapInfo);

	MetaMemory::SyncHeap (heapInfo);

	Patches::ExitMemMgr ("MemHeapFreeByOwnerID");
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrTailpatch::MemHeapScramble
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void MemMgrTailpatch::MemHeapScramble (void)
{
	// Err MemHeapScramble(UInt heapID)

	struct StackFrame
	{
		UInt heapID;
	};

	UInt	heapID = GET_PARAMETER (heapID);

	uaecptr	heapHdr = PalmHeap::GetHeapHeader (heapID);

	PalmHeap::HeapInfo	heapInfo;
	PalmHeap::GetHeapInfo (heapHdr, &heapInfo);

	MetaMemory::SyncHeap (heapInfo);

	Patches::ExitMemMgr ("MemHeapScramble");
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrTailpatch::MemKernelInit
 *
 * DESCRIPTION:	MemKernelInit is called just after AMXDriversInit, which
 *				is where the exception vectors are set up.  After those
 *				vectors are installed, there really shouldn't be any
 *				more memory accesses to that range of memory. Thus,
 *				memory access there is flagged as invalid.
 *
 *				While we're here, let's mark some other memory
 *				locations, too.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void MemMgrTailpatch::MemKernelInit (void)
{
	MetaMemory::MarkLowMemory (		MetaMemory::GetLowMemoryBegin (),
									MetaMemory::GetLowMemoryEnd ());
	MetaMemory::MarkSystemGlobals (	MetaMemory::GetSysGlobalsBegin (),
									MetaMemory::GetSysGlobalsEnd ());
	MetaMemory::MarkHeapHeader (	MetaMemory::GetHeapHdrBegin (0),
									MetaMemory::GetHeapHdrEnd (0));

	Patches::ExitMemMgr ("MemKernelInit");
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrTailpatch::MemLocalIDToLockedPtr
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void MemMgrTailpatch::MemLocalIDToLockedPtr (void)
{
	// VoidPtr MemLocalIDToLockedPtr(LocalID local,  UInt cardNo)

	VoidPtr	p = (VoidPtr) m68k_areg (regs, 0);

	MetaMemory::MemLocalIDToLockedPtr (p);

	Patches::ExitMemMgr ("MemLocalIDToLockedPtr");
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrTailpatch::MemPtrNew
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void MemMgrTailpatch::MemPtrNew (void)
{
	// VoidPtr MemPtrNew(ULong size) 

	struct StackFrame
	{
		ULong size;
	};

//	uae_u32	size = GET_PARAMETER (size);
	uaecptr	p = m68k_areg (regs, 0);

#ifdef FILL_BLOCKS

	if (p)
	{
		if (gPrefs->FillNewBlocks ())
		{
			CEnableFullAccess	munge;
			uae_memset (p, kPtrNewValue, size);
		}
	}

#endif

	MetaMemory::MemPtrNew ((VoidPtr) p);

	Patches::ExitMemMgr ("MemPtrNew");
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrTailpatch::MemPtrResetLock
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void MemMgrTailpatch::MemPtrResetLock (void)
{
	// Err MemPtrResetLock(VoidPtr p) 

	struct StackFrame
	{
		VoidPtr p;
	};

	VoidPtr	p = (VoidPtr) GET_PARAMETER (p);

	MetaMemory::MemPtrResetLock (p);

	Patches::ExitMemMgr ("MemPtrResetLock");
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrTailpatch::MemPtrResize
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void MemMgrTailpatch::MemPtrResize (void)
{
	// Err MemPtrResize(VoidPtr p, ULong newSize)

	struct StackFrame
	{
		VoidPtr p;
		ULong newSize;
	};

	uaecptr	p = GET_PARAMETER (p);
//	uae_u32	newSize = GET_PARAMETER (newSize);

#ifdef FILL_BLOCKS

	if (p)
	{
		if (newSize > gResizeOrigSize && gPrefs->FillResizedBlocks ())
		{
			CEnableFullAccess	munge;
			uae_memset (p + gResizeOrigSize, kPtrResizeValue, newSize - gResizeOrigSize);
		}
	}

#endif

	MetaMemory::MemPtrResize ((VoidPtr) p, gResizeOrigSize);

	Patches::ExitMemMgr ("MemPtrResize");
}


/***********************************************************************
 *
 * FUNCTION:	MemMgrTailpatch::MemPtrUnlock
 *
 * DESCRIPTION:	
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void MemMgrTailpatch::MemPtrUnlock (void)
{
	// Err MemPtrUnlock(VoidPtr p) 

	struct StackFrame
	{
		VoidPtr p;
	};

	VoidPtr	p = (VoidPtr) GET_PARAMETER (p);

	MetaMemory::MemPtrUnlock (p);

	Patches::ExitMemMgr ("MemPtrUnlock");
}


