/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "SystemPacket.h"

#include "ATraps.h"				// ATrap
#include "DebugMgr.h"			// Debug::
#include "HostControl.h"		// hostSelectorWaitForIdle
#include "Logging.h"			// LogAppendMsg
#include "EmRPC.h"				// slkSocketRPC
#include "SLP.h"				// SLP
#include "UAE_Utils.h"			// uae_memcpy


/*
	Crib notes on the supported packets:

	+- Called by Debugger
	|	+- Implemented
	|	|								Command
	|	|	Command name				Number		Sent to
	------- --------------------------- -----------	-----------------
	*	*	sysPktStateCmd				0x00		slkSocketDebugger	SysPktStateCmdType				SysPktStateRspType
	*	*	sysPktReadMemCmd			0x01		slkSocketDebugger	SysPktReadMemCmdType			SysPktReadMemRspType
	*	*	sysPktWriteMemCmd			0x02		slkSocketDebugger	SysPktWriteMemCmdType			SysPktWriteMemRspType
	*	.	sysPktSingleStepCmd			0x03
	*	*	sysPktGetRtnNameCmd			0x04		slkSocketDebugger	SysPktRtnNameCmdType			SysPktRtnNameRspType
	*	*	sysPktReadRegsCmd			0x05		slkSocketDebugger	SysPktReadRegsCmdType			SysPktReadRegsRspType
	*	*	sysPktWriteRegsCmd			0x06		slkSocketDebugger	SysPktWriteRegsCmdType			SysPktWriteRegsRspType
	*	*	sysPktContinueCmd			0x07		slkSocketDebugger	SysPktContinueCmdType			<no response>
	*	*	sysPktRPCCmd				0x0A		both*				SysPktRPCType
		*	sysPktGetBreakpointsCmd		0x0B							SysPktGetBreakpointsCmdType		SysPktGetBreakpointsRspType
	*	*	sysPktSetBreakpointsCmd		0x0C		slkSocketDebugger	SysPktSetBreakpointsCmdType		SysPktSetBreakpointsRspType
		.	sysPktRemoteUIUpdCmd		0x0C							SysPktRemoteUIUpdCmdType
		.	sysPktRemoteEvtCmd			0x0D							SysPktRemoteEvtCmdType
	*	*	sysPktDbgBreakToggleCmd		0x0D		slkSocketDebugger	SysPktDbgBreakToggleCmdType		SysPktDbgBreakToggleRspType
	*	.	sysPktFlashCmd				0x0E							SysPktFlashWriteType
	*	.	sysPktCommCmd				0x0F		both*				SysPktCommCmdType				SysPktCommRspType
	*	*	sysPktGetTrapBreaksCmd		0x10		slkSocketDebugger	SysPktGetTrapBreaksCmdType		SysPktGetTrapBreaksRspType
	*	*	sysPktSetTrapBreaksCmd		0x11		slkSocketDebugger	SysPktSetTrapBreaksCmdType		SysPktSetTrapBreaksRspType
		+	sysPktGremlinsCmd			0x12		slkSocketConsole	SysPktGremlinsCmdType
	*	*	sysPktFindCmd				0x13		slkSocketDebugger	SysPktFindCmdType				SysPktFindRspType
		*	sysPktGetTrapConditionsCmd	0x14		slkSocketDebugger	SysPktGetTrapConditionsCmdType	SysPktGetTrapConditionsRspType
		*	sysPktSetTrapConditionsCmd	0x15		slkSocketDebugger	SysPktSetTrapConditionsCmdType	SysPktSetTrapConditionsRspType
		.	sysPktChecksumCmd			0x16		slkSocketDebugger	SysPktChecksumType
		.	sysPktExecFlashCmd			0x17		slkSocketDebugger	sysPktExecFlashType
	*	*	sysPktRemoteMsgCmd			0x7f		both				SysPktRemoteMsgCmdType

	"both*" = if attached, sent to slkSocketDebugger, else slkSocketConsole
*/

#define sysPktBadFormatRsp		0xFF


#define ENTER_CODE(fnName, cmdType, rspType)		\
	PRINTF ("Entering SystemPacket::" fnName ".");	\
	cmdType&	packet = (cmdType&) slp.Body ()


#define ENTER_PACKET(fnName, cmdType, rspType)		\
	PRINTF ("Entering SystemPacket::" fnName ".");	\
	cmdType&	packet = (cmdType&) slp.Body ();	\
	rspType		response


#define EXIT_PACKET(fnName, code, bodySize)			\
	response.command = code;						\
	response._filler = 0;							\
	ErrCode result = SystemPacket::SendPacket (slp, &response, bodySize);	\
	PRINTF ("Exiting SystemPacket::" fnName ".");	\
	return result


#define EXIT_CODE(fnName, code)						\
	ErrCode result = SystemPacket::SendResponse (slp, code);	\
	PRINTF ("Exiting SystemPacket::" fnName ".");	\
	return result


#define PRINTF	if (!LogHLDebugger ()) ; else LogAppendMsg


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::SendState
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::SendState (SLP& slp)
{
	// Do this function entry by hand instead of using the ENTER_PACKET macro.
	// That macro will call SLP::Body, which returns a reference to the
	// packet body that was sent to us.  However, SendState (a) doesn't
	// need the body reference, so there's no need to set it up, and (b)
	// isn't necessarily called in response to a "SendState" command,
	// and so the reference may be NULL.

	PRINTF ("Entering SystemPacket::SendState.");
	SysPktStateRspType		response;

	// Get the resetted boolean.  This flag is true if the "device" has
	// been resetted since the last time we sent a packet.	The debugger
	// needs to know this in order to update its notion of the device's
	// state (like, for instance, if breakpoints are installed).

	response.resetted	= gDebuggerGlobals.firstEntrance;
	gDebuggerGlobals.firstEntrance = false;

	// Get the exception id.

	response.exceptionId	= gDebuggerGlobals.excType;

	// Get the registers.

	SystemPacket::GetRegs (response.reg);

	// Get the breakpoints.

	for (int i = 0; i < dbgTotalBreakpoints; i++)
	{
		response.bp[i] = gDebuggerGlobals.bp[i];
	}

	// Get some code beginning at PC.

	for (int ii = 0; ii < sysPktStateRspInstWords; ++ii)
	{
		// Do this as two fetches in case the PC is odd.
		response.inst[ii] = get_byte (m68k_getpc () + ii * 2) * 256 +
			get_byte (m68k_getpc () + ii * 2 + 1);
	}

	// Get the routine name and address range info.

	memset (response.name, 0, sizeof (response.name));
	::FindFunctionName (	m68k_getpc () & 0xFFFFFFF0, // Protect against odd addresses.
							response.name,
							(uaecptr*) &response.startAddr,
							(uaecptr*) &response.endAddr);

	// Send the rev of the trap table.

	response.trapTableRev	= LowMem_GetGlobal (sysDispatchTableRev);

	EXIT_PACKET ("SendState", sysPktStateRsp, sizeof (response));
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::ReadMem
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::ReadMem (SLP& slp)
{
	ENTER_PACKET ("ReadMem", SysPktReadMemCmdType, SysPktReadMemRspType2);

	uae_memcpy ((void*) (&response.data), (uaecptr) packet.address,
				packet.numBytes);

	EXIT_PACKET ("ReadMem", sysPktReadMemRsp, sizeof (SysPktReadMemRspType) + packet.numBytes);
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::WriteMem
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::WriteMem (SLP& slp)
{
	ENTER_CODE ("WriteMem", SysPktWriteMemCmdType, SysPktWriteMemRspType);

	uae_memcpy ((uaecptr) packet.address, (const void*) ((&packet) + 1),
				packet.numBytes);

	// If we just altered low memory, recalculate the low-memory checksum.
	// Make sure we're on a ROM that has this field!  Determine this by
	// seeing that the address of sysLowMemChecksum is below the memCardInfo
	// fields that come after the FixedGlobals.
	//
	// !!! This chunk of code should be in some more generally accessible location.

	if (LowMem_Location (sysLowMemChecksum) < LowMem_GetGlobal (memCardInfoP))
	{
		if (packet.address < (void*) 0x100)
		{
			DWord		checksum	= 0;
			uaecptr		csP		= UAE_NULL;

			// First, calculate the checksum

			while (csP < (uaecptr) 0x100)
			{
				DWord	data = get_long (csP);

				// Don't do these trap vectors since they change whenever the
				// debugger is set to break on any a-trap or breakpoint.

				if (csP == offsetof (M68KExcTableType, trapN[sysDispatchTrapNum]))
					data = 0;

				if (csP == offsetof (M68KExcTableType, trapN[sysDbgBreakpointTrapNum]))
					data = 0;

				if (csP == offsetof (M68KExcTableType, trace))
					data = 0;

				checksum += data;
				csP += 4;
			}

			// Save new checksum

			LowMem_SetGlobal (sysLowMemChecksum, checksum);
		}
	}

	EXIT_CODE ("WriteMem", sysPktWriteMemRsp);
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::SendRoutineName
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::SendRoutineName (SLP& slp)
{
	ENTER_PACKET ("SendRoutineName", SysPktRtnNameCmdType, SysPktRtnNameRspType);

	response.address	= (void*) (((uae_u32) packet.address) & ~1);	// Protect against odd address.
	response.startAddr	= NULL;
	response.endAddr	= NULL;
	response.name[0]	= 0;

	::FindFunctionName( (uaecptr) response.address,
						response.name,
						(uaecptr*) &response.startAddr,
						(uaecptr*) &response.endAddr);


	EXIT_PACKET ("SendRoutineName", sysPktGetRtnNameRsp, sizeof (response));
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::ReadRegs
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::ReadRegs (SLP& slp)
{
	ENTER_PACKET ("ReadRegs", SysPktReadRegsCmdType, SysPktReadRegsRspType2);

	SystemPacket::GetRegs (response.reg);

	EXIT_PACKET ("ReadRegs", sysPktReadRegsRsp, sizeof (response));
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::WriteRegs
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::WriteRegs (SLP& slp)
{
	ENTER_CODE ("WriteRegs", SysPktWriteRegsCmdType, SysPktWriteRegsRspType);

	SystemPacket::SetRegs (packet.reg);

	EXIT_CODE ("ReadRegs", sysPktWriteRegsRsp);
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::Continue
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::Continue (SLP& slp)
{
	ENTER_CODE ("Continue", SysPktContinueCmdType, SysPktContinueCmdType);

	SystemPacket::SetRegs (packet.regs);

	gDebuggerGlobals.stepSpy	= packet.stepSpy != 0;
	gDebuggerGlobals.ssAddr		= packet.ssAddr;
	// ssCount is ignored?
	gDebuggerGlobals.ssValue	= packet.ssCheckSum;

	ErrCode result = Debug::ExitDebugger ();

	// Perform any platform-specific actions.

	if (result == errNone)
	{
		Platform::ExitDebugger ();
	}

	// No reply...

	PRINTF ("Exiting SystemPacket::Continue.");

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::RPC
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::RPC (SLP& slp)
{
	ENTER_PACKET ("RPC", SysPktRPCType2, SysPktRPCType2);

	ATrap	trap;

	SysPktRPCParamInfo* param = (SysPktRPCParamInfo*) &packet.param;

	for (Word ii = 0; ii < packet.numParams; ++ii)
	{
		if (param->byRef)
		{
			trap.PushLong ((uae_u32) (&param->data));
		}
		else
		{
			if (param->size == 1)
			{
				uae_u8	value = *(uae_u8*) param->data;
				Canonical (value);
				trap.PushByte (value);
			}
			else if (param->size == 2)
			{
				uae_u16 value = *(uae_u16*) param->data;
				Canonical (value);
				trap.PushWord (value);
			}
			else if (param->size == 4)
			{
				uae_u32 value = *(uae_u32*) param->data;
				Canonical (value);
				trap.PushLong (value);
			}
		}

		param = (SysPktRPCParamInfo*) (((char*) param) + offsetof (SysPktRPCParamInfo, data) +
									   ((param->size + 1) & ~1));
	}

	// Map in the memory pointed to by the reference parameters
	StMemoryMapper	mapper (&packet, ((char*) param) - ((char*) &packet));


	// Call the trap.  Special case the call to SysReset, as merely calling
	// that function will not return from it.

	if (packet.trapWord == sysTrapSysReset)
	{
		// Set flags that we're no longer in the debugger.  This is the same
		// as the last few steps of ExitDebugger.

		gDebuggerGlobals.excType = 0;
		LowMem_SetGlobal (dbgInDebugger, false);
		gDebuggerGlobals.inDebugger = false;

		// Reset the emulator (causes it to "reboot")

		Emulator::Reset();

		// That's it!  No reply!

		return kError_NoError;
	}

	trap.Call (packet.trapWord);

	response = packet;

	response.resultD0 = trap.GetD0 ();
	response.resultA0 = trap.GetA0 ();

	long numBytes = ((char*) param) - ((char*) &packet);	// or get from header...

	EXIT_PACKET ("RPC", sysPktRPCRsp, numBytes);
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::RPC2
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::RPC2 (SLP& slp)
{
	ENTER_PACKET ("RPC2", SysPktRPC2Type, SysPktRPC2Type);

	ATrap	trap;

	// Extract any D-registers from the packet.

	UInt32*	regs = packet.Regs;
	int		mask = 0x01;
	int		regNum = 0;

	while (mask < 0x100)
	{
		if ((mask & packet.DRegMask) != 0)
		{
			UInt32	reg = *regs++;
			trap.SetNewDReg (regNum, reg);
		}

		++regNum;
		mask <<= 1;
	}

	// Extract any A-registers from the packet.

	mask = 0x01;
	regNum = 0;
	while (mask < 0x100)
	{
		if ((mask & packet.ARegMask) != 0)
		{
			UInt32	reg = *regs++;
			trap.SetNewAReg (regNum, reg);
		}

		++regNum;
		mask <<= 1;
	}

	// Extract any stack-based parameters from the packet.

	UInt16*	numParams = (UInt16*) regs;

	SysPktRPCParamInfo* param = (SysPktRPCParamInfo*) (numParams + 1);

	for (Word ii = 0; ii < *numParams; ++ii)
	{
		if (param->byRef)
		{
			trap.PushLong ((uae_u32) (&param->data));
		}
		else
		{
			if (param->size == 1)
			{
				uae_u8	value = *(uae_u8*) param->data;
				Canonical (value);
				trap.PushByte (value);
			}
			else if (param->size == 2)
			{
				uae_u16 value = *(uae_u16*) param->data;
				Canonical (value);
				trap.PushWord (value);
			}
			else if (param->size == 4)
			{
				uae_u32 value = *(uae_u32*) param->data;
				Canonical (value);
				trap.PushLong (value);
			}
		}

		param = (SysPktRPCParamInfo*) (((char*) param) + offsetof (SysPktRPCParamInfo, data) +
									   ((param->size + 1) & ~1));
	}

	// Map in the memory pointed to by the reference parameters
	StMemoryMapper	mapper (&packet, ((char*) param) - ((char*) &packet));


	// Call the trap.  Special case the call to SysReset, as merely calling
	// that function will not return from it.

	if (packet.trapWord == sysTrapSysReset)
	{
		// Set flags that we're no longer in the debugger.  This is the same
		// as the last few steps of ExitDebugger.

		gDebuggerGlobals.excType = 0;
		LowMem_SetGlobal (dbgInDebugger, false);
		gDebuggerGlobals.inDebugger = false;

		// Reset the emulator (causes it to "reboot")

		Emulator::Reset();

		// That's it!  No reply!

		return kError_NoError;
	}

	trap.Call (packet.trapWord);

	response = packet;

	response.resultD0			= trap.GetD0 ();
	response.resultA0			= trap.GetA0 ();
	response.resultException	= 0;

	long numBytes = ((char*) param) - ((char*) &packet);	// or get from header...

	EXIT_PACKET ("RPC2", sysPktRPC2Rsp, numBytes);
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::GetBreakpoints
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::GetBreakpoints (SLP& slp)
{
	ENTER_PACKET ("GetBreakpoints", SysPktGetBreakpointsCmdType, SysPktGetBreakpointsRspType2);

	memcpy (response.bp, gDebuggerGlobals.bp, sizeof (gDebuggerGlobals.bp));

	EXIT_PACKET ("GetBreakpoints", sysPktGetBreakpointsRsp, sizeof (response));
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::SetBreakpoints
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::SetBreakpoints (SLP& slp)
{
	ENTER_CODE ("SetBreakpoints", SysPktSetBreakpointsCmdType2, SysPktSetBreakpointsRspType);

	for (int ii = 0; ii < dbgTotalBreakpoints; ++ii)
	{
		if (packet.bp[ii].enabled)
		{
			Debug::SetBreakpoint (ii, (uaecptr) packet.bp[ii].addr, NULL);
		}
		else
		{
			Debug::ClearBreakpoint (ii);
		}
	}

	EXIT_CODE ("GetBreakpoints", sysPktSetBreakpointsRsp);
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::ToggleBreak
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::ToggleBreak (SLP& slp)
{
	ENTER_PACKET ("ToggleBreak", SysPktDbgBreakToggleCmdType, SysPktDbgBreakToggleRspType);

	gDebuggerGlobals.ignoreDbgBreaks = !gDebuggerGlobals.ignoreDbgBreaks;

	response.newState	= gDebuggerGlobals.ignoreDbgBreaks;

	EXIT_PACKET ("ToggleBreak", sysPktDbgBreakToggleRsp, sizeof (response));
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::GetTrapBreaks
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::GetTrapBreaks (SLP& slp)
{
	ENTER_PACKET ("GetTrapBreaks", SysPktGetTrapBreaksCmdType, SysPktGetTrapBreaksRspType2);

	memcpy (response.trapBP, gDebuggerGlobals.trapBreak,
			sizeof (gDebuggerGlobals.trapBreak));

	EXIT_PACKET ("GetTrapBreaks", sysPktGetTrapBreaksRsp, sizeof (response));
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::SetTrapBreaks
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::SetTrapBreaks (SLP& slp)
{
	ENTER_CODE ("SetTrapBreaks", SysPktSetTrapBreaksCmdType2, SysPktSetTrapBreaksRspType);

	memcpy (gDebuggerGlobals.trapBreak, packet.trapBP,
			sizeof (gDebuggerGlobals.trapBreak));

	gDebuggerGlobals.breakingOnATrap = false;
	for (int ii = 0; ii < dbgTotalTrapBreaks; ++ii)
	{
		if (gDebuggerGlobals.trapBreak[ii])
		{
			gDebuggerGlobals.breakingOnATrap = true;
			break;
		}
	}

	EXIT_CODE ("SetTrapBreaks", sysPktSetTrapBreaksRsp);
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::Find
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::Find (SLP& slp)
{
	ENTER_PACKET ("Find", SysPktFindCmdType, SysPktFindRspType);

	response.addr		= (DWord) NULL;
	response.found		= false;
	
	{
		BytePtr dataP = (BytePtr) (&packet + 1);
		DWord	startP = packet.firstAddr;

		while (startP <= packet.lastAddr)
		{
			Word	ii;

			for (ii = 0; ii < packet.numBytes; ++ii)
			{
				Byte	b1 = get_byte(startP + ii);
				Byte	b2 = dataP[ii];

				// Handle caseless conversion

				if (packet.caseInsensitive)
				{
					if (isalpha (b1))
						b1 = tolower (b1);

					if (isalpha (b2))
						b2 = tolower (b2);
				}

				if (b1 != b2)			// if different -->
					break;
			}

			// Check if matched

			if (ii >= packet.numBytes)
			{
				response.addr = startP;
				response.found = true;
				break;
			}

			startP++;					// advance start of search pointer
		}
	}

	EXIT_PACKET ("Find", sysPktFindRsp, sizeof (response));
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::GetTrapConditions
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::GetTrapConditions (SLP& slp)
{
	ENTER_PACKET ("GetTrapConditions", SysPktGetTrapConditionsCmdType, SysPktGetTrapConditionsRspType2);

	memcpy (response.trapParam, gDebuggerGlobals.trapParam,
			sizeof (gDebuggerGlobals.trapParam));

	EXIT_PACKET ("GetTrapConditions", sysPktGetTrapConditionsRsp, sizeof (response));
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::SetTrapConditions
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::SetTrapConditions (SLP& slp)
{
	ENTER_CODE ("SetTrapConditions", SysPktSetTrapConditionsCmdType2, SysPktSetTrapConditionsRspType);

	memcpy (gDebuggerGlobals.trapParam, packet.trapParam,
			sizeof (gDebuggerGlobals.trapParam));

	EXIT_CODE ("SetTrapConditions", sysPktSetTrapConditionsRsp);
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::SendMessage
 *
 * DESCRIPTION: Sends a text string to the external client.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::SendMessage (SLP& slp, const char* msg)
{
//	ENTER_PACKET ("SendMessage", SysPktBodyType, SysPktBodyType);

	// Do this function entry by hand instead of using the ENTER_PACKET macro.
	// That macro will call SLP::Body, which returns a reference to the
	// packet body that was sent to us.  However, SendMessage (a) doesn't
	// need the body reference, so there's no need to set it up, and (b)
	// isn't necessarily called in response to a "SendMessage" command,
	// and so the reference may be NULL.

	PRINTF ("Entering SystemPacket::SendMessage.");
	SysPktBodyType		response;

	size_t	len = strlen (msg) + 1;
	strcpy ((char*) &response.data[0], msg);

	EXIT_PACKET ("SendMessage", sysPktRemoteMsgCmd,
					sizeof (SysPktRemoteMsgCmdType) + len);
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::SendResponse
 *
 * DESCRIPTION: Sends a simple null-body message to the external debugger.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::SendResponse (SLP& slp, Byte code)
{
	SysPktEmptyRspType	response;

	response.command = code;
	response._filler = 0;

	return SystemPacket::SendPacket (slp, &response, sizeof (response));
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::SendPacket
 *
 * DESCRIPTION: Sends the given packet to the external client.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode SystemPacket::SendPacket (SLP& slp, const void* body, long bodySize)
{
	PRINTF ("Entering SystemPacket::SendPacket.");

	ErrCode result = slp.SendPacket (body, bodySize);

	PRINTF ("Exiting SystemPacket::SendPacket.");

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	SystemPacket::GetRegs
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void SystemPacket::GetRegs (M68KRegsType& debuggerRegs)
{
	debuggerRegs.d[0]	= m68k_dreg (regs, 0);
	debuggerRegs.d[1]	= m68k_dreg (regs, 1);
	debuggerRegs.d[2]	= m68k_dreg (regs, 2);
	debuggerRegs.d[3]	= m68k_dreg (regs, 3);
	debuggerRegs.d[4]	= m68k_dreg (regs, 4);
	debuggerRegs.d[5]	= m68k_dreg (regs, 5);
	debuggerRegs.d[6]	= m68k_dreg (regs, 6);
	debuggerRegs.d[7]	= m68k_dreg (regs, 7);

	debuggerRegs.a[0]	= m68k_areg (regs, 0);
	debuggerRegs.a[1]	= m68k_areg (regs, 1);
	debuggerRegs.a[2]	= m68k_areg (regs, 2);
	debuggerRegs.a[3]	= m68k_areg (regs, 3);
	debuggerRegs.a[4]	= m68k_areg (regs, 4);
	debuggerRegs.a[5]	= m68k_areg (regs, 5);
	debuggerRegs.a[6]	= m68k_areg (regs, 6);
//	debuggerRegs.a[7]	= m68k_areg (regs, 7);

	// Update the usp, isp, and/or msp before returning it.
	// (from m68k_dumpstate() in newcpu.c)
	
	if (regs.s == 0)
		regs.usp = m68k_areg (regs, 7);
	else
		regs.isp = m68k_areg (regs, 7);

	// Make sure the Status Register is up-to-date before moving it
	// into the debugger register set.

	Registers::UpdateSRFromRegisters ();

	// Copy the remaining registers from the UAE register set to the
	// debugger's register set.

	debuggerRegs.usp	= regs.usp;
	debuggerRegs.ssp	= regs.isp;
	debuggerRegs.sr		= regs.sr;
	debuggerRegs.pc		= m68k_getpc ();
}


/***********************************************************************
 *
 * FUNCTION:	Debug::SetRegs
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void SystemPacket::SetRegs (const M68KRegsType& debuggerRegs)
{
	m68k_dreg (regs, 0) = debuggerRegs.d[0];
	m68k_dreg (regs, 1) = debuggerRegs.d[1];
	m68k_dreg (regs, 2) = debuggerRegs.d[2];
	m68k_dreg (regs, 3) = debuggerRegs.d[3];
	m68k_dreg (regs, 4) = debuggerRegs.d[4];
	m68k_dreg (regs, 5) = debuggerRegs.d[5];
	m68k_dreg (regs, 6) = debuggerRegs.d[6];
	m68k_dreg (regs, 7) = debuggerRegs.d[7];

	m68k_areg (regs, 0) = debuggerRegs.a[0];
	m68k_areg (regs, 1) = debuggerRegs.a[1];
	m68k_areg (regs, 2) = debuggerRegs.a[2];
	m68k_areg (regs, 3) = debuggerRegs.a[3];
	m68k_areg (regs, 4) = debuggerRegs.a[4];
	m68k_areg (regs, 5) = debuggerRegs.a[5];
	m68k_areg (regs, 6) = debuggerRegs.a[6];
//	m68k_areg (regs, 7) = debuggerRegs.a[7];

	regs.usp	= debuggerRegs.usp;
	regs.isp	= debuggerRegs.ssp;
	regs.sr		= debuggerRegs.sr;
	m68k_setpc (debuggerRegs.pc);

	// Make sure 's' bit is correct.  This will also set SPCFLAG_INT, and
	// if either the t0 or t1 bits are set, SPCFLAG_TRACE, too.

	Registers::UpdateRegistersFromSR ();

	// I think we need to do a little extra handling here of the TRACE
	// mode.  If the TRACE mode bit is set, UpdateRegistersFromSR will
	// set its SPCFLAG_TRACE flag, which tells the CPU loop to finish
	// executing the current instruction, and then set the SPCFLAG_DOTRACE
	// flag afterwards, which will then cause execution to stop after
	// the NEXT instruction.  We have a bit of an edge condition here
	// when we set the SPCFLAG_TRACE flag BETWEEN instructions.  Doing it
	// this way will cause the next instruction to be treated as the
	// "current" instruction, and so execution will stop after executing
	// the second opcode.  We want execution to stop with the first
	// opcode, so fix up the flags here.  This code is the same as in
	// Emulator::ExecuteSpecial for managing this stuff.

	if (regs.spcflags & SPCFLAG_TRACE)
	{
		regs.spcflags &= ~SPCFLAG_TRACE;
		regs.spcflags |= SPCFLAG_DOTRACE;
	}

	// Finally, update the stack pointer from the usp or ssp,
	// as appropriate.

	if (regs.s == 0)
		m68k_areg (regs, 7) = debuggerRegs.usp;
	else
		m68k_areg (regs, 7) = debuggerRegs.ssp;
}


