/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "StreamHandle.h"

#include "ErrorHandling.h"		// this->Throw
#include "Platform_Files.h"		// FileReference



// ===========================================================================
//		 StreamHandle
//
//		A StreamHandle is an abstract handle to streams of data.
//		
// ===========================================================================


/***********************************************************************
 *
 * FUNCTION:	StreamHandle()
 *
 * DESCRIPTION:	StreamHandle constructor.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

StreamHandle::StreamHandle (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	~StreamHandle()
 *
 * DESCRIPTION:	StreamHandle destructor.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

StreamHandle::~StreamHandle (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	Throw()
 *
 * DESCRIPTION:	StreamHandle exception thrower. StreamHandle methods will
 *				call this to throw exceptions.
 *
 * PARAMETERS:	err - error code.
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void			
StreamHandle::Throw (ErrCode err)
{
	Errors::Throw (err);
}


/***********************************************************************
 *
 * FUNCTION:	Write
 *
 * DESCRIPTION:	Writing to a StreamHandle DOES NOTHING.
 *
 * PARAMETERS:	length - number of bytes to write
 *				buffer - location of buffer to read out of
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void
StreamHandle::Write (uae_s32 length, const void* buffer)
{
	UNUSED_PARAM (length);
	UNUSED_PARAM (buffer);
}


/***********************************************************************
 *
 * FUNCTION:	GetFileRef()
 *
 * DESCRIPTION:	All StreamHandles support getting the underlying file
 *				reference, but not all StreamHandles use files. So by
 *				by default a StreamHandle will pass back an empty
 *				FileReference and return false.
 *
 * PARAMETERS:	f - FileReference to be passed back.
 *
 * RETURNED:	TRUE if a file reference was returned.
 *				FALSE otherwise.
 *
 ***********************************************************************/

Bool
StreamHandle::GetFileRef (FileReference& f)
{
	// pass back a default (empty) FileReference
	FileReference r;

	f = r;
	return false;
}


// ===========================================================================
//		 BufferHandle
//
//		A BufferHandle is a specialization of a StreamHandle which handles
//		streams in memory buffers. The BufferHandle does not "own" the memory
//		that is passed to it, because this memory can be (and typically is) a
//		read-only resource. Therefore, if the memory you pass to the
//		BufferHandle needs to be freed, it is your responsibility to do that.
//		
// ===========================================================================

/***********************************************************************
 *
 * FUNCTION:	BufferHandle()
 *
 * DESCRIPTION:	BufferHandle constructor.
 *
 * PARAMETERS:	pData - pointer to the start of the buffer
 *				iSize - size of the buffer
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

BufferHandle::BufferHandle (void* data, uae_s32 size)
{
	fData	= (char*) data;
	fSize	= size;
	fPos	= 0;
}


/***********************************************************************
 *
 * FUNCTION:	~BufferHandle
 *
 * DESCRIPTION:	BufferHandle destructor. Does not free the memory that
 *				handle represents.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

BufferHandle::~BufferHandle (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	SetPos
 *
 * DESCRIPTION:	Sets the "cursor" position in the buffer, where the next
 *				read will begin.
 *
 * PARAMETERS:	offset - change in cursor position (signed)
 *				seekMode - 	pass in {kSeekSet, kSeekCur, kSeekEnd}
 *							representing where to begin the offset
 *							(front, current position, end)
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void
BufferHandle::SetPos (uae_s32 offset, SeekMode seekMode)
{
	// used signed value to detect mishaps
	uae_s32	pos = fPos;

	if (seekMode == kSeekSet)
	{
		pos = offset;
	}
	else if (seekMode == kSeekEnd)
	{
		pos = fSize - 1 + offset;
	}
	else
	{
		pos += offset;
	}

	// Note that as with files, it is not an error to set the position
	// past the end of the buffer--only before it. Memory buffers cannot
	// be written to, and our code to read from it will automatically
	// ignore reads past the end of the file, so we are safe.

	if (pos < 0)
	{
//		this->Throw (paramErr);
		assert (false);	// We don't have any cross-platform value to throw
						// here, so let's just assert.
	}

	fPos =  pos;
}


/***********************************************************************
 *
 * FUNCTION:	Read
 *
 * DESCRIPTION:	Reads a specified number of characters from this buffer
 *				object into a specified memory buffer. Will only read up
 *				to the end of this buffer object, or the length specified,
 *				whichever is smaller.
 *
 * PARAMETERS:	length - number of bytes to read
 *				buffer - location of buffer to read into
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void
BufferHandle::Read (uae_s32 length, void* buffer)
{
	uae_s32	copyLength = min (length, (fSize - fPos));

	if (copyLength > 0)
	{
		memcpy (buffer, fData + fPos, copyLength);
		fPos += copyLength;
	}
}
