/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#ifndef _PLATFORM_H_
#define _PLATFORM_H_


#ifdef __cplusplus
extern "C"
{
#endif

	// Define these wrapper functions for C modules.

void*	Platform_AllocateMemory (size_t size);
void*	Platform_ReallocMemory	(void* p, size_t size);
void	Platform_DisposeMemory	(void* p);

#ifdef __cplusplus
}	// extern "C"

	// Do all the rest of this for C++ studs.

#include "ErrorHandling.h"		// ErrCode
#include "LoadApplication.h"	// LoadableList
#include "omnithread.h" 		// omni_mutex
#include "UART.h"				// UART::Parity
#include "EmulatorTypes.h"		// DeviceType

class ChunkFile;
class CPU;
class CSocket;
class StMemory;


// Public interface to our byte queue.	This queue is used to collect bytes
// going in an out of the serial ports, as well as characters to be passed
// from the keyboard to the emulated ROM.
//
// Most of the implementation of this class is provided by a platform-
// dependent, private class.  The reason for splitting the class up into
// two parts is because the multi-threaded Windows version of the emulator
// needs some synchronization when bytes are added and removed.

class TByteQueuePrivate;

class TByteQueue
{
	public:
								TByteQueue	();
								~TByteQueue ();

		void					Put 		(uae_u8 b);
		uae_u8					Get 		(void);
		int 					GetUsed 	(void);
		int 					GetFree 	(void);
		Bool					WaitForDataAvailable (int timeoutms);

	private:
		TByteQueuePrivate*		fImplementation;
};


// Globals

extern unsigned long	gPseudoTickcount;
extern Bool 			gEnableSounds;		// accessed by CPU_setsounds
extern bool 			gShowingDialog;

int _stricmp(const char *s1, const char *s2);
int _strnicmp(const char *s1, const char *s2, int n);
char* _strdup (const char* s);

// Function prototypes.

class Platform
{
	public:

		static void 			Initialize			(void);
		static void 			Reset				(void);
		static void 			Save				(SessionFile&);
		static void 			Load				(SessionFile&);
		static void 			Dispose 			(void);

	// Dialog-related functions

		static int				CommonDialog			(const char*, const char*, int flags);

	// Resource-related functions

		static string			GetString				(int id);
		static int				GetIDForError			(ErrCode error);
		static int				GetIDForRecovery		(ErrCode error);
	
		static string			GetShortVersionString	(void);

		static Bool 			ROMResourcePresent		(void);
		static Bool 			GetROMResource			(void*& rom, uae_s32& romSize);
		static Bool 			PSFResourcePresent		(void);
		static Bool 			GetPSFResource			(void*& psf, uae_s32& psfSize);

		static Bool 			ConfigResourcePresent	(void);
		static Bool 			GetConfigResource		(void*& config, uae_s32& configSize);
		static Bool 			GetEmbeddedDeviceType	(DeviceType& device);
		static Bool 			GetEmbeddedRAMSize		(RAMSizeType& ramSize);

	// Time-related functions

		static uae_u32			GetMilliseconds 		(void);

	// External debugger-related functions

		static CSocket* 		CreateDebuggerSocket	(void);
		static void 			ExitDebugger			(void);

	// Graphics-related functions

		static void 			ViewDrawLine			(int xStart, int yStart, int xEnd, int yEnd);
		static void 			ViewDrawPixel			(int xPos, int yPos);

	// Session-reated functions

		static Bool				SessionRunning			(void);

			// These two functions are specific to the Drag And Drop
			// facility.  They should eventually get rolled into some
			// other facility (like being merged with the Startup stuff).
		static ErrCode			CreateSession			(void);
		static ErrCode			OpenSession				(void);

	// Serial-related functions

		static ErrCode			OpenSerialPort			(void);
		static ErrCode			SetSerialPortSettings	(UART::Parity	parity,
														 int			stopBits,
														 int			dataBits,
														 uae_u32		baud,
														 Bool			hwHandshaking);
		static void 			GetSerialPortSettings	(UART::Parity&	parity,
														 int&			stopBits,
														 int&			dataBits,
														 uae_u32&		baud,
														 Bool&			hwHandshaking);
		static Bool 			SerialPortOpen			(void);
		static void 			CloseSerialPort 		(void);
		static void 			TransmitTxFIFO			(void);
		static void 			ReceiveRxFIFO			(void);

	// "Load File Progress Window"

		static void 			LFPW_Open				(void);
		static void 			LFPW_SetNumFiles		(int);
		static void 			LFPW_StartNewFile		(const char*);
		static void 			LFPW_SetFileMax 		(int);
		static void 			LFPW_SetFileValue		(int);
		static void 			LFPW_Close				(void);
		static Bool 			LFPW_UserAborts 		(void);

	// "Gremlin Control Window"

		static void 			GCW_Open				(void);
		static void 			GCW_Close				(void);

	// Whatever....
	
		static Bool				PinToScreen				(AbsRectType&);

		static Bool				QueryNewDocument		(Configuration& cfg);

		static CPU*				GetCPU					(void);

		static void 			ToHostEOL				(StMemory& dest, long& destLen,
														 const char* src, long srcLen);

		static void 			CreatePortNameCache 	(StringList& results);

		static FileReference	CreateReferenceInEmulatorPath		(const char*);
		static FileReference	CreateReferenceInGremlinStatePath	(const char*);
		static FileReference	GetPreferenceFile		(void);

		static void				SetGremlinStatePathFromControlFile	(FileReference& controlFile);

		static void 			GetLoadableFileList 	(string directoryName,
														 FileRefList& fileList);
		static Bool 			ReadROMFileReference	(ChunkFile& docFile,
														 FileReference& f);
		static void 			WriteROMFileReference	(ChunkFile& docFile,
														 const FileReference& f);

		static void 			UseNewAutoSaveDirectory (void);
		static void 			AutoSaveState			(const FileReference& fileRef);

		static void 			Delay					(void);
		static Bool 			TimeForCycle			(Bool sleeping);
		static void 			Cycle					(void);

		static void*			RealAllocateMemory	 	(size_t size, Bool clear, const char* file, int line);
		static void*			RealReallocMemory		(void* p, size_t size, const char* file, int line);
		template <class T>
		static void 			DisposeMemory			(T*& p)
								{
									// Normally, I don't put functions inline like
									// this.  However, if I try defining this function
									// in a separate body, VC++ complains that DisposeMemory
									// doesn't take one parameter.	Go figure...

									if (p)
									{
										RealDisposeMemory ((void*) p);
										p = NULL;
									}
								}
		static void 			RealDisposeMemory		(void* p);

			// Aliases for DisposeMemory, because I can never remember
			// what the real name is...
		template <class T>
		static void 			DeleteMemory			(T*& p) { DisposeMemory(p); }

		template <class T>
		static void 			FreeMemory				(T*& p) { DisposeMemory(p); }

		static Bool				ForceStartupScreen		(void);
		static Bool 			StopOnResetKeyDown		(void);

		static void 			Debugger				(void);
};

#define AllocateMemory(size)	\
	RealAllocateMemory(size, false, __FILE__, __LINE__)

#define AllocateMemoryClear(size)	 \
	RealAllocateMemory(size, true, __FILE__, __LINE__)

#define ReallocMemory(p, size)	  \
	RealReallocMemory(p, size, __FILE__, __LINE__)


#if defined (__MACOS__)

	extern unsigned long	gLastCycleTime;

	inline Bool Platform::TimeForCycle (Bool sleeping)
	{
		// Calling LMGetTicks every time through the CPU loop is excessive.
		// On a 250MHz PowerMac, we can execute about 30K opcodes/tick, so
		// let's check in with LMGetTicks only that often.	Checking in every
		// 32768 opcodes, we get about a 9% performance increase (1850 msecs
		// vs. 1690 msecs in the tests I ran).

		static instructionCount;
		instructionCount++;
		if (!sleeping && (instructionCount & 0x7FFF) != 0)
			return false;

		return ((LMGetTicks () - gLastCycleTime) >= EVENT_THRESHHOLD);
	}

	inline void Platform::Debugger (void)
	{
		::Debugger ();
	}

#endif


#if defined (_WINDOWS) || defined (UNIX)
	inline Bool Platform::TimeForCycle (Bool)
	{
		static unsigned long	lastPseudoTickcount;

		if (lastPseudoTickcount != gPseudoTickcount)
		{
			lastPseudoTickcount = gPseudoTickcount;
			return true;
		}

		return false;
	}
#endif

#if defined (_WINDOWS)
	inline void Platform::Debugger (void)
	{
		__asm int 3;
	}

#endif

#if defined(UNIX)
	inline void Platform::Debugger(void)
	{
	}
	
#endif

#endif	// extern "C"

#endif /* _PLATFORM_H_ */
