/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "LoadApplication.h"

#include "Byteswapping.h"
#include "ErrorHandling.h"		// Errors::ThrowIfPalmError, kError_OutOfMemory
#include "Miscellaneous.h"		// StMemory
#include "Platform.h"			// Platform::AllocateMemory, Platform::LFPW...
#include "PreferenceMgr.h"		// gEmuPrefs->UpdatePRCMRU
#include "RAM_ROM.h"			// Memory::MapPhysicalMemory
#include "Strings.r.h"			// kStr_ values
#include "TrapPatches.h"		// Patches::UIInitialized
#include "UAE_Utils.h"			// uae_memcpy

Bool PrvLauncherActive (void);
Bool PrvSwitchToApp (UInt32 type, UInt32 creator);

static Bool PrvIsResources(Word attributes)
{
	return (attributes & dmHdrAttrResDB);
}

static Bool PrvIsResources(DatabaseHdrPtr hdrP)
{
	return PrvIsResources(hdrP->attributes);
}

static UInt32 PrvGetEntrySize(DatabaseHdrPtr hdrP)
{
	if (PrvIsResources(hdrP))
		return sizeof (RsrcEntryType);
	
	return sizeof (RecordEntryType);
}

/************************************************************
 *
 *	FUNCTION: PrvValidateDatabaseHeader
 *
 *	DESCRIPTION: Validates database header (to be used before
 *		any of the rest of the database is byteswapped or
 *		otherwise processed)
 *
 *	PARAMETERS: 
 *			hdrP - pointer to database header
 *			bufferSize - size of the buffer holding the database
 *
 *	RETURNS: Err - error code
 *
 *************************************************************/

static ErrCode PrvValidateDatabaseHeader(DatabaseHdrPtr hdrP, UInt32 bufferSize)
{
	UInt	i;
	UInt	len = strlen((CharPtr) hdrP->name);
	
	// Establish the upper and lower ranges for localChunkIDs.	These IDs are
	// offsets from the beginning of the buffer/file, so they can't be larger
	// than the size of the buffer/file, nor can they be smaller than the
	// starting offset of the chunks.
	ULong	upperRange = bufferSize;
	ULong	lowerRange = offsetof (DatabaseHdrType, recordList.firstEntry) +
							PrvGetEntrySize(hdrP) * hdrP->recordList.numRecords;

	// check to see that name is null-terminated correctly
	if (len > dmDBNameLength)
		return dmErrCorruptDatabase;

	// check to see that name consists of printable characters
	for (i = 0; i < len; i++)
	{
		if (hdrP->name[i] < ' ')
			return dmErrCorruptDatabase;
	}

	// check that the resource/database records headers fit within the space
	// of the buffer before we even byteswap so that we do not crash while
	// byteswapping them

	// check if first entry is too low
	if ((ULong)(&(hdrP->recordList.firstEntry)) < lowerRange)
		return dmErrCorruptDatabase;

	// check to see if last entry is too high
	if (hdrP->recordList.numRecords * PrvGetEntrySize(hdrP) +
		offsetof (DatabaseHdrType, recordList.firstEntry) > bufferSize)
		return dmErrCorruptDatabase;
	
	// if we got here, we're okay
	return 0;
}

/************************************************************
 *
 *	FUNCTION: PrvValidateEntries
 *
 *	DESCRIPTION: Validates RESOURCE entries in a database
 *
 *	PARAMETERS: 
 *			hdrP - pointer to database header
 *			bufferSize - size of the buffer holding the database
 *
 *	RETURNS: Err - error code
 *
 *************************************************************/

static ErrCode PrvValidateEntries(DatabaseHdrPtr hdrP, UInt32 bufferSize)
{
	// We use entryP to refer to both records OR resources; it is declared
	// as a BytePtr so that we can do byte-arithmetic to increment the appropriate
	// amount depending on whether we are dealing with records or resources
	BytePtr 		entryP	= (BytePtr)(&hdrP->recordList.firstEntry);

	// Establish the upper and lower ranges for localChunkIDs.	These IDs are
	// offsets from the beginning of the buffer/file, so they can't be larger
	// than the size of the buffer/file, nor can they be smaller than the
	// starting offset of the chunks.
	ULong			upperRange = bufferSize;
	ULong			lowerRange = offsetof (DatabaseHdrType, recordList.firstEntry) +
									PrvGetEntrySize(hdrP) * hdrP->recordList.numRecords;
	StMemory		marks;

	// Create a mirror buffer to check for overlapping records.
	marks.Adopt ((char*) Platform::AllocateMemoryClear (bufferSize));

	// nextRecordListID not supported for now.
	if (hdrP->recordList.nextRecordListID != 0)
		return dmErrCorruptDatabase;

	for (int i = 0; i < hdrP->recordList.numRecords; i++)
	{
		LocalID 	id;
		long		itsSize;	// Use a signed value to detect non-increasing LocalIDs

		// Check for negative sizes and get the LocalID.
		// ---------------------------------------------
		// Though the code is similar for resources and for records, we have separate
		// blocks so that we can safely cast to different types to access the localChunkID
		// field.
		if (PrvIsResources(hdrP))
		{
			RsrcEntryPtr rsrcEntryP = (RsrcEntryPtr) entryP;

			if (i < hdrP->recordList.numRecords - 1)
			{
				itsSize = (ULong) rsrcEntryP[1].localChunkID;
				itsSize -= (ULong) rsrcEntryP[0].localChunkID;
			}
			else
				itsSize = bufferSize - (ULong) rsrcEntryP[0].localChunkID;

			// Check for negative sizes (zero is NOT okay for resources).
			if (itsSize <= 0)
				return dmErrCorruptDatabase;

			id = rsrcEntryP->localChunkID;
		}
		else
		{
			RecordEntryPtr recordEntryP = (RecordEntryPtr) entryP;
			id = recordEntryP->localChunkID;

			if (i < hdrP->recordList.numRecords - 1)
			{
				itsSize = (ULong) recordEntryP[1].localChunkID;
				itsSize -= (ULong) recordEntryP[0].localChunkID;
			}
			else
				itsSize = bufferSize - (ULong) recordEntryP[0].localChunkID;

			// Check for negative sizes (zero is okay for resources).
			if (itsSize < 0)
				return dmErrCorruptDatabase;

			id = recordEntryP->localChunkID;
		}

		// Test that the beginning and ending of the chunk are in range.
		// Note the special case for zero-length objects appearing at
		// the end of the file.
		if (itsSize > 0)
		{
			if (id >= upperRange || id + itsSize < lowerRange)
				return dmErrCorruptDatabase;
		}
		else if (id > upperRange || id + itsSize < lowerRange)
			return dmErrCorruptDatabase;

		// Check that the range of this record is not occupied by
		// another record.
		for (ULong ii = id; ii < id + itsSize; ++ii)
		{
			if (marks.Get () [ii] != 0)
				return dmErrCorruptDatabase;

			marks.Get () [ii] = 1;	// Mark this byte as occupied.
		}

		// next entry
		entryP += PrvGetEntrySize(hdrP);
	}

	// if we got here, we're okay
	return 0;
}


/************************************************************
 *
 *	FUNCTION: PrvValidateDatabase
 *
 *	DESCRIPTION: 
 *
 *	PARAMETERS: 
 *			hdrP - pointer to database header
 *			bufferSize - size of the buffer holding the database
 *
 *	RETURNS: Err - error code
 *
 *************************************************************/

static ErrCode PrvValidateDatabase (DatabaseHdrPtr hdrP, UInt32 size)
{
	if (size < sizeof (DatabaseHdrType))
		return dmErrCorruptDatabase;

	Canonical(*hdrP);	// Header now in little-endian format

	ErrCode err;

	err = ::PrvValidateDatabaseHeader(hdrP, size);
	if (err != errNone)
		return err;

	if (PrvIsResources(hdrP))
	{
		RsrcEntryPtr	rsrcP;

		// Byteswap the resource entry headers

		rsrcP  = (RsrcEntryPtr) (&hdrP->recordList.firstEntry);
		for (int i = 0; i < hdrP->recordList.numRecords; ++i)
		{
			Canonical(*rsrcP);
			++rsrcP;
		}
	}
	else
	{
		RecordEntryPtr	recordP;

		// Byteswap the record entry headers

		recordP  = (RecordEntryPtr)(&hdrP->recordList.firstEntry);
		for (int i = 0; i < hdrP->recordList.numRecords; ++i)
		{
			Canonical (*recordP);
			++recordP;
		}
	}

	// Check out the record entries; make sure they're okay
	err = ::PrvValidateEntries(hdrP, size);

	return err;
}


/************************************************************
 *
 *	FUNCTION: DmCreateDatabaseFromImage
 *
 *	DESCRIPTION: Can be called to create an entire database
 *			from a single resource that contains in image of the database. 
 *			Usually, this would be called from an app's reset action code during
 *			boot.
 *
 *	PARAMETERS: 
 *			bufferP - pointer to locked resource containing image of the database
 *
 *	RETURNS: void
 *
 *	CREATED: 3/13/95
 *
 *************************************************************/

static Err	My_DmCreateDatabaseFromImage(Ptr bufferP, UInt32 size, UInt& cardNo, LocalID& dbID)
{
	DmOpenRef		dbP = 0;
	Err 			err = 0;
	BytePtr 		srcP;
	BytePtr 		dstP;
	DatabaseHdrPtr	hdrP = (DatabaseHdrPtr)bufferP;
	UInt			i;
	DWord			uniqueIDSeed;
	DmAccessPtr 	dbP2;
	DmOpenInfoPtr	openP2;
	DatabaseHdrPtr	hdrP2;

	// Get the size and create it
	err = DmCreateDatabase(cardNo, (CharPtr)hdrP->name, hdrP->creator, hdrP->type, 
						PrvIsResources(hdrP));
	if (err)
		goto Exit;

	// Set Database info
	dbID = DmFindDatabase(cardNo, (CharPtr)hdrP->name);
	if (!dbID)
	{
		err = dmErrCantFind;
		goto Exit;
	}

	err = DmSetDatabaseInfo(cardNo, dbID, 0,
				&hdrP->attributes, &hdrP->version, &hdrP->creationDate,
				&hdrP->modificationDate, &hdrP->lastBackupDate,
				&hdrP->modificationNumber, 0,
				0, &hdrP->type,
				&hdrP->creator);

	if (err)
		goto Exit;
	
	
	// Open the new database
	dbP = DmOpenDatabase(cardNo, dbID, dmModeReadWrite);
	if (!dbP)
	{
		err = dmErrCantOpen;
		goto Exit;
	}

	// The new database has been created with a random number in it.
	// We don't want the records copied into the new database to have
	// different unique id's between each hard reset or else the records
	// with the same contents but with different unique id's will appear
	// distinct to HotSync and so will accumulate with hard resets and syncs.
	// To stop this behavior we always set the seed to a constant number,
	// copy the records into the new database, and then we restore the
	// random seed.
	MemSemaphoreReserve(true);
//	uniqueIDSeed = ((DmAccessPtr)dbP)->openP->hdrP->uniqueIDSeed;
//	((DmAccessPtr)dbP)->openP->hdrP->uniqueIDSeed = dmDefaultRecordsID << 12;
	dbP2	= (DmAccessPtr) get_real_address((uaecptr) dbP);
	openP2	= (DmOpenInfoPtr) get_real_address(do_get_mem_long(&dbP2->openP));
	hdrP2	= (DatabaseHdrPtr) get_real_address(do_get_mem_long(&openP2->hdrP));
	uniqueIDSeed = hdrP2->uniqueIDSeed;
	do_put_mem_long(&hdrP2->uniqueIDSeed, dmDefaultRecordsID << 12);
	MemSemaphoreRelease(true);

	
	//------------------------------------------------------------
	// If it's a resource database, Add the Resources
	//------------------------------------------------------------
	if (PrvIsResources(hdrP)) {
		RsrcEntryPtr	rsrcP;
		ULong			resSize;
		Handle			newResH;

		Platform::LFPW_SetFileMax (hdrP->recordList.numRecords);

		// Pointer to first resource
		rsrcP  = (RsrcEntryPtr)(&hdrP->recordList.firstEntry);
		for (i = 0; i < hdrP->recordList.numRecords; i++)
		{
		
			Platform::LFPW_SetFileValue (i);

			// Calculate the size of the resource
			if (i < hdrP->recordList.numRecords - 1)
			{
				resSize = (ULong)rsrcP[1].localChunkID;
				resSize -= (ULong)rsrcP[0].localChunkID;
			}
			else
				resSize = size - (ULong)rsrcP[0].localChunkID;

			// (Should never get here, since the .prc file validity check
			//	above should filter out files with zero-sized resources.)
			if (resSize == 0)
			{
				rsrcP++;
				continue;
			}

			// Allocate the new resource
			newResH = (Handle) DmNewResource(dbP, rsrcP->type, rsrcP->id, resSize);
			if (!newResH)
			{
				err = dmErrMemError;
				goto Exit;
			}

			// Copy the data in
			srcP = (BytePtr)bufferP + (ULong)rsrcP->localChunkID;
			dstP = (BytePtr)MemHandleLock((VoidHand) newResH);
			DmWrite(dstP, 0, srcP, resSize);
			MemHandleUnlock((VoidHand) newResH);
			// Done with it
			DmReleaseResource((VoidHand) newResH);

			// next one
			rsrcP++;
		}
	}

	//------------------------------------------------------------
	// If it's a Record database, Add the Records
	//------------------------------------------------------------
	else {
		Handle			newRecH;
		RecordEntryPtr	recordP;
		ULong			recSize;
		LocalID 		appInfoID;

		// Pointer to first record
		recordP  = (RecordEntryPtr)(&hdrP->recordList.firstEntry);

		// If there's an appInfo, add that
		if (hdrP->appInfoID)
		{
			if (hdrP->sortInfoID)
				recSize = (ULong)hdrP->sortInfoID - (ULong)hdrP->appInfoID;
			else if (hdrP->recordList.numRecords > 0)
				recSize = (ULong)recordP[0].localChunkID - (ULong)hdrP->appInfoID;
			else
				recSize = size - (ULong)hdrP->appInfoID;

			// Allocate the chunk
			newRecH = (Handle)DmNewHandle(dbP, recSize);
			if (!newRecH)
			{
				err = dmErrMemError;
				goto Exit;
			}

			// Copy the data in
			srcP = (BytePtr)bufferP + (ULong)hdrP->appInfoID;
			dstP = (BytePtr)MemHandleLock((VoidHand) newRecH);
			DmWrite(dstP, 0, srcP, recSize);
			MemHandleUnlock((VoidHand) newRecH);

			// Store it in the header
			appInfoID = MemHandleToLocalID((VoidHand) newRecH);
			DmSetDatabaseInfo(0, dbID, 0,
				0, 0, 0,
				0, 0,
				0, &appInfoID,
				0, 0, 
				0);
		}

		Platform::LFPW_SetFileMax (hdrP->recordList.numRecords);

		// Add Each of the records.
		for (i=0; i<hdrP->recordList.numRecords; i++)
		{
			UInt		index;
			UInt		attr;
			ULong		id;

			Platform::LFPW_SetFileValue (i);

			// Calculate the size of the record
			if (i < hdrP->recordList.numRecords - 1)
			{
				recSize = (ULong)recordP[1].localChunkID;
				recSize -= (ULong)recordP[0].localChunkID;
			}
			else
				recSize = size - (ULong)recordP[0].localChunkID;

			//	Only add non-nil records. When the default databases are created
			//	they sometimes have nil records if the user had to delete a
			//	record they didn't want.
			if (recSize)
			{
				// Allocate the new record
				index = 0xFFFF;
				newRecH = (Handle)DmNewRecord(dbP, &index, recSize);
				if (!newRecH)
				{
					err = dmErrMemError;
					goto Exit;
				}

				// Copy the data in
				srcP = (BytePtr)bufferP + (ULong)recordP->localChunkID;
				dstP = (BytePtr)MemHandleLock((VoidHand) newRecH);
				DmWrite(dstP, 0, srcP, recSize);
				MemHandleUnlock((VoidHand) newRecH);

				// Set the attributes
				attr = recordP->attributes;
				id = recordP->uniqueID[0];
				id = (id << 8) | recordP->uniqueID[1];
				id = (id << 8) | recordP->uniqueID[2];
				DmSetRecordInfo(dbP, index, &attr, &id);

				// Done with it
				DmReleaseRecord(dbP, index, true);
			}

			// next one
			recordP++;
		}
	}


		// Fixup the modification # to match what was in the image.
	DmSetDatabaseInfo(cardNo, dbID, 0,
				0, 0,0,
				0, 0,
				&hdrP->modificationNumber, 0,
				0, 0,
				0);


	// Restore the database's uniqueIDSeed to it's random value.
	MemSemaphoreReserve(true);
//	((DmAccessPtr)dbP)->openP->hdrP->uniqueIDSeed = uniqueIDSeed;
	// (Refresh the pointers; they may have changed since we initialized them.)
	dbP2	= (DmAccessPtr) get_real_address((uaecptr) dbP);
	openP2	= (DmOpenInfoPtr) get_real_address(do_get_mem_long(&dbP2->openP));
	hdrP2	= (DatabaseHdrPtr) get_real_address(do_get_mem_long(&openP2->hdrP));
	hdrP2->uniqueIDSeed = uniqueIDSeed;
	MemSemaphoreRelease(true);

	// If we got to here, no error 
	DmCloseDatabase(dbP);
	dbP = 0;


Exit:
	if (dbP)
	{
		DmCloseDatabase(dbP);
		if (dbID)
			DmDeleteDatabase(cardNo, dbID);
	}

	return err;
}


// ---------------------------------------------------------------------------
//		 LoadPalmFile
// ---------------------------------------------------------------------------
// Loads a Palm OS program or database file into the emulator.

void LoadPalmFile (FileHandle& appFile)
{
	UInt cardNo = 0;
	LocalID dbID = 0;
	Err err = 0;

	// Get the size of the file.

	uae_s32 fileSize = appFile.GetLength ();

	// Create a buffer to hold the file data.

	StMemory	fileBuffer (fileSize);

	// Load the file into memory.

	appFile.Read (fileSize, fileBuffer.Get ());

	// Check for corrupt databases.

	ErrCode errCode = ::PrvValidateDatabase ((DatabaseHdrPtr) fileBuffer.Get (), fileSize);
	Errors::ThrowIfPalmError (errCode);

	// Let the emulated ROM access the memory in the file buffer.

	StMemoryMapper	mapper (fileBuffer.Get (), fileSize);

	// See if there's already a database with this name.

	LocalID tempID = DmFindDatabase (cardNo, fileBuffer.Get ());

	// If the database already exists, delete it.

	if (tempID != 0)
	{
		err = DmDeleteDatabase (cardNo, tempID);

		if (err != dmErrROMBased)
			Errors::ThrowIfPalmError (err);
	}

	// Create the new database.

	err = My_DmCreateDatabaseFromImage (fileBuffer.Get (), fileSize, cardNo, dbID);
	Errors::ThrowIfPalmError (err);
}



// ---------------------------------------------------------------------------
//		 LoadPalmFileList
// ---------------------------------------------------------------------------

void LoadPalmFileList (const FileRefList& fileList)
{
	if (fileList.size () > 0)
	{
		Bool	restoreLauncher =	::PrvLauncherActive () &&
									::PrvSwitchToApp (sysFileTApplication, sysFileCCalculator);

		Platform::LFPW_Open ();

		Platform::LFPW_SetNumFiles (fileList.size ());

		FileReference	fileRef;

		try
		{
			FileRefList::const_iterator	iter = fileList.begin ();
			while (iter != fileList.end () && !Platform::LFPW_UserAborts())
			{
				fileRef = *iter;
				FileHandle	file (fileRef, kOpenExisting | kOpenRead);

				Platform::LFPW_StartNewFile (fileRef.GetFileName ().c_str ());

				::LoadPalmFile (file);

				gEmuPrefs->UpdatePRCMRU (fileRef);

				++iter;
			}
		}
		catch (ErrCode errCode)
		{
			if (fileList.size () > 1)
			{
				Errors::ReportIfError (kStr_CmdInstallMany, errCode, 0, false);
			}
			else
			{
				Errors::SetParameter ("%filename", fileRef.GetFileName ());
				Errors::ReportIfError (kStr_CmdInstall, errCode, 0, false);
			}
		}

		Platform::LFPW_Close ();

		if (restoreLauncher)
		{
			::PrvSwitchToApp (sysFileTApplication, sysFileCLauncher);
		}
	}
}


/************************************************************
 * Export a resource or record database as a normal data file in 
 *	Pilot format.
 *************************************************************/

static void My_ShlExportAsPilotFile(FileHandle& fh, UInt cardNo, const char* nameP)
{
	DmOpenRef		dbP = 0;
	Err 			err;
	UInt				numRecords;
	int 			size;
	DatabaseHdrPtr	hdrP;
	long				offset;
	int 			i;
	
	
try
{
	//------------------------------------------------------------
	// Locate the Database and see what kind of database it is
	//------------------------------------------------------------

	LocalID dbID;
	dbID = ::DmFindDatabase(cardNo, (CharPtr) nameP);
	if (!dbID) Errors::ThrowIfPalmError(DmGetLastErr());
	
	UInt		attributes, version;
	ULong		crDate, modDate, bckUpDate;
	ULong		type, creator;
	ULong		modNum;
	LocalID appInfoID, sortInfoID;
	err = ::DmDatabaseInfo(cardNo, dbID, 0,
				&attributes, &version, &crDate,
				&modDate, &bckUpDate,
				&modNum, &appInfoID,
				&sortInfoID, &type,
				&creator);
	Errors::ThrowIfPalmError(err);
	
	dbP = ::DmOpenDatabase(cardNo, dbID, dmModeReadOnly);
	if (!dbP) Errors::ThrowIfPalmError(DmGetLastErr());


	//------------------------------------------------------------
	// Write out a resource database
	//------------------------------------------------------------
	if (PrvIsResources(attributes)) {
		ULong	resType;
		Int resID;
		ULong	resSize;

		numRecords = DmNumRecords(dbP);
		
		size = sizeof(DatabaseHdrType) + numRecords * sizeof(RsrcEntryType);
		StMemory	outP (size, true);
		
		// Fill in header
		hdrP = (DatabaseHdrPtr)outP.Get ();
		
		strcpy((char*)hdrP->name, nameP);
		hdrP->attributes = attributes;
		hdrP->version = version;
		hdrP->creationDate = crDate;
		hdrP->modificationDate = modDate;
		hdrP->lastBackupDate = bckUpDate;
		hdrP->modificationNumber = modNum;
		hdrP->appInfoID = 0;
		hdrP->sortInfoID = 0;
		hdrP->type = type;
		hdrP->creator = creator;
	
		hdrP->recordList.nextRecordListID = 0;
		hdrP->recordList.numRecords = numRecords;
	
	
		// Fill in the info on each resource into the header
		RsrcEntryPtr	entryP = (RsrcEntryPtr)(&hdrP->recordList.firstEntry);
		offset = size;
		for (i=0; i<numRecords; i++)  {
			
			DmResourceInfo(dbP, i, &resType, &resID, 0);
			VoidHand	resH = DmGetResourceIndex(dbP, i);
			if (resH)  {
				resSize = MemHandleSize(resH);
				DmReleaseResource(resH);
				}
			else {
				resSize = 0;
				}
			entryP->type = resType;
			entryP->id =resID;
			entryP->localChunkID = offset;
			
			offset += resSize;
			entryP++;
			}

		// Byteswap the resource entry headers
		entryP	= (RsrcEntryPtr)(&hdrP->recordList.firstEntry);
		for (i=0; i<hdrP->recordList.numRecords; i++)
		{
			Canonical(*entryP);
			entryP++;
		}
		Canonical(*hdrP);	// Now in big-endian format
	
		// Write out  entry table
		fh.Write (size, outP.Get ());

		// Write out each resource
		for (i=0; i<numRecords; i++) {
			VoidHand	srcResH;
			LocalID resChunkID;
			
			DmResourceInfo(dbP, i, &resType, &resID, &resChunkID);

			if (resChunkID) {
				DWord	srcP;
				srcResH = DmGetResourceIndex(dbP, i);
				if (!srcResH) {
					Errors::ThrowIfPalmError(-1);
					}
				resSize = MemHandleSize(srcResH);

				StMemory	outP (resSize);
				srcP = (DWord)MemHandleLock(srcResH);
				uae_memcpy((void*) outP.Get (), srcP, resSize);
				MemPtrUnlock((Ptr)srcP);
				fh.Write (resSize, outP.Get ());

				DmReleaseResource(srcResH);
				}
			}

		} // Resource database
		
		

	//------------------------------------------------------------
	// Write out a records database
	//------------------------------------------------------------
	else {
		UInt		attr;
		ULong		uniqueID;
		VoidHand	srcH;
		
		numRecords = DmNumRecords(dbP);
		
		size = sizeof(DatabaseHdrType) + numRecords * sizeof(RecordEntryType);
		StMemory	outP(size);
		
		// Fill in header
		hdrP = (DatabaseHdrPtr)outP.Get ();
		
		strcpy((char*)hdrP->name, nameP);
		hdrP->attributes = attributes;
		hdrP->version = version;
		hdrP->creationDate = crDate;
		hdrP->modificationDate = modDate;
		hdrP->lastBackupDate = bckUpDate;
		hdrP->modificationNumber = modNum;
		hdrP->appInfoID = 0;
		hdrP->sortInfoID = 0;
		hdrP->type = type;
		hdrP->creator = creator;
	
		hdrP->recordList.nextRecordListID = 0;
		hdrP->recordList.numRecords = numRecords;
		
		
		// Get the size of the appInfo and sort Info if they exist
		offset = size;
		VoidHand	appInfoH=0, sortInfoH=0;
		ULong		appInfoSize=0, sortInfoSize=0;
		if (appInfoID) {
			hdrP->appInfoID = offset;
			appInfoH = (VoidHand)MemLocalIDToGlobal(appInfoID, cardNo);
			if (!appInfoH) {
				Errors::ThrowIfPalmError(-1);
				}
			appInfoSize = MemHandleSize(appInfoH);
			offset += appInfoSize;
			}
			
		if (sortInfoID) {
			hdrP->sortInfoID = offset;
			sortInfoH = (VoidHand)MemLocalIDToGlobal(sortInfoID, cardNo);
			if (!sortInfoH) {
				Errors::ThrowIfPalmError(-1);
				}
			sortInfoSize = MemHandleSize(sortInfoH);
			offset += sortInfoSize;
			}			
	
	
		// Fill in the info on each resource into the header
		RecordEntryPtr	entryP = (RecordEntryPtr)(&hdrP->recordList.firstEntry);
		for (i=0; i<numRecords; i++)  {
			
			err = DmRecordInfo(dbP, i, &attr, &uniqueID, 0);
			if (err) {
				Errors::ThrowIfPalmError(-1);
				}
			
			srcH = DmQueryRecord(dbP, i);

			entryP->localChunkID = offset;
			entryP->attributes = attr;
			entryP->uniqueID[0] = (uniqueID >> 16) & 0x00FF;	// vmk 10/16/95 fixed: && 0x00FF --> & 0x00FF
			entryP->uniqueID[1] = (uniqueID >> 8) & 0x00FF;
			entryP->uniqueID[2] = uniqueID	& 0x00FF;
			
			if (srcH)
				offset += MemHandleSize(srcH);
			entryP++;
			}

		// Byteswap the resource entry headers
		entryP	= (RecordEntryPtr)(&hdrP->recordList.firstEntry);
		for (i=0; i<hdrP->recordList.numRecords; i++)
		{
			Canonical(*entryP);
			entryP++;
		}
		Canonical(*hdrP);	// Now in big-endian format
	
		// Write out  entry table
		fh.Write (size, outP.Get());

		// Write out the appInfo followed by sortInfo, if they exist
		if (appInfoID && appInfoSize) {
			DWord		srcP;
			StMemory	outP(appInfoSize);
			srcP = (DWord)MemHandleLock(appInfoH);
			uae_memcpy((void*) outP.Get(), srcP, appInfoSize);
			MemPtrUnlock((Ptr)srcP);
			fh.Write (appInfoSize, outP.Get());
			}
			
		if (sortInfoID && sortInfoSize) {
			DWord		srcP;
			StMemory	outP(sortInfoSize);
			srcP = (DWord)MemHandleLock(sortInfoH);
			uae_memcpy((void*) outP.Get(), srcP, sortInfoSize);
			MemPtrUnlock((Ptr)srcP);
			fh.Write (sortInfoSize, outP.Get());
			}
	
	
	
		// Write out each record
		for (i=0; i<numRecords; i++) {
			ULong		recSize;
			
			err = DmRecordInfo(dbP, i, &attr, &uniqueID, 0);
			if (err) {
				Errors::ThrowIfPalmError(-1);
				}
				
			srcH = DmQueryRecord(dbP, i);
				
			if (srcH) {
				DWord	srcP;
				
				recSize = MemHandleSize(srcH);
				StMemory	outP(recSize);
				srcP = (DWord)MemHandleLock(srcH);
				uae_memcpy((void*) outP.Get(), srcP, recSize);
				MemPtrUnlock((Ptr)srcP);
				fh.Write (recSize, outP.Get());
				}
			}
	
			
		}
		
	// Clean up
	::DmCloseDatabase(dbP);
}
catch (...)
{
	if (dbP)
		::DmCloseDatabase (dbP);

	throw;
}
}


// ---------------------------------------------------------------------------
//		 SavePalmFile
// ---------------------------------------------------------------------------
// Saves a Palm OS program or database file to a file.

void SavePalmFile (FileHandle& appFile, UInt cardNo, const char* databaseName)
{
	My_ShlExportAsPilotFile (appFile, cardNo, databaseName);
}



Bool PrvLauncherActive (void)
{
	if (!Patches::UIInitialized ())
		return false;

	EmuAppInfo	info = Patches::GetRootAppInfo ();

	ULong	type;
	ULong	creator;
	Err 	err = ::DmDatabaseInfo (info.fCardNo, info.fDBID,
									NULL,		// const CharPtr nameP,
									NULL,		// UIntPtr attributesP,
									NULL,		// UIntPtr versionP,
									NULL,		// ULongPtr crDateP,
									NULL,		// ULongPtr modDateP,
									NULL,		// ULongPtr bckUpDateP,
									NULL,		// ULongPtr modNumP,
									NULL,		// LocalID* appInfoIDP,
									NULL,		// LocalID* sortInfoIDP,
									&type,		// ULongPtr typeP,
									&creator);	// ULongPtr creatorP)

	if (type == sysFileTApplication && creator == sysFileCLauncher)
		return true;

	return false;
}


Bool PrvSwitchToApp (UInt32 /*type*/, UInt32 /*creator*/)
{
#if 0
	Err 				err;
	DmSearchStateType	searchInfo;
	UInt16				cardNo;
	LocalID 			dbID;

	err = ::DmGetNextDatabaseByTypeCreator (true, &searchInfo,
			 type, creator, true, &cardNo, &dbID);

	if (err == errNone)
	{
		err = ::SysUIAppSwitch (cardNo, dbID, sysAppLaunchCmdNormalLaunch, NULL);
	}

	if (err == errNone)
	{
		Patches::StartWatchingAppLaunch ();

		do {
			Emulator::ExecuteUntilATrap ();
		} while (!Patches::AppFinishedLaunching ());
	}

	return err == errNone;
#else
	return false;
#endif
}
