/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#ifndef _HOSTCONTROL_H_
#define _HOSTCONTROL_H_

#include <SysTraps.h>

#ifdef __cplusplus
extern "C" {
#endif


/*
	Set the base value for selectors. Note that this value MUST be
	two bytes long and have the high byte be non-zero. The reason
	for this has to do with the way SysGremlins was originally
	declared. It took a GremlinsSelector enumerated value. Originally,
	there was only one value, and it was zero. The way the 68K compiler
	works, it decides that GremlinsSelectors are only one byte long,
	so a call to SysGremlins would push one byte onto the stack. Because
	all values on the stack need to be word-aligned, the processor
	subtracts 1 from the stack before pushing on the byte. Therefore,
	the stack looks like:

			previous contents
			garbage byte
			selector
			return address

	With this setup, we have two choices: leave the selector size at
	one byte and limit ourselves to 256 functions, or define the selector
	to be a two byte value, with the first 256 values (all those with 0x00
	in the upper byte) to be GremlinIsOn. The latter sounds preferable, so
	we start the new selectors at 0x0100.
*/

#define hostSelectorBase	0x0100
#define hostSelectorChunk	0x0100
#define hostSection(x)		(hostSelectorBase + (hostSelectorChunk * (x)))

typedef enum
{
	// Host information selectors

	hostSelectorGetHostVersion = hostSection(0),
	hostSelectorGetHostID,
	hostSelectorGetHostPlatform,
	hostSelectorIsSelectorImplemented,
	hostSelectorGestalt,
	hostSelectorIsCallingTrap,

	// Profiler selectors

	hostSelectorProfileInit = hostSection(1),
	hostSelectorProfileStart,
	hostSelectorProfileStop,
	hostSelectorProfileDump,
	hostSelectorProfileCleanup,
	hostSelectorProfileDetailFn,

	// Std C Library wrapper selectors

	hostSelectorErrNo = hostSection(2),

	hostSelectorFClose,
	hostSelectorFEOF,
	hostSelectorFError,
	hostSelectorFFlush,
	hostSelectorFGetC,
	hostSelectorFGetPos,
	hostSelectorFGetS,
	hostSelectorFOpen,
	hostSelectorFPrintF,		// Floating point not yet supported in Poser
	hostSelectorFPutC,
	hostSelectorFPutS,
	hostSelectorFRead,
	hostSelectorRemove,			// Not yet implemented in Poser
	hostSelectorRename,			// Not yet implemented in Poser
	hostSelectorFReopen,		// Not yet implemented in Poser
	hostSelectorFScanF,			// Not yet implemented
	hostSelectorFSeek,
	hostSelectorFSetPos,
	hostSelectorFTell,
	hostSelectorFWrite,
	hostSelectorTmpFile,
	hostSelectorTmpNam,			// Not yet implemented in Poser
	hostSelectorGetEnv,

	hostSelectorMalloc,			// Not yet implemented in Poser
	hostSelectorRealloc,		// Not yet implemented in Poser
	hostSelectorFree,			// Not yet implemented in Poser
	
	// Gremlin selectors

	hostSelectorGremlinIsRunning = hostSection(3),
	hostSelectorGremlinNumber,
	hostSelectorGremlinCounter,
	hostSelectorGremlinLimit,
	hostSelectorGremlinNew,

	// Database selectors

	hostSelectorImportFile = hostSection(4),
	hostSelectorExportFile,

	// Preferences selectors

	hostSelectorGetPreference = hostSection(5),		// Not yet implemented
	hostSelectorSetPreference,						// Not yet implemented

	// Logging selectors

	hostSelectorLogFile = hostSection(6),
	hostSelectorSetLogFileSize,

	// RPC selectors

	hostSelectorSessionCreate = hostSection(7),
	hostSelectorSessionOpen,
	hostSelectorSessionClose,
	hostSelectorSessionQuit,
	hostSelectorSignalSend,
	hostSelectorSignalWait,
	hostSelectorSignalResume,

	// External tracing tool support

	hostSelectorTraceInit = hostSection(8),
	hostSelectorTraceClose,
	hostSelectorTraceOutputT,
	hostSelectorTraceOutputTL,
	hostSelectorTraceOutputVT,
	hostSelectorTraceOutputVTL,
	hostSelectorTraceOutputB,

	hostSelectorLastTrapNumber
} HostControlTrapNumber;


struct HostFILE
{
	long	_field;
};

typedef struct HostFILE HostFILE;


typedef long	HostBool;
typedef long	HostErr;
typedef long	HostID;
typedef long	HostPlatform;
typedef long	HostSignal;

#ifndef hostErrorClass
	#define	hostErrorClass				0x1C00	// Host Control Manager
#else
	#if hostErrorClass != 0x1C00
		#error "You cannot change hostErrorClass without telling us."
	#endif
#endif

enum	// HostErr values
{
	hostErrNone = 0,

	hostErrBase = hostErrorClass,

	hostErrUnknownGestaltSelector,
	hostErrDiskError,
	hostErrOutOfMemory,
	hostErrMemReadOutOfRange,
	hostErrMemWriteOutOfRange,
	hostErrMemInvalidPtr,
	hostErrInvalidParameter,
	hostErrTimeout,
	hostErrInvalidDeviceType,
	hostErrInvalidRAMSize,
	hostErrFileNotFound,
	hostErrRPCCall,				// Issued if the following functions are not called remotely:
								//		HostSessionCreate
								//		HostSessionOpen
								//		HostSessionClose
								//		HostSessionQuit
								//		HostWaitForSignal
								//		HostResumeFromSignal
								//		HostMemRead
								//		HostMemWrite
	hostErrSessionRunning,		// Issued by HostSessionCreate, HostSessionOpen, and
								// HostSessionQuit if a session is running.
	hostErrSessionNotRunning,	// Issued by HostSessionClose if no session is running.
	hostErrNoSignalWaiters,		// Issued by HostSendSignal if no one's waiting for a signal.
	hostErrSessionNotPaused		// Issued when HostResumeFromSignal, but the session was not
								// halted from a HostSendSignal call.
};


enum	// HostID values
{
	hostIDPalmOS,			// The plastic thingy
	hostIDPalmOSEmulator,	// The Copilot thingy
	hostIDPalmOSSimulator	// The Mac libraries you link with thingy
};


enum	// HostPlatform values
{
	hostPlatformPalmOS,
	hostPlatformWindows,
	hostPlatformMacintosh,
	hostPlatformUnix
};

enum	// HostSignal values
{
	hostSignalReserved,
	hostSignalIdle,
	hostSignalQuit,
#if 0
	// (Proposed...not supported yet)
	hostSignalSessionStarted,
	hostSignalSessionStopped,
	hostSignalHordeStarted,
	hostSignalGremlinStarted,
	hostSignalGremlinSuspended,
	hostSignalGremlinResumed,
	hostSignalGremlinStopped,
	hostSignalHordeStopped,
#endif
	hostSignalUser	= 0x40000000	// User-defined values start here and go up.
};


// Use these to call FtrGet to see if you're running under the
// Palm OS Emulator.  If not, FtrGet will return ftrErrNoSuchFeature.

#define kPalmOSEmulatorFeatureCreator	('pose')
#define kPalmOSEmulatorFeatureNumber	(0)


struct HostGremlinInfo
{
	long		fFirstGremlin;
	long		fLastGremlin;
	long		fSaveFrequency;
	long		fSwitchDepth;
	long		fMaxDepth;
	char		fAppNames[200];	// Comma-seperated list of application names
								// to run Gremlins on.  If the string is empty,
								// all applications are fair game.  If the string
								// begins with a '-' (e.g., "-Address,Datebook"),
								// then all applications named in the list are
								// excluded instead of included.
};

typedef struct HostGremlinInfo HostGremlinInfo;


// Define this in case SysTraps.h doesn't have it.

#define sysTrapHostControl (sysTrapBase + 836)

#define HOST_TRAP(selector)													\
	FIVEWORD_INLINE(														\
		0x3F3C, selector,					/* MOVE.W #selector, -(A7)	*/	\
		m68kTrapInstr + sysDispatchTrapNum,	/* TRAP $F					*/	\
		sysTrapHostControl,					/* sysTrapHostControl		*/	\
		0x544F)								/* ADD.Q #2, A7				*/ 


/* ==================================================================== */
/* Host environment-related calls										*/
/* ==================================================================== */

long			HostGetHostVersion(void)
						HOST_TRAP(hostSelectorGetHostVersion);

HostID			HostGetHostID(void)
						HOST_TRAP(hostSelectorGetHostID);

HostPlatform	HostGetHostPlatform(void)
						HOST_TRAP(hostSelectorGetHostPlatform);

HostBool		HostIsSelectorImplemented(long selector)
						HOST_TRAP(hostSelectorIsSelectorImplemented);

HostErr			HostGestalt(long gestSel, long* response)
						HOST_TRAP(hostSelectorGestalt);

HostBool		HostIsCallingTrap(void)
						HOST_TRAP(hostSelectorIsCallingTrap);


/* ==================================================================== */
/* Profiling-related calls												*/
/* ==================================================================== */

HostErr			HostProfileInit(long maxCalls, long maxDepth)
						HOST_TRAP(hostSelectorProfileInit);

HostErr			HostProfileDetailFn(void* addr, HostBool logDetails)
						HOST_TRAP(hostSelectorProfileDetailFn);

HostErr			HostProfileStart(void)
						HOST_TRAP(hostSelectorProfileStart);

HostErr			HostProfileStop(void)
						HOST_TRAP(hostSelectorProfileStop);

HostErr			HostProfileDump(const char* filename)
						HOST_TRAP(hostSelectorProfileDump);

HostErr			HostProfileCleanup(void)
						HOST_TRAP(hostSelectorProfileCleanup);


/* ==================================================================== */
/* Std C Library-related calls											*/
/* ==================================================================== */

long			HostErrNo(void)
						HOST_TRAP(hostSelectorErrNo);


long			HostFClose(HostFILE* f)
						HOST_TRAP(hostSelectorFClose);

long			HostFEOF(HostFILE* f)
						HOST_TRAP(hostSelectorFEOF);

long			HostFError(HostFILE* f)
						HOST_TRAP(hostSelectorFError);

long			HostFFlush(HostFILE* f)
						HOST_TRAP(hostSelectorFFlush);

long			HostFGetC(HostFILE* f)
						HOST_TRAP(hostSelectorFGetC);

long			HostFGetPos(HostFILE* f, long* posP)
						HOST_TRAP(hostSelectorFGetPos);

char*			HostFGetS(char* s, long n, HostFILE* f)
						HOST_TRAP(hostSelectorFGetS);

HostFILE*		HostFOpen(const char* name, const char* mode)
						HOST_TRAP(hostSelectorFOpen);

long			HostFPrintF(HostFILE* f, const char* fmt, ...)
						HOST_TRAP(hostSelectorFPrintF);

long			HostFPutC(long c, HostFILE* f)
						HOST_TRAP(hostSelectorFPutC);

long			HostFPutS(const char* s, HostFILE* f)
						HOST_TRAP(hostSelectorFPutS);

long			HostFRead(void* buffer, long size, long count, HostFILE* f)
						HOST_TRAP(hostSelectorFRead);

long			HostRemove(const char* name)
						HOST_TRAP(hostSelectorRemove);

long			HostRename(const char* oldName, const char* newName)
						HOST_TRAP(hostSelectorRename);

HostFILE*		HostFReopen(const char* name, const char* mode, HostFILE *f)
						HOST_TRAP(hostSelectorFReopen);

long			HostFScanF(HostFILE* f, const char *fmt, ...)
						HOST_TRAP(hostSelectorFScanF);

long			HostFSeek(HostFILE* f, long offset, long origin)
						HOST_TRAP(hostSelectorFSeek);

long			HostFSetPos(HostFILE* f, long* pos)
						HOST_TRAP(hostSelectorFSetPos);

long			HostFTell(HostFILE* f)
						HOST_TRAP(hostSelectorFTell);

long			HostFWrite(const void* buffer, long size, long count, HostFILE* f)
						HOST_TRAP(hostSelectorFWrite);

HostFILE*		HostTmpFile(void)
						HOST_TRAP(hostSelectorTmpFile);

char*			HostTmpNam(char *name)
						HOST_TRAP(hostSelectorTmpNam);

char*			HostGetEnv(const char*)
						HOST_TRAP(hostSelectorGetEnv);


void*			HostMalloc(long size)
						HOST_TRAP(hostSelectorMalloc);

void*			HostRealloc(void* p, long size)
						HOST_TRAP(hostSelectorRealloc);

void			HostFree(void* p)
						HOST_TRAP(hostSelectorFree);


/* ==================================================================== */
/* Gremlin-related calls												*/
/* ==================================================================== */

HostBool		HostGremlinIsRunning(void)
						HOST_TRAP(hostSelectorGremlinIsRunning);

long			HostGremlinNumber(void)
						HOST_TRAP(hostSelectorGremlinNumber);

long			HostGremlinCounter(void)
						HOST_TRAP(hostSelectorGremlinCounter);

long			HostGremlinLimit(void)
						HOST_TRAP(hostSelectorGremlinLimit);

HostErr			HostGremlinNew(const HostGremlinInfo*)
						HOST_TRAP(hostSelectorGremlinNew);


/* ==================================================================== */
/* Import/export-related calls											*/
/* ==================================================================== */

HostErr			HostImportFile(const char* fileName, long cardNum)
						HOST_TRAP(hostSelectorImportFile);

HostErr			HostExportFile(const char* fileName, long cardNum, const char* dbName)
						HOST_TRAP(hostSelectorExportFile);


/* ==================================================================== */
/* Preference-related calls												*/
/* ==================================================================== */

HostBool		HostGetPreference(const char*, char*)
						HOST_TRAP(hostSelectorGetPreference);

void			HostSetPreference(const char*, const char*)
						HOST_TRAP(hostSelectorSetPreference);


/* ==================================================================== */
/* Logging-related calls												*/
/* ==================================================================== */

HostFILE*		HostLogFile(void)
						HOST_TRAP(hostSelectorLogFile);

void			HostSetLogFileSize(long)
						HOST_TRAP(hostSelectorSetLogFileSize);


/* ==================================================================== */
/* RPC-related calls													*/
/* ==================================================================== */

HostErr			HostSessionCreate(const char* device, long ramSize, const char* romPath)
						HOST_TRAP(hostSelectorSessionCreate);

HostErr			HostSessionOpen(const char* psfFileName)
						HOST_TRAP(hostSelectorSessionOpen);

HostErr			HostSessionClose(const char* saveFileName)
						HOST_TRAP(hostSelectorSessionClose);

HostErr			HostSessionQuit(void)
						HOST_TRAP(hostSelectorSessionQuit);

HostErr			HostSignalSend(HostSignal signalNumber)
						HOST_TRAP(hostSelectorSignalSend);

HostErr			HostSignalWait(long timeout, HostSignal* signalNumber)
						HOST_TRAP(hostSelectorSignalWait);

HostErr			HostSignalResume(void)
						HOST_TRAP(hostSelectorSignalResume);

/* ==================================================================== */
/* Tracing calls														*/
/* ==================================================================== */


void			HostTraceInit(void)
						HOST_TRAP(hostSelectorTraceInit);

void			HostTraceClose(void)
						HOST_TRAP(hostSelectorTraceClose);

void			HostTraceOutputT(unsigned short, const char*, ...)
						HOST_TRAP(hostSelectorTraceOutputT);

void			HostTraceOutputTL(unsigned short, const char*, ...)
						HOST_TRAP(hostSelectorTraceOutputTL);

void			HostTraceOutputVT(unsigned short, const char*, char* /*va_list*/)
						HOST_TRAP(hostSelectorTraceOutputVT);

void			HostTraceOutputVTL(unsigned short, const char*, char* /*va_list*/)
						HOST_TRAP(hostSelectorTraceOutputVTL);

void			HostTraceOutputB(unsigned short, const unsigned char*, unsigned long/*size_t*/)
						HOST_TRAP(hostSelectorTraceOutputB);


#ifdef __cplusplus 
}
#endif

#endif /* _HOSTCONTROL_H_ */
