/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#ifndef _EMULATORTYPES_H_
#define _EMULATORTYPES_H_

#include "Platform_Files.h"		// FileReference

#include <deque>
#include <map>
#include <string>
#include <vector>


// ======================================
// ========== ENUMERATED TYPES ==========
// ======================================

// ---------- DeviceType ----------

// -*- NEW DEVICE -*-

enum DeviceType 
{
	kDeviceUnspecified	= 0,
	kDevicePilot1000,
	kDevicePilot5000,
	kDevicePalmPilotPersonal,
	kDevicePalmPilotProfessional,
	kDevicePalmPilotWithUpgrade,
	kDevicePalmIII,
	kDevicePalmVII,
	kDeviceUnused1,
	kDeviceQualComm,
	kDevicePalmV,
	kDevicePalmIIIx,
	kDeviceUnused2,
	kDeviceAustin,
	kDevicePalmVIIEZ,
	kDeviceLast
};
typedef vector<DeviceType>	DeviceList;


// ---------- <Less of a type, more of a constant...> ----------

enum
{
	MRU_COUNT = 9
};

// ---------- CloseActionType ----------

enum CloseActionType
{
	kSaveAlways,
	kSaveAsk,
	kSaveNever
};
typedef vector<CloseActionType>	CloseActionList;


// ---------- LoggingType ----------

enum LoggingType
{
	kNormalLogging = 0x01,
	kGremlinLogging = 0x02,
	kAllLogging = kNormalLogging | kGremlinLogging,
	kCurrentLogging = 0x80
};
typedef vector<LoggingType>	LoggingList;


// =====================================
// ========== SIMPLE TYPEDEFS ==========
// =====================================

typedef uae_u8	ScaleType;
typedef uae_s32	RAMSizeType;

typedef vector<ScaleType>		ScaleList;
typedef vector<RAMSizeType>		RAMSizeList;

typedef deque<FileReference>	FileRefList;	// Need to push_front at times.
typedef vector<unsigned char>	ByteList;
typedef vector<string>			StringList;
typedef vector<RectangleType>	RectangleList;
typedef vector<PointType>		PointLess;

typedef map<string, string>		StringStringMap;


// ================================
// ========== STRUCTURES ==========
// ================================

// ---------- RGBType ----------

struct RGBType
{
	RGBType (void) {}
	RGBType (uae_u8 red, uae_u8 green, uae_u8 blue) :
		fRed (red),
		fGreen (green),
		fBlue (blue)
	{}

	bool operator== (const RGBType& rhs) const
	{
		return fRed == rhs.fRed && fGreen == rhs.fGreen && fBlue == rhs.fBlue;
	}

	bool operator!= (const RGBType& rhs) const
	{
		return !(*this == rhs);
	}

	uae_u8		fRed;
	uae_u8		fGreen;
	uae_u8		fBlue;
};
typedef vector<RGBType>	RGBList;


// ---------- SystemCallContext ----------

struct SystemCallContext
{
	uaecptr		fPC;			// PC at which the system call was made.
	uaecptr		fNextPC;		// PC of instruction after system call.
	uaecptr		fDestPC;		// Address of system function.
	uae_u16		fTrapWord;		// Trapword, even for SYSTRAP_FAST calls.
	uae_u16		fTrapIndex;		// Trapword with the high 4 bits removed.
	uaecptr		fExtra;			// RefNum for library calls, D2 for "dispatch" calls.
	Bool		fViaTrap;		// True if called via TRAP $F.
	Bool		fViaJsrA1;		// True if called via SYSTRAP_FASTER.
};
typedef vector<SystemCallContext>	SystemCallContextList;


// ---------- DatabaseInfo ----------

struct DatabaseInfo
{
	ULong		creator;
	ULong		type;
	UInt		version;
	LocalID		dbID;
	UInt 		cardNo;
	ULong		modDate;
	Word		dbAttrs;

	char		name[dmDBNameLength];
	char		dbName[dmDBNameLength];

	Bool operator < (const DatabaseInfo& other) const
		{ if (creator < other.creator) return true;
		  if (creator > other.creator) return false;
		  if (type < other.type) return true;
		  return false; }

	Bool operator == (const DatabaseInfo& other) const
		{ return (creator == other.creator) &&
				 (type == other.type); }
};
typedef vector<DatabaseInfo>	DatabaseInfoList;


// ---------- Configuration ----------

struct Configuration
{
	Configuration () :
		fDeviceType (kDeviceUnspecified),
		fRAMSize (1024),
		fROMFile ()
	{
	}

	Configuration (DeviceType dt, long size, const FileReference& rom) :
		fDeviceType (dt),
		fRAMSize (size),
		fROMFile (rom)
	{
	}

	DeviceType				fDeviceType;
	RAMSizeType				fRAMSize;
	FileReference			fROMFile;
};
typedef vector<Configuration>	ConfigurationList;


// ---------- GremlinInfo ----------

struct GremlinInfo
{
	GremlinInfo () :
		fNumber (0),
		fSteps (-1),
		fSaveFrequency (10000),
		fAppList ()
	{
	}

	long				fNumber;
	long				fSteps;
	long				fSaveFrequency;
	DatabaseInfoList	fAppList;
};
typedef vector<GremlinInfo>	GremlinInfoList;


// ---------- HordeInfo ----------

struct HordeInfo
{
	HordeInfo () :
		fStartNumber (0),
		fStopNumber (0),
		fSaveFrequency (10000),
		fSwitchDepth (30000),
		fMaxDepth (1000000),
		fAppList ()
	{
	}

	long				fStartNumber;
	long				fStopNumber;
	long				fSaveFrequency;
	long				fSwitchDepth;
	long				fMaxDepth;
	DatabaseInfoList	fAppList;
};
typedef vector<HordeInfo>	HordeInfoList;


// ----------------------------------------------------------------------
// For reasons of "convenience", the official type declarations for these
// packets omit their bodies.  For the emulator, we'd really like to have
// the bodies, so we define these parallel types.
// ----------------------------------------------------------------------

#include "PalmPack.h"

struct SysPktReadMemRspType2
{
	_sysPktBodyCommon;
	Byte			data[sysPktMaxBodySize-2];
};

struct SysPktReadRegsRspType2
{
	_sysPktBodyCommon;								// Common Body header
	M68KRegsType	reg;							// <- return registers
};

struct SysPktGetBreakpointsRspType2
{
	_sysPktBodyCommon;								// Common Body header
	BreakpointType	bp[dbgTotalBreakpoints];
};

struct SysPktSetBreakpointsCmdType2
{
	_sysPktBodyCommon;								// Common Body header
	BreakpointType	bp[dbgTotalBreakpoints];
};

struct SysPktGetTrapBreaksRspType2
{
	_sysPktBodyCommon;								// Common Body header
	Word			trapBP[dbgTotalTrapBreaks];
};

struct SysPktSetTrapBreaksCmdType2
{
	_sysPktBodyCommon;								// Common Body header
	Word			trapBP[dbgTotalTrapBreaks];
};

struct SysPktGetTrapConditionsRspType2
{
	_sysPktBodyCommon;								// Common Body header
	Word			trapParam[dbgTotalTrapBreaks];
};

struct SysPktSetTrapConditionsCmdType2
{
	_sysPktBodyCommon;								// Common Body header
	Word			trapParam[dbgTotalTrapBreaks];
};

struct SysPktRPCType2
{
	_sysPktBodyCommon;					// Common Body header
	Word	trapWord;					// which trap to execute
	DWord	resultD0;					// result from D0 placed here
	DWord	resultA0;					// result from A0 placed here
	Word	numParams;					// how many parameters follow
	// Following is a variable length array of SlkRPCParamInfo's
//	SysPktRPCParamType	param[1];
	char	param[sysPktMaxBodySize];
};

struct SysPktEmptyRspType
{
	_sysPktBodyCommon;			// Common Body header
};



struct SysPktRPC2Type
{
	_sysPktBodyCommon;				// Common Body header

	UInt16		trapWord;			// which trap to execute
	UInt32		resultD0;			// result from D0 placed here
	UInt32		resultA0;			// result from A0 placed here
	UInt16		resultException;

	Byte		DRegMask;
	Byte		ARegMask;
	UInt32		Regs[1];

	UInt16		numParams;		   // how many parameters follow
//	SysPktRPCParamType	param[1];
	char	param[sysPktMaxBodySize];
};

#include "PalmPackPop.h"


// ============================================
// ========== Windows socket mapping ==========
// ============================================

#if !defined (_WINDOWS)
	typedef int				SOCKET;

	#define WSAEINVAL		EINVAL
	#define INVALID_SOCKET	((SOCKET) (~0))
	#define SOCKET_ERROR	(-1)
	#define closesocket		close
	#define ioctlsocket		ioctl

	#ifndef INADDR_LOOPBACK
	#define INADDR_LOOPBACK	0x7f000001
	#endif
#endif

#if UNIX
	#include <sys/socket.h>			// socklen_t
#else
	typedef int			socklen_t;
#endif


#endif	// _EMULATORTYPES_H_
