/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "Bank_ROM.h"

#include "Byteswapping.h"		// Canonical
#include "CPU_REG.h"			// Software::BusError
#include "Crc.h"				// Crc16CalcBigBlock
#include "PreferenceMgr.h"		// Preference
#include "RAM_ROM.h"			// Memory::InitializeBanks
#include "SessionFile.h"		// WriteROMFileReference
#include "Strings.r.h"			// kStr_BadChecksum
#include "UAE_Utils.h"			// uae_memset


// Private function declarations


class Card
{
	public:
		static Bool 	CheckCardHeader 		(const CardHeaderType& cardHdr);
		static Bool 	CheckChecksum			(const void* p, uae_u32 romLength);
		static Bool 	Supports328 			(const void* p);
		static Bool 	SupportsEZ				(const void* p);
};


// ===========================================================================
//		 ROM Bank Accessors
// ===========================================================================
// These functions provide fetch and store access to the emulator's read only
// memory.

static AddressBank	gROMAddressBank =
{
	ROMBank::GetLong,
	ROMBank::GetWord,
	ROMBank::GetByte,
	ROMBank::SetLong,
	ROMBank::SetWord,
	ROMBank::SetByte,
	ROMBank::GetRealAddress,
	ROMBank::ValidAddress,
	ROMBank::GetMetaAddress,
	ROMBank::AddOpcodeCycles
};

static uae_u32	gROMBank_Size;
static uae_u32	gManagedROMSize;
static uae_u32	gROMImage_Size;
static uae_u32	gROMBank_Mask;
static uae_u8*	gROM_Memory;
static uae_u8*	gROM_MetaMemory;


/***********************************************************************
 *
 * FUNCTION:	ROMBank::Initialize
 *
 * DESCRIPTION: Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void ROMBank::Initialize (StreamHandle& hROM)
{
	ROMBank::LoadROM (hROM);
	ROMBank::SetBankHandlers ();
}

/***********************************************************************
 *
 * FUNCTION:	ROMBank::Reset
 *
 * DESCRIPTION: Standard reset function.  Sets the sub-system to a
 *				default state.	This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void ROMBank::Reset (void)
{
	memset (gROM_MetaMemory, 0, gROMImage_Size);
}


/***********************************************************************
 *
 * FUNCTION:	ROMBank::Save
 *
 * DESCRIPTION: Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void ROMBank::Save (SessionFile& f)
{
	Preference<Configuration>	cfg (kPrefKeyLastConfiguration);
	f.WriteROMFileReference (cfg->fROMFile);

	StWordSwapper	swapper1 (gROM_MetaMemory, gROMImage_Size);
	f.WriteMetaROMImage (gROM_MetaMemory, gROMImage_Size);
}


/***********************************************************************
 *
 * FUNCTION:	ROMBank::Load
 *
 * DESCRIPTION: Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void ROMBank::Load (SessionFile& f)
{
	assert (gROM_MetaMemory != NULL);

	if (f.ReadMetaROMImage (gROM_MetaMemory))
	{
		ByteswapWords (gROM_MetaMemory, gROMImage_Size);
	}
	else
	{
		f.SetCanReload (false);
	}
}


/***********************************************************************
 *
 * FUNCTION:	ROMBank::Dispose
 *
 * DESCRIPTION: Standard dispose function.	Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void ROMBank::Dispose (void)
{
	Platform::DisposeMemory (gROM_Memory);
	Platform::DisposeMemory (gROM_MetaMemory);
}


/***********************************************************************
 *
 * FUNCTION:    ROMBank::SetBankHandlers
 *
 * DESCRIPTION: Set the bank handlers UAE uses to dispatch memory
 *				access operations.
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void ROMBank::SetBankHandlers (void)
{
	gManagedROMSize = HWRegisters::GetROMSize ();

	// Memory banks 0x10C0 to <mumble> are managed by the functions
	// in ROMBank.	<mumble> is based on the amount of ROM being emulated.

	uae_u32	first_bank	= bankindex (kROMMemoryStart);
	uae_u32	last_bank	= bankindex (kROMMemoryStart + gManagedROMSize - 1);

	Memory::InitializeBanks (	gROMAddressBank, first_bank,
								last_bank - first_bank + 1);
}


// ---------------------------------------------------------------------------
//		 ROMBank::GetLong
// ---------------------------------------------------------------------------

uae_u32 ROMBank::GetLong (uaecptr iAddress)
{
#if PROFILE_MEMORY
	gMemoryAccess[kROMLongRead]++;
	if (iAddress & 2)
		gMemoryAccess[kROMLongRead2]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

#if (VALIDATE_ROM_GET)
	if (gMemAccessFlags.fValidate_ROMGet && !ValidAddress (iAddress, sizeof (long)))
	{
		InvalidAccess (iAddress, sizeof (long), true);
	}
#endif

#if (PREVENT_USER_ROM_GET || PREVENT_SYSTEM_ROM_GET)
	if (IsPCInRAM ())
	{
		if (PREVENT_USER_ROM_GET && gMemAccessFlags.fProtect_ROMGet)
		{
			InvalidAccess (iAddress, sizeof (long), true);
		}
	}
	else
	{
		if (PREVENT_SYSTEM_ROM_GET && gMemAccessFlags.fProtect_SysROMGet)
		{
			InvalidAccess (iAddress, sizeof (long), true);
		}
	}
#endif

#if HAS_PROFILING
	CYCLE_GETLONG (WAITSTATES_ROM);
#endif

	iAddress &= gROMBank_Mask;

	return do_get_mem_long (gROM_Memory + iAddress);
}


// ---------------------------------------------------------------------------
//		 ROMBank::GetWord
// ---------------------------------------------------------------------------

uae_u32 ROMBank::GetWord (uaecptr iAddress)
{
#if PROFILE_MEMORY
	gMemoryAccess[kROMWordRead]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

#if (VALIDATE_ROM_GET)
	if (gMemAccessFlags.fValidate_ROMGet && !ValidAddress (iAddress, sizeof (short)))
	{
		InvalidAccess (iAddress, sizeof (short), true);
	}
#endif

#if (PREVENT_USER_ROM_GET || PREVENT_SYSTEM_ROM_GET)
	if (IsPCInRAM ())
	{
		if (PREVENT_USER_ROM_GET && gMemAccessFlags.fProtect_ROMGet)
		{
			InvalidAccess (iAddress, sizeof (short), true);
		}
	}
	else
	{
		if (PREVENT_SYSTEM_ROM_GET && gMemAccessFlags.fProtect_SysROMGet)
		{
			InvalidAccess (iAddress, sizeof (short), true);
		}
	}
#endif

#if HAS_PROFILING
	CYCLE_GETWORD (WAITSTATES_ROM);
#endif

	iAddress &= gROMBank_Mask;

	return do_get_mem_word (gROM_Memory + iAddress);
}


// ---------------------------------------------------------------------------
//		 ROMBank::GetByte
// ---------------------------------------------------------------------------

uae_u32 ROMBank::GetByte (uaecptr iAddress)
{
#if PROFILE_MEMORY
	gMemoryAccess[kROMByteRead]++;
#endif

#if (VALIDATE_ROM_GET)
	if (gMemAccessFlags.fValidate_ROMGet && !ValidAddress (iAddress, sizeof (char)))
	{
		InvalidAccess (iAddress, sizeof (char), true);
	}
#endif

#if (PREVENT_USER_ROM_GET || PREVENT_SYSTEM_ROM_GET)
	if (IsPCInRAM ())
	{
		if (PREVENT_USER_ROM_GET && gMemAccessFlags.fProtect_ROMGet)
		{
			InvalidAccess (iAddress, sizeof (char), true);
		}
	}
	else
	{
		if (PREVENT_SYSTEM_ROM_GET && gMemAccessFlags.fProtect_SysROMGet)
		{
			InvalidAccess (iAddress, sizeof (char), true);
		}
	}
#endif

#if HAS_PROFILING
	CYCLE_GETBYTE (WAITSTATES_ROM);
#endif

	iAddress &= gROMBank_Mask;

	return do_get_mem_byte (gROM_Memory + iAddress);
}


// ---------------------------------------------------------------------------
//		 ROMBank::SetLong
// ---------------------------------------------------------------------------

void ROMBank::SetLong (uaecptr iAddress, uae_u32 iLongValue)
{
#if PROFILE_MEMORY
	gMemoryAccess[kROMLongWrite]++;
	if (iAddress & 2)
		gMemoryAccess[kROMLongWrite2]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

	assert (ValidAddress (iAddress, sizeof (long)));

#if (PREVENT_USER_ROM_SET || PREVENT_SYSTEM_ROM_SET)
	if (IsPCInRAM ())
	{
		if (PREVENT_USER_ROM_SET && gMemAccessFlags.fProtect_ROMSet)
		{
			InvalidAccess (iAddress, sizeof (long), false);
		}
	}
	else
	{
		if (PREVENT_SYSTEM_ROM_SET && gMemAccessFlags.fProtect_SysROMSet)
		{
			InvalidAccess (iAddress, sizeof (long), false);
		}
	}
#endif

#if HAS_PROFILING
	// writing to ROM?? don't know what to make of this
	CYCLE_PUTLONG (WAITSTATES_ROM);
#endif

	iAddress &= gROMBank_Mask;

	do_put_mem_long (gROM_Memory + iAddress, iLongValue);
}


// ---------------------------------------------------------------------------
//		 ROMBank::SetWord
// ---------------------------------------------------------------------------

void ROMBank::SetWord (uaecptr iAddress, uae_u32 iWordValue)
{
#if PROFILE_MEMORY
	gMemoryAccess[kROMWordWrite]++;
#endif

#if (CHECK_FOR_ADDRESS_ERROR)
	if ((iAddress & 1) != 0)
	{
		Software::AddressError ();
	}
#endif

	assert (ValidAddress (iAddress, sizeof (short)));

#if (PREVENT_USER_ROM_SET || PREVENT_SYSTEM_ROM_SET)
	if (IsPCInRAM ())
	{
		if (PREVENT_USER_ROM_SET && gMemAccessFlags.fProtect_ROMSet)
		{
			InvalidAccess (iAddress, sizeof (short), false);
		}
	}
	else
	{
		if (PREVENT_SYSTEM_ROM_SET && gMemAccessFlags.fProtect_SysROMSet)
		{
			InvalidAccess (iAddress, sizeof (short), false);
		}
	}
#endif

#if HAS_PROFILING
	// writing to ROM?? don't know what to make of this
	CYCLE_PUTWORD (WAITSTATES_ROM);
#endif

	iAddress &= gROMBank_Mask;

	do_put_mem_word (gROM_Memory + iAddress, iWordValue);
}


// ---------------------------------------------------------------------------
//		 ROMBank::SetByte
// ---------------------------------------------------------------------------

void ROMBank::SetByte (uaecptr iAddress, uae_u32 iByteValue)
{
#if PROFILE_MEMORY
	gMemoryAccess[kROMByteWrite]++;
#endif

	assert (ValidAddress (iAddress, sizeof (char)));
	
#if (PREVENT_USER_ROM_SET || PREVENT_SYSTEM_ROM_SET)
	if (IsPCInRAM ())
	{
		if (PREVENT_USER_ROM_SET && gMemAccessFlags.fProtect_ROMSet)
		{
			InvalidAccess (iAddress, sizeof (char), false);
		}
	}
	else
	{
		if (PREVENT_SYSTEM_ROM_SET && gMemAccessFlags.fProtect_SysROMSet)
		{
			InvalidAccess (iAddress, sizeof (char), false);
		}
	}
#endif

#if HAS_PROFILING
	// writing to ROM?? don't know what to make of this
	CYCLE_PUTBYTE (WAITSTATES_ROM);
#endif

	iAddress &= gROMBank_Mask;

	do_put_mem_byte (gROM_Memory + iAddress, iByteValue);
}


// ---------------------------------------------------------------------------
//		 ROMBank::ValidAddress
// ---------------------------------------------------------------------------

int ROMBank::ValidAddress (uaecptr iAddress, uae_u32 iSize)
{
	iAddress -= kROMMemoryStart;
	int	result = (iAddress + iSize) <= (kROMMemoryStart + gROMBank_Size);

//	assert (result);

	return result;
}


// ---------------------------------------------------------------------------
//		 ROMBank::GetRealAddress
// ---------------------------------------------------------------------------

uae_u8* ROMBank::GetRealAddress (uaecptr iAddress)
{
	// Strip the uppermost bit of the address.

	iAddress &= gROMBank_Mask;

	return (uae_u8*) &gROM_Memory[iAddress];
}


// ---------------------------------------------------------------------------
//		 ROMBank::GetMetaAddress
// ---------------------------------------------------------------------------

uae_u8* ROMBank::GetMetaAddress (uaecptr iAddress)
{
	// Strip the uppermost bit of the address.

	iAddress &= gROMBank_Mask;

	return (uae_u8*) &(gROM_MetaMemory[iAddress]);
}


// ---------------------------------------------------------------------------
//		 ROMBank::AddOpcodeCycles
// ---------------------------------------------------------------------------

void ROMBank::AddOpcodeCycles (void)
{
#if (HAS_PROFILING)
	CYCLE_GETWORD (WAITSTATES_ROM);
#endif
}


// ---------------------------------------------------------------------------
//		 ROMBank::InvalidAccess
// ---------------------------------------------------------------------------

void ROMBank::InvalidAccess (uaecptr iAddress, long size, Bool forRead)
{
	int button = Errors::ReportWroteToROM (iAddress, size, forRead);
	Emulator::HandleDlgButton (button, m68k_getpc ());
}


/***********************************************************************
 *
 * FUNCTION:	ROMBank::LoadROM
 *
 * DESCRIPTION: Does the work of actually loading (and validating a ROM
 *				from a stream.
 *
 * PARAMETERS:	StreamHandle hROM
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void ROMBank::LoadROM (StreamHandle& hROM)
{
	// Make sure the file is big enough to have a card header.

	if (hROM.GetLength() < sizeof (CardHeaderType))
		Errors::Throw (kError_BadROM);

	// Read the card header.

	CardHeaderType	cardHeader;
	hROM.Read(sizeof (CardHeaderType), &cardHeader);
	Canonical (cardHeader);

	// Validate the card header.

	if (!Card::CheckCardHeader (cardHeader))
		Errors::Throw (kError_BadROM);

	// The ROM is made up of two parts: a Big ROM and a Small ROM.	ROM image
	// files can contain both parts or just the Big ROM part.  If it contains
	// both parts, then everything is just fine, and we can use it no problem.
	// If the Small ROM part is missing, we have to detect that, dummy up a
	// small ROM for correct operation, and relocate the Big ROM higher in memory.

	// Both the Big ROM and Small ROM start off with a CardHeaderType struct.
	// We just read in one of them.  Let's look at where the Big ROM _should_
	// be to see if there's another CardHeader there.  If so, we have both parts.
	// If not, we have just the Big ROM.

	uae_u32 bigROMOffset;

	if (cardHeader.hdrVersion == 1)
	{
		bigROMOffset = 0x3000;
	}
	else
	{
		// Version 2 CardHeaders contain a "bigROMOffset" field that contains the
		// offset from the beginning of the card to where the ROM should
		// appear in memory.  Normally the card appears at 0x10000000 (which
		// would make bigROMOffset something like 0x00C08000), but that
		// isn't always the case.  The problem is, we don't always know
		// where the card beginning will be.  For now, we'll assume that
		// the ROM wants to live at 0x10C00000, and that the Big ROM will
		// want to begin somewhere soon after that.  This means we can get by
		// with using only the lower bits of bigROMOffset.

		bigROMOffset = cardHeader.bigROMOffset;
		bigROMOffset &= 0x000FFFFF; 	// Allows for 1 Meg offset.
	}

	// Make sure the file is big enough to have a Big ROM.

	if (hROM.GetLength() < bigROMOffset)
		Errors::Throw (kError_BadROM);

	// We'll use bigROMOffset for later to get to some Big ROM data.
	// But we'll also need a value when we read the ROM image into
	// memory.	This value will be used as an offset into a buffer
	// where we'll be reading the ROM image.  If we have a Small ROM,
	// the offset will be zero.  If we have only a Big ROM, the offset
	// will be the same value as bigROMOffset.

	uae_u32 bufferOffset = bigROMOffset;

	// See if there's a card header there, too.

	CardHeaderType	cardHeader2;
	hROM.SetPos (bigROMOffset, kSeekSet);
	hROM.Read (sizeof (CardHeaderType), &cardHeader2);
	Canonical (cardHeader2);

	if (Card::CheckCardHeader (cardHeader2))
	{
		// Looks like, so we don't have to relocate the ROM image.

		bufferOffset = 0;
	}

	// The ROM size is now the size of the file plus any offset for
	// any small ROM that we have to add and dummy up.

	gROMImage_Size	= hROM.GetLength() + bufferOffset;
	gROMBank_Size	= ::NextPowerOf2 (gROMImage_Size);

	// Read in the ROM image.

	StMemory	romImage (gROMImage_Size);
	StMemory	romMetaImage (gROMImage_Size);

	hROM.SetPos (0, kSeekSet);
	hROM.Read (hROM.GetLength(), romImage.Get () + bufferOffset);

	// If we only had a Big ROM, dummy up the Small ROM.  All we really
	// need to do here is copy the Big ROM's card header to the Small
	// ROM area.
	//
	// Also, clear out this area to 0xFF to look like new Flash RAM.

	if (bufferOffset)
	{
		memset (romImage, 0xFF, bigROMOffset);

		*(CardHeaderType*) romImage.Get () = *(CardHeaderType*) (romImage.Get () + bigROMOffset);
	}

	// See if the checksum looks OK.

	Card::CheckChecksum (romImage.Get () + bigROMOffset, gROMImage_Size - bigROMOffset);

	// Check that the ROM we just loaded can be run on this device.

	if (Card::SupportsEZ (romImage.Get ()))
	{
		if (!Emulator::EZMode ())
		{
			Errors::Throw (kError_WrongROMForType);
		}
	}
	else
	{
		if (Emulator::EZMode ())
		{
			Errors::Throw (kError_WrongROMForType);
		}
	}

	// Byteswap all the words in the ROM (if necessary). Most accesses
	// are 16-bit accesses, so we optimize for that case.

	ByteswapWords (romImage.Get (), gROMImage_Size);

	// Everything seems to be OK.  Save the ROM data in some global
	// variables for the CPU emulator to access.  Make sure that
	// gROMBank_Size is a power-of-2.  The ROMBank memory routines
	// require this.

	gROM_Memory 	= (uae_u8*) romImage.Release ();
	gROM_MetaMemory = (uae_u8*) romMetaImage.Release ();
	gROMBank_Mask	= gROMBank_Size - 1;
}


#pragma mark -

// ===========================================================================
//		 Flash Bank Accessors
// ===========================================================================
// These functions provide fetch and store access to the emulator's read only
// memory.

/*
	There are five types of flash that our flash manager routines identify:
	Mitsubishi, Hitachi, Intel, AMD, and Fujitsu.  Of these, only the last
	two are really supported right now.  Unfortunately, they are the hardest
	to emulate.  :-(

	To identify the kind of flash being used, our ROM routines:

		- Read a word from FLASHBASE
		- Write 0x00FF to FLASHBASE
		- Read a word from FLASHBASE
		- Write 0x0090 to FLASHBASE
		- Read a manufacturer ID (word) from FLASHBASE
		- Read a device ID (word) from FLASHBASE + 2
		- Write 0x00FF to FLASHBASE

	Mitsubishi: manufacturer == 0x001C, device == 0x005E
	Hitachi:	manufacturer == 0x0007, device == 0x0086
	Intel:		manufacturer == 0x0089, device == 0x0091

	If the flash is not one of those, our ROM routines:

		- Read a word from FLASHBASE
		- Write 0x00F0 to FLASHBASE
		- Write 0x00AA to FLASHBASE + 0xAAAA
		- Write 0x0055 to FLASHBASE + 0x5554
		- Write 0x0090 to FLASHBASE + 0xAAAA
		- Read a manufacturer ID (word) from FLASHBASE
		- Read a device ID (word) from FLASHBASE + 2
		- Write 0x00F0 to FLASHBASE

	AMD:		manufacturer == 0x0001, device == 0x0049
	Fujitsu:	manufacturer == 0x0004, device == 0x0049


	To erase a word of flash, our ROM routines:

		AMD, Fujitsu:

		- Read a word from FLASHBASE
		- Write 0x00F0 to FLASHBASE
		- Write 0x00AA to FLASHBASE + 0xAAAA
		- Write 0x0055 to FLASHBASE + 0x5554
		- Write 0x0080 to FLASHBASE + 0xAAAA
		- Write 0x00AA to FLASHBASE + 0xAAAA
		- Write 0x0055 to FLASHBASE + 0x5554
		- Write 0x0030 to location to be erased
		- Check erase location for 0x0080
		- Read from FLASHBASE
		- Write 0x00F0 to FLASHBASE

		Mitsubishi, Hitachi:

		- Read a word from FLASHBASE
		- Write 0x00FF to FLASHBASE
		- Write 0x0020 to FLASHBASE
		- Write 0x00D0 to location to be erased
		- Check erase location for 0x0080
			-- If 0x0020 is also set, an error occured
		- Read from FLASHBASE
		- Write 0x00FF to FLASHBASE

		Intel

		- Not supported

	To program a block of flash:

		AMD, Fujitsu:

		- Read a word from FLASHBASE
		- Write 0x00F0 to FLASHBASE
		- For each word to write
			- If the word is already there, continue
			- Write 0x00AA to FLASHBASE + 0xAAAA
			- Write 0x0055 to FLASHBASE + 0x5554
			- Write 0x00A0 to FLASHBASE + 0xAAAA
			- Write the word to the dest location
			- Check write location for 0x0080
		- Read from FLASHBASE
		- Write 0x00F0 to FLASHBASE

		Mitsubishi, Hitachi, Intel:

		- Not supported
*/


enum
{
	kAMDState_Normal,		// Read-only mode. Acts like a normal ROM.
							// Almost any write of 0x00F0 to any address will get us here.
							// Also enter here automatically after Erase or Program operation.
							// A write of 0x00AA to 0xAAAA will put us in kAMDState_Unlocked1.

	kAMDState_Unlocked1,	// After first unlock cycle (0x00AA written to 0xAAAA)
							// Looking for the second unlock cycle (0x0055 written to 0x5554);
							// If we find one, go to kAMDState_Unlocked2.
							// ??? What happens on other operations?

	kAMDState_Unlocked2,	// After second unlock cycle (0x0055 written to 0x5554)
							// Now looking for a command to be written to 0xAAAA
							// If we find 0x0090, go to kAMDState_Autoselect.
							// If we find 0x0080, set fgEraseIsSetup and go to kAMDState_Normal. ??? When should fgEraseIsSetup get cleared?
							// If we find 0x0030, if fgEraseIsSetup erase the sector and go to kAMDState_EraseDone.
							// If we fine 0x00A0, go to kAMDState_Program.
							// ??? What happens on other operations?

	kAMDState_Autoselect,	// After a write of 0x0090 to 0x5555.
							// A read of 0x0000 returns 0x0001 (manufacturer ID).
							// A read of 0x0002 returns 0x0049 (device ID).
							// A write of 0x00F0 to any address returns us to kAMDState_Normal.
							// ??? What happens on other operations?

	kAMDState_Program,		// After a program sequence was entered.
							// Accept any write operation and go to kAMDState_ProgramDone.
							// ??? What happens on other operations?

	kAMDState_EraseDone,	// After a Erase operation.
							// On the next read, return successful operation and return to kAMDState_Normal
							// ??? What happens on other operations?

	kAMDState_ProgramDone,	// After a Program operation.
							// On the next read, return successful operation and return to kAMDState_Normal
							// ??? What happens on other operations?
	kAMDState_Dummy
};


static AddressBank	gFlashAddressBank =
{
	FlashBank::GetLong,
	FlashBank::GetWord,
	FlashBank::GetByte,
	FlashBank::SetLong,
	FlashBank::SetWord,
	FlashBank::SetByte,
	FlashBank::GetRealAddress,
	FlashBank::ValidAddress,
	FlashBank::GetMetaAddress,
	FlashBank::AddOpcodeCycles
};

#define FLASHBASE	kROMMemoryStart

static int		gState = kAMDState_Normal;
static Bool 	gEraseIsSetup;


/***********************************************************************
 *
 * FUNCTION:	FlashBank::Initialize
 *
 * DESCRIPTION: Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void FlashBank::Initialize (void)
{
	// Memory banks 0x10C0 to <mumble> are managed by the functions
	// in FlashBank.  <mumble> is based on the amount of ROM being emulated.

	Memory::InitializeBanks (	gFlashAddressBank,
								bankindex (kROMMemoryStart),
								1);
}


/***********************************************************************
 *
 * FUNCTION:	FlashBank::Reset
 *
 * DESCRIPTION: Standard reset function.  Sets the sub-system to a
 *				default state.	This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void FlashBank::Reset (void)
{
	gState = kAMDState_Normal;
}


/***********************************************************************
 *
 * FUNCTION:	FlashBank::Save
 *
 * DESCRIPTION: Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void FlashBank::Save (SessionFile&)
{
	// !!! Save gState?
}


/***********************************************************************
 *
 * FUNCTION:	FlashBank::Load
 *
 * DESCRIPTION: Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void FlashBank::Load (SessionFile&)
{
	// !!! Load gState?
}


/***********************************************************************
 *
 * FUNCTION:	FlashBank::Dispose
 *
 * DESCRIPTION: Standard dispose function.	Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void FlashBank::Dispose (void)
{
}


// ---------------------------------------------------------------------------
//		 FlashBank::GetLong
// ---------------------------------------------------------------------------

uae_u32 FlashBank::GetLong (uaecptr iAddress)
{
	return ROMBank::GetLong (iAddress);
}


// ---------------------------------------------------------------------------
//		 FlashBank::GetWord
// ---------------------------------------------------------------------------

uae_u32 FlashBank::GetWord (uaecptr iAddress)
{
	int		orgState = gState;

	switch (gState)
	{
		case kAMDState_Normal:
			// Read-only mode. Acts like a normal ROM.
			// Almost any write of 0x00F0 to any address will get us here.
			// Also enter here automatically after Erase or Program operation.
			// A write of 0x00AA to 0xAAAA will put us in kAMDState_Unlocked1.

			break;

		case kAMDState_Unlocked1:
			// After first unlock cycle (0x00AA written to 0xAAAA)
			// Looking for the second unlock cycle (0x0055 written to 0x5554);
			// If we find one, go to kAMDState_Unlocked2.
			// ??? What happens on other operations?

			break;

		case kAMDState_Unlocked2:
			// After second unlock cycle (0x0055 written to 0x5554)
			// Now looking for a command to be written to 0xAAAA
			// If we find 0x0090, go to kAMDState_Autoselect.
			// If we find 0x0080, set fgEraseIsSetup and go to kAMDState_Normal. ??? When should fgEraseIsSetup get cleared?
			// If we find 0x0030, if fgEraseIsSetup erase the sector and go to kAMDState_ProgramDone.
			// If we fine 0x00A0, go to kAMDState_Program.
			// ??? What happens on other operations?

			break;

		case kAMDState_Autoselect:
			// After a write of 0x0090 to 0x5555.
			// A read of 0x0000 returns 0x0001 (manufacturer ID).
			// A read of 0x0002 returns 0x0049 (device ID).
			// A write of 0x00F0 to any address returns us to kAMDState_Normal.
			// ??? What happens on other operations?

			if (iAddress == FLASHBASE)
			{
				return 0x0001;
			}
			else if (iAddress == FLASHBASE + 2)
			{
				return 0x0049;
			}
			break;

		case kAMDState_Program:
			// After a program sequence was entered.
			// Accept any write operation and go to kAMDState_ProgramDone.
			// ??? What happens on other operations?

			break;

		case kAMDState_EraseDone:
			// After a Program or Erase operation.
			// On the next read, return successful operation and return to kAMDState_Normal
			// ??? What happens on other operations?

			// !!! We should really check that iAddress is the same as the
			// one specified in the Erase command.

			gState = kAMDState_Normal;
			return 0x0080;

		case kAMDState_ProgramDone:
			// After a Program or Erase operation.
			// On the next read, return successful operation and return to kAMDState_Normal
			// ??? What happens on other operations?

			// !!! We should really check that iAddress is the same as the
			// one specified in the Program command.

			gState = kAMDState_Normal;
			return 0x0080 & ROMBank::GetWord (iAddress);
	}

	return ROMBank::GetWord (iAddress);
}


// ---------------------------------------------------------------------------
//		 FlashBank::GetByte
// ---------------------------------------------------------------------------

uae_u32 FlashBank::GetByte (uaecptr iAddress)
{
	return ROMBank::GetByte (iAddress);
}


// ---------------------------------------------------------------------------
//		 FlashBank::SetLong
// ---------------------------------------------------------------------------

void FlashBank::SetLong (uaecptr iAddress, uae_u32 iLongValue)
{
	ROMBank::SetLong (iAddress, iLongValue);
}


// ---------------------------------------------------------------------------
//		 FlashBank::SetWord
// ---------------------------------------------------------------------------

void FlashBank::SetWord (uaecptr iAddress, uae_u32 iWordValue)
{
	int		orgState = gState;

	switch (gState)
	{
		case kAMDState_Normal:
			// Read-only mode. Acts like a normal ROM.
			// Almost any write of 0x00F0 to any address will get us here.
			// Also enter here automatically after Erase or Program operation.
			// A write of 0x00AA to 0xAAAA will put us in kAMDState_Unlocked1.

			if (iAddress == FLASHBASE + 0xAAAA && iWordValue == 0x00AA)
			{
				gState = kAMDState_Unlocked1;
				return;
			}

			// Allow these.  PrvIdentifyFlashCode probes with these values
			// to determine what Flash part we've got.

			if (iAddress == FLASHBASE && (iWordValue == 0x00FF || iWordValue == 0x00F0 || iWordValue == 0x0090))
			{
				return;
			}
			break;

		case kAMDState_Unlocked1:
			// After first unlock cycle (0x00AA written to 0xAAAA)
			// Looking for the second unlock cycle (0x0055 written to 0x5554);
			// If we find one, go to kAMDState_Unlocked2.
			// ??? What happens on other operations?

			if (iWordValue == 0x00F0)
			{
				gState = kAMDState_Normal;
				return;
			}
			else if (iAddress == FLASHBASE + 0x5554 && iWordValue == 0x0055)
			{
				gState = kAMDState_Unlocked2;
				return;
			}
			break;

		case kAMDState_Unlocked2:
			// After second unlock cycle (0x0055 written to 0x5554)
			// Now looking for a command to be written to 0xAAAA
			// If we find 0x0090, go to kAMDState_Autoselect.
			// If we find 0x0080, set fgEraseIsSetup and go to kAMDState_Normal. ??? When should fgEraseIsSetup get cleared?
			// If we find 0x0030, if fgEraseIsSetup erase the sector and go to kAMDState_EraseDone.
			// If we fine 0x00A0, go to kAMDState_Program.
			// ??? What happens on other operations?

			if (iWordValue == 0x00F0)
			{
				gState = kAMDState_Normal;
				return;
			}
			else if (iWordValue == 0x0030 && gEraseIsSetup)
			{
				const int	kEraseValue 		= 0xFF;
#if 0
				DWord	readWriteParmsOffset	= get_long (0x10C00000 + offsetof (CardHeaderType, readWriteParmsOffset));
				DWord	readWriteParmsSize		= get_long (0x10C00000 + offsetof (CardHeaderType, readWriteParmsSize));
				DWord	readOnlyParmsOffset 	= get_long (0x10C00000 + offsetof (CardHeaderType, readOnlyParmsOffset));
				DWord	readOnlyParmsSize		= 0x00002000;
				DWord	readWriteWorkingOffset	= get_long (0x10C00000 + offsetof (CardHeaderType, readWriteWorkingOffset));
				DWord	readWriteWorkingSize	= get_long (0x10C00000 + offsetof (CardHeaderType, readWriteWorkingSize));

				if (iAddress >= 0x10000000 + readWriteParmsOffset &&
					iAddress < 0x10000000 + readWriteParmsOffset + readWriteParmsSize)
				{
					uae_memset (0x10000000 + readWriteParmsOffset, kEraseValue, readWriteParmsSize);
				}
				else if (iAddress >= 0x10000000 + readOnlyParmsOffset &&
					iAddress < 0x10000000 + readOnlyParmsOffset + readOnlyParmsSize)
				{
					uae_memset (0x10000000 + readOnlyParmsOffset, kEraseValue, readOnlyParmsSize);
				}
				else if (iAddress >= 0x10000000 + readWriteWorkingOffset &&
					iAddress < 0x10000000 + readWriteWorkingOffset + readWriteWorkingSize)
				{
					uae_memset (0x10000000 + readWriteWorkingOffset, kEraseValue, readWriteWorkingSize);
				}
#endif

				const unsigned long kSector1Start	= 0x10C00000;
				const unsigned long kSector1Size	= 0x00004000;
				const unsigned long kSector2Start	= 0x10C04000;
				const unsigned long kSector2Size	= 0x00002000;
				const unsigned long kSector3Start	= 0x10C06000;
				const unsigned long kSector3Size	= 0x00002000;
				const unsigned long kSector4Start	= 0x10C08000;
				const unsigned long kSector4Size	= 0x00008000;

				CEnableFullAccess	munge;

				if (iAddress == kSector1Start)
				{
					uae_memset (kSector1Start, kEraseValue, kSector1Size);
				}
				else if (iAddress == kSector2Start)
				{
					uae_memset (kSector2Start, kEraseValue, kSector2Size);
				}
				else if (iAddress == kSector3Start)
				{
					uae_memset (kSector3Start, kEraseValue, kSector3Size);
				}
				else if (iAddress == kSector4Start)
				{
					uae_memset (kSector4Start, kEraseValue, kSector4Size);
				}

				gState = kAMDState_EraseDone;
				return;
			}
			else if (iAddress == FLASHBASE + 0xAAAA)
			{
				if (iWordValue == 0x0090)
				{
					gState = kAMDState_Autoselect;
					return;
				}
				else if (iWordValue == 0x0080)
				{
					gEraseIsSetup = true;
					gState = kAMDState_Normal;
					return;
				}
				else if (iWordValue == 0x00A0)
				{
					gState = kAMDState_Program;
					return;
				}
			}
			break;

		case kAMDState_Autoselect:
			// After a write of 0x0090 to 0x5555.
			// A read of 0x0000 returns 0x0001 (manufacturer ID).
			// A read of 0x0002 returns 0x0049 (device ID).
			// A write of 0x00F0 to any address returns us to kAMDState_Normal.
			// ??? What happens on other operations?

			if (iWordValue == 0x00F0)
			{
				gState = kAMDState_Normal;
				return;
			}
			break;

		case kAMDState_Program:
			// After a program sequence was entered.
			// Accept any write operation and go to kAMDState_ProgramDone.
			// ??? What happens on other operations?

			iAddress &= gROMBank_Mask;
			do_put_mem_word (gROM_Memory + iAddress, iWordValue);

			gState = kAMDState_ProgramDone;
			return;

		case kAMDState_EraseDone:
			// After a Program or Erase operation.
			// On the next read, return successful operation and return to kAMDState_Normal
			// ??? What happens on other operations?

			return;

		case kAMDState_ProgramDone:
			// After a Program or Erase operation.
			// On the next read, return successful operation and return to kAMDState_Normal
			// ??? What happens on other operations?

			return;
	}

	ROMBank::SetWord (iAddress, iWordValue);
}


// ---------------------------------------------------------------------------
//		 FlashBank::SetByte
// ---------------------------------------------------------------------------

void FlashBank::SetByte (uaecptr iAddress, uae_u32 iByteValue)
{
	ROMBank::SetByte (iAddress, iByteValue);
}


// ---------------------------------------------------------------------------
//		 FlashBank::ValidAddress
// ---------------------------------------------------------------------------

int FlashBank::ValidAddress (uaecptr iAddress, uae_u32 iSize)
{
	return ROMBank::ValidAddress (iAddress, iSize);
}


// ---------------------------------------------------------------------------
//		 FlashBank::GetRealAddress
// ---------------------------------------------------------------------------

uae_u8* FlashBank::GetRealAddress (uaecptr iAddress)
{
	// Strip the uppermost bit of the address.

	iAddress &= gROMBank_Mask;

	return (uae_u8*) &gROM_Memory[iAddress];
}


// ---------------------------------------------------------------------------
//		 FlashBank::GetMetaAddress
// ---------------------------------------------------------------------------

uae_u8* FlashBank::GetMetaAddress (uaecptr iAddress)
{
	// Strip the uppermost bit of the address.

	iAddress &= gROMBank_Mask;

	return (uae_u8*) &(gROM_MetaMemory[iAddress]);
}


// ---------------------------------------------------------------------------
//		 FlashBank::AddOpcodeCycles
// ---------------------------------------------------------------------------

void FlashBank::AddOpcodeCycles (void)
{
#if (HAS_PROFILING)
	CYCLE_GETWORD (WAITSTATES_ROM);
#endif
}


#pragma mark -

// ===========================================================================
//		 Card
// ===========================================================================

// ---------------------------------------------------------------------------
//		 Card::CheckCardHeader
// ---------------------------------------------------------------------------
// Determines if the memory pointed to looks like a card header.

Bool Card::CheckCardHeader (const CardHeaderType& cardHdr)
{
	// Make sure the stack size isn't something ludicrous.

	if (cardHdr.initStack == 0 || cardHdr.initStack > 0x000FFFFF)	// 1 Meg
		return false;

	// Make sure the signature is right.

	if (cardHdr.signature != sysCardSignature)
		return false;

	// Make sure the header version isn't ludicrous.

	if (cardHdr.hdrVersion == 0 || cardHdr.hdrVersion > 255)	// arbitrary value
		return false;

	// Make sure this isn't a "RAM only" card.

	if (cardHdr.flags & memCardHeaderFlagRAMOnly)
		return false;

	return true;
}


// ---------------------------------------------------------------------------
//		 Card::CheckChecksum
// ---------------------------------------------------------------------------
// Determines if the memory pointed to looks like a card header.

Bool Card::CheckChecksum (const void* p, uae_u32 fileLength)
{
	CardHeaderType	cardHdr = *(CardHeaderType*) p;

	Canonical (cardHdr);

	if (cardHdr.hdrVersion == 1)
	{
		static const Word	kDebug20 = 0x9BED;

#if 0
		static const Word	kROMChecksums[] =
		{
			0x67C8, 	// 1.0 - English - v30 ROMs - ROM5Rel - 1/9/96
			0x83FC, 	// 1.0 - French - v10 ROMs - ROM5Rel - 3/28/96
			0xE1D9, 	// 1.0 - French - v11 ROMs - ROM5Rel - 4/4/96
			0x41BE, 	// 1.0 - French - v13 ROMs - ROM5Rel - 4/11/96
			0xC2AE, 	// 1.0 - German - v10 ROMs - ROM5Rel - 4/1/96
			0x1FCE, 	// 1.0 - German - v12 ROMs - ROM5Rel - 4/10/96
			0x3D33, 	// 1.0 - German - v13 ROMs - ROM5Rel - 4/10/96

			kDebug20,	// 2.0 - Debug - 12/23/97
			0x3EA3, 	// 2.0 - ROMA - ROM5Rel - 1/28/97
			0xEFB6, 	// 2.0 - ROMB - English - ROM.2.0B.GM.US - 2/18/97
			0x11A9, 	// 2.0 - ROMB - WorkPad - ROMf2wProFatRel - 7/7/97
			0x9573, 	// 2.0 - International ROMA - French - ROMf2fRel - 4/29/97
			0x2422, 	// 2.0 - International ROMA - German - ROMf2gRel - 3/7/97
			0x0BFC, 	// 2.0 - International ROMB - French - ROMf1fProRel - 5/28/97
			0xFCBB, 	// 2.0 - International ROMB - German - ROMf2gProRel - 5/29/97
			0x2E10		// 2.0 - International ROMB - Spanish - ROMf1sFatRel - 11/18/97
		};
#endif

		Word	checksumValue = Crc16CalcBigBlock ((/*non-const*/ void*) p, fileLength, 0);

		// Special hack for 2.0 Debug ROMs.  The version with this checksum was
		// bogus, so let's make sure we never load it.

		if (checksumValue == kDebug20)
		{
			// Throw a special error for this one.

			Errors::Throw (kError_UnsupportedROM);
		}

#if 0
		// Can't do this, yet, as the above set of checksums may not match
		// what's actually in shipping devices (I only know that they match
		// what we have on our servers around here).

		int ii;
		for (int ii = 0; ii < (sizeof (kROMChecksums) / sizeof (kROMChecksums[0])); ++ii)
		{
			if (kROMChecksums[ii] == checksumValue)
			{
				return true;
			}
		}
#else
		return true;
#endif
	}
	else
	{
		// The checksum is the cumulative checksum of the ROM image before
		// the stored checksum value and the ROM image following the checksum
		// value.  First, calculate the first part.

		DWord	chunkSize = offsetof (CardHeaderType, checksumValue);
		Word	checksumValue = Crc16CalcBigBlock ((/*non-const*/ void*) p, chunkSize, 0);

		// Now calculate the second part.

		checksumValue = Crc16CalcBigBlock (
			((char*) p) + chunkSize + sizeof (cardHdr.checksumValue),
			cardHdr.checksumBytes - chunkSize - sizeof (cardHdr.checksumValue),
			checksumValue);

		if (checksumValue == cardHdr.checksumValue)
		{
			return true;
		}
	}

	Platform::CommonDialog (NULL, Platform::GetString (kStr_BadChecksum).c_str (), Errors::kOK);

	return false;
}


// ---------------------------------------------------------------------------
//		 Card::Supports328
// ---------------------------------------------------------------------------

Bool Card::Supports328 (const void* p)
{
	CardHeaderType	cardHdr = *(CardHeaderType*) p;
//	ByteswapWords (&cardHdr, sizeof (CardHeaderType));
	Canonical (cardHdr);

	Bool	dbMode = false;

	if (cardHdr.hdrVersion < 3 || (cardHdr.flags & memCardHeaderFlag328))
	{
		dbMode = true;
	}

	return dbMode;
}


// ---------------------------------------------------------------------------
//		 Card::SupportsEZ
// ---------------------------------------------------------------------------

Bool Card::SupportsEZ (const void* p)
{
	CardHeaderType	cardHdr = *(CardHeaderType*) p;
//	ByteswapWords (&cardHdr, sizeof (CardHeaderType));
	Canonical (cardHdr);

	Bool	ezMode = false;

	if (cardHdr.hdrVersion >= 3 && (cardHdr.flags & memCardHeaderFlagEZ))
	{
		ezMode = true;
	}

	return ezMode;
}


