/* 
 * Apple // emulator for Linux: 
 *   Functions for memory-access indirection table and
 *   low-level framebuffer output.
 *
 * Copyright 1994 Alexander Jean-Claude Bottema
 * Copyright 1995 Stephen Lee
 * Copyright 1997, 1998 Aaron Culliney
 * Copyright 1998, 1999 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 2 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define __ASSEMBLY__
#include <linux/linkage.h>
#include "apple2.h"


	/* --------------------------------------------------------------------
	   ....................................................................
	    Apple ][+ Memory documentation (//e differs mostly in softswitches)
	  
	    I use two jump tables; one for get_memory and one for set_memory
	    respectively. The jump tables contain exactly 64k entries to
            avoid unnecessary range checking. Furthermore, special access
	    or to serve memory mapped I/O services are easily obtained by
	    patching the jump table with various routines.
	  
	    Apple 64k address space:
	  
	    (Two byte addresses are represented with least significant
	     byte first, e.g. address FA59 is represented as 59 FA)
	  
	    Address		Description
	  
	    0000 - 00FF   	Zero page RAM
	    0100 - 01FF		Stack
	    0200 - 03EF		RAM
	    03F0 - 03F1		Address for BRK instruction
	                        (normally 59 FA = FA59)
	    03F2 - 03F3         Soft entry vector (normally BF 9D = 9DBF)
	    03F4                Power-up byte
	    03F5 - 03F7         Jump instruction to the subroutine which
	                        handles Applesoft II &-commands
	   			(normally 4C 58 FF = JMP FF58)
	    03F8 - 03FA		Jump instruction to the subroutine which
	                        handles User CTRL-Y commands
	                        (normally 4C 65 FF = JMP FF65)
	    03FB - 03FD		Jump instruction to the subroutine which
	                        handles Non-Maskable Interrupts (NMI)
	                        (normally 4C 65 FF = JMP FF65)
	    03FE - 03FF         Address of the subroutine which handles
	                        Interrupt ReQuests (IRQ)
	                        (normally 65 FF = FF65)
	    0400 - 07FF         Basically primary video text page
	    0478 - 047F		I/O Scratchpad RAM Addresses for Slot 0 - 7
	    04F8 - 04FF		               - " " -
	    0578 - 057F		               - " " -
	    05F8 - 05FF		               - " " -
	    0678 - 067F		               - " " -
	    06F8 - 06FF		               - " " -
	    0778 - 077F		               - " " -
	    07F8 - 07FF		               - " " -
	    ----- These two addresses are pretty strange; the values
	          provided were inconsistent at least on my Apple II
	    05F8		Holds the slot number of the disk
	  			controller card from which DOS was
	                        booted.
	    07F8		Holds the slot number (CX, X = Slot #)
	                        of the slot that is currently active.
	  
	    0800 - 0BFF		Basically secondary video text page
	  
	    0C00 - 1FFF		Plain RAM
	    2000 - 3FFF		Primary video hires page (RAM)
	    4000 - 5FFF         Secondary video Hi-Res page (RAM)
	    6000 - BFFF		Plain RAM
	                        (Normally the operating system is
	                         loaded into ~= 9C00 - BFFF)
	    C000 - C00F		Keyboard data
	                        (C00X contains the character ASCII
	                         code of the pressed key. The
	                         value is available at any C00X
	                         address)
	    C010 - C01F         Clear Keyboard strobe
	    C020 - C02F         Cassette output toggle
	    C030 - C03F         Speaker toggle
	    C040 - C04F         Utility strobe
	    C050		Set graphics mode
	    C051		Set text mode
	    C052		Set all text or graphics
	    C053		Mix text and graphics
	    C054		Display primary page
	    C055		Display secondary page
	    C056		Display low-res graphics
	    C057		Display hi-res graphics
	    C058 - C05F		Annunciator outputs
	    C060		Cassette input
	    C061 - C063		Pushbutton inputs (button 1, 2 and 3)
	    C064 - C067		Game controller inputs
	    C068 - C06F		Same as C060 - C067
	    C070 - C07F		Game controller strobe
	    C080 - C08F		Slot 0 I/O space (usually a language card)
	                        --- If language card
	    C080		    Reset language card
	                            Read mode enabled
	  			    Write mode disabled
	                            Read from language card
	  			    Write to ROM (impossible of course)
	    C081		--- First access
	  			    Read mode disabled
	                            Read from ROM
	  			--- On second access
	                            Write mode enabled
	  			    Write to language card
	    C082		--- (Disable language card)
	                            Read mode disabled
	                            Write mode disabled
	                            Read from ROM
	                            Write to ROM
	    C083		--- First access
	                            Read mode enabled
	  			    Read from language card
	  		 	--- On second access
	  			    Write mode enabled
	  			    Write to language card
	    C088 - C08B		Same as C080 - C083, but
	  			switch to second bank, i.e.
	  			map addresses D000-DFFF to
	  			other 4k area.
	    C100 - C1FF		Slot 1 PROM
	    C200 - C2FF		Slot 2 PROM
	    C300 - C3FF		Slot 3 PROM
	    C400 - C4FF		Slot 4 PROM
	    C500 - C5FF		Slot 5 PROM
	    C600 - C6FF		Slot 6 PROM
	    C700 - C7FF		Slot 7 PROM
	    C800 - CFFF		Expansion ROM (for peripheral cards)
	    CFFF		Disable access to expansion ROM for
	  			ALL peripheral cards.
	    D000 - DFFF		ROM or 4k RAM if language card is
	  			enabled. However, there are TWO 4k
	  			banks that can be mapped onto addresses
	  			D000 - DFFF. See C088 - C08B.
	    E000 - FFFF		ROM or 8k RAM if language card is
	  			enabled.
	   ----------------------------------------------------------------- */


/* -------------------------------------------------------------------------
    Graphics routines.
    Care has been taken to isolate the dimension-dependent (320x200 vs 640x400)
    routines.
   ------------------------------------------------------------------------- */


/* -------------------------------------------------------------------------
 * Plot exatly 7 pixels from FROM to TO.
 * ecx: scratch
 * ------------------------------------------------------------------------- */
#define Plot7Pixels(FROM,TO)\
		movl	(FROM), %ecx;			/* long -> GM */      \
		movl	%ecx, (TO);				      \
		addl	$4, FROM;			/* inc pointers */    \
		addl	$4, TO;					      \
		movw	(FROM), %cx;			/* word -> GM */      \
		movw	%cx, (TO);				      \
		addl	$2, FROM;			/* inc pointers */    \
		addl	$2, TO;					      \
		movb	(FROM), %cl;			/* byte -> GM */      \
		movb	%cl, (TO);


#ifdef _640x400

#define LoadHiresTableRef(TABLE)\
		leal	SYMBOL_NAME(expanded_col_hires_##TABLE), %ebx;\
		shll	$4, %eax;/* *16 */\
		addl	%eax, %ebx;

/* -------------------------------------------------------------------------
 * Plot a normal swath of pixels.
 * For 640x400 this is 2 rows of 14 pixels.
 * ebx: from table
 * eax: to graphics memory
 * ------------------------------------------------------------------------- */
#define PlotPixels\
		PlotPixels640\
		subl	$12, %ebx;\
		addl	$SCANWIDTH-12, %eax;\
		PlotPixels640\
		subl	$SCANWIDTH, %eax;\
		addl	$1, %eax;
#define PlotPixels640\
		movl	(%ebx), %ecx;			/* long -> GM */      \
		movl	%ecx, (%eax);				      \
		addl	$4, %eax;			/* inc pointers */    \
		addl	$4, %ebx;					      \
		movl	(%ebx), %ecx;			/* long -> GM */      \
		movl	%ecx, (%eax);				      \
		addl	$4, %eax;			/* inc pointers */    \
		addl	$4, %ebx;					      \
		movl	(%ebx), %ecx;			/* long -> GM */      \
		movl	%ecx, (%eax);				      \
		addl	$4, %eax;			/* inc pointers */    \
		addl	$4, %ebx;					      \
		movw	(%ebx), %cx;			/* word -> GM */      \
		movw	%cx, (%eax);

/* -------------------------------------------------------------------------
 * Plot a dynamically interpolated swath of pixels.
 * For 640x400 this is 2 rows of 18 pixels.
 * ebx: from table
 * eax: to graphics memory
 * ecx: scratch
 * ------------------------------------------------------------------------- */
#define PlotPixelsExtra\
		subl	$2, %eax;\
		pushl	%edx;\
		xorl	%edx, %edx;\
		PlotPixelsExtraRow;\
		addl	$SCANWIDTH-18, %eax;\
		subl	$9, %ebx;\
		PlotPixelsExtraRow;\
		popl	%edx;
#define PlotPixelsExtraRow\
		movb	$9, %dl;\
	1:	movb	(%ebx), %cl;\
		movb	%cl, %ch;\
		movw	%cx, (%eax);\
		incl	%ebx;\
		addl	$2, %eax;\
		decb	%dl;\
		jnz	1b;


/* -------------------------------------------------------------------------
 * Plot an 80 column character row.  We can do this only in 640x400 resolution.
 * For 640x400 this is 2 rows of 7 pixels.
 * esi: from table
 * eax: to graphics memory
 * ------------------------------------------------------------------------- */
#define PlotCharacter80Row\
		Plot7Pixels(%esi,%eax);\
		addl	$ SCANWIDTH-6, %eax;/* Go to next row */\
		subl	$6, %esi;\
		Plot7Pixels(%esi,%eax);\

/* -------------------------------------------------------------------------
 * Plot a 40 column character row.
 * For 640x400 this is 2 rows of 14 pixels.
 * esi: from table
 * eax: to graphics memory
 * ------------------------------------------------------------------------- */
#define PlotCharacter40Row\
		PlotCharacter40Row640\
		subl	$12, %esi;\
		addl	$ SCANWIDTH-12, %eax;\
		PlotCharacter40Row640\
		addl	$2, %esi;
#define PlotCharacter40Row640\
		movl	(%esi), %ecx;	\
		movl	%ecx, (%eax);	\
		addl	$4, %esi;	\
		addl	$4, %eax;	\
		movl	(%esi), %ecx;	\
		movl	%ecx, (%eax);	\
		addl	$4, %esi;	\
		addl	$4, %eax;	\
		movl	(%esi), %ecx;	\
		movl	%ecx, (%eax);	\
		addl	$4, %esi;	\
		addl	$4, %eax;	\
		movw	(%esi), %cx;	\
		movw	%cx, (%eax);

/* -------------------------------------------------------------------------
 * Plot a 40 column row of lores graphics.
 * For 640x400 this is 2 rows of 14 pixels.
 * esi: from table
 * eax: to graphics memory
 * ------------------------------------------------------------------------- */
#define PlotBlockRow				\
		PlotBlockRow640			\
		addl	$ SCANWIDTH-12, %eax;	\
		PlotBlockRow640
#define PlotBlockRow640\
		movl	%edx, (%eax);		\
		addl	$4, %eax;		\
		movl	%edx, (%eax);		\
		addl	$4, %eax;		\
		movl	%edx, (%eax);		\
		addl	$4, %eax;		\
		movw	%dx, (%eax);

/* -------------------------------------------------------------------------
 * Get the adjancent color bytes in memory.
 * For 640x400 mode, we need to remember to move around by a factor of 2.
 * ebx: graphics memory index
 * eax: temp buffer for comparison
 * ------------------------------------------------------------------------- */
#define GrabAdjGMBytes\
		subl	$3, %ebx;\
		movw	(%ebx), %cx;\
		movw	%cx, (%eax);			/* GM -> temp */\
		addl	$9, %eax;\
		addl	$18, %ebx;\
		movw	(%ebx), %cx;\
		movw	%cx, (%eax);			/* GM -> temp */\
		decl	%eax;

/* -------------------------------------------------------------------------
 * Plots a normalized byte of dhires color directly into graphics memory.
 * eax: graphics memory index
 * ebx: dhires_colors index
 * edx: scratch
 * ------------------------------------------------------------------------- */
#define PlotDHiresByte\
		movb	SYMBOL_NAME(dhires_colors2)(,%ebx,1), %dl;\
		movb	%dl, %dh;\
		shll	$16, %edx;\
		movb	SYMBOL_NAME(dhires_colors1)(,%ebx,1), %dl;\
		movb	%dl, %dh;\
		movl	%edx, (%eax);\
		movl	%edx, SCANWIDTH(%eax);\
		addl	$4, %eax;

#define PlotDHiresFirstByte\
		subl	$4, %eax;\
		PlotDHiresByte

#else /* if ! _640x400 */

#define	expanded_col8_hires_even expanded_col_hires_even
#define expanded_col8_hires_odd  expanded_col_hires_odd

#define LoadHiresTableRef(TABLE)\
		leal	SYMBOL_NAME(expanded_col_hires_##TABLE)(,%eax,8), %ebx;


/* -------------------------------------------------------------------------
 * Plot a normal swath of pixels.
 * For 320x200 this is exactly 7 pixels.
 * ebx: from table
 * eax: to graphics memory
 * ------------------------------------------------------------------------- */
#define PlotPixels\
		Plot7Pixels(%ebx,%eax)
#define PlotCharacter40Row		\
		Plot7Pixels(%esi,%eax)
#define PlotBlockRow				\
		movl	%edx, (%eax);		\
		addl	$4, %eax;		\
		movw	%dx, (%eax);		\
		addl	$2, %eax;		\
		movb	%dl, (%eax);

/* -------------------------------------------------------------------------
 * Plot a dynamically interpolated swath of pixels
 * For 320x200 this is exactly 9 pixels.
 * ebx: from table
 * eax: to graphics memory
 * ecx: scratch
 * ------------------------------------------------------------------------- */
#define PlotPixelsExtra\
		decl	%eax;\
		movl	(%ebx), %ecx;\
		movl	%ecx, (%eax);\
		addl	$4, %eax;\
		addl	$4, %ebx;\
		movl	(%ebx), %ecx;\
		movl	%ecx, (%eax);\
		addl	$4, %eax;\
		addl	$4, %ebx;\
		movb	(%ebx), %cl;\
		movb	%cl, (%eax);

/* -------------------------------------------------------------------------
 * Get the adjancent color bytes in memory.
 * ebx: graphics memory index
 * eax: temp buffer for comparison
 * ------------------------------------------------------------------------- */
#define GrabAdjGMBytes\
		subl	$2, %ebx;\
		movw	(%ebx), %cx;\
		movw	%cx, (%eax);			/* GM -> temp */\
		addl	$9, %eax;\
		addl	$9, %ebx;\
		movw	(%ebx), %cx;\
		movw	%cx, (%eax);			/* GM -> temp */\
		decl	%eax;

/* -------------------------------------------------------------------------
 * Plots a normalized byte of dhires color directly into graphics memory.
 * eax: graphics memory index
 * ebx: dhires_colors index
 * edx: scratch
 * ------------------------------------------------------------------------- */
#define PlotDHiresByte							\
		movb	SYMBOL_NAME(dhires_colors1)(,%ebx,1), %dl;	\
		movb	SYMBOL_NAME(dhires_colors2)(,%ebx,1), %dh;	\
		movw	%dx, (%eax);					\
		addl	$2, %eax;

#define PlotDHiresFirstByte\
		subl	$2, %eax;\
		PlotDHiresByte

#endif/*_640x400*/


/* -------------------------------------------------------------------------
 * Calculate the graphics memory offset based on EffectiveAddr.
 * BASE 0x2000, 0x4000
 * PTR register to store the offset
 * ------------------------------------------------------------------------- */
#define CalcHiresGM(BASE,PTR)\
		movl	EffectiveAddr_E, %ecx;		/* ecx = mem addrs */ \
		subw	BASE, EffectiveAddr;		/* - graphics base */ \
		movl	SYMBOL_NAME(hires_row_offset)			      \
			(,EffectiveAddr_E,4), PTR;	/* PTR = GM offset */ \
		movl	%ecx, EffectiveAddr_E;		/* + graphics base */ \
		addl	SYMBOL_NAME(GM), PTR;		/* PTR += GM base */


/* -------------------------------------------------------------------------
 * PlotByte - macro to plot a hires byte into graphics memory.
 * BASE = 0x2000,0x4000.
 * TABLE = expanded_col_hires_even, expanded_col_hires_odd.
 * OPP_TABLE = opposite table
 * INTERP_COLOR = even_colors, odd_colors
 * ALT_INTERP_COLOR = opposite colors
 * ------------------------------------------------------------------------- */
#define PlotByte(BASE,X,TABLE,OPP_TABLE,INTERP_COLOR,ALT_INTERP_COLOR)	      \
		pushl	%eax;				/* save regs */	      \
		pushl	%ebx;						      \
		pushl	%ecx;						      \
		\
		xorb	%ah, %ah;			/* clear noise */     \
		testb	$0xFF, SYMBOL_NAME(strict_color);		      \
		jnz	PB_dynamic##X;			/* dynamic color mode */\
		LoadHiresTableRef(TABLE);\
		CalcHiresGM(BASE,%eax);			/* eax = GM */\
		PlotPixels;				/* temp -> GM */\
		jmp	PB_exit##X;\
		\
PB_dynamic##X:\
		leal	SYMBOL_NAME(expanded_col8_hires_##TABLE)(,%eax,8), %ebx;\
		leal	SYMBOL_NAME(temp), %eax;	/* eax = temp */\
		addl	$2, %eax;\
		Plot7Pixels(%ebx,%eax);			/* 7bytes -> temp+2 */\
		\
		subl	$8, %eax;\
		CalcHiresGM(BASE,%ebx);			/* ebx = GM */\
		/* copy adjacent color bytes into temp array */\
		GrabAdjGMBytes;\
		\
		/* calculate dynamic colors in temp array */\
		DynamicCalculateColor(X,OPP_TABLE,INTERP_COLOR,ALT_INTERP_COLOR);\
PB_plot_dynamic##X:\
		leal	SYMBOL_NAME(temp), %ebx;	/* ebx = temp */\
		incl	%ebx;\
		CalcHiresGM(BASE,%eax);			/* eax = GM */\
		PlotPixelsExtra				/* temp -> GM: 1 + 7 + 1 */\
PB_exit##X:\
		popl	%ecx;				/* restore regs */    \
		popl	%ebx;						      \
		popl	%eax;


/* -------------------------------------------------------------------------
 * Dynamic calculation of color at the edges of bytes.
 * ------------------------------------------------------------------------- */
#define DynamicCalculateColor(X,OPP_TABLE,INTERP_COLOR,ALT_INTERP_COLOR);\
		movw	(%eax), %cx;					      \
		testb	$0xFF, %ch;	/* check right color */		      \
		jz	PB_next0##X;	/* right black, do other end */	      \
	movw	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %cx;\
		andb	$1, %ch;					      \
		jz	PB_black0##X;	/* right black */		      \
		andb	$0x40, %cl;					      \
		jz	PB_black0##X;	/* inside black, right colored */     \
		movw	$0x3737, (%eax); /* edge is white (#55) */	      \
		jmp	PB_next0##X;					      \
PB_black0##X:								      \
	movzwl	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %ecx;\
		movb	%ch, %cl;					      \
		xorb	%ch, %ch;					      \
		leal	SYMBOL_NAME(expanded_col8_hires_##OPP_TABLE)	      \
			(,%ecx,8), %ebx;				      \
		incl	%eax;						      \
		movb	(%ebx), %cl;					      \
		movb	%cl, (%eax);					      \
		decl	%eax;						      \
PB_next0##X:								      \
		decw	EffectiveAddr;		/* previous byte */	      \
		subl	$7, %eax;	/* left edge of byte */		      \
		movb	(%eax), %cl;					      \
		testb	$0xFF, %cl;	/* check left color */		      \
		jz	PB_next1##X;	/* left black, done */		      \
	movw	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %cx;\
		andb	$0x40, %cl;					      \
		jz	PB_black1##X;	/* left black */		      \
		andb	$0x1, %ch;					      \
		jz	PB_black1##X;	/* left colored, inside black */      \
		movw	$0x3737, (%eax); /* edge is white (#55) */	      \
		jmp	PB_next1##X;					      \
PB_black1##X:								      \
	movzbl	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %ecx;\
		leal	SYMBOL_NAME(expanded_col8_hires_##OPP_TABLE)	      \
			(,%ecx,8), %ebx;				      \
		addb	$6, %ebx;					      \
		movb	(%ebx), %cl;					      \
		movb	%cl, (%eax);					      \
PB_next1##X:								      \
		incw	EffectiveAddr;                                        \
		/* do extra calculation for interpolated colors */	      \
		cmpb	$2, SYMBOL_NAME(strict_color);			      \
		jne	PB_plot_dynamic##X;				      \
									      \
		decw	EffectiveAddr;					      \
		CalculateInterpColor(X,2,ALT_INTERP_COLOR);		      \
PB_next2##X:								      \
		incw	EffectiveAddr;					      \
		incl	%eax;						      \
		CalculateInterpColor(X,3,INTERP_COLOR);			      \
PB_next3##X:								      \
		addl	$6, %eax;					      \
		CalculateInterpColor(X,4,INTERP_COLOR);			      \
PB_next4##X:								      \
		incw	EffectiveAddr;					      \
		incl	%eax;						      \
		CalculateInterpColor(X,5,ALT_INTERP_COLOR);		      \
PB_next5##X:								      \
	decw	EffectiveAddr;


/* -------------------------------------------------------------------------
 * Calculates the color at the edge of interpolated bytes.
 * Done 4 times in little endian order (...7 0...7 0...)
 * ------------------------------------------------------------------------- */
#define CalculateInterpColor(X,Y,INTERP_COLOR)				\
		testb	$0xFF, (%eax);					\
		jnz	PB_next##Y##X;	/* not black, next */		\
		movw	(%eax), %cx;	/* off+1 in %ch */		\
		testb	$0xFF, %ch;					\
		jz	PB_next##Y##X;	/* off+1 is black, next */	\
		movb	-1(%eax), %cl;	/* off-1 in %cl */		\
		testb	$0xFF, %cl;					\
		jz	PB_next##Y##X;	/* off-1 is black, next */	\
		cmpb	$55, %cl;	/* off-1 is white? */		\
		je	PB_white0##X##Y;				\
		movb	%cl, (%eax);	/* store non-white */		\
		jmp	PB_next##Y##X;	/* next */			\
PB_white0##X##Y:							\
		cmpb	$55, %ch;	/* off+1 is white? */		\
		je	PB_white##X##Y;					\
		movb	%ch, (%eax);	/* store non-white */		\
		jmp	PB_next##Y##X;	/* next */			\
PB_white##X##Y:				/* both sides are white */	\
	movzbl	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %ecx;\
		shrb	$7, %cl;					\
 		movb	SYMBOL_NAME(INTERP_COLOR)(,%ecx,1), %bl;	\
		movb	%bl, (%eax);


/* -------------------------------------------------------------------------
 * compeletely update all the hires/dhires page rows.
 * X=0,1
 * OFFSET=0x2027,0x4027 (page 1, 2)
 * ------------------------------------------------------------------------- */
#define UpdateHiresRows(PRE,X,OFFSET)					 \
update_hires_rows_##X:							 \
		movl	$20, %ecx;	/* ECX: 40 column counter */	 \
		movl	%ebx, %edx;	/* EBX: pixel row counter */	 \
		shrb	$3, %dl;	/* normalize to 0 - 23 */	 \
					/* EDI: row offset */		 \
		movw	SYMBOL_NAME(video_line_offset)(,%edx,2),	 \
			EffectiveAddr;					 \
		movl	%ebx, %edx;					 \
		andb	$0x7, %dl;					 \
		shlw	$10, %dx;	/* EDX: offset range 0 - 1C00 */ \
		addw	OFFSET, %dx;	/* add base end-row offset */	 \
		addw	%dx, EffectiveAddr;	/* EDI: mem address */	 \
update_hires_columns_##X:						 \
	movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al;\
		cmpw	$159, %bx;	/* mixed mode boundary */	 \
		jg	update_hires_mixed_##X;				 \
		call	SYMBOL_NAME(PRE##ram_hires_page##X##_odd);	 \
		decw	EffectiveAddr;	/* previous address */		 \
	movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al;\
		call	SYMBOL_NAME(PRE##ram_hires_page##X##_even);	 \
		jmp	update_hires_cont_##X;				 \
update_hires_mixed_##X:							 \
		call	SYMBOL_NAME(PRE##ram_hires_mixed##X##_odd);	 \
		decw	EffectiveAddr;	/* previous address */		 \
	movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al;\
		call	SYMBOL_NAME(PRE##ram_hires_mixed##X##_even);	 \
update_hires_cont_##X:							 \
		decw	EffectiveAddr;	/* previous address */		 \
		decb	%cl;		/* dec column counter */	 \
		jnz	update_hires_columns_##X;			 \
		decw	%bx;		/* dec row */			 \
		jns	update_hires_rows_##X;


/* -------------------------------------------------------------------------
 * compeletely update all the text page rows.
 * X=0,1
 * OFF=0x427,0x827 (page 1, 2)
 * ------------------------------------------------------------------------- */
#define UpdateRows(PRE,X,OFFSET)				\
update_rows_##X:							\
		movl	$39, %ecx;					\
		movw	SYMBOL_NAME(video_line_offset)(,%ebx,2),	\
			EffectiveAddr;					\
		addw	OFFSET, EffectiveAddr;				\
update_columns_##X:							\
	movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al;\
		cmpb	$19, %bl;					\
		jg	update_mixed_##X;				\
		call	SYMBOL_NAME(PRE##ram_text_page##X);		\
		jmp	update_cont_##X;				\
update_mixed_##X:							\
		call	SYMBOL_NAME(PRE##ram_text_mixed##X);		\
update_cont_##X:							\
		decw	%di;						\
		decb	%cl;						\
		jns	update_columns_##X;				\
		decb	%bl;						\
		jns	update_rows_##X;


/* -------------------------------------------------------------------------
 * Plot a full double hires color byte into GM
 * OFF 0x2000, 0x4000
 * PROBLEMS:
 * 	graphics artifiacts are not implemented correctly.
 * ------------------------------------------------------------------------- */
#define PlotDHires(OFF,X)						      \
		pushl	%eax;				/* save regs */	      \
		pushl	%ebx;						      \
		pushl	%ecx;						      \
		pushl	%edx;						      \
		pushl	EffectiveAddr_E;				      \
									      \
		andw	$0xFFFF, EffectiveAddr;		/* erase offset */    \
		btr	$0, EffectiveAddr_E;		/* normalize */	      \
		movl	EffectiveAddr_E, %ecx;		/* ecx = mem addrs */ \
		subw	OFF, EffectiveAddr;		/* - graphics base */ \
		movl	SYMBOL_NAME(hires_row_offset)			      \
			(,EffectiveAddr_E,4), %eax;	/* eax = GM offset */ \
		movb	SYMBOL_NAME(hires_col_offset)			      \
			(,EffectiveAddr_E,1), %bl;			      \
		addl	SYMBOL_NAME(GM), %eax;		/* eax += GM base */  \
									      \
	leal	SYMBOL_NAME(apple_ii_64k), EffectiveAddr_E;\
		addl	%ecx, EffectiveAddr_E;				      \
		movl	EffectiveAddr_E, %ecx;				      \
		addl	BANK2, %ecx;					      \
									      \
		testb	$0xFF, %bl;					      \
		jz	plot_dhires##X##_cont;				      \
		movzbl	-1(EffectiveAddr_E), %ebx;			      \
		movb	0(%ecx), %bh;					      \
		btr	$7, %ebx;					      \
		shrb	$3, %bl;					      \
		shlb	$4, %bh;					      \
		orb	%bh, %bl;					      \
		xorb	%bh, %bh;					      \
		PlotDHiresFirstByte					      \
									      \
plot_dhires##X##_cont:							      \
		movl	%ecx, %edx;					      \
		movb	2(%edx), %cl;					      \
		shll	$28, %ecx;					      \
									      \
		movzbl	1(EffectiveAddr_E), %ebx;			      \
		btr	$7, %ebx;		/* erase msb */		      \
		shll	$21, %ebx;					      \
		orl	%ebx, %ecx;					      \
									      \
		movzbl	1(%edx), %ebx;					      \
		btr	$7, %ebx;		/* erase msb */		      \
		shll	$14, %ebx;					      \
		orl	%ebx, %ecx;					      \
									      \
		movzbl	0(EffectiveAddr_E), %ebx;			      \
		btr	$7, %ebx;		/* erase msb */		      \
		shll	$7, %ebx;					      \
		orl	%ebx, %ecx;					      \
									      \
		movzbl	0(%edx), %ebx;					      \
		btr	$7, %ebx;		/* erase msb */		      \
		orl	%ebx, %ecx;					      \
		/* 00000001 11111122 22222333 3333xxxx */		      \
									      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		popl	EffectiveAddr_E;				      \
		andl	$0xFFFF, EffectiveAddr_E;/* for safety */	      \
		popl	%edx;						      \
		popl	%ecx;			 /* restore regs */	      \
		popl	%ebx;						      \
		popl	%eax;						      \
		ret;

#define offset_RAMRD_RAMWRT						\
		movl	SYMBOL_NAME(ramrd_offset), %eax;		\
		movl	%eax, SYMBOL_NAME(read_text_page_offset);	\
		movl	%eax, SYMBOL_NAME(read_hires_page_offset);	\
		movl	SYMBOL_NAME(ramwrt_offset), %eax;		\
		movl	%eax, SYMBOL_NAME(write_text_page_offset);	\
		movl	%eax, SYMBOL_NAME(write_hires_page_offset);


/* -------------------------------------------------------------------------
 * setup to plot the text/lores stuff.
 * eax: graphics memory pointer
 * ------------------------------------------------------------------------- */
#define PlotTextPagePre(OFF,OP,PAGE)\
		pushal;					/*Save everything -MUST BE MATCHED!*/\
		xorb	%ah, %ah;\
		movl	%eax, %esi;			/*ESI=EAX=Chr code*/\
		subw	OFF, EffectiveAddr;		/*Normalize scrn addr*/\
		testb	$0xFF, SYMBOL_NAME(vmode_active);/*Current SVGA page*/\
		OP	10f;\
		movl	PAGE, %eax;			/*Choose page PAGE*/\
		call	set_page;			/*Set page PAGE*/\
	10:\
							/*Compute row*/\
		movl	SYMBOL_NAME(text_row_offset)(,EffectiveAddr_E,4), %eax;\
		addl	SYMBOL_NAME(GM), %eax;		/*Graphic addr*/


/* -------------------------------------------------------------------------
 * Common code for plotting an 80 column character.
 * Only 640x400 resolution can do this.
 * eax: graphics memory pointer
 * esi: precalculated font pointer
 * OFF 0x400, 0x800
 * PAGE 0, 1
 * ------------------------------------------------------------------------- */
#define Plot80Character(TAG,OFF,OP,PAGE)\
		PlotTextPagePre(OFF,OP,PAGE)/* does a pushal */\
plot_80character_correct_page##TAG:\
    		addw	%bx, %ax;				/*screen offset*/\
		shll	$6, %esi;				/* * 64 = 8cols * 8rows*/\
		addl	$ SYMBOL_NAME(intermediate_font), %esi;	/*Font addr*/\
		\
		PlotCharacter80Row;\
		addl	$2, %esi;\
		addl	$ SCANWIDTH-6, %eax;			/*Go to next row*/\
		\
		PlotCharacter80Row;\
		addl	$2, %esi;\
		addl	$ SCANWIDTH-6, %eax;			/*Go to next row*/\
		\
		PlotCharacter80Row;\
		addl	$2, %esi;\
		addl	$ SCANWIDTH-6, %eax;			/*Go to next row*/\
		\
		PlotCharacter80Row;\
		addl	$2, %esi;\
		addl	$ SCANWIDTH-6, %eax;			/*Go to next row*/\
		\
		PlotCharacter80Row;\
		addl	$2, %esi;\
		addl	$ SCANWIDTH-6, %eax;			/*Go to next row*/\
		\
		PlotCharacter80Row;\
		addl	$2, %esi;\
		addl	$ SCANWIDTH-6, %eax;			/*Go to next row*/\
		\
		PlotCharacter80Row;\
		addl	$2, %esi;\
		addl	$ SCANWIDTH-6, %eax;			/*Go to next row*/\
		\
		PlotCharacter80Row;\
		\
		popal;	/* MATCHES pushal from PlotTextPagePre */



	/* -----------------------------------------------------------------
	 * Scan through video memory (text & graphics) and call the updating
	 * routines.  Depending on softswitch settings, either text or
	 * graphics, page 1 or 2 will be rendered.
	 * This is called on exit from menu screens, etc.
	 * ebx: number of rows (counting down)
	 * ----------------------------------------------------------------- */
ENTRY(update_video_screen)
		pushal

		xorl	%eax, %eax
		xorl	%edi, %edi

		/* 24 rows text/lores page 0 */
		movl	$23, %ebx
#ifdef APPLE_IIE
		UpdateRows(iie_soft_write_,0,$0x427)
#else
		UpdateRows(write_,0,$0x427)
#endif

		/* 24 rows text/lores page 1 */
		movl	$23, %ebx
#ifdef APPLE_IIE
		UpdateRows(iie_soft_write_,1,$0x827)
#else
		UpdateRows(write_,1,$0x827)
#endif

		/* 192 rows hires page 0 */
		movl	$191, %ebx
#ifdef APPLE_IIE
		UpdateHiresRows(iie_soft_write_,0,$0x2027)
#else
		UpdateHiresRows(write_,0,$0x2027)
#endif

		/* 192 rows hires page 1 */
		movl	$191, %ebx
#ifdef APPLE_IIE
		UpdateHiresRows(iie_soft_write_,1,$0x4027)
#else
		UpdateHiresRows(write_,1,$0x4027)
#endif

		popal
		ret
	

	/* --------------------------------------------------------------------
	 * This is a convenience wrapper around c_setpage() (video.h).
	 * Call set_page with appropriate content in eax.
	 * eax: page number (0, 1)
	 * ----------------------------------------------------------------- */
		.align 4
set_page:
		movb	%al, SYMBOL_NAME(vmode_active)# set apple2 active page
		orb	%al, %al		# Check content of AL
		jnz	set_page_1		# AL = 1 ?
set_page_0:
		pushal
		pushl	$0
set_page_c:	call	SYMBOL_NAME(c_setpage)
		addl	$4, %esp
		popal
		ret
set_page_1:
		pushal
		pushl	$1
		jmp	set_page_c
		/******************************************/

/* ram_nop, read_ram_default, write_ram_default: default ram/rom
   read/write functions. */
ENTRY(ram_nop)
		ret

ENTRY(read_ram_default)
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		ret

ENTRY(write_ram_default)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		ret

ENTRY(read_unmapped_softswitch)
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
    		ret

ENTRY(write_unmapped_softswitch)
    		ret

#ifdef APPLE_IIE
/* iie_read_ram_default - read from 128k ram */
ENTRY(iie_read_ram_default)
		orl	SYMBOL_NAME(ramrd_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)\
			(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* iie_write_ram_default - read from 128k ram */
ENTRY(iie_write_ram_default)
		orl	SYMBOL_NAME(ramwrt_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
		ret
#endif

		/******************************************/

/* write to LC RAM 0xe000 -> 0xffff.  we check to make sure that we
   can indeed write to the language card. */
ENTRY(write_ram_lc)
		cmpb	$0, SYMBOL_NAME(language_card_write)
		je	SYMBOL_NAME(ram_nop)
		pushl	EffectiveAddr_E
		subw	$0xE000, EffectiveAddr
		movb	%al, SYMBOL_NAME(language_card)(,EffectiveAddr_E,1)
		popl	EffectiveAddr_E
		ret

#ifdef APPLE_IIE
/* IIE write to LC RAM 0xe000 -> 0xffff.  we check to make sure that
   we can indeed write to the language card. */
ENTRY(iie_write_ram_lc)
		cmpb	$0, SYMBOL_NAME(language_card_write)
		je	SYMBOL_NAME(ram_nop)
		pushl	EffectiveAddr_E
		subw	$0xE000, EffectiveAddr
		orl	SYMBOL_NAME(lc_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(language_card)(,EffectiveAddr_E,1)
		popl	EffectiveAddr_E
		ret
#endif

/* write to LC RAM 0xd000 -> 0xdfff banks. we check to make sure that
   we can indeed write to the language card. */
ENTRY(write_ram_bank)
		cmpb	$0, SYMBOL_NAME(language_card_write)
		je	SYMBOL_NAME(ram_nop)
		pushl	EffectiveAddr_E
		cmpb	$0, SYMBOL_NAME(language_current_bank)
		jne	write_bank1
		subw	$0xD000, EffectiveAddr
		movb	%al, SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1)
		popl	EffectiveAddr_E
		ret
write_bank1:	subw	$0xC000, EffectiveAddr
		movb	%al, SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1)
		popl	EffectiveAddr_E
		ret

#ifdef APPLE_IIE
/* write to LC RAM 0xd000 -> 0xdfff banks. we check to make sure that
   we can indeed write to the language card. */
ENTRY(iie_write_ram_bank)
		cmpb	$0, SYMBOL_NAME(language_card_write)
		je	SYMBOL_NAME(ram_nop)
		pushl	EffectiveAddr_E
		cmpb	$0, SYMBOL_NAME(language_current_bank)
		jne	iie_write_bank1
		subw	$0xD000, EffectiveAddr
		orl	SYMBOL_NAME(lc_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1)
		popl	EffectiveAddr_E
		ret
iie_write_bank1:
		subw	$0xC000, EffectiveAddr
		orl	SYMBOL_NAME(lc_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1)
		popl	EffectiveAddr_E
		ret
#endif

		/******************************************/

/* read LC RAM or ROM 0xe000 -> 0xffff. */
ENTRY(read_ram_lc)
		cmpb	$0, SYMBOL_NAME(language_card_read)
		je	SYMBOL_NAME(read_ram_default)
		pushl	EffectiveAddr_E
		subw	$0xE000, EffectiveAddr
		movb	SYMBOL_NAME(language_card)(,EffectiveAddr_E,1), %al
		popl	EffectiveAddr_E
		ret

#ifdef APPLE_IIE
/* read LC RAM or ROM 0xe000 -> 0xffff. */
ENTRY(iie_read_ram_lc)
		cmpb	$0, SYMBOL_NAME(language_card_read)
		je	iie_read_ram_lc_rom
		pushl	EffectiveAddr_E
		subw	$0xE000, EffectiveAddr
		orl	SYMBOL_NAME(lc_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(language_card)(,EffectiveAddr_E,1), %al
		popl	EffectiveAddr_E
		ret
iie_read_ram_lc_rom:	/* don't need offsets in ROM! */
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		ret
#endif

/* read to LC RAM or ROM 0xd000 -> 0xdfff banks */
ENTRY(read_ram_bank)
		cmpb	$0, SYMBOL_NAME(language_card_read)
		je	SYMBOL_NAME(read_ram_default)
		pushl	EffectiveAddr_E
		cmpb	$0, SYMBOL_NAME(language_current_bank)
		jne	read_bank1
		subw	$0xD000, EffectiveAddr
		movb	SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1), %al
		popl	EffectiveAddr_E
		ret
read_bank1:	subw	$0xC000, EffectiveAddr
		movb	SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1), %al
		popl	EffectiveAddr_E
		ret

#ifdef APPLE_IIE
/* read to LC RAM or ROM 0xd000 -> 0xdfff banks */
ENTRY(iie_read_ram_bank)
		cmpb	$0, SYMBOL_NAME(language_card_read)
		je	iie_read_ram_bank_rom
		pushl	EffectiveAddr_E
		cmpb	$0, SYMBOL_NAME(language_current_bank)
		jne	iie_read_bank1
		subw	$0xD000, EffectiveAddr
		orl	SYMBOL_NAME(lc_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1), %al
		popl	EffectiveAddr_E
		ret
iie_read_bank1:	subw	$0xC000, EffectiveAddr
		orl	SYMBOL_NAME(lc_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1), %al
		popl	EffectiveAddr_E
		ret
iie_read_ram_bank_rom:	/* don't need offsets in ROM! */
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		ret

/* read/write //e zpage and stack */
ENTRY(iie_read_ram_zpage_and_stack)
		orl	SYMBOL_NAME(zp_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

ENTRY(iie_write_ram_zpage_and_stack)
		orl	SYMBOL_NAME(zp_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
		ret
#endif

		/******************************************/

ENTRY(write_ram_text_page0)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	plot_character0
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # lores mode?
		jz	plot_block0
		ret

ENTRY(write_ram_text_mixed0)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	plot_character0
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	plot_character0
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # Not hires mode?
		jz	plot_block0
		ret

ENTRY(write_ram_text_page1)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	plot_character1
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # lores mode?
		jz	plot_block1
		ret

ENTRY(write_ram_text_mixed1)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	plot_character1
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	plot_character1
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # Not hires mode?
		jz	plot_block1
		ret

#ifdef APPLE_IIE
/* iie_write_ram_text_page0 - handle text page //e specific */
ENTRY(iie_write_ram_text_page0)
		orl	SYMBOL_NAME(write_text_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_text_page0:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jz	iie_write_lores0		# graphics
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jnz	plot_80character0		# 80 col text

		/* check for screen update */
		testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	plot_character0			# 40 col text
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_text_page_offset)
		jnz	ram_nop				# NOP (in auxram)
		jmp	plot_character0			# 40 col text

iie_write_lores0:
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # lores mode?
		jz	plot_block0			# lores
		ret

/* iie_read_ram_text_page0 - read value from text page 0 */
ENTRY(iie_read_ram_text_page0)
		orl	SYMBOL_NAME(read_text_page_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* iie_read_screen_hole_text_page0 - read value out of screen hole */
ENTRY(iie_read_screen_hole_text_page0)
		orl	SYMBOL_NAME(read_text_page_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* iie_write_screen_hole_text_page0 - write value into screen hole */
ENTRY(iie_write_screen_hole_text_page0)
		orl	SYMBOL_NAME(write_text_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* iie_write_ram_text_mixed0 - handle mixed text page //e specific */
ENTRY(iie_write_ram_text_mixed0)
		orl	SYMBOL_NAME(write_text_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_text_mixed0:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	iie_mixed_text0			# text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jz	iie_mixed_lores0

iie_mixed_text0:
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
		jnz	plot_80character0
		/* check for screen update */
		testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	plot_character0			# 40 col text
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_text_page_offset)
		jnz	ram_nop				# NOP (in auxram)
		jmp	plot_character0			# 40 col text

iie_mixed_lores0:
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # Not hires mode?
		jz	plot_block0			# lores
		ret

/* iie_write_ram_text_page1 - handle text page1 //e specific */
ENTRY(iie_write_ram_text_page1)
		orl	SYMBOL_NAME(ramwrt_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_text_page1:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jz	iie_write_lores1		# graphics
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jnz	plot_80character1		# 80 col text

		/* check for screen update */
		testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	plot_character1			# 40 col text
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_text_page_offset)
		jnz	ram_nop				# NOP (in auxram)
		jmp	plot_character1			# 40 col text

iie_write_lores1:
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # lores mode?
		jz	plot_block1			# lores
		ret


/* iie_write_ram_text_mixed1 - handle mixed page 1 //e specific */
ENTRY(iie_write_ram_text_mixed1)
		orl	SYMBOL_NAME(ramwrt_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_text_mixed1:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	iie_mixed_text1			# text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jz	iie_mixed_lores1

iie_mixed_text1:
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
		jnz	plot_80character1
		/* check for screen update */
		testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	plot_character1			# 40 col text
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_text_page_offset)
		jnz	ram_nop				# NOP (in auxram)
		jmp	plot_character1			# 40 col text

iie_mixed_lores1:
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # Not hires mode?
		jz	plot_block1			# lores
		ret


#endif /* APPLE_IIE */

ENTRY(write_ram_hires_page0_even)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_even_byte0
		ret

ENTRY(write_ram_hires_mixed0_even)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_even_byte0
		ret

ENTRY(write_ram_hires_page0_odd)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_odd_byte0
		ret

ENTRY(write_ram_hires_mixed0_odd)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_odd_byte0
		ret

#ifdef APPLE_IIE
/* iie_write_ram_hires_page0_even - handle hires page //e specific */
ENTRY(iie_write_ram_hires_page0_even)
		orl	SYMBOL_NAME(write_hires_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_page0_even:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires0		# dhires

		/* hires page is in main memory, draw it if we're
                   repainting the screen. */
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page0_even
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
_iie_plot_hires_page0_even:
		jmp	plot_even_byte0			# plot hires


/* iie_write_ram_hires_mixed0_even - handle mixed hires page //e specific */
ENTRY(iie_write_ram_hires_mixed0_even)
		orl	SYMBOL_NAME(write_hires_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_mixed0_even:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires0		# dhires

		/* hires page is in main memory, draw it if we're
                   repainting the screen. */
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page0_even
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
		jmp	plot_even_byte0			# plot hires

/* iie_write_ram_hires_page0_odd - handle hires page //e specific */
ENTRY(iie_write_ram_hires_page0_odd)
		orl	SYMBOL_NAME(write_hires_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_page0_odd:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires0		# dhires

		/* hires page is in main memory, draw it if we're
                   repainting the screen. */
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page0_odd
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
_iie_plot_hires_page0_odd:
		jmp	plot_odd_byte0			# plot hires

/* iie_write_ram_hires_mixed0_odd - handle mixed hires page //e specific */
ENTRY(iie_write_ram_hires_mixed0_odd)
		orl	SYMBOL_NAME(write_hires_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_mixed0_odd:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires0		# dhires

		/* hires page is in main memory, draw it if we're
                   repainting the screen. */
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page0_odd
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
		jmp	plot_odd_byte0			# plot hires

/* iie_read_ram_hires_page0 - read value from hires page */
ENTRY(iie_read_ram_hires_page0)
		orl	SYMBOL_NAME(read_hires_page_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* iie_read_screen_hole_hires_page0 - read a value from hires screen hole */
ENTRY(iie_read_screen_hole_hires_page0)
		orl	SYMBOL_NAME(read_hires_page_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* iie_write_screen_hole_hires_page0 - write a value into hires screen hole */
ENTRY(iie_write_screen_hole_hires_page0)
		orl	SYMBOL_NAME(write_hires_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
		ret
#endif

ENTRY(write_ram_hires_page1_even)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_even_byte1
		ret

ENTRY(write_ram_hires_mixed1_even)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_even_byte1
		ret

ENTRY(write_ram_hires_page1_odd)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_odd_byte1
		ret

ENTRY(write_ram_hires_mixed1_odd)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_odd_byte1
		ret

#ifdef APPLE_IIE
/* iie_write_ram_hires_page1_even - write hires page1 //e specific */
ENTRY(iie_write_ram_hires_page1_even)
		orl	SYMBOL_NAME(ramwrt_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_page1_even:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires1		# dhires
		/* hires page is in main memory, draw it if we're
                   repainting the screen. */
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page1_even
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
_iie_plot_hires_page1_even:
		jmp	plot_even_byte1			# plot hires

/* iie_write_ram_hires_mixed1_even - write hires page1 //e specific */
ENTRY(iie_write_ram_hires_mixed1_even)
		orl	SYMBOL_NAME(ramwrt_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_mixed1_even:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires1		# dhires
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page1_even
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
		jmp	plot_even_byte1			# plot hires

/* iie_write_ram_hires_page1_odd - write hires page1 //e specific */
ENTRY(iie_write_ram_hires_page1_odd)
		orl	SYMBOL_NAME(ramwrt_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_page1_odd:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires1		# dhires
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page1_odd
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
_iie_plot_hires_page1_odd:
		jmp	plot_odd_byte1			# plot hires

/* iie_write_ram_hires_mixed1_odd - write hires page1 //e specific */
ENTRY(iie_write_ram_hires_mixed1_odd)
		orl	SYMBOL_NAME(ramwrt_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_mixed1_odd:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires1		# dhires
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page1_odd
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
		jmp	plot_odd_byte1			# plot hires

		.align	4
iie_plot_dhires0:
		testb	$0xFF, SYMBOL_NAME(vmode_active)# Current SVGA page
		jz	iie_plot_dhires0_correct_page
		pushal
		xorl	%eax, %eax			# Choose page 0
		call	set_page			# Set page 0
		popal
iie_plot_dhires0_correct_page:
		PlotDHires($0x2000,0)
		ret

iie_plot_dhires1:
		testb	$0xFF, SYMBOL_NAME(vmode_active)# Current SVGA page
		jnz	iie_plot_dhires1_correct_page
		pushal
		xorl	%eax, %eax
		movb	$1, %eax			# Choose page 1
		call	set_page			# Set page 1
		popal
iie_plot_dhires1_correct_page:
		PlotDHires($0x4000,1)
		ret


#ifdef _640x400
		.align 4
plot_80character0:
		pushl	%ebx
		pushl	EffectiveAddr_E

		orl	$0x10000, EffectiveAddr_E	# aux ram
		movl	$0, %ebx			# +0 screen offset
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		Plot80Character(0a,$0x400,jz,$0)

		popl	EffectiveAddr_E			# main ram
		movl	$7, %ebx			# +7 screen offset
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		Plot80Character(0b,$0x400,jz,$0)

		popl	%ebx
		ret

		.align 4
plot_80character1:
		pushl	%ebx
		pushl	EffectiveAddr_E

		orl	$0x10000, EffectiveAddr_E	# aux ram
		movl	$0, %ebx			# +0 screen offset
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		Plot80Character(1a,$0x800,jnz,$1)

		popl	EffectiveAddr_E			# main ram
		movl	$7, %ebx			# +7 screen offset
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		Plot80Character(1b,$0x800,jnz,$1)

		popl	%ebx
		ret

#else /* !640x400 resolution is not sufficient for 80 column */
		.align 4
plot_80character0:
		andl	$0xFFFF, EffectiveAddr_E/* for safety */
		ret

		.align 4
plot_80character1:
		andl	$0xFFFF, EffectiveAddr_E/* for safety */
		ret
#endif /* _640x400 */

#endif /* APPLE_IIE */


/* plot character on first text page */
		.align 4
plot_character0:
		PlotTextPagePre($0x400,jz,$0)
plot_character_correct_page:
#ifdef _640x400
		shll	$7, %esi				# * 128
#else
		shll	$6, %esi				# * 64
#endif
		addl	$ SYMBOL_NAME(expanded_font), %esi	# Font addr

		PlotCharacter40Row
		addl	$2, %esi;
		addl	$ SCANSTEP, %eax			# Go to next row

		PlotCharacter40Row
		addl	$2, %esi;
		addl	$ SCANSTEP, %eax			# Go to next row

		PlotCharacter40Row
		addl	$2, %esi;
		addl	$ SCANSTEP, %eax			# Go to next row

		PlotCharacter40Row
		addl	$2, %esi;
		addl	$ SCANSTEP, %eax			# Go to next row

		PlotCharacter40Row
		addl	$2, %esi;
		addl	$ SCANSTEP, %eax			# Go to next row

		PlotCharacter40Row
		addl	$2, %esi;
		addl	$ SCANSTEP, %eax			# Go to next row

		PlotCharacter40Row
		addl	$2, %esi;
		addl	$ SCANSTEP, %eax			# Go to next row

		PlotCharacter40Row

		popal
		ret


/* plot character on second text page */
		.align 4
plot_character1:
		PlotTextPagePre($0x800,jnz,$1)
		jmp	plot_character_correct_page	# same as page 0


/* plot lores block first page */
	
plot_block0:
		PlotTextPagePre($0x400,jz,$0)
plot_block_correct_page:
		movw	%si, %dx			# Compute color
		andb	$0x0F, %dl
		shlb	$4, %dl
		movb	%dl, %dh
		shll	$16, %edx
		movw	%si, %dx
		andb	$0x0F, %dl
		shlb	$4, %dl
		movb	%dl, %dh

		PlotBlockRow
		addl	$ SCANSTEP, %eax		# Go to next row

		PlotBlockRow
		addl	$ SCANSTEP, %eax		# Go to next row

		PlotBlockRow
		addl	$ SCANSTEP, %eax		# Go to next row

		PlotBlockRow
		addl	$ SCANSTEP, %eax		# Go to next row

		movw	%si, %dx			# Compute color
		andb	$0xF0, %dl
		movb	%dl, %dh
		shll	$16, %edx
		movw	%si, %dx
		andb	$0xF0, %dl
		movb	%dl, %dh

		PlotBlockRow
		addl	$ SCANSTEP, %eax		# Go to next row

		PlotBlockRow
		addl	$ SCANSTEP, %eax		# Go to next row

		PlotBlockRow
		addl	$ SCANSTEP, %eax		# Go to next row

		PlotBlockRow

		popal
		ret

		.align 4
plot_block1:
		PlotTextPagePre($0x800,jnz,$1)
    		jmp plot_block_correct_page
	
	

/* plot even column hires byte on page 0 */
		.align 4
plot_even_byte0:
		testb	$0xFF, SYMBOL_NAME(vmode_active)# Current SVGA page
		jz	plot_even_byte0_correct_page
		pushal
		xorl	%eax, %eax			# Choose page 0
		call	set_page			# Set page 0
		popal
plot_even_byte0_correct_page:
		PlotByte($0x2000,0,even,odd,even_colors,odd_colors)
		ret
	
/* plot odd column hires byte on page 0 */
		.align 4
plot_odd_byte0:
		testb	$0xFF, SYMBOL_NAME(vmode_active)# Current SVGA page
		jz	plot_odd_byte0_correct_page
		pushal
		xorl	%eax, %eax			# Choose page 0
		call	set_page			# Set page 0
		popal
plot_odd_byte0_correct_page:
		PlotByte($0x2000,1,odd,even,odd_colors,even_colors)
		ret

/* plot even column hires byte on page 1 */	
		.align 4
plot_even_byte1:
		testb	$0xFF, SYMBOL_NAME(vmode_active)# Current SVGA page
		jnz	plot_even_byte1_correct_page
		pushal
		movl	$1, %eax			# Choose page 1
		call	set_page			# Set page 1
		popal
plot_even_byte1_correct_page:
		PlotByte($0x4000,2,even,odd,even_colors,odd_colors)
		ret

/* plot odd column hires byte on page 1 */
		.align 4
plot_odd_byte1:
		testb	$0xFF, SYMBOL_NAME(vmode_active)# Current SVGA page
		jnz	plot_odd_byte1_correct_page
		pushal
		movl	$1, %eax			# Choose page 1
		call	set_page			# Set page 1
		popal
plot_odd_byte1_correct_page:
		PlotByte($0x4000,3,odd,even,odd_colors,even_colors)
		ret
	
	

ENTRY(read_keyboard)
		movb	SYMBOL_NAME(apple_ii_64k)+0xC000, %al
        	ret

ENTRY(read_keyboard_strobe)
		andb	$0x7f, SYMBOL_NAME(apple_ii_64k)+0xC000
		andb	$0x7f, SYMBOL_NAME(apple_ii_64k)+0x1C000
		movb	SYMBOL_NAME(apple_ii_64k)+0xC000, %al /* HACK: necessary? */
		ret

ENTRY(read_random)
		pushal
		call	SYMBOL_NAME(c_read_random)
		popal
		movb	SYMBOL_NAME(random_value), %al
		ret

ENTRY(read_speaker_toggle_pc)
		inb	$0x61, %al
		xorb	$0x2, %al
		outb	%al, $0x61
		ret

ENTRY(read_switch_primary_page)
		testb	$0xFF, SYMBOL_NAME(vmode_page2)
		jnz	_read_switch_primary
		ret
_read_switch_primary:
		movb	$0, SYMBOL_NAME(vmode_page2)
		movb	$0, SYMBOL_NAME(vmode_screen)
		pushal
		pushl	$0
		call	SYMBOL_NAME(c_setscreen)
		addl	$4, %esp
		popal
		ret

ENTRY(read_switch_secondary_page)
		testb	$0xFF, SYMBOL_NAME(vmode_page2)
		jz	_read_switch_secondary
		ret
_read_switch_secondary:
		movb	$1, SYMBOL_NAME(vmode_page2)
		movb	$1, SYMBOL_NAME(vmode_screen)
		pushal
		pushl	$1
		call	SYMBOL_NAME(c_setscreen)
		addl	$4, %esp
		popal
		ret

#ifdef APPLE_IIE
/* PAGE2 off. if 80STORE on then we use main text page1, and if HIRES
	on we also use main hires page1, regardless of RAMRD and
	RAMWRT. */
ENTRY(iie_page2_off)
		testb	$0xFF, SYMBOL_NAME(vmode_page2)	# already off?
		jz	ram_nop
		movb	$0, SYMBOL_NAME(vmode_page2)
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)
		jz	_iie_page2_off_hires_off	# no 80STORE
		movl	$0, SYMBOL_NAME(read_text_page_offset)
		movl	$0, SYMBOL_NAME(write_text_page_offset)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jz	_iie_page2_off_hires_off	# no HIRES
		movl	$0, SYMBOL_NAME(read_hires_page_offset)
		movl	$0, SYMBOL_NAME(write_hires_page_offset)
_iie_page2_off_hires_off:
		movb	$0, SYMBOL_NAME(vmode_screen)
		pushal
		pushl	$0				# page 1 main
		call	SYMBOL_NAME(c_setscreen)
		addl	$4, %esp
		popal
/* 		call	SYMBOL_NAME(update_video_screen)# redraw */
		ret


/* PAGE2 on.  if 80STORE on then we use aux text page 1, and if HIRES
	on we also use aux hires page 1, regardless of RAMRD and
	RAMWRT. */
ENTRY(iie_page2_on)
		testb	$0xFF, SYMBOL_NAME(vmode_page2)	# already on?
		jnz	ram_nop
		movb	$1, SYMBOL_NAME(vmode_page2)
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)
		jz	_iie_page2_on_80store_off	# no 80STORE
		movl	BANK2, SYMBOL_NAME(read_text_page_offset)
		movl	BANK2, SYMBOL_NAME(write_text_page_offset)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jz	_iie_page2_on_cont
		movl	BANK2, SYMBOL_NAME(read_hires_page_offset)
		movl	BANK2, SYMBOL_NAME(write_hires_page_offset)
_iie_page2_on_cont:
		movb	$0, SYMBOL_NAME(vmode_screen)
		pushal
		pushl	$0				# page 1 aux
		call	SYMBOL_NAME(c_setscreen)
		addl	$4, %esp
		popal
/* 		call	SYMBOL_NAME(update_video_screen)# redraw */
		ret

_iie_page2_on_80store_off:				# no 80STORE
		movb	$1, SYMBOL_NAME(vmode_screen)
		pushal
		pushl	$1				# page 2 main
		call	SYMBOL_NAME(c_setscreen)
		addl	$4, %esp
		popal
/* 		call	SYMBOL_NAME(update_video_screen)# redraw */
		ret
#endif

ENTRY(read_switch_graphics)
		testb	$0xFF, SYMBOL_NAME(vmode_text)
		jz	ram_nop
		movb	$0, SYMBOL_NAME(vmode_text)
		call	SYMBOL_NAME(update_video_screen)
		ret		

ENTRY(read_switch_text)
		testb	$0xFF, SYMBOL_NAME(vmode_text)
		jnz	ram_nop
		movb	$1, SYMBOL_NAME(vmode_text)
		call	SYMBOL_NAME(update_video_screen)
		ret

ENTRY(read_switch_no_mixed)
		testb	$0xFF, SYMBOL_NAME(vmode_mixed)
		jz	ram_nop
		movb	$0, SYMBOL_NAME(vmode_mixed)
		call	SYMBOL_NAME(update_video_screen)
		ret		

ENTRY(read_switch_mixed)
		testb	$0xFF, SYMBOL_NAME(vmode_mixed)
		jnz	ram_nop
		movb	$1, SYMBOL_NAME(vmode_mixed)
		call	SYMBOL_NAME(update_video_screen)
		ret

ENTRY(read_switch_lores)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jz	ram_nop
		movb	$0, SYMBOL_NAME(vmode_hires)
		call	SYMBOL_NAME(update_video_screen)
		ret

ENTRY(read_switch_hires)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jnz	ram_nop
		movb	$1, SYMBOL_NAME(vmode_hires)
		call	SYMBOL_NAME(update_video_screen)
		ret

#ifdef APPLE_IIE
/* HIRES off. use RAMRD/RAMWRT offsets for hires page 1. */
ENTRY(iie_hires_off)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)		# already off?
		jz	ram_nop
		movb	$0, SYMBOL_NAME(vmode_hires)		# no hires
		movl	SYMBOL_NAME(ramrd_offset), %eax
		movl	%eax, SYMBOL_NAME(read_hires_page_offset)
		movl	SYMBOL_NAME(ramwrt_offset), %eax
		movl	%eax, SYMBOL_NAME(write_hires_page_offset)
		call	SYMBOL_NAME(update_video_screen)	# update screen
		ret

/* HIRES on. if 80STORE on, use PAGE2 offset for hires page 1. */
ENTRY(iie_hires_on)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)		# already on?
		jnz	ram_nop
		movb	$1, SYMBOL_NAME(vmode_hires)		# hires on
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)
		jz	iie_hires_on_80store_off		# no 80STORE

		xorl	%eax, %eax
		movb	SYMBOL_NAME(vmode_page2), %ah
		shll	$8, %eax				# 0 or 0x10000
		movl	%eax, SYMBOL_NAME(read_hires_page_offset)
		movl	%eax, SYMBOL_NAME(write_hires_page_offset)
iie_hires_on_80store_off:
		call	SYMBOL_NAME(update_video_screen)	# update screen
		ret
#endif

/*****************************************************************************/

.comm		joy_trigger0, 2
.comm		joy_trigger1, 2
.comm		joy_trigger2, 2
.comm		joy_trigger3, 2

ENTRY(read_button0)
		movb	SYMBOL_NAME(joy_button0), %al
		ret

ENTRY(read_button1)
		movb	SYMBOL_NAME(joy_button1), %al
		ret

ENTRY(read_button2)
		movb	SYMBOL_NAME(joy_button2), %al
		ret

ENTRY(read_gc0)
		cmpw	$0xFF, joy_trigger0
		je	read_gc0_cont
		incw	joy_trigger0
read_gc0_cont:
		movw	joy_trigger0, %ax
		cmpw	%ax, SYMBOL_NAME(joy_x)
		jge	read_gc0_ff	/* XXXX is this correct? */
		movb	$0, %al
		ret
read_gc0_ff:
		movb	$0xFF, %al
		ret

ENTRY(read_gc1)
		cmpw	$0xFF, joy_trigger1
		je	read_gc1_cont
		incw	joy_trigger1
read_gc1_cont:
		movw	joy_trigger1, %ax
		cmpw	%ax, SYMBOL_NAME(joy_y)
		jge	read_gc1_ff
		movb	$0, %al
		ret
read_gc1_ff:
		movb	$0xFF, %al
		ret

/* HACK not doing anything... */
ENTRY(iie_read_gc2)
		ret

ENTRY(iie_read_gc3)
		ret

ENTRY(read_gc_strobe)
		movb	$0, joy_trigger0
		movb	$0, joy_trigger1
		movb	$0, joy_trigger2
		movb	$0, joy_trigger3
		ret

/* c080: read RAM; no write; use $D000 bank 2. */
ENTRY(lc_c080)
		movb	$0, SYMBOL_NAME(language_card_second)
		movb	$0, SYMBOL_NAME(language_card_write)
		movb	$1, SYMBOL_NAME(language_card_read)
		movb	$0, SYMBOL_NAME(language_current_bank)
		ret

/* c081: read ROM; write RAM; use $D000 bank 2. */
ENTRY(lc_c081)
		movb	$0, SYMBOL_NAME(language_card_read)
		cmpb	$0, SYMBOL_NAME(language_card_second)
		je	lc_c081_exit
		movb	$0, SYMBOL_NAME(language_current_bank)
		movb	$1, SYMBOL_NAME(language_card_write)
lc_c081_exit:
		movb	$1, SYMBOL_NAME(language_card_second)
		ret

/* c082: read ROM; no write; use $D000 bank 2. */
ENTRY(lc_c082)
		movb	$0, SYMBOL_NAME(language_card_second)
		movb	$0, SYMBOL_NAME(language_card_read)
		movb	$0, SYMBOL_NAME(language_card_write)
		ret

/* c083: read and write RAM; no write; use $D000 bank 2. */
ENTRY(lc_c083)
		movb	$1, SYMBOL_NAME(language_card_read)
		movb	$0, SYMBOL_NAME(language_current_bank)
		cmpb	$0, SYMBOL_NAME(language_card_second)
		je	lc_c083_exit
		movb	$1, SYMBOL_NAME(language_card_write)
lc_c083_exit:
		movb	$1, SYMBOL_NAME(language_card_second)
		ret

/* c088: read RAM; no write; use $D000 bank 1. */
ENTRY(lc_c088)
		movb	$0, SYMBOL_NAME(language_card_second)
		movb	$1, SYMBOL_NAME(language_card_read)
		movb	$0, SYMBOL_NAME(language_card_write)
		movb	$1, SYMBOL_NAME(language_current_bank)
		ret

/* c089: read ROM; write RAM; use $D000 bank 1. */
ENTRY(lc_c089)
		movb	$0, SYMBOL_NAME(language_card_read)
		cmpb	$0, SYMBOL_NAME(language_card_second)
		jz	lc_c089_exit
		movb	$1, SYMBOL_NAME(language_card_write)
		movb	$1, SYMBOL_NAME(language_current_bank)
lc_c089_exit:
		movb	$1, SYMBOL_NAME(language_card_second)
		ret

/* c08a: read ROM; no write; use $D000 bank 1. */
ENTRY(lc_c08a)
		movb	$0, SYMBOL_NAME(language_card_second)
		movb	$0, SYMBOL_NAME(language_card_read)
		movb	$0, SYMBOL_NAME(language_card_write)
		ret

/* c08b: read and write RAM; use $D000 bank 1. */
ENTRY(lc_c08b)
		movb	$1, SYMBOL_NAME(language_card_read)
		movb	$1, SYMBOL_NAME(language_current_bank)
		cmpb	$0, SYMBOL_NAME(language_card_second)
		jz	lc_c08b_exit
		movb	$1, SYMBOL_NAME(language_card_write)
lc_c08b_exit:
		movb	$1, SYMBOL_NAME(language_card_second)
		ret
/* -------------------------------------------------------------------------
 * misc //e functions
 * ------------------------------------------------------------------------- */

#ifdef APPLE_IIE

/* 80STORE off. use RAMRD/RAMWRT for text/hires display page 1 */
ENTRY(iie_80store_off)
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)	# already off?
		jz	ram_nop
		movb	$0, SYMBOL_NAME(eighty_store_flag)
		offset_RAMRD_RAMWRT	# RAMRD/RAMWRT are text/hires offset
		movzbl	SYMBOL_NAME(vmode_page2), %eax
		movb	%al, SYMBOL_NAME(vmode_screen)
		pushal
		pushl	%eax
		call	SYMBOL_NAME(c_setscreen)
		addl	$4, %esp
		popal
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret

/* 80STORE on. we use PAGE2 offset to access text page 1.  if HIRES on
	we also use PAGE2 offset to access hires page 1 */
ENTRY(iie_80store_on)
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)	# already on?
		jnz	ram_nop
		movb	$0x80, SYMBOL_NAME(eighty_store_flag)
		testb	$0xFF, SYMBOL_NAME(vmode_page2)
		jnz	iie_80store_on_page2_on

		movl	$0, SYMBOL_NAME(read_text_page_offset)
		movl	$0, SYMBOL_NAME(write_text_page_offset)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jz	iie_80store_on_cont
		movl	$0, SYMBOL_NAME(read_hires_page_offset)
		movl	$0, SYMBOL_NAME(write_hires_page_offset)
		jmp	iie_80store_on_cont

iie_80store_on_page2_on:
		movl	BANK2, SYMBOL_NAME(read_text_page_offset)
		movl	BANK2, SYMBOL_NAME(write_text_page_offset)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jz	iie_80store_on_cont
		movl	BANK2, SYMBOL_NAME(read_hires_page_offset)
		movl	BANK2, SYMBOL_NAME(write_hires_page_offset)
iie_80store_on_cont:
		movb	$0, SYMBOL_NAME(vmode_screen)
		pushal
		pushl	$0					# page 1
		call	SYMBOL_NAME(c_setscreen)
		addl	$4, %esp
		popal
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret

ENTRY(iie_check_80store)
		movb	SYMBOL_NAME(eighty_store_flag), %al
		ret

/* RAMRD main. if 80STORE off then text/hires display pages are in
	main memory. */
ENTRY(iie_ramrd_main)
		testb	$0xFF, SYMBOL_NAME(ramrd_flag)		# already on?
		jz	ram_nop
		movb	$0, SYMBOL_NAME(ramrd_flag)
		movl	$0, SYMBOL_NAME(ramrd_offset)
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)	# 80store?
		jnz	iie_ramrd_main_80store_on
		movl	$0, SYMBOL_NAME(read_text_page_offset)
		movl	$0, SYMBOL_NAME(read_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret
iie_ramrd_main_80store_on:
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jnz	ram_nop					# already set
		movl	$0, SYMBOL_NAME(read_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret

/* RAMRD aux. if 80STORE off then text/hires display pages are in aux
	memory. */
ENTRY(iie_ramrd_aux)
		testb	$0xFF, SYMBOL_NAME(ramrd_flag)		# already off?
		jnz	ram_nop
		movb	$0x80, SYMBOL_NAME(ramrd_flag)
		movl	BANK2, SYMBOL_NAME(ramrd_offset)
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)	# 80store?
		jnz	iie_ramrd_aux_80store_on
		movl	BANK2, SYMBOL_NAME(read_text_page_offset)
		movl	BANK2, SYMBOL_NAME(read_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret
iie_ramrd_aux_80store_on:
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jnz	ram_nop					# already set
		movl	BANK2, SYMBOL_NAME(read_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret

ENTRY(iie_check_ramrd)
		movb	SYMBOL_NAME(ramrd_flag), %al
		ret

/* RAMWRT main. if 80STORE off then text/hires display pages are in
	main memory. */
ENTRY(iie_ramwrt_main)
		testb	$0xFF, SYMBOL_NAME(ramwrt_flag)		# already on?
		jz	ram_nop
		movb	$0, SYMBOL_NAME(ramwrt_flag)
		movl	$0, SYMBOL_NAME(ramwrt_offset)	# main RAM
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)	# 80store?
		jnz	iie_ramwrt_main_80store_on
		movl	$0, SYMBOL_NAME(write_text_page_offset)
		movl	$0, SYMBOL_NAME(write_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret
iie_ramwrt_main_80store_on:
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jnz	ram_nop					# already set
		movl	$0, SYMBOL_NAME(write_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret

/* RAMWRT aux. if 80STORE off then write to text/hires display pages
	are in aux memory. */
ENTRY(iie_ramwrt_aux)
		testb	$0xFF, SYMBOL_NAME(ramwrt_flag)		# already off?
		jnz	ram_nop
		movb	$0x80, SYMBOL_NAME(ramwrt_flag)
		movl	BANK2, SYMBOL_NAME(ramwrt_offset)	# aux RAM
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)	# 80store?
		jnz	iie_ramwrt_aux_80store_on
		movl	BANK2, SYMBOL_NAME(write_text_page_offset)
		movl	BANK2, SYMBOL_NAME(write_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret
iie_ramwrt_aux_80store_on:
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jnz	ram_nop					# already set
		movl	BANK2, SYMBOL_NAME(write_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret

ENTRY(iie_check_ramwrt)
		movb	SYMBOL_NAME(ramwrt_flag), %al
		ret
	
ENTRY(iie_altzp_main)
		movb	$0, SYMBOL_NAME(altzp_flag)
		movl	$0, SYMBOL_NAME(lc_offset)
		movl	$0, SYMBOL_NAME(zp_offset)
		andl	$0xFFFF, SP_Reg
		ret

ENTRY(iie_altzp_aux)
		movb	$0x80, SYMBOL_NAME(altzp_flag)
		movl	$0x2000, SYMBOL_NAME(lc_offset)
		movl	BANK2, SYMBOL_NAME(zp_offset)
		orl	BANK2, SP_Reg
		ret

ENTRY(iie_check_altzp)
		movb	SYMBOL_NAME(altzp_flag), %al
		ret

ENTRY(iie_80col_off)
    		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	ram_nop
		movb	$0, SYMBOL_NAME(eighty_col_flag)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	SYMBOL_NAME(update_video_screen)# redraw text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	SYMBOL_NAME(update_video_screen)# redraw text
		ret

ENTRY(iie_80col_on)
    		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jnz	ram_nop
		movb	$0x80, SYMBOL_NAME(eighty_col_flag)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	SYMBOL_NAME(update_video_screen)# redraw text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	SYMBOL_NAME(update_video_screen)# redraw text
		ret

ENTRY(iie_check_80col)
		movb	SYMBOL_NAME(eighty_col_flag), %al
		ret

ENTRY(iie_altchar_off)
		testb	$0xff, SYMBOL_NAME(altchar_flag)
    		jz	ram_nop
		movb	$0, SYMBOL_NAME(altchar_flag)
		pushal
		call	SYMBOL_NAME(c_set_primary_char)
		popal
		ret

ENTRY(iie_altchar_on)
		testb	$0xff, SYMBOL_NAME(altchar_flag)
    		jnz	ram_nop
		movb	$0x80, SYMBOL_NAME(altchar_flag)
		pushal
		call	SYMBOL_NAME(c_set_altchar)
		popal
		ret

ENTRY(iie_check_altchar)
		movb	SYMBOL_NAME(altchar_flag), %al
		ret

/* unset ioudis flag and read_gc_strobe */
ENTRY(iie_ioudis_off)
		movb	$0, SYMBOL_NAME(ioudis_flag)
		jmp	SYMBOL_NAME(read_gc_strobe)

/* set ioudis flag and read_gc_strobe */
ENTRY(iie_ioudis_on)
		movb	$0x80, SYMBOL_NAME(ioudis_flag)
		jmp	SYMBOL_NAME(read_gc_strobe)

/* check ioudis flag and read_gc_strobe */
ENTRY(iie_check_ioudis)
		call	SYMBOL_NAME(read_gc_strobe)
		movb	SYMBOL_NAME(ioudis_flag), %al
		ret

ENTRY(iie_dhires_on)
		testb	$0xFF, SYMBOL_NAME(ioudis_flag)
		jz	ram_nop
		movb	$1, SYMBOL_NAME(vmode_dhires)
		movb	$0x80, SYMBOL_NAME(dhires_flag)
		call	SYMBOL_NAME(update_video_screen)
		ret

ENTRY(iie_dhires_off)
		testb	$0xFF, SYMBOL_NAME(ioudis_flag)
		jz	ram_nop
		movb	$0, SYMBOL_NAME(vmode_dhires)
		movb	$0, SYMBOL_NAME(dhires_flag)
		call	SYMBOL_NAME(update_video_screen)
		ret

/* check dhires flag and read_gc_strobe */
ENTRY(iie_check_dhires)
		call	SYMBOL_NAME(read_gc_strobe)
		movb	SYMBOL_NAME(dhires_flag), %al
		ret

ENTRY(iie_check_text)
		xorw	%ax, %ax
		movb	SYMBOL_NAME(vmode_text), %ah
		shrw	%ax
		ret

ENTRY(iie_check_mixed)
		xorw	%ax, %ax
		movb	SYMBOL_NAME(vmode_mixed), %ah
		shrw	%ax
		ret

ENTRY(iie_check_page2)
		xorw	%ax, %ax
		movb	SYMBOL_NAME(vmode_page2), %ah
		shrw	%ax
		ret

ENTRY(iie_check_hires)
		xorw	%ax, %ax
		movb	SYMBOL_NAME(vmode_hires), %ah
		shrw	%ax
		ret

ENTRY(iie_check_bank)
		xorw	%ax, %ax
		movb	SYMBOL_NAME(language_current_bank), %ah
		shrw	%ax
		xorb	$0x80, %al	/* I'm backwards */
		ret

ENTRY(iie_check_lcram)
		xorw	%ax, %ax
		movb	SYMBOL_NAME(language_card_read), %ah
		shrw	%ax
		ret

/* HACK not doing anything... */
ENTRY(iie_check_vbl)
		ret

	/* ---------------------------------- */

ENTRY(iie_c3rom_peripheral)
		movb	$0x80, SYMBOL_NAME(c3rom_flag)
		movl	$0, SYMBOL_NAME(c3rom_offset)
		ret

ENTRY(iie_c3rom_internal)
		movb	$0, SYMBOL_NAME(c3rom_flag)
		movl	BANK2, SYMBOL_NAME(c3rom_offset)
		ret

ENTRY(iie_check_c3rom)
		movb	SYMBOL_NAME(c3rom_flag), %al
		ret

ENTRY(iie_cxrom_peripheral)
		movb	$0, SYMBOL_NAME(cxrom_flag)
		movl	$0, SYMBOL_NAME(cxrom_offset)
		ret

ENTRY(iie_cxrom_internal)
		movb	$0x80, SYMBOL_NAME(cxrom_flag)
		movl	BANK2, SYMBOL_NAME(cxrom_offset)
	/* HACK - do we reset c3rom_offset when we go to peripheral? */
		movl	BANK2, SYMBOL_NAME(c3rom_offset)
		ret

ENTRY(iie_check_cxrom)
		movb	SYMBOL_NAME(cxrom_flag), %al
		ret

ENTRY(iie_read_slot3)
		orl	SYMBOL_NAME(c3rom_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

ENTRY(iie_read_slot6)
		orl	SYMBOL_NAME(cxrom_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

ENTRY(iie_read_slotx)
		orl	SYMBOL_NAME(cxrom_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* HACK!!!!! - how does the expansion slot get referenced? need to handle other roms requesting to use this area!!!! */
ENTRY(iie_read_slot_expansion)
		orl	SYMBOL_NAME(c8rom_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* HACK!!!!! - how does the expansion slot get referenced? */
ENTRY(iie_disable_slot_expansion)
		ret

#endif /* APPLE_IIE */
