/*
 * (C) Copyright Keith Visco 1998, 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package xslp.xml.parsers;

import org.w3c.dom.*;
import com.ibm.xml.parser.Parser;
import com.ibm.xml.parser.Source;
import com.ibm.xml.parser.StreamProducer;
import org.xml.sax.InputSource;

import java.io.InputStream;
import java.io.Reader;
import java.io.PrintWriter;

import com.kvisco.net.URIUtils;
import com.kvisco.util.List;

/**
 * Creates a DOM Package for use with IBM's XML4J 1.x package
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class XML4J_1xParser 
    implements com.kvisco.xml.parser.DOMPackage
{
    
    private boolean       validate    = false;
    private PrintWriter   pw          = null;
    private SimpleCatalog catalog     = null;
    
      //----------------/    
     //- Constructors -/
    //----------------/    
    
    /**
     * Creates a new XML4J_1xParser
    **/
    public XML4J_1xParser() {
        super();
        //-- load catalog, based on code from Michel Casabianca
        catalog = new SimpleCatalog(System.getProperty("xml.catalog"));
    } //-- XML4J_1xParser

    
      //---------------------------------------------------/    
     //- Interfaces for com.kvisco.xml.parser.DOMPackage -/
    //---------------------------------------------------/    
    
    /**
     * Creates a DOM Document 
     * @return the new Document
    **/
    public Document createDocument() {
        return new com.ibm.xml.parser.TXDocument();
    } //-- createDocument
    
    /**
     * Creates a DOM DocumentType using the DOM package of this DOMReader
     * @param document the document to create the DocumentType for
     * @param systemId the System ID of the DOCTYPE
     * @return the new DocumentType
    **/
    public void setDocumentType(Document document, String systemId) {
        //-- not yet available
    } //-- setDocumentType
    
    /**
     * Sets whether or not to Validate the Document
     * @param validate a boolean indicating whether or not to
     * validate the Document
    **/
    public void setValidation(boolean validate) {
        this.validate = validate;
    } //-- setValidation

      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * Reads an XML Document from the given Reader
     * @param reader the Reader for reading the XML stream
     * @param filename
     * @param errorWriter the PrintWriter to write all errors to
    **/
    public Document readDocument
        (Reader reader, String filename, PrintWriter errorWriter) 
    {
        this.pw = errorWriter;
        SimpleStreamProducer sp = new SimpleStreamProducer(filename);
        
	    try {
	        Parser parser = new Parser(filename, sp, sp);
	        //-- fix for XML4J 2.0.11+ compatibiliy
	        //-- Jason Mathews
	        parser.setProcessNamespace(true);
	        //-- end fix for XML4j 2.0.11+
            parser.setExpandEntityReferences(true);
            Document doc = parser.readStream(reader);
            return doc;
        }
        catch(Exception ex) {
            errorWriter.println("unable to read XML document: " + filename);
            ex.printStackTrace(errorWriter);
        }
		return null;
	} //-- readDocument
    
    class SimpleStreamProducer implements
        com.ibm.xml.parser.StreamProducer,
        com.ibm.xml.parser.ErrorListener
        
    {
        
        String documentBase = null;
        String filename     = null;
        
        SimpleStreamProducer(String filename) {
            this.filename = filename;
            this.documentBase = URIUtils.getDocumentBase(filename);
        } //-- SimpleStreamProducer
        
        //--------------------------------------------------/
        //-- methods from com.ibm.xml.parser.ErrorListener -/
        //--------------------------------------------------/
        
        /**
        * Defined by ErrorListener
        * @see com.ibm.xml.parser.ErrorListener
        **/
        public int error(java.lang.String fileName, int lineNo, int charOffset,
                    java.lang.Object key, java.lang.String msg) {

            
            if (fileName == null) fileName = this.filename;
            pw.print("error reading xml document: " + fileName + " (");
            pw.println(lineNo + ", " + charOffset + ")");
            pw.println(" - " + msg);
            pw.flush();
            return 1;

        } //-- error
        
        //---------------------------------------------------/
        //-- methods from com.ibm.xml.parser.StreamProducer -/
        //---------------------------------------------------/
    
        public void closeInputStream(Source source) {
            //-- do nothing for now
        } //-- closeInputStream
        
        public Source getInputStream
            (String name, String publicId, String systemId)
        {
            if (publicId != null) {
                InputSource is = (InputSource)catalog.get(publicId);
                return new Source(is.getCharacterStream());
            }
            
            //-- should we cache stream in case we see entity again?
            try {
                InputStream is = URIUtils.getInputStream(systemId, 
                    this.documentBase);
                return new Source(is);
            }
            catch(java.io.IOException ioe) {
                pw.println("unable to open source: " + systemId);
            }
            return null;
            
        } //-- getInputStream
        
        public void loadCatalog(Reader reader) {
            catalog.loadCatalog(reader);
        } //-- loadCatalog
    } //-- SimpleStreamProducer
    
} //-- DOMParser

