/*
 * (C) Copyright Keith Visco 1998  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl;

import org.w3c.dom.*;
import java.io.PrintWriter;
import java.util.Hashtable;

import com.kvisco.xsl.util.*;
import com.kvisco.util.List;

/**
 * This class represents a UnionExpr
 * <PRE>
 * UnionExpr ::= PathExpr | (PathExpr '|' UnionExpr)
 * </PRE>
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
class UnionExpr extends List implements Expr, MatchExpr {


     
      //----------------/
     //- Constructors -/
    //----------------/
    
    /**
     * Creates an empty UnionExpr
    **/
    protected UnionExpr() {
        super();
    }
    
      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * adds the given PathExpr to this UnionExpr
     * @param expr the PathExpr to append
    **/
    public void add(PathExpr expr) {
        super.add(expr);
    }
    
    /**
     * Inserts the given PathExpr at the given index
     * @param expr the PathExpr to insert
     * @param index the index to insert the PathExpr at
    **/
    public void add(PathExpr expr, int index) {
        super.add(index, expr);
    } //-- add
    
    /**
     * Evaluates this Expr using the given context Node and ProcessorState
     * @param context the current context Node
     * @param ps the ProcessorState that contains the current processing 
     * environment
     * @return the ExprResult
    **/
    public ExprResult evaluate(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        
        NodeSet nodes = new NodeSet();
        
        for (int i = 0; i < size(); i++) {
            PathExpr pathExpr = (PathExpr)get(i);
            NodeSet temp = (NodeSet)pathExpr.evaluate(context, ps);
            nodes.add(temp);
        }
        //-- for now
        return nodes;
    } //-- evaluate
    
    /**
     * Returns the type of Expr this Expr represents
     * @return the type of Expr this Expr represents
    **/
    public short getExprType() {
        return Expr.NODE_SET;
    } //-- getExprType
    
    /**
     * Returns the String representation of this UnionExpr
     * @return the String representation of this UnionExpr
    **/
    public String toString() {
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < size(); i++) {
            if (i != 0) sb.append(" | ");
            sb.append( ((PathExpr)get(i)).toString() );
        }
        return sb.toString();
        
    } //-- toString

    /**
     * Retrieves the PathExpr that matches the given node. If more
     * than one PathExpr matches the given node, the most specific
     * PathExpr will be returned.
     * @param node the node to test for matching
     * @return the matching PathExpr or null if none match
    **/
    public PathExpr getMatchingExpr
        (Node node, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        PathExpr match = null;
        for (int i = 0; i < size(); i++) {
            PathExpr pathExpr = (PathExpr)get(i);
            //-- we may need to look further than the
            //-- first expression, incase this unionExpr contains
            //-- two path expressions that match the same node
            if (pathExpr.matches(node, context, ps)) {
                if (match == null) match = pathExpr;
                else {
                    if (pathExpr.getDefaultPriority() > 
                        match.getDefaultPriority())
                        match = pathExpr;
                }
            }
        }
        return match;
    } //-- getMatchingExpr
    
    
    /**
     * Determines if the given node is matched by this MatchExpr with
     * respect to the given context node.
     * @param node the node to determine a match for
     * @param context the Node which represents the current context
     * @param ps the current ProcessorState
     * @return true if the given node is matched by this MatchExpr
    **/
    public boolean matches(Node node, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        return (getMatchingExpr(node, context, ps) != null);
    } //-- matches
    
} // -- UnionExpr