/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The contents of this file are released under an Open Source 
 * Definition (OSD) compliant license; you may not use this file 
 * execpt in compliance with the license. Please see license.txt, 
 * distributed with this file. You may also obtain a copy of the
 * license at http://www.clc-marketing.com/xslp/license.txt
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 *
 * 
 */
package com.kvisco.xsl;

import com.kvisco.util.*;
import com.kvisco.xsl.util.*;
import com.kvisco.xsl.functions.*;

import java.io.*;

/**
 * A class for parsing expression strings
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class ExpressionParser {

    private static final String PATTERN_SEP         = "|";

    private static WildCardExpr wildCardExpr = new WildCardExpr();

      //------------------/
     //- Public Methods -/
    //------------------/


    public static Expr createExpr(String exprString)
        throws InvalidExprException
    {
        ExprLexer lexer
            = new ExprLexer(exprString);

        Expr expr = createExpr(lexer);
        return expr;
    } //-- createExpr

    private static Expr createExpr (ExprLexer lexer)
        throws InvalidExprException
    {

        if (!lexer.hasMoreTokens()) missingExpr(lexer.toString(), null);

        Expr  expr       = null;
        QuickStack  exprs      = new QuickStack();
        QuickStack  ops        = new QuickStack();
        
        boolean cFlag = true;
        int currentPos = lexer.getPosition();
        
        while(lexer.hasMoreTokens() && cFlag) {

            Token tok = lexer.nextToken();

            //System.out.println("tok: " + tok);

            if (lexer.isBinaryOp(tok)) {
                if (!exprs.empty()) {
                    if (!hasGreaterPrecedence(tok, (Token)ops.peek())) {
                        expr = createBinaryExpr((Expr)exprs.pop(), expr, 
                            (Token)ops.pop());
                    }
                }
                exprs.push(expr);
                ops.push(tok);
                continue;
            }
            switch (tok.type) {
                case Token.R_PAREN:
                case Token.R_BRACKET:
                case Token.COMMA:
                    lexer.pushBack();
                    cFlag = false;
                    break;
                case Token.LITERAL:
                    PrimaryExpr pExpr = new PrimaryExpr(PrimaryExpr.LITERAL);
                    pExpr.setLiteral(tok.value);
                    expr = pExpr;
                    break;
                case Token.UNION_OP:

                    if (expr == null) {
                        throw new InvalidExprException("| cannot start an expr");
                    }
                    else if (expr instanceof PathExpr ) {
                        UnionExpr unionExpr = createUnionExpr(lexer);
                        unionExpr.add((PathExpr)expr, 0);
                        expr = unionExpr;
                    }
                    else {
                        throw new InvalidExprException();
                    }
                    break;
                case Token.ANCESTOR_OP:
                case Token.PARENT_OP:

                    if (expr == null) {
                        //-- fixed for absolute PathExpr
                        //-- submitted by Rolande Kendal
                        lexer.pushBack();
                        expr = createPathExpr(lexer);
                    }
                    else {
                        throw new InvalidExprException();
                    }
                    break;
                default:

                    lexer.pushBack();
                    //System.out.println("#default: " + tok);
                    //-- try for PathExpr
                    if (isLocationStepToken(tok)) {
                        if (expr == null) {
                            expr = createPathExpr(lexer);
                        }
                        else if (expr instanceof PathExpr) {
                            ((PathExpr)expr).add(createLocationStep(lexer));
                        }
                        else {
                            throw new InvalidExprException();
                        }
                        break;
                    }
                    
                    PrimaryExpr primaryExpr = createPrimaryExpr(lexer);
                    //-- try FilterExpr
                    //-- look ahead for predicate list
                    Token nextTok = lexer.lookAhead(0);
                    if ((nextTok != null) &&
                        (nextTok.type == Token.L_BRACKET)) {
                        PathExpr pathExpr = null;
                        FilterExpr filter = new FilterExpr();
                        filter.setPrimaryExpr(primaryExpr);
                        parsePredicates(filter, lexer);
                        pathExpr = new PathExpr();
                        pathExpr.add(filter);
                        expr = pathExpr;
                    }
                    else expr = primaryExpr;
                    break;
            } //-- end switch
        } //-- end while more tokens
        
        //-- finish Binary expressions
        while (!exprs.empty()) {
            expr = createBinaryExpr(
                (Expr)exprs.pop(), expr,(Token)ops.pop());
        }

        return expr;

    } //-- createExpr

    private static boolean hasGreaterPrecedence(Token op1, Token op2) {
        
        if (ExprLexer.isMultiplicativeOp(op1)) {
            if (ExprLexer.isMultiplicativeOp(op2)) return false;
            return true;
        }
        else if (ExprLexer.isAdditiveOp(op1)) {
            if (ExprLexer.isAdditiveOp(op2)) return false;
            return !ExprLexer.isMultiplicativeOp(op2);
        }
        else if (ExprLexer.isRelationalOp(op1)) {
            if (ExprLexer.isRelationalOp(op2)) return false;
            else if (ExprLexer.isAdditiveOp(op2)) return false;
            return !ExprLexer.isMultiplicativeOp(op2);
        }
        else if (ExprLexer.isEqualityOp(op1)) {
            if (ExprLexer.isEqualityOp(op2)) return false;
            else if (ExprLexer.isRelationalOp(op2)) return false;
            else if (ExprLexer.isAdditiveOp(op2)) return false;
            else return !ExprLexer.isMultiplicativeOp(op2);
        }
        else if (op1.type == Token.AND_OPNAME) {
            if (op2.type == Token.AND_OPNAME) return false;
            return (op2.type == Token.OR_OPNAME);
        }
        if (op2.type == Token.OR_OPNAME) return false;
        return true;
    } //-- hasGreaterPrecedence
    
    /**
     * Creates a new BinaryExpr
    **/
    private static Expr createBinaryExpr(Expr left, Expr right, Token op)
        throws InvalidExprException
    {
        if (left == null)
            throw new InvalidExprException("Missing left side of expression: ");
        if (right == null)
            throw new InvalidExprException("Missing right side of expression: ");

        switch (op.type) {
            case Token.OR_OPNAME:
                return new OrExpr(left, right);
            case Token.AND_OPNAME:
                return new AndExpr(left, right);
            case Token.EQUALS_OP:
                return new EqualityExpr(left, right, EqualityExpr.EQUAL);
            case Token.LESS_THAN_OP:
                return new EqualityExpr(left, right, EqualityExpr.LESS_THAN);
            case Token.GREATER_THAN_OP:
                return new EqualityExpr(left, right, EqualityExpr.GREATER_THAN);
            case Token.LESS_OR_EQ_OP:
                return new EqualityExpr(left, right, EqualityExpr.LT_OR_EQUAL);
            case Token.GREATER_OR_EQ_OP:
                return new EqualityExpr(left, right, EqualityExpr.GT_OR_EQUAL);
            case Token.ADDITION_OP:
                return new AdditiveExpr(left, right, AdditiveExpr.ADD);
            case Token.SUBTRACTION_OP:
                return new AdditiveExpr(left, right, AdditiveExpr.SUBTRACT);
            case Token.MULTIPLY_OP:
                return new MultiplicativeExpr(left, right,
                    MultiplicativeExpr.MULTIPLY);
            case Token.MOD_OPNAME:
                return new MultiplicativeExpr(left, right,
                    MultiplicativeExpr.MODULUS);
            case Token.DIV_OPNAME:
                return new MultiplicativeExpr(left, right,
                    MultiplicativeExpr.DIVIDE);
            case Token.QUO_OPNAME:
                return new MultiplicativeExpr(left, right,
                    MultiplicativeExpr.QUOTIENT);
            default:
                break;
        }

        throw new InvalidExprException
            ("Invalid binary expr: " + left + op + right);
    } //-- createBinaryExpr

    /**
     * Creates a PrimaryExpr using the given String
     * @param exprString the String to create the PrimaryExpr from
     * @return the new PrimaryExpr
     * @exception XSLException
    **/
    private static PrimaryExpr createPrimaryExpr(ExprLexer lexer)
        throws InvalidExprException
    {

        PrimaryExpr pExpr  = null;

        if (!lexer.hasMoreTokens())
            missingExpr(lexer.toString(), null);

        Token tok = lexer.nextToken();

        //-- Grouped Expression '(' Expr ')'
        switch (tok.type) {

            case Token.L_PAREN:
                pExpr = new PrimaryExpr(PrimaryExpr.EXPR);
                pExpr.setExpr(createExpr(lexer));
                tok = lexer.nextToken();
                if (tok.type != Token.R_PAREN)
                    unexpectedToken(lexer.toString(), tok);
                break;
            case Token.VAR_REFERENCE:
                pExpr = new PrimaryExpr(PrimaryExpr.VARIABLE_REFERENCE);
                pExpr.setLiteral(tok.value);
                break;
            case Token.LITERAL:
                pExpr = new PrimaryExpr(PrimaryExpr.LITERAL);
                pExpr.setLiteral(tok.value);
                break;
            case Token.FUNCTION_NAME:
                String name = tok.value;
                List params = parseParams(lexer);
                FunctionCall fnCall = createFunctionCall(name,params);
                pExpr = new PrimaryExpr(PrimaryExpr.FUNCTION_CALL);
                pExpr.setExpr(fnCall);
                break;
            case Token.NUMBER:
                try {
                    Double dbl = Double.valueOf(tok.value);
                    pExpr = new PrimaryExpr(PrimaryExpr.NUMBER);
                    pExpr.setNumber(dbl.doubleValue());
                }
                catch (NumberFormatException nfe) {
                    pExpr = new PrimaryExpr(PrimaryExpr.LITERAL);
                    pExpr.setLiteral(tok.value);
                }
                break;
        }
        if (pExpr == null)
            throw new InvalidExprException("Invalid PrimaryExpr: " +
                lexer.toString()+ " ->{"+tok.value+"}");

        return pExpr;

    } //-- createPrimaryExpr


    /**
     * creates a FilterExpr from the given string.
     * @param pattern the String to create the FilterExpr from
     * @returns the new FilterExpr
     * @exception InvalidExprException
    **/
    public static FilterExpr createFilterExpr(String pattern)
        throws InvalidExprException
    {
        return createFilterExpr(new ExprLexer(pattern));

    } //-- createFilterExpr

    /**
     * Creates a FilterExpr from the given string.
     * @param tokenizer the set of tokens to create the FilterExpr from
     * @param tokenStack the current Stack of group tokens
     * @returns the new FilterExpr
     * @exception InvalidExprException
    **/
    private static FilterExpr createFilterExpr (ExprLexer lexer)
        throws InvalidExprException
    {

        FilterExpr filterExpr = new FilterExpr();

        PrimaryExpr primaryExpr = createPrimaryExpr(lexer);
        if (primaryExpr == null) {
            throw new InvalidExprException(lexer.toString());
        }
        filterExpr.setPrimaryExpr(primaryExpr);
        parsePredicates(filterExpr, lexer);
        return filterExpr;

    } //-- createFilterExpr


    /**
     * Creates a LocationStep from the given string.
     * @param pattern the String to create the LocationStep from
     * @returns the new LocationStep
     * @exception InvalidExprException
    **/
    public static LocationStep createLocationStep(String pattern)
        throws InvalidExprException
    {

        return createLocationStep(new ExprLexer(pattern));

    } //-- createLocationStep


    public static boolean isLocationStepToken(Token token) {

        if (token == null) return false;
        if (isAxisIdentifierToken(token)) return true;
        return isNodeTypeToken(token);

    } //-- isLocationStep;

    public static boolean isAxisIdentifierToken(Token token) {
        if (token == null) return false;
        switch(token.type) {
            case Token.ANCESTORS_AXIS:
            case Token.ANCESTORS_OR_SELF:
            case Token.ATTRIBUTES_AXIS:
            case Token.CHILDREN_AXIS:
            case Token.DESCENDANTS_AXIS:
            case Token.DESCENDANTS_OR_SELF:
            case Token.FOLLOWING_AXIS:
            case Token.FOLLOWING_SIBLINGS_AXIS:
            case Token.PARENT_AXIS:
            case Token.PRECEDING_AXIS:
            case Token.PRECEDING_SIBLINGS_AXIS:
            case Token.SELF_AXIS:
                return true;
            default:
                return false;
        }

    } //--  isAxisIdentifierToken

    public static boolean isNodeTypeToken(Token token) {
        if (token == null) return false;
        switch(token.type) {
            case Token.TEXT:
            case Token.COMMENT:
            case Token.PI:
            case Token.NODE:
            case Token.AT_SYMBOL:
            case Token.SELF_NODE:
            case Token.PARENT_NODE:
            case Token.WILDCARD:
            case Token.CNAME:
                return true;
            default:
                return false;
        }
    } //-- isNodeTypeToken

    /**
     * Creates a LocationStep from the given string.
     * @param tokenizer the set of tokens to create the LocationStep from
     * @param tokenStack the current Stack of group tokens
     * @returns the new LocationStep
     * @exception InvalidExprException
    **/
    private static LocationStep createLocationStep (ExprLexer lexer)
        throws InvalidExprException
    {

        LocationStep locationStep = new LocationStep();

        Token cTok = lexer.lookAhead(0);
        
        short axisIdentifier = LocationStep.CHILDREN_AXIS;

        int offset = 1;
        switch (cTok.type) {
            case Token.ANCESTORS_AXIS:
                axisIdentifier = LocationStep.ANCESTORS_AXIS;
                break;
            case Token.ANCESTORS_OR_SELF:
                axisIdentifier = LocationStep.ANCESTORS_OR_SELF_AXIS;
                break;
            case Token.ATTRIBUTES_AXIS:
                axisIdentifier = LocationStep.ATTRIBUTES_AXIS;
                break;
            case Token.CHILDREN_AXIS:
                axisIdentifier = LocationStep.CHILDREN_AXIS;
                break;
            case Token.DESCENDANTS_AXIS:
                axisIdentifier = LocationStep.DESCENDANTS_AXIS;
                break;
            case Token.DESCENDANTS_OR_SELF:
                axisIdentifier = LocationStep.DESCENDANTS_OR_SELF_AXIS;
                break;
            case Token.FOLLOWING_AXIS:
                axisIdentifier = LocationStep.FOLLOWING_AXIS;
                break;
            case Token.FOLLOWING_SIBLINGS_AXIS:
                axisIdentifier = LocationStep.FOLLOWING_SIBLINGS_AXIS;
                break;
            case Token.PARENT_AXIS:
                axisIdentifier = LocationStep.PARENT_AXIS;
                break;
            case Token.PRECEDING_AXIS:
                axisIdentifier = LocationStep.PRECEDING_AXIS;
                break;
            case Token.PRECEDING_SIBLINGS_AXIS:
                axisIdentifier = LocationStep.PRECEDING_SIBLINGS_AXIS;
                break;
            case Token.SELF_AXIS:
                axisIdentifier = LocationStep.SELF_AXIS;
                break;
            default:
                offset = 0;
        }
        if (offset == 1) {
            if (lexer.lookAhead(1).type == Token.L_PAREN) offset = 2;
        }

        lexer.advance(offset);

        NodeExpr nodeExpr = createNodeExpr(lexer);
        if (nodeExpr == null) {
            throw new InvalidExprException(lexer.toString());
        }
        locationStep.setNodeExpr(nodeExpr);
        parsePredicates(locationStep, lexer);
        if (offset > 0) {
            cTok = lexer.nextToken();
            if (cTok.type != Token.R_PAREN)
                throw new InvalidExprException("missing closing parenthesis ')'");
        }
        else {
            //-- adjust axis identifer if necessary
            switch (nodeExpr.getNodeExprType()) {
                case NodeExpr.IDENTITY_EXPR:
                    axisIdentifier = LocationStep.SELF_AXIS;
                    break;
                case NodeExpr.PARENT_EXPR:
                    axisIdentifier = LocationStep.PARENT_AXIS;
                    break;
                case NodeExpr.ATTRIBUTE_EXPR:
                    axisIdentifier = LocationStep.ATTRIBUTES_AXIS;
                    break;
                default:
                    break;

            }
        }
        locationStep.setAxisIdentifier(axisIdentifier);

        return locationStep;

    } //-- createLocationStep

    /**
     * Parses the set of predicates for the given FilterBase
     * @param filterBase the FilterBase to add Predicates to
     * @param tokenizer the set of tokens to create the Predicates from
     * @param tokenStack the current Stack of group tokens
     * @exception InvalidExprException
    **/
    private static void parsePredicates
        (FilterBase filterBase, ExprLexer lexer)
            throws InvalidExprException
    {
        //-- handle predicates

        while ( lexer.hasMoreTokens() &&
                (lexer.lookAhead(0).type == Token.L_BRACKET) ) {

            lexer.nextToken(); //-- remove '['
            Expr expr = createExpr(lexer);
            filterBase.addPredicate(expr);
            //-- make sure we have end of predicate ']'
            Token tok = lexer.nextToken();
            if ((tok == null) || (tok.type != Token.R_BRACKET))
                unexpectedToken(lexer.toString(), tok);
        }

    } //-- parsePredicates

    /**
     * Creates the appropriate FunctionCall based on the given name
     * @param name the name of the function to call
     * @param params the List of Expr paramaters for the function call
     * @return the new FunctionCall
    **/
    public static FunctionCall createFunctionCall
        (String name, List params)
    {

        FunctionCall fnCall = null;

        name = name.intern();

        // BooleanFunctionCall
        if (name == Names.TRUE_FN)
            return new TrueFunctionCall();
        else if (name == Names.FALSE_FN)
            return new FalseFunctionCall();
        else if (name == Names.NOT_FN)
            fnCall = new NotFunctionCall();
        else if (name == Names.BOOLEAN_FN)
            fnCall = new BooleanFunctionCall();
        else if (name == Names.LANG_FN)
            fnCall = new LangFunctionCall();

        // NodeSetFunctionCall

        else if (name == Names.POSITION_FN)
            return new PositionFunctionCall();
        else if (name == Names.LAST_FN)
            return new LastFunctionCall();
        else if (name == Names.COUNT_FN)
            fnCall = new CountFunctionCall();
        else if (name == Names.LOCAL_PART_FN)
            fnCall = new XMLNamesFunctionCall(XMLNamesFunctionCall.LOCAL_PART);
        else if (name == Names.NAMESPACE_FN) 
            fnCall = new XMLNamesFunctionCall(XMLNamesFunctionCall.NAMESPACE);
        // -- new for XSLT WD 19990709 
        else if (name == Names.NAME_FN)
            fnCall = new XMLNamesFunctionCall();
        else if (name == Names.QNAME_FN)
            fnCall = new XMLNamesFunctionCall();
        else if (name == Names.GENERATE_ID_FN)
            fnCall = new GenerateIDFunctionCall();
            
            
        else if (name == Names.ID_FN)
            fnCall = new IdFunctionCall();
        else if (name == Names.IDREF_FN)
            fnCall = new IdRefFunctionCall();
            
        else if (name == Names.DOC_FN) {
            fnCall = new DocumentFunctionCall();
        }
        // -- new for XSLT WD 19990709 
        else if (name == Names.DOCUMENT_FN) {
            fnCall = new DocumentFunctionCall();
        }

        // NodeTest

        else if (name == Names.TEXT_FN)
            return new TextFunctionCall();

        // String Functions
        else if (name == Names.CONCAT_FN) {
            fnCall = new Concat();
        }
        else if (name == Names.CONTAINS_FN) {
            fnCall = new Contains();
        }
        else if (name == Names.FORMAT_NUMBER_FN) {
            fnCall = new FormatNumber();
        }
        else if (name == Names.NORMALIZE_FN) {
            fnCall = new Normalize();
        }
        else if (name == Names.STARTS_WITH_FN) {
            fnCall = new StartsWith();
        }
        else if (name == Names.STRING_FN) {
            fnCall = new StringFunctionCall();
        }
        // -- string-length() new for XSLT WD 19990709 
        else if (name == Names.STRING_LENGTH_FN) {
            fnCall = new StringLength();
        }
        // -- substring new for XSLT WD 19990709 
        else if (name == Names.SUBSTRING_FN) {
            fnCall = new Substring();
        }
        else if (name == Names.SUBSTRING_BEFORE_FN) {
            fnCall = new SubstringBefore();
        }
        else if (name == Names.SUBSTRING_AFTER_FN) {
            fnCall = new SubstringAfter();
        }
        else if (name == Names.TRANSLATE_FN) {
            fnCall = new Translate();
        }

        //-- NumberFunctionCalls
        else if (name == Names.SUM_FN) {
            fnCall = new SumFunctionCall();
        }
        else if (name == Names.NUMBER_FN) {
            fnCall = new NumberFunctionCall();
        }
        else if (name == Names.FLOOR_FN) {
            fnCall = new NumberFunctionCall(NumberFunctionCall.FLOOR);
        }
        else if (name == Names.CEILING_FN) {
            fnCall = new NumberFunctionCall(NumberFunctionCall.CEILING);
        }
        else if (name == Names.ROUND_FN) {
            fnCall = new NumberFunctionCall(NumberFunctionCall.ROUND);
        }

        //-- system property
        else if (name == Names.SYSTEM_PROPERTY_FN) {
            fnCall = new SystemFunctionCall();
        }
        else if (name == Names.FUNCTION_AVAILABLE_FN) {
            fnCall = new SystemFunctionCall
                            (SystemFunctionCall.FUNCTION_AVAILABLE);
        }


        if (fnCall == null) {
            fnCall = new ExtensionFunctionCall(name);
        }

        for (int i = 0; i < params.size(); i++) {
            fnCall.addExprParameter( (Expr)params.get(i) );
        }
        return fnCall;

    } //-- createFunctionCall

    /**
     * Parses the a pattern String into a MatchExpr
     * @param matchString the pattern string to create the MatchExpr from
     * @return the new MatchExpr
     * @exception InvalidExprException
    **/
    public static MatchExpr createMatchExpr(String matchString)
        throws InvalidExprException
    {
        MatchExpr matchExpr = createUnionExpr(matchString);
        return matchExpr;
    } //-- createMatchExpr

    /**
     * creates a NodeExpr from the given string argument
     * @param pattern the string to create the RefrencePattern from
     * @return the NodeExpr or null if the string was not a
     *  valid NodeExpr
     * @exception InvalidExprException
    **/
    public static NodeExpr createNodeExpr (ExprLexer lexer)
        throws InvalidExprException
    {

        NodeExpr nodeExpr = null;


        Token tok = lexer.nextToken();

        if (tok == null) return null;

        if (lexer.hasMoreTokens() &&
            (lexer.lookAhead(0).type == Token.L_PAREN)) {

            MatchExpr matchExpr = null;
            lexer.advance(1);

            //-- read param
            Token pTok = lexer.nextToken();
            if (pTok == null)
                unexpectedToken(lexer.toString(), null);

            String param = null;
            while (pTok.type != Token.R_PAREN) {

                if (param == null) param = pTok.value;
                else param += pTok.value;

                pTok = lexer.nextToken();
                if (pTok == null)
                    unexpectedToken(lexer.toString(), null);
            }

            switch (tok.type) {

                case Token.TEXT:
                    nodeExpr = new TextFunctionCall();
                    break;
                case Token.COMMENT:
                    nodeExpr = new CommentExpr();
                    break;
                case Token.PI:
                    nodeExpr = new PIExpr(param);
                    break;
                case Token.NODE:
                    nodeExpr = new AnyNodeExpr();
                    break;
                default:
                    throw new InvalidExprException();
            }
        }
        else {
            switch (tok.type) {
                case Token.AT_SYMBOL:
                    Token nTok = lexer.nextToken();
                    switch (nTok.type) {
                        case Token.CNAME:
                        case Token.WILDCARD:
                            break;
                        default:
                            unexpectedToken(lexer.toString(), nTok);
                            break;
                    }
                    nodeExpr = new AttributeExpr(nTok.value);
                    break;
                case Token.ATTRIBUTES_AXIS:
                    nodeExpr = new AttributeExpr(tok.value);
                    break;
                case Token.SELF_NODE:
                    nodeExpr = new IdentityExpr();
                    break;
                case Token.PARENT_NODE:
                    nodeExpr = new ParentExpr();
                    break;
                case Token.WILDCARD:
                    nodeExpr = wildCardExpr;
                    break;
                case Token.NUMBER:
                    throw new InvalidExprException();
                default:
                    nodeExpr = new ElementExpr(tok.value);
                    break;
            }
        }

        return nodeExpr;

    } // -- createNodeExpr


    /**
     * Creates a PathExpr from the string argument.
     * @param pattern the string to create the PathExpr from
     * @return the new PathExpr
     * @exception InvalidExprException
    **/
    public static PathExpr createPathExpr(String pattern) 
        throws InvalidExprException
    {
        return createPathExpr(new ExprLexer(pattern));
    } //-- createPathExpr;
    
    /**
     * Creates a PathExpr from the string argument.
     * @param lexer the ExprLexer to create the PathExpr from
     * @return the new PathExpr
     * @exception InvalidExprException
    **/
    public static PathExpr createPathExpr(ExprLexer lexer) 
        throws InvalidExprException 
    {
        
        
        PathExpr pathExpr = new PathExpr();
        
        //-- look for empty PathExpr
        if (!lexer.hasMoreTokens()) return pathExpr;
        
        boolean isAbsolute = false;
        
        
        Token tok = lexer.lookAhead(0);
        
        if (tok.type == Token.PARENT_OP) {
            if (lexer.lookAhead(1) == null) {
                lexer.nextToken();
                return new RootExpr();
            }
            isAbsolute = true;
        }
        else isAbsolute = (tok.type == Token.ANCESTOR_OP);
        
        int ancestryOp = FilterBase.NO_OP;
        
        boolean done = false;
        while (lexer.hasMoreTokens() && (!done) ) {
            
            tok = lexer.nextToken();
            
            if (lexer.isBinaryOp(tok)) {
                lexer.pushBack();
                break;
            }
            switch (tok.type) {
                //--
                case Token.COMMA:
                case Token.R_PAREN:
                case Token.R_BRACKET:
                case Token.UNION_OP:
                    lexer.pushBack();
                    done = true;
                    break;
                case Token.PARENT_OP:
                    ancestryOp = FilterBase.PARENT_OP;
                    break;
                case Token.ANCESTOR_OP:
                    ancestryOp = FilterBase.ANCESTOR_OP;
                    break;
                default: 
                    lexer.pushBack();
                    FilterBase filterBase = null;
                    //-- try to create a LocationStep
                    if (isLocationStepToken(tok)) {
                        filterBase = createLocationStep(lexer);
                    }
                    else {
                        filterBase = createFilterExpr(lexer);
                    }
                    //-- set ancestry op
                    filterBase.setAncestryOp(ancestryOp);
                    
                    //-- add filterbase to pathExpr
                    pathExpr.add(filterBase);
                    
                    //-- reset ancestry op
                    ancestryOp = FilterBase.NO_OP;
                    break;
            }
        } //-- end while
        
        return pathExpr;
    } // -- createPathExpr
    
    /**
     * Parses the a pattern String into a SelectExpr
     * @param selectString the pattern string to create the SelectExpr from
     * @return the new SelectExpr
     * @exception XSLException
    **/
    public static SelectExpr createSelectExpr(String selectString) 
        throws InvalidExprException
    {
        SelectExpr selectExpr = new SelectExpr();
        selectExpr.setUnionExpr(createUnionExpr(selectString));
        return selectExpr;
    } //-- createSelectExpr

    /**
     * Creates a StringExpr from the given String
     * @param attValue the String to create the StringExpr from
     * @return the new StringExpr
     * @exception InvalidExprException
    **/
    public static StringExpr createStringExpr(String attValue) 
        throws InvalidExprException 
    {  
        return new StringExpr(attValue);
        
    } //-- createStringExpr

    /**
     * Creates a UnionExpr from the given string argument.
     * @param pattern the string to create the UnionExpr from
     * @return the new UnionExpr
     * @exception InvalidExprException
    **/
    public static UnionExpr createUnionExpr(String pattern) 
        throws InvalidExprException
    {
        return createUnionExpr(new ExprLexer(pattern));
    } //-- createUnionExpr
    
    /**
     * Creates a UnionExpr from the given string argument.
     * @param lexer the ExprLexer to create the UnionExpr from
     * @return the new UnionExpr
     * @exception InvalidExprException
    **/
    private static UnionExpr createUnionExpr(ExprLexer lexer) 
        throws InvalidExprException
    {
        
        UnionExpr unionExpr = new UnionExpr();
        
        boolean done = false;
        while (lexer.hasMoreTokens() && (!done)) {
            
            Token tok = lexer.nextToken();
            
            if (lexer.isBinaryOp(tok)) {
                //-- should create an error
                lexer.pushBack();
                break;
            }
            switch(tok.type) {
                case Token.R_BRACKET:
                case Token.R_PAREN:
                    lexer.pushBack();
                    done = true;
                    break;
                case Token.UNION_OP:
                    if (unionExpr.size() == 0)
                        throw new InvalidExprException();
                    break;
                default:
                    lexer.pushBack();
                    unionExpr.add(createPathExpr(lexer));
                    break;
            }
        }
        //System.out.println("#unionExpr: " + unionExpr);
        
        return unionExpr;

    } //-- createUnionExpr
    
    
    
      //-------------------/
     //- Private Methods -/
    //-------------------/
    
    /**
     * Parses a list of parameters 
    **/
    private static List parseParams (ExprLexer lexer) 
        throws InvalidExprException
    {
        List params = new List();
        
        
        Token tok = lexer.nextToken();
        
        //-- look for start of parameter list '('
        if (tok.type != Token.L_PAREN)
            missingExpr(lexer.toString(), tok);

        //-- read all parameters
        
        while (true) {
            if (!lexer.hasMoreTokens())
                missingExpr(lexer.toString(), tok);
                
            //-- look for ending of parameter list
            tok = lexer.lookAhead(0);
            if (tok.type == Token.R_PAREN) {
                lexer.advance(1);
                break;
            }
            //-- create parameter
            params.add(createExpr(lexer));
            
            tok = lexer.nextToken();
            if (tok.type == Token.R_PAREN) break;
            else if (tok.type != Token.COMMA)
                unexpectedToken(lexer.toString(),tok);
                    
        }
        return params;
    } //-- parseParams
    

    private static final String INVALID_EXPR = "Invalid expression: ";


    /**
     * This method will throws and InvalidExprException. It is used
     * when an Non Unary Expr is missing an Expr component.
     * @param exprString the current expression String being parsed
     * @param op the binary operator
     * @exception InvalidExprException always
    **/
    private static void missingExpr(String exprString, Token token) 
        throws InvalidExprException
    {
        StringBuffer error = new StringBuffer(INVALID_EXPR);
        error.append(exprString);
        error.append("; '");
        if (token != null) error.append(token.value);
        error.append("' -> missing remainder of expression.");
        throw new InvalidExprException(error.toString());
    } //-- missingExpr

    /**
     * This method will throws and InvalidExprException. It is used
     * when an Expr operator appears in an unexpected place.
     * @param exprString the current expression String being parsed
     * @param op the binary operator
     * @exception InvalidExprException always
    **/
    private static void unexpectedToken(String exprString, Token token) 
        throws InvalidExprException
    {
        StringBuffer error = new StringBuffer(INVALID_EXPR);
        error.append(exprString);
        error.append("; '");
        if (token != null) error.append(token.value);
        else error.append("NULL");
        error.append("' is unexpected.");
        throw new InvalidExprException(error.toString());
    } //-- unexpectedToken
    
    
    /* For Debugging */
    
    public static void main(String[] args) 
        throws InvalidExprException
    {
        
        String[] UNION_EXPRS = {
            "//pi()",
            "cellphone.e-catalog",
            "//elementA/elementB",
            "../elementA[@name=$name][@type='foo']",
            "elementA",
            "elementA | elementB",
            "element-A | element-B | element-C",
            "element-A[@attr and (not(position() = 1))]",
        };

        String[] EXPRS = {
            "position() mod 2 = 0",
            "(position() mod 2) = 0",
            "count((//@name)[1])"
        };
        
        //-- UnionExpr tests
        System.out.println("UnionExpr Tests\n");
        
        for (int i = 0; i < UNION_EXPRS.length; i++) {
            System.out.println("Test:   " + UNION_EXPRS[i]);
            System.out.print("Result: ");
            System.out.println(createUnionExpr(UNION_EXPRS[i]));
        }
        System.out.println();
        //-- Expr tests
        System.out.println("Expr Tests\n");
        for (int i = 0; i < EXPRS.length; i++) {
            System.out.println("Test:   " + EXPRS[i]);
            System.out.print("Result: ");
            System.out.println(createExpr(EXPRS[i]));
        }
        
    } //-- main
    
} //-- ExpressionParser