/*
 * (C) Copyright Keith Visco 1998, 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xml;

import com.kvisco.xml.parser.*;

import org.w3c.dom.*;
import java.io.*;

/**
 * This class is a wrapper for whatever dom package is being used. 
 * @author Keith Visco (kvisco@ziplink.net)
**/
public class DOMReader {
    
    
    private static final String DEFAULT_DOM_PACKAGE 
        = "xslp.xml.parsers.XML4J_1xParser";
    
    /**
     * DOM package class to use
    **/
    private String domPackageClassName = DEFAULT_DOM_PACKAGE;
    
    /**
     * DOM package to use
    **/
    private DOMPackage domPackage = null;
    
    /**
     * Creates a new DOMReader for the default DOM Package
     * @exception ClassNotFoundException when the default DOMPackage
     * could not be found.
    **/
    public DOMReader() 
        throws Exception
    {
        this(null);
    }
    
    /**
     * Creates a new DOMReader for the specified DOM Package
     * @exception ClassNotFoundException when the specified DOMPackage
     * could not be found.
    **/
    public DOMReader(String domPackageClassName) 
        throws Exception
    {
        super();
        if (domPackageClassName != null) {
            this.domPackageClassName = domPackageClassName;
        }
        initDOMPackage();
    } //-- DOMReader
    
      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * Creates a DOM Document using the DOM Package of this DOMReader
     * @return the new Document
    **/
    public Document createDocument() {
        return domPackage.createDocument();
    } //-- createDocument
    
    /**
     * Returns the class name of the default DOMPackage 
     * @return the class name of the default DOMPackage 
    **/
    public static String getDefaultDOMPackageClassName() {
        return DEFAULT_DOM_PACKAGE;
    } //-- getDOMPackageName
    
    /**
     * Returns the DOMPackage being used
     * @return the DOMPackage being used
    **/
    public DOMPackage getDOMPackage() {
        return domPackage;
    } //-- getDOMPackage
    
    /**
     * Returns the class name of the DOMPackage being used
     * @return the class name of the DOMPackage being used
    **/
    public String getDOMPackageClassName() {
        return domPackageClassName;
    } //-- getDOMPackageName

    
	/**
	 * Reads an XML Document from the given InputStream. 
	 * @param xmlInput the XML InputStream
	 * @param filename the filename to use during error reporting.
     * @retutn the new Document
	**/
	public Document readDocument(InputStream xmlInput, String filename, boolean validate) {
        return readDocument(xmlInput, filename, validate, new PrintWriter(System.out,true));
	} //-- readDocument

	/**
	 * Reads an XML Document from the given InputStream. 
	 * @param xmlInput the XML InputStream
	 * @param filename the filename to use during error reporting.
	 * @param errorOuput is a Writer used for reporting errors
     * @retutn the new Document
	**/
	public Document readDocument (
	    InputStream xmlInput, 
	    String filename, 
	    boolean validate, 
	    Writer errorWriter ) 
	{
	    //- Performance Testing
	    /* *
	    long stime = System.currentTimeMillis();
	    /* */
	        
	    domPackage.setValidation(validate);
        Document doc = domPackage.readDocument(new InputStreamReader(xmlInput), 
                                           filename, 
                                           new PrintWriter(errorWriter,true));
                                           
	    //- Performance Testing
	    /* *
	    long etime = System.currentTimeMillis();
	    System.out.println("DOMReader#readDocument(" + filename + ") time: " + (etime-stime) + "(ms)");
	    /* */
        return doc;
		
	} //-- readDocument
    
    private void initDOMPackage() throws Exception {
        Class dpClass = Class.forName(domPackageClassName);
        domPackage = (DOMPackage)dpClass.newInstance();
    } //-- initDOMPackage
    
} //-- DOMReader