package com.jtauber.fop.apps;

// FOP
import com.jtauber.fop.layout.AreaTree;
import com.jtauber.fop.layout.FontInfo;
import com.jtauber.fop.render.Renderer;
import com.jtauber.fop.render.pdf.PDFRenderer; // for default
import com.jtauber.fop.fo.FOTreeBuilder;
import com.jtauber.fop.fo.StandardElementMapping;
import com.jtauber.fop.svg.SVGElementMapping;

// Java
import java.io.PrintWriter;
import java.io.IOException;

/**
 * base class for FOP Processors.
 *
 * should be subclassed for each way of acquiring a formatting object tree
 * format(PrintWriter) should trigger the methods of FOTreeBuilder then run
 * layoutput(PrintWriter) (implemented in this class)
 */
public abstract class FOProcessor {
    
    /** the FO tree builder */
    protected FOTreeBuilder treeBuilder;

    /** the renderer to use */
    protected Renderer renderer;

    /** the error handler to use */
    protected ErrorHandler errorHandler;

    /** 
     * create a new FOP Processor
     *
     * @param Renderer the renderer to use
     */
    protected FOProcessor(Renderer renderer) {
	this.renderer = renderer;
	this.treeBuilder = new FOTreeBuilder();

	/* element mappings (the element mapping class names should
	   eventually be in a property */
	StandardElementMapping.addToBuilder(this.treeBuilder);
	SVGElementMapping.addToBuilder(this.treeBuilder);

	/* not yet implemented */
	//this.errorHandler = new DefaultErrorHandler();
    }

    /**
     * create a new FOP Processor with the default renderer
     */
    protected FOProcessor() {
	this(new PDFRenderer(com.jtauber.fop.Version.getVersion()));
    }

    /**
     * this method, when implemented, should trigger the methods of
     * FOTreeBuilder then run layoutput(PrintWriter) (implemented in
     * this class)
     */
    public abstract void format(PrintWriter writer)
	throws FOPException;

    /**
     * lays out the document and renders it.
     *
     * generally called by a subclasses's format(PrintWriter) method
     */
    protected final void layoutput(PrintWriter writer) throws FOPException {

	/* set up a font info object, because area tree needs it for metrics */
	FontInfo fontInfo = new FontInfo();
	this.renderer.setupFontInfo(fontInfo);

	AreaTree areaTree = new AreaTree();
	areaTree.setFontInfo(fontInfo);

	this.treeBuilder.format(areaTree);
		
	try {
	    this.renderer.render(areaTree, writer);
	} catch (IOException e) {
	    throw new FOPException(e.getMessage());
	}
    }
}
