/*
 *  Sarien AGI :: Copyright (C) 1999 Dark Fiber 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else
#include "getopt.h"
#endif

#include "sarien.h"
#include "font.h"
#include "sound.h"

UINT8	optForce;		/* force loading of all resources */
UINT8	optCache;		/* cache loaded resources */
UINT8	optGameRun;		/* status for game */
UINT8	optPCJrGFX;
UINT8	optShowScreenDraw;
UINT8	optShowKeyPress;
UINT16	optEmuVersion;
#ifndef NO_DEBUG
UINT8	optDebug;
#endif
UINT8	optScale;		/* window scale factor */
UINT8	optAGDS;
UINT8	optAmiga;
UINT8	optFullScreen;		/* full screen mode if available */
UINT8	optNoSound;		/* disable sound */
#ifdef MITSHM
UINT8	optMitShm;		/* use shared memory extension */
#endif
#ifdef XF86DGA
UINT8   optDGA;			/* use XFree86 DGA extension */
#endif
UINT8	optSoundEmu;		/* Sound emulation mode */

extern char *optarg;

static void help (int argc, char **argv)
{

	printf (
"Syntax is -:- %s [switches] [gamedir]\n"
"\n"
"Where [switches] are optionally:\n"
#ifdef AGDS_SUPPORT
"  -a --agds          Enables sarien to play AGDS created games.\n"
#endif
"  -A --amiga         Forces the game to be seen as an Amiga word padded game.\n"
#if 0
"  -c --cache-loading Cache loading, static resources remain cached in memory\n"
"                     but are only loaded as needed.\n"
#endif
"  -d --list-dictionary\n"
"                     List dictionary words.\n"
#ifndef NO_DEBUG
"  -D --debug         Turn on debugging.\n"
#endif
"  -E --emulate-sound {type}\n"
"                     Emulate the sound of Sierra AGI running in different\n"
"                     computers. Valid emulations are: pc, tandy and mac\n"
#if 0
"  -l -force-loading  Force loading of all volume information at start of\n"
"                     game which will give you faster play, this also implies\n"
"                     the caching.\n"
#endif
"  -F --full-screen   Run in full-screen mode if allowed by the graphics device\n"
#if 0
"  -g --dga           Use XFree86 DGA extension (if available).\n"
#endif
"  -h --help          Display this help screen.\n"
"  -n --no-sound      Disable sound output.\n"
"  -o --list-objects  List objects.\n"
"  -p --picture-viewer\n"
"                     Interactive picture viewer.\n"
"  -ps --show-drawing Show screen as its drawn.\n"
"  -pk --wait-key     Wait for keypress on each action. (forces -ps)\n"
"  -S --scale {num}   Window size scale (only for windowed graphics).\n"
"  -v --emulate-version {version}\n"
"                     Force version to emulate. Valid v2 game versions are:\n"
"                     2.089, 2.272, 2.440, 2.917, 2.936. Valid v3 games are:\n"
"                     3.002.086, 3.002.149.\n"
"  -V --version       Display version information.\n"
"  -x --no-x-shm      Disable X shared memory extension (if available).\n"
"\n"
"[gamedir] is optionally the directory the game is in, if no directory is\n"
"specified, the current directory is assumed.\n"
"\n"
"Example: %s -v2.272 /games/sierra/sq1\n", argv[0], argv[0]);
}


UINT16 parse_cli (int argc, char **argv)
{
	struct vstrings {
		UINT16	vers;
		UINT8	*string;
	} cmp_versions[]= {
		{0x2089, (UINT8*)"2.089"},
		{0x2272, (UINT8*)"2.272"},
		{0x2440, (UINT8*)"2.440"},
		{0x2917, (UINT8*)"2.917"},
		{0x2936, (UINT8*)"2.936"},

		{0x3086, (UINT8*)"3.002.086"},
		{0x3149, (UINT8*)"3.002.149"},
		{0x0, (UINT8*)""}
	};
	int o, optidx = 0;
#define OPTIONS "VhaAv:DdE:FlnopskS:x"
	static struct option lopt[] = {
		{ "version",		0, 0, 'V' },
		{ "help",		0, 0, 'h' },
#ifdef AGDS_SUPPORT
		{ "agds",		0, 0, 'a' },
#endif
		{ "amiga",		0, 0, 'A'},
		{ "emulate-version",	1, 0, 'v' },
#if 0
		{ "force-loading",	0, 0, 'l' },
		{ "cache-loading",	0, 0, 'c' },
#endif
		{ "list-dictionary",	0, 0, 'd' },
		{ "debug",		0, 0, 'D' },
		{ "full-screen",	0, 0, 'F' },
		{ "no-sound",		0, 0, 'n' },
		{ "list-objects",	0, 0, 'o' },
		{ "emulate-sound",	1, 0, 'E' },
		{ "picture-viewer",	0, 0, 'p' },
		{ "show-drawing",	0, 0, 's' },
		{ "wait-key",		0, 0, 'k' },
		{ "no-x-shm",		0, 0, 'x' },
		{ "scale",		1, 0, 'S' }
	};

	UINT16 xc;
	UINT16 ec=err_OK;

	/* FIXME: Add support for a rc file for UNIX */

	/* Set defaults */
	optGameRun = gRUN_GAME;
	optCache = __FALSE;
	optForce = __FALSE;
	optPCJrGFX = __FALSE;
	optShowKeyPress = __FALSE;
	optShowScreenDraw = __FALSE;
#ifndef NO_DEBUG
	optDebug = __FALSE;
#endif
	optEmuVersion = 0x0;
	optScale = 2;
	optAGDS = __FALSE;
	optAmiga = __FALSE;
	optFullScreen = __FALSE;
	optNoSound = __FALSE;
#ifdef MITSHM
	optMitShm = __TRUE;
#endif
	optSoundEmu = SOUND_EMU_NONE;

	while ((o = getopt_long (argc, argv, OPTIONS, lopt, &optidx)) != -1) {
		switch (o) {
		case 'V':
			printf ("Compiled on " __DATE__ "; " __TIME__ "\n");
/*
#ifdef TRADITIONAL_SIERRA
			printf ("Traditional Sierra colouring defined\n");
#else
			printf ("Alternate colouring defined\n");
#endif
*/
			printf ("Disable Copyprotection is ");
#ifdef DISABLE_COPYPROTECTION
			printf ("enabled\n");
#else
			printf ("disabled\n");
#endif
			printf("AGDS games and Russian/Cryllic font is ");
#ifdef AGDS_SUPPORT
			printf("supported\n");
#else
			printf("not supported\n");
#endif
			exit (0);
		case 'A':
			optAmiga = __TRUE;
			break;
#ifndef NO_DEBUG
		case 'D':
			optDebug = __TRUE;
			break;
#endif
		case 'E':
			if (!strcmp (optarg, "pc"))
				optSoundEmu = SOUND_EMU_PC;
			else if (!strcmp (optarg, "mac"))
				optSoundEmu = SOUND_EMU_MAC;
			else {
				fprintf (stderr, "Sound emulation \"%s\" is "
					"unknown\n", optarg);
				exit (0);
			}
			break;
		case 'F':
			optFullScreen = __TRUE;
			break;
		case 'd':
			optGameRun = gSHOW_WORDS;
			break;
		case 'o':
			optGameRun = gSHOW_OBJECTS;
			break;
		case 'n':
			optNoSound = __TRUE;
			break;
		case 'v':
			for (xc = 0; xc != 0xfff0 && cmp_versions[xc].vers; xc++) {
				if (!strcmp ((char*)optarg, (char*)cmp_versions[xc].string)) {
					optEmuVersion = cmp_versions[xc].vers;
					xc = 0xfff0 - 1;
				}
			}
			if (!optEmuVersion) {
				printf ("Error: bad AGI version number \"%s\"\n",
					optarg);
				exit (-1);
			}
			break;
		case 'a':
			optAGDS=__TRUE;
			break;
		case 'c':
			optCache= __TRUE;
			break;
		case 'f':
			optForce = __TRUE;
			optCache = __TRUE;
			break;
		case 'S':
			optScale = strtoul (optarg, NULL, 0);
			if (optScale < 1)
				optScale = 1;
			if (optScale > 4)
				optScale = 4;
			break;
		case 'k':
			optShowKeyPress=__TRUE;
		case 's':
			optShowScreenDraw = __TRUE;
		case 'p':
			optGameRun = gVIEW_PICTURES;
			break;
#ifdef MITSHM
		case 'x':
			optMitShm = __FALSE;
			break;
#endif
		case 'h':
			help (argc, argv);
		default:
			exit (-1);
		}
	}

	return ec;
}
