/*

------------------------------------------------------------------------------

A license is hereby granted to reproduce this software source code and
to create executable versions from this source code for personal,
non-commercial use.  The copyright notice included with the software
must be maintained in all copies produced.

THIS PROGRAM IS PROVIDED "AS IS". THE AUTHOR PROVIDES NO WARRANTIES
WHATSOEVER, EXPRESSED OR IMPLIED, INCLUDING WARRANTIES OF
MERCHANTABILITY, TITLE, OR FITNESS FOR ANY PARTICULAR PURPOSE.  THE
AUTHOR DOES NOT WARRANT THAT USE OF THIS PROGRAM DOES NOT INFRINGE THE
INTELLECTUAL PROPERTY RIGHTS OF ANY THIRD PARTY IN ANY COUNTRY.

Copyright (c) 1995, 1996, John Conover, All Rights Reserved.

Comments and/or bug reports should be addressed to:

    john@johncon.com (John Conover)

------------------------------------------------------------------------------

searchfile.c, read and search a file

int searchfile (char *filename, BMHPATTERN *list);

    open and search a file, named filename, reading the file, for
    instances of pattern, unlock the file, close the file, and
    deallocate any memory allocated by this module

    a shutdown procedure is implemented by using static variables for
    the file descriptor, file name, and memory reference page; if, for
    example under interrupt conditions, the program is aborted, the
    module relclose() (which was installed as an interrupt handler,)
    is called, which in turn calls a routine, int_searchfile(), to
    close the file

The algorithm is as follows:

    open the file for reading

    lock the file for reading

    stat the file for its size

    allocate memory for the file

    read the file

    search the file for the pattern

    unlock the file, if that fails

    close the file, if that fails

    deallocate memory for the file

Usage is a call with the file name, for example:

    int retval;

    if ((retval = searchfile (name, pattern_stack)) != NO_ERROR)
    {
        return (retval);
    }

For a detailed description of the applications of the file related
POSIX functions used, see "POSIX Programmer's Guide," Donald
A. Lewine, Editor: Dale Dougherty, O'Reilly & Associates, Inc., 103
Morris Street, Suite A, Sebastopol, CA 95472, 1991, ISBN
0-937175-73-0, Chapter 4, pp 63.

The argument, name, is the name of the file to be searched, the
argument pattern_stack is the search stack, compiled in bmhsearch.c.

Zero is returned if successful, non-zero if not.

To test this module, compile the module source with -DTEST_SEARCHFILE

$Revision: 1.3 $
$Date: 1996/09/13 13:47:23 $
$Id: searchfile.c,v 1.3 1996/09/13 13:47:23 john Exp $
$Log: searchfile.c,v $
Revision 1.3  1996/09/13 13:47:23  john
Added handling of circularly linked directories and subdirectories in searchpath.c
Cosmetic changes to bmhsearch.c, postfix.c, rel.c, searchfile.c, translit.c, uppercase.c., version.c.

Revision 1.2  1996/02/08 02:54:34  john
Added hyphenation, backspace, and multiple whitespace capability.
Changes to files: uppercase.c translit.c searcfile.c rel.c and version.c-required for hyphenation, backspace, and multiple whitespace capability.

Revision 1.1  1995/12/23 23:11:10  john
Changed variable, count, to integer for lint compatability with 3rd argument of read () in some compilers; changed cast in third argument of read () to unsigned for lint compatability with some compilers.
Changes to files: Makefile, searchpath.c, searchfile.c, message.c, version.c-specifically to control program behavior under certain file system exceptions; specifics for the GNU gcc compiler, and ANSI C cosmetic formalities.

 * Revision 1.0  1995/04/22  05:13:18  john
 * Initial revision
 *

*/

#include "rel.h"

#ifndef LINT /* include rcsid only if not running lint */

static char rcsid[] = "$Id: searchfile.c,v 1.3 1996/09/13 13:47:23 john Exp $"; /* module version */
static char rcsid_h[] = SEARCHFILE_H_ID; /* module include version */

#endif

static char *name = (char *) 0; /* reference to filename */

static unsigned char *page = (unsigned char *) 0; /* reference to page from file */

static int infile = -1; /* input file descriptor */

#ifdef __STDC__

int searchfile (char *filename, BMHPATTERN *bmhlist)

#else

int searchfile (filename, bmhlist)
    char *filename;
    BMHPATTERN *bmhlist;

#endif

{
    int retval = URFIL_ERR, /* return value, assume error opening file */
        count; /* count of bytes read from the file */

    struct flock lockfile; /* file locking structure */

    struct stat buf; /* structure to obtain file size */

    name = filename; /* reference the filename */

    lockfile.l_type = F_RDLCK; /* lock/unlock entire file for read */
    lockfile.l_whence = (short int) SEEK_SET;
    lockfile.l_start = (off_t) 0;
    lockfile.l_len = (off_t) 0;

    if ((infile = open (filename, O_RDONLY, S_IREAD)) != -1) /* open the file */
    {
        retval = URLCK_ERR; /* assume error locking file */

        if (fcntl (infile, F_SETLKW, &lockfile) == 0) /* read lock the file */
        {
            retval = URSTA_ERR; /* assume error stating file */

            if (fstat (infile, &buf) == 0) /* get the stat structrure for the file */
            {
                count = (int) buf.st_size; /* save the size of the file */
                retval = URMEM_ERR; /* assume error allocating memory */

                /* allocate the memory page, plus a leading space, plus a two trailing spaces, plus an EOS */

                if ((page = (unsigned char *) malloc (((size_t) (count + 4)) * sizeof (unsigned char))) != (unsigned char *) 0)
                {
                    retval = URRED_ERR; /* assume error reading from file */

                    if (read (infile, (char *) &page[1], (unsigned) count) != -1) /* read the file into core */
                    {
                        retval = NO_ERROR; /* assume no errors */
                        count = (int) transliterate (&page[1], count); /* transliterate the page */
                        page[0] = (unsigned char) ' '; /* add a leading space to the page */
                        page[count + 1] = (unsigned char) ' '; /* add a trailing space to the page */
                        page[count + 2] = (unsigned char) ' '; /* add a second trailing space to the page */
                        page[count + 3] = (unsigned char) '\0'; /* add an EOS to the page */
                        bmhsearch_list (page, count, bmhlist); /* search the page */
                    }

                    free (page); /* free the page */
                    page = (unsigned char *) 0; /* reset the reference to page from file */
                }

            }

            lockfile.l_type = F_UNLCK; /* unlock the file */

            if (fcntl (infile, F_SETLKW, &lockfile) != 0) /* unlock the file */
            {

                if (retval != NO_ERROR) /* couldn't unlock the file, pending error? */
                {
                    message (retval, filename); /* yes, print the error */
                }

                retval = URUCK_ERR; /* assume error unlocking file */
            }

        }

        if (close (infile) != 0) /* close the file */
        {

            if (retval != NO_ERROR) /* couldn't close the file, pending error? */
            {
                message (retval, filename); /* yes, print the error */
            }

            retval = URCLS_ERR; /* assume error closing file */
        }

        infile = -1; /* reset the input file descriptor */
        name = (char *) 0; /* reset the reference to filename */
    }

    if (retval != NO_ERROR) /* pending error? */
    {
        message (retval, filename); /* yes, print the error */
    }

    return (retval); /* for formality, return any errors */
}

/*

void int_searchfile (void);

since searchfile() could be interrupted, a shutdown procedure is
necessary to close any open file, and deallocate the memory page-this
routine should be installed as part of the interrupt handling process

The algorithm is as follows:

    if the file is open, close it

    if the memory page is allocated, deallocate it

There are no arguments, and no return value from this function

*/

#ifdef __STDC__

void int_searchfile (void)

#else

void int_searchfile ()

#endif

{

    if (infile != -1) /* input file open? */
    {

        if (close (infile) != 0) /* yes, close the input file */
        {
            message (URCLS_ERR, name); /* couldn't close the file, print the error */
        }

        infile = -1; /* reset the input file descriptor */
        name = (char *) 0; /* reset the reference to filename */
    }

    if (page != (unsigned char *) 0) /* memory page allocated */
    {
        free (page); /* yes, free the page */
        page = (unsigned char *) 0; /* reset the reference to page from file */
    }

}

#ifdef TEST_SEARCHFILE

/*

simple exerciser for testing searchfile (); open and input the file
specified on the command line, and read it-the search argument must be
in uppercase; ignore the:

declared global, could be static
    searchfile          searchfile.c(xxx)

from lint

*/

#ifdef __STDC__

int main (int argc, char *argv[])

#else

int main (argc, argv)
    int argc;
    char *argv[];

#endif

{
    unsigned char tokens[2 * BUFSIZ]; /* buffer containing tokens from infix notation string */

    int retval = URARG_ERR, /* return value, assume argument error */
     value, /* number of matches in a file */
     file_ctr; /* file counter */

    ELEMENT *postfix_stack; /* reference to postfix stack */

    BMHPATTERN *pattern_stack; /* reference to pattern stack */

    if (argc > 2) /* enough arguments? */
    {
        retval = NO_ERROR; /* assume no errors */

        if (make_uppercase () != (unsigned char *) 0) /* setup the uppercase array */
        {

            if ((postfix_stack = postfix (argv[1], tokens)) != (ELEMENT *) 0) /* translate first argument to postfix notation */
            {

                if ((pattern_stack = bmhcompile_postfix (postfix_stack)) != (BMHPATTERN *) 0)
                {

                    for (file_ctr = 2; file_ctr < argc; file_ctr++) /* for each file specified */
                    {

                        if ((retval = searchfile (argv[file_ctr], pattern_stack)) != NO_ERROR) /* search the file */
                        {
                            break; /* if any errors quit */
                        }

                        if ((value = postfix_eval (postfix_stack)) > 0) /* evaluate the postfix stack */
                        {
                            (void) printf ("%s = %d\n", argv[file_ctr], value); /* if any matches, print the number */
                        }

                    }

                }

            }

        }

    }

    else
    {
        message (retval, argv[0]); /* not enough arguments, print the error */
        retval = NO_ERROR; /* assume no error */
    }

    exit (retval); /* return any errors */

#ifdef LINT /* include only if running lint */

    return (0); /* for LINT formality */

#endif

}

#endif
