 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /************************************************************************/
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)            */
 /*                   Date: Tue Apr 22 1997                              */
 /************************************************************************/
 /*                                                                      */
 /* Synthesise from an LPC analysis and possibly a residual.             */
 /*                                                                      */
 /************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include "EST.h"
#include "EST_sigpr.h"
#include "EST_Wave.h"
#include "EST_wave_aux.h"
#include "EST_track_aux.h"
#include "EST_Track.h"
#include "EST_error.h"

#define DEFAULT_FRAME_SIZE 0.01
#define DEFAULT_ORDER 20
#define DEFAULT_WINDOW "hamming"

static void print_help_exit(const char *message, int status);

void make_mod_track(EST_Track &lpcs, 
		    int sample_rate, float pitch_mod, float duration_mod, 
		    EST_Track *pitch_track,
		    int jitter,
		    EST_Track &mod_track);

EST_PpMod::Func *pp_mod_func(EST_String name)
{
  if (name == "chop_1_3")
    return EST_PpMod::chop_1_3;
  else if (name == "stretch_1_3")
    return EST_PpMod::stretch_1_3;
  else if (name == "stretch")
    return EST_PpMod::stretch;
  else if (name == "stretch_mid")
    return EST_PpMod::stretch_mid;
  else if (name == "resample")
    return EST_PpMod::resample;
  else if (name == "none")
    return NULL;

  EST_error("no such mod function %s", (const char *)name);
  return NULL;
}

int main(int argc, char *argv[]) 
{ 
  EST_StrList files;
  EST_Option settings, cmd_line;

  parse_command_line(argc, argv, 
		     EST_String("-help -r:s -o:s  -otype:s -labels:s -pitch:i -f0:s -duration:i -pp_mod_func:s -pp_mod_func_u:s -jitter:i -split"),
		 files, cmd_line);

  init_lib_ops(cmd_line, settings);

  if (cmd_line.present("-help"))
    print_help_exit(NULL, 0);

  if (files.length() > 2 )
    print_help_exit("Too many file names", 1);

  EST_String out_file (cmd_line.present("-o") ? cmd_line.val("-o") : (EST_String)"-");
  EST_String res_file (cmd_line.present("-r") ? cmd_line.val("-r") : (EST_String)"");
  EST_String lab_file (cmd_line.present("-labels") ? cmd_line.val("-labels") : (EST_String)"");

  EST_String pitch_file (cmd_line.present("-f0") ? cmd_line.val("-f0") : (EST_String)"");
  
  EST_Wave signal;
  EST_Track lpcs;
  EST_Track *pitch_track = NULL;
  EST_Track mod_track;
  EST_Wave residual;

  if (read_track(lpcs, files.first(), settings))
    exit(1);

  if (lab_file != "")
    if (lpcs.load_channel_names(lab_file) != format_ok)
      {
	cerr << "error reading labels\n";
	exit(1);
      }

  if (pitch_file != "")
    {
      pitch_track = new EST_Track;
      settings.override_val("time_channel", "track0");
      settings.override_fval("time_scale", 1.0);

      if (read_track(*pitch_track, pitch_file, settings))
	{
	  cerr << "error reading f0\n";
	  exit(1);
	}
    }

  lpcs.create_map();

  if (res_file == "")
    EST_error("Must have Residual file\n");
  else if(read_wave(residual, res_file, settings))
    exit(1);

  if (cmd_line.present("-pitch") || cmd_line.present("-duration") || pitch_track)
    {
      make_mod_track(lpcs, residual.sample_rate(),
		     settings.fval("pitch_mod"), settings.fval("duration_mod"),
		     pitch_track,
		      settings.ival("jitter"),
		     mod_track);

      EST_PpMod::Func *pp_mod = pp_mod_func(settings.sval("pp_mod_func"));
      EST_PpMod::Func *pp_mod_u = NULL;

      if (settings.present("split_synthesis"))
	lpc_resynth_split(signal, lpcs, residual, 
		    mod_track, pp_mod, pp_mod_u);
      else
	lpc_resynth(signal, lpcs, residual, 
		    mod_track, pp_mod, pp_mod_u);
    }
  else
      lpc_resynth(signal, lpcs, residual);


  signal.save(out_file, cmd_line.val("-otype", 0));
	
  return 0;
}

// description of tracks created by make_mod_track

static EST_ChannelMapping mod_mapping =
{
  { channel_frame, 0 },		// which lpc frame
  { channel_length,  1 },	// new length
  { channel_unknown},	
};
static EST_TrackMap ModTrackMap(mod_mapping);


static int sum_lengths(EST_Track &t)
{
  int l=0;

  for(int i=0; i<t.num_frames(); i++)
    l += (int)t.a(i, channel_length);

return l;
}

#define min(x,y) ((x<y)?(x):(y))
#define feq(f1, f2) (fabs((f1)-(f2)) < 0.0001)

void make_mod_track(EST_Track &lpcs, 
		    int sample_rate, float pitch_mod, float duration_mod, 
		    EST_Track *pitch_track,
		    int jitter,
		    EST_Track &mod_track)
{

  if (pitch_track)
    duration_mod = pitch_track->end()/sum_lengths(lpcs)*sample_rate;
  int orig_length = sum_lengths(lpcs);
  int target_length = (int)(orig_length * duration_mod + 0.5);
  int frame_space = lpcs.num_frames();
  struct frame {int f; int pp;}  *frames = new struct frame [frame_space];
  float period_mod = 1.0/pitch_mod;

    

  cout << "Modify: pitch " << pitch_mod << " duration " << duration_mod << "\n";

  int frame_i=0, lpc_i=0;
  float length=0, lpc_time=0;

  bool have_voiced = lpcs.has_channel(channel_voiced);
  bool have_f0 = (bool)(pitch_track && pitch_track->has_channel(channel_f0));

  cout << " voiced " << have_voiced << " f0 " << have_f0 << "\n";

  while (length < target_length)
    {
     if (frame_space <= frame_i)
       {
	 struct frame *new_space = new struct frame [frame_space *2];
	 memcpy(new_space, frames, sizeof(struct frame)*frame_space);
	 delete[] frames;
	 frame_space = frame_space*2;
	 frames = new_space;
       }
     frames[frame_i].f = lpc_i;
     
     if (have_voiced && lpcs.a(lpc_i, channel_voiced) < 0.5)
       frames[frame_i].pp = (int)(lpcs.a(lpc_i, channel_length) * min(duration_mod, 1)+ 0.5);
     else if (!pitch_track)
       frames[frame_i].pp = (int)(lpcs.a(lpc_i, channel_length) * period_mod + 0.5);
     else
       {
	 float f0;
	 if (have_f0)
	   f0 = pitch_track->a((float)length/sample_rate, channel_f0, it_linear_nz);
	 else
	   f0 = pitch_track->a((float)length/sample_rate, 1, it_linear_nz);

	 // cout << " F0 = " << f0 << "\n";

	 frames[frame_i].pp = f0 > 10?(int)(sample_rate / (f0 *pitch_mod) + 0.5): (int)(lpcs.a(lpc_i, channel_length) * min(duration_mod, 1)+ 0.5);
       }

     if (jitter > 0)
       frames[frame_i].pp += (int)(((float)rand() * (float)jitter * 2.0 / 0x7fff) - jitter);

     // cout << "at " << length << " pick " << frames[frame_i].f << " length " << frames[frame_i].pp << "\n";


     length += frames[frame_i].pp;

     float lpcs_pos = feq(duration_mod, 1.0)? length : (length/target_length * orig_length);
     while (lpc_i < lpcs.num_frames() && lpcs_pos >= lpc_time + lpcs.a(lpc_i, channel_length))
       {
	 lpc_time += lpcs.a(lpc_i, channel_length);
	 lpc_i++;
       }
     if ( lpc_i >= lpcs.num_frames())
       lpc_i = lpcs.num_frames()-1;
     frame_i++;
    }

  mod_track.assign_map(ModTrackMap);

  mod_track.resize(frame_i, 2);
  for(int i=0; i<frame_i; i++)
    {
      mod_track.a(i, channel_frame) = frames[i].f;
      mod_track.a(i, channel_length) = frames[i].pp;
      // cout << "picked frame " << (int)mod_track.a(i, channel_frame) << ", " <<  (int)mod_track.a(i, channel_length) << "\n";

    }
  delete [] frames;
}

void override_lib_ops(EST_Option &settings, EST_Option &cmd_line)
{
  settings.override_fval("pitch_mod", 1.0);
  settings.override_fval("duration_mod", 1.0);
  settings.override_val("pp_mod_func", "none");
  settings.override_ival("jitter", 0);

  if (cmd_line.present("-pitch"))
    settings.override_fval("pitch_mod", cmd_line.ival("-pitch", 1)/100.0);
  if (cmd_line.present("-duration"))
    settings.override_fval("duration_mod", cmd_line.ival("-duration", 1)/100.0);
  if (cmd_line.present("-pp_mod_func"))
    settings.override_val("pp_mod_func", cmd_line.sval("-pp_mod_func", 1));
  if (cmd_line.present("-jitter"))
    settings.override_ival("jitter", cmd_line.ival("-jitter", 1));
  if (cmd_line.present("-split"))
    settings.override_val("split_synthesis", "yes");
}

static void print_help_exit(const char *message, int status)
{

  if (message)
    cout << message << "\n";
  cout <<
"Useage: lpc_synthesis [options] lpcs_file [result_file_name]\n\
    Options:\n\
	-o output_file \n\
	-otype output_file_format \n\
	-r residual_file_name\n\
	-duration percentage\n\
	-pitch percentage\n\
	-f0 F0 track_file \n\
	-pp_mod_func name \n\
	-pp_mod_func_u name (for unvoiced periods, if known)\n\
	-split (use two pass synthesis) \n\
	-help \n\
";

  exit(status);
}


