/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, and modify this software and its            */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                      Author :  Paul Taylor                            */
/*                      Date   :  April 1994                             */
/*-----------------------------------------------------------------------*/
/*                 Intonation Detection Algroithm                        */
/*                                                                       */
/*=======================================================================*/

#include "EST.h"
#include "EST_pda.h"
#include "EST_cmd_line_options.h"

void default_options(EST_Option &al);
EST_Track abs_F0(EST_Track &raw_fz, EST_Track &speech, EST_Option &op);
EST_Track diff_F0(EST_Track &raw_fz, EST_Track &speech, EST_Option &op);

int main (int argc, char *argv[])
{
    EST_Track raw_fz, speech, sm_fz;
    EST_Wave sig;
    EST_StrList files;
    EST_Option al, op;
    EST_String bc_class, out_file;
    EST_Stream sil_lab;
    float end_point;

    parse_command_line2(argc, argv, 
       EST_String("Usage:   ")+
       "icda  <input file> -o <output file> <options>\n"+
       "Summary: pitch track waveform files to produce smooth contours\n"+
       "use \"-\" to make input and output files stdin/out\n"+
       "-h               Options help\n"+
       "-abs             produce absolute differentiated contour\n" +
       "-class <string>  name of phoneme set for speech detection\n" +
       "-diff            differentiated contour\n" +
       "-fi              full interpolation through everything\n" +
       "-f0in            input file is raw f0 contour\n" +
       "-lab <string>    name of speech label file\n" +
       "-ndiff           normalise differentiated contour\n" + 
       "-ni              no interpolation through unvoiced regions\n" +
       "-sp <string>     speech class (for pause detection)\n" +
       "-sf              set output type to simple F0\n" +
       "-w1 <float>      size of smoothing window prior to interpolation\n" +
       "-w2 <float>      size of smoothing window after to interpolation\n" + 
       options_wave_input()+ "\n"+
       options_pda_general()+ "\n"+
       options_pda_srpd()+ "\n"+
       "-color <string> Color of output line\n" +
       "-o <ofile>       Output filename, if not specified output is\n"+
       "                 to stdout\n"+
       "-otype <string>  Output file type, (optional).  If no type is\n"+
       "                 Specified the type of the input file is assumed\n"+
       "                 types are: "+options_track_filetypes()+"\n",
			files, al);

/*    parse_command_line(argc, argv, "-H:f -L -N -P -R -abs -c:s -class:s 
		      -color:s-d:f -diff -f:i -fi -fmax:f -fmin:f -forder:i -itype:s
		       -j:s -l:f -lab:s -m:f -n:f -ni -ndiff -o:s
		      -otype:s -ops -owave:s -r:f -s:f -sp:s -t:f -sf
		      -u:f -w1:f -w2:f", files, al);
*/

    default_pda_options(op);
    init_lib_ops(al, op);
    out_file = al.present("-o") ? al.val("-o") : (EST_String)"-";

    if (al.present("-f0in"))
    {
	if (read_track(raw_fz, files.first(), al) == -1)
	    exit(-1);
	end_point = raw_fz.end();
    }
    else
    {
	if (read_wave(sig, files.first(), al) == -1)
	    exit(-1);
	raw_fz = pda(sig, op);
//	end_point = raw_fz.end();
	end_point = sig.end();
    }

    if (al.present("-lab"))
    {
	sil_lab.load(al.val("-lab"));
	cout << sil_lab;
	convert_to_broad_class(sil_lab, al.val("-class") ,op);
	merge_all_label(sil_lab, "pos");
	cout << sil_lab;
	speech = label_to_track(sil_lab, op.fval("pda_frame_shift"));
    }
    else if (al.present("-fi"))
    {
	EST_Stream_Item all;
	all.set_name("pos");
	all.set_end(end_point);
	sil_lab.append(all);
	cout << sil_lab;
	op.override_val("icda_fi", "1");
	speech = label_to_track(sil_lab, op.fval("pda_frame_shift"));
    }
    else if (!al.present("-ni")) 
    {
	cerr << "Broad classifier disabled: using -fi, -ni or -lab options\n";
	exit(0);
    }

    sm_fz = smooth_phrase(raw_fz, speech, op);
//    sm_fz.set_field_name("F0", 0);

    if (al.present("-color"))
	sm_fz.color = al.val("-color");
    if (al.present("-sf")) // "simple f0 file", no esps fancy stuff
	sm_fz.set_contour_type(EST_ContourType::SIMPLE_F0);

    sm_fz.save(out_file, op.val("f0_file_type", 0));

    if (al.present("-abs"))
    {
	sm_fz = abs_F0(sm_fz, speech, op);
	sm_fz.save(out_file + ".abs", op.val("f0_file_type", 0));
    }
    else if (al.present("-diff"))
    {
	sm_fz = diff_F0(sm_fz, speech, op);
	sm_fz.save(out_file + ".diff", op.val("f0_file_type", 0));
    }
    if (al.present("-owave"))
	sig.save(al.val("-owave"), op.val("wave_file_type", 0));
    return 0;
}

EST_Track abs_F0(EST_Track &fz, EST_Track &speech, EST_Option &op)
{
    EST_FVector mean, sd;
    EST_Track diff, sm_diff, abs_diff;
    (void)speech;

    // fz= smooth_phrase(raw_fz, speech, op);
    // Differentiated F0 contour

    diff   = differentiate(fz);
    sm_diff = diff;
    time_mean_smooth(sm_diff, op.fval("diff_smooth", 1));
    abs_diff = sm_diff;
    absolute(abs_diff);

    time_mean_smooth(abs_diff, op.fval("abs_diff_smooth", 1));
    meansd(abs_diff, mean, sd);
    normalise(abs_diff, mean, sd);
    abs_diff.set_contour_type(EST_ContourType::make("abs_F0"));

    return abs_diff;
}

EST_Track diff_F0(EST_Track &fz, EST_Track &speech, EST_Option &op)
{
    EST_FVector  mean, sd;
    EST_Track diff;
    (void)speech;

//    cout << "Fz\n" << fz;
    diff = differentiate(fz);

//    cout << "diff " << diff;
    time_mean_smooth(diff, op.fval("diff_smooth", 1));
//    cout << "tms " << diff;
    
    if (op.val("norm_diff", 0) == "true")
    {
	meansd(diff, mean, sd);
	normalise(diff, mean, sd);
    }
    diff.set_contour_type(EST_ContourType::DIFF_F0);

    return diff;
}
void override_lib_ops(EST_Option &a_list, EST_Option &al)
{
    // general options
    a_list.override_val("sample_rate", al.val("-f", 0));
    a_list.override_val("min_pitch",  al.val("-fmin", 0));
    a_list.override_val("max_pitch",  al.val("-fmax", 0));
    a_list.override_val("pda_frame_shift", al.val("-s", 0));
    a_list.override_val("ms_frame_length",al.val("-l", 0));
    
    // low pass filtering options.
    if (al.val("-L", 0) == "true")
	a_list.override_val("do_low_pass", "true");
    if (al.val("-R", 0) == "true")
	a_list.override_val("do_low_pass", "false");
    a_list.override_val("lpf_cutoff",al.val("-u", 0));
    a_list.override_val("lpf_order",al.val("-forder", 0));
    a_list.override_val("icda_no_interp",al.val("-ni", 0));
    a_list.override_val("window_length", al.val("-w1", 0));
    a_list.override_val("second_length", al.val("-w2", 0));
    
    //sprd options
    a_list.override_val("decimation", al.val("-d", 0));
    a_list.override_val("noise_floor",   al.val("-n", 0));
    a_list.override_val("min_v2uv_coef_thresh", al.val("-m", 0));
    a_list.override_val("v2uv_coef_thresh_ratio", al.val("-r", 0));
    a_list.override_val("v2uv_coef_thresh", al.val("-H", 0));
    a_list.override_val("anti_doubling_thresh", al.val("-t", 0));
    a_list.override_val("peak_tracking", al.val("-P", 0));

    // differentiation options.
    if (al.present("-ndiff"))
	a_list.override_val("norm_diff", "true");
    
    // file options
    a_list.override_val("f0_file_type", al.val("-otype", 0));
    a_list.override_val("wave_file_type", al.val("-itype", 0));
}

void print_help(int argc, char *argv[])
{
    for (int i = 0; i < argc; ++i)
	if (strncmp(argv[i],"-help", 2) == 0)
	{
	    cout << "Usage:\n";
	    cout << "icda  <input file> <output file> <options>\n";
	    cout << "use \"-\" to make input and output files stdin/out\n";

	    cout << "General Options:\n";
	    cout << "-h      options help\n";
	    cout << "-N      do NOT read environment options file\n";
	    cout << "-c      <options file> for system params \n";
	    cout << "-color  <string> Color of output line\n";
	    cout << "-f      <int> sample rate in Hertz\n";
	    cout << "-itype  <input type: 'raw' 'vox' 'esps' 'sd' 'nist'>\n";
	    cout << "-ops    print relevant system options\n";
	    cout << "-otype  <output file type 'xmg' 'ascii' 'esps'>\n";
	    cout << "-owave  <filename> Save output (lpf) waveform\n";

	    cout << "\nPitch tracking options\n";
	    cout << "-L      perform low pass filtering on input\n";
	    cout << "-P      perform peak tracking\n";
	    cout << "-R      don't perform low pass filtering on input\n";
	    cout << "-fmin   <int> miniumum F0 value\n";
	    cout << "-fmax   <int> maxiumum F0 value\n";
	    cout << "-j      <\"on\" \"off\"> join points in output\n";
	    cout << "-l      <float> frame length in seconds.\n";
	    cout << "-forder lp filter order (as an exponent of 2, eg 5)\n";
	    cout << "-s      <float> frame shift in seconds.\n";
	    cout << "-srpd   use srpd method\n";
	    cout << "-u      lp filter cutoff (eg 600\n";

	    cout << "srpd options:\n";
	    cout << "-d      decimation threshold (srpd)\n";
	    cout << "-n      noise floor\n";
	    cout << "-H      unvoiced to voiced coeff threshold\n";
	    cout << "-m      min voiced to unvoiced coeff threshold\n";
	    cout << "-r      voiced to unvoiced coeff threshold-ratio\n";
	    cout << "-t      anti pitch doubling/halving threshold (sprd\n";

	    cout << "\nSmoothing Options\n";
	    cout << "-abs    produce absolute differentiated contour\n";
	    cout << "-class  <string> name of phoneme set for speech detection\n";
	    cout << "-diff   differentiated contour\n";
	    cout << "-lab    <string> name of speech label file\n";
	    cout << "-ni     no interpolation through unvoiced regions\n";
	    cout << "-fi     full interpolation through everything\n";
	    cout << "-ndiff  normalise differentiated contour\n";
	    cout << "-sp     speech class (for pause detection)\n";
	    cout << "-w1     size of smoothing window prior to interpolation\n";
	    cout << "-w2     size of smoothing window after to interpolation\n";
	    exit (0);
	}
}
