/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                 Authors:  Alan W Black                                */
/*                 Date   :  July 1996                                   */
/*-----------------------------------------------------------------------*/
/*  A viterbi decoder                                                    */
/*                                                                       */
/*  User provides the candidates, target and combine score function      */
/*  and it searches for the best path through the candidates              */
/*                                                                       */
/*=======================================================================*/

#ifndef __VERTERBI_H__
#define __VERTERBI_H__

#include "EST_cutils.h"
#include "EST_Stream.h"

class EST_VTCandidate {
  private:
  public:
    EST_VTCandidate() {score=0.0; next=0; s=0; }
    ~EST_VTCandidate() {if (next != 0) delete next;}
    float score;
    EST_Val name;
    int pos;
    EST_Stream_Item *s;
    EST_VTCandidate *next;
};

class EST_VTPath {
  private:
  public:
    EST_VTPath() {score=0.0; from=0; next=0; c=0;}
    ~EST_VTPath() {if (next != 0) delete next;}
    double score;   /* cummulative score for path */
    int state;
    KVL <EST_String,EST_Val> features;
    EST_VTCandidate *c;
    EST_VTPath *from;
    EST_VTPath *next;
    EST_Val feature(const EST_String &name)
         {return features.val_def(name,EST_Val("0"));}
    void set_feature(const EST_String &name, int ival)
       { EST_Val pv(ival); features.add_item(name,pv); }
    void set_feature(const EST_String &name, float fval)
       { EST_Val pv(fval); features.add_item(name,pv); }
    void set_feature(const EST_String &name, double dval)
       { EST_Val pv((float)dval); features.add_item(name,pv); }
    void set_feature(const EST_String &name, EST_String sval)
       { EST_Val pv(sval); features.add_item(name,pv); }
    void set_feature(const EST_String &name, const char *cval)
       { EST_Val pv(cval); features.add_item(name,pv); }
};

class EST_VTPoint {
  private:
  public:
    EST_VTPoint() {next=0; s=0; paths=0; num_paths=0; cands=0; st_paths=0; num_states=0;}
    ~EST_VTPoint();
    EST_Stream_Item *s;
    int num_states;
    int num_paths;
    EST_VTCandidate *cands;
    EST_VTPath *paths;
    EST_VTPath **st_paths;
    EST_VTPoint *next;
};    

typedef EST_VTCandidate *(*uclist_f_t)(EST_Stream_Item &s);
typedef EST_VTPath *(*unpath_f_t)(EST_VTPath *p,EST_VTCandidate *c);
class EST_Viterbi_Decoder {
  private:
    int num_states;
    int debug;
    int beam_width;
    int cand_width;
    int big_is_good;
    uclist_f_t user_clist;
    unpath_f_t user_npath;
    EST_VTPoint *timeline;
    void add_path(EST_VTPoint *p, EST_VTPath *np);
    void vit_add_path(EST_VTPoint *p, EST_VTPath *np);
    EST_VTPath *find_best_end() const;
    const int betterthan(const float a,const float b) const;
  public:
    EST_Viterbi_Decoder(uclist_f_t a, unpath_f_t b);
    EST_Viterbi_Decoder(uclist_f_t a, unpath_f_t b, int num_states);
    ~EST_Viterbi_Decoder();
    void set_beam_width(int w) {beam_width = w;}
    void set_cand_width(int w) {cand_width = w;}
    void set_debug(int d) {debug = d;}
    void set_user_clist(uclist_f_t f);
    void set_user_npath(unpath_f_t f);
    void set_big_is_good(int f) { big_is_good = f; }
    EST_VTCandidate *add_cand_prune(EST_VTCandidate *newcand,
				    EST_VTCandidate *allcands);
    void initialise(EST_Stream &w);
    void search(void);
    void result(const EST_String &n);
    void copy_feature(const EST_String &n);
};


#endif // __VERTERBI_H__
