/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                     Copyright (c) 1995,1996                           */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                      Author :  Paul Taylor                            */
/*                      Date   :  March 1995                             */
/*-----------------------------------------------------------------------*/
/*                Signal Procssing Library Header File                   */
/*                                                                       */
/*=======================================================================*/
#ifndef __EST_SIGPR_H__
#define __EST_SIGPR_H__

#include "EST_Wave.h"
#include "EST_Track.h"
#include "EST_FMatrix.h"
#include "EST_Option.h"
#include "EST_TNamedEnum.h"
#include "EST_PpMod.h"

// Window types

typedef enum EST_WindowType {
  wf_none=0,
  wf_rectangle=1,
  wf_triangle=2,
  wf_hanning=3,
  wf_hamming=4
} EST_WindowType;

typedef EST_FVector EST_WindowFunction(int);

struct EST_window_info {
 EST_WindowFunction *fn;
};

extern EST_TNamedEnumI<EST_WindowType,EST_window_info> EST_WindowTypeMap;

EST_Track make_cepstra_unnorm(EST_Wave &sig, EST_Option &op);
EST_Track make_cepstra(EST_Wave &sig, EST_Option &op);
int init_cepstra(EST_Option &op, EST_String bc_file);
void normalise(EST_Track &fvs);
void normalise(EST_Track &fvs, float mean, float sd);
void normalise(EST_Track &tr);
void normalise(EST_Track &tr, float mean, float sd);

EST_Track rms_energy(EST_Track &pow);
EST_Track rms_energy(EST_Wave &sig, float shift=10.0, float length=32.0);
EST_Track power(EST_Wave &sig, float shift=10.0, float length=32.0);

// stuff for delta coefficients

// max. order of derivatives (2 means allow 1st and 2nd differential only)
# define MAX_DELTA_ORDER 2

// max. number of points on which the delta co-eff is based
# define MAX_REGRESSION_LENGTH 4

// find slope of best-fit straight line through set of points
float compute_gradient(float* x, int num_points);

// compute delta coefficients (derivatives computed by regression)
void delta(EST_Track &EST_Track, int regression_length = 3);

EST_Wave make_lpc_residual(EST_Wave &sig, EST_Track &lpc, EST_Option &op);

void lpc_cepstra(EST_Track &c, EST_Wave &sig, float length, float
		 shift, int lporder, int genlpc = 0);

// New analysis and resynthesis functions by RJC.

// lpc_analyse is controled by the presence or absence of a pitchmark
// track and by what channels are defined in the description

void lpc_analyse(EST_Track &lpcs, EST_Wave &sig,
		 EST_WindowType window_type, 
		 float length, float shift, 
		 EST_TrackMap *description=NULL,
		 EST_Track *pms=NULL);

inline
void lpc_analyse(EST_Track &lpcs, EST_Wave &sig,
		 EST_WindowType window_type, 
		 float length, float shift, 
		 EST_TrackMap *description,
		 EST_Track &pms)
	{ lpc_analyse(lpcs, sig,
		 window_type, length, shift, description, &pms); };

void lpc_residual(EST_Wave &residual, EST_Track &lpcs, EST_Wave &sig);

void lpc_resynth(EST_Wave &signal, EST_Track &lpcs, 
		 EST_Wave &residual, EST_Track *modifications=NULL, 
		 EST_PpMod::Func* pp_mod=NULL, EST_PpMod::Func *pp_mod_u=NULL);

inline void lpc_resynth(EST_Wave &signal, EST_Track &lpcs, 
			EST_Wave &residual, EST_Track &modifications, 
			EST_PpMod::Func *pp_mod=NULL, 
			EST_PpMod::Func *pp_mod_u=NULL)
	{ lpc_resynth(signal, lpcs, residual, &modifications, 
		      pp_mod, pp_mod_u); };

void lpc_resynth_chunk(short *sig,
		       EST_Track &lpcs,
		       const short *res,
		       int num_samples, 
		       int start_frame,
		       int num_frames, 
		       int order,	       
		       EST_Track *modifications=NULL,
		       EST_PpMod::Func *pp_mod=NULL,
		       EST_PpMod::Func *pp_mod_u=NULL,
		       short *res_return=NULL);

inline void lpc_resynth_chunk(short *sig,
			      EST_Track &lpcs,
			      const short *res,
			      int num_samples, 
			      int start_frame,
			      int num_frames, 
			      int order,
			      EST_Track &modifications,
			      EST_PpMod::Func *pp_mod=NULL,
			      EST_PpMod::Func *pp_mod_u=NULL,
			      short *res_return=NULL)
	{ lpc_resynth_chunk(sig, lpcs, res,
			    num_samples, start_frame, num_frames, order, 
			    &modifications, pp_mod, pp_mod_u, res_return); };


void lpc_resynth_split(EST_Wave &signal, EST_Track &lpcs, 
		       EST_Wave &residual, EST_Track *modifications=NULL, 
		       EST_PpMod::Func* pp_mod=NULL, EST_PpMod::Func *pp_mod_u=NULL);

inline void lpc_resynth_split(EST_Wave &signal, EST_Track &lpcs, 
			      EST_Wave &residual, EST_Track &modifications, 
			      EST_PpMod::Func *pp_mod=NULL, EST_PpMod::Func *pp_mod_u=NULL)
	{ lpc_resynth_split(signal, lpcs, residual, &modifications, 
			    pp_mod, pp_mod_u); };

void lpc_resynth_chunk_split(short *sig,
			     EST_Track &lpcs,
			     const short *res,
			     int num_samples, 
			     int start_frame,
			     int num_frames, 
			     int order,	       
			     EST_Track *modifications=NULL,
			     EST_PpMod::Func *pp_mod=NULL,
			     EST_PpMod::Func *pp_mod_u=NULL,
			     short *res_return=NULL);

inline void lpc_resynth_chunk_split(short *sig,
			      EST_Track &lpcs,
			      const short *res,
			      int num_samples, 
			      int start_frame,
			      int num_frames, int order,
			      EST_Track &modifications,
			      EST_PpMod::Func *pp_mod=NULL,
			      EST_PpMod::Func *pp_mod_u=NULL,
			      short *res_return=NULL)
	{ lpc_resynth_chunk_split(sig, lpcs, res,
				  num_samples, start_frame, num_frames, order, 
				  &modifications, pp_mod, pp_mod_u, res_return); };


// filter functions, return true for
bool FIRfilter(EST_Wave &sigin, EST_FVector numerator, int delay_correction=0);
bool FIRlowpass_filter(EST_Wave &sigin, int cutoff_frequency, int order=199);
bool FIRhighpass_filter(EST_Wave &sigin, int cutoff_frequency, int order=199);

// filter design functions

EST_FVector design_FIR_filter(EST_FVector &frequency_response, int
			      filter_order);

EST_FVector design_lowpass_FIR_filter(int sample_rate, int
				      cutoff_freq, int order);

EST_FVector design_highpass_FIR_filter(int sample_rate, int
				       cutoff_freq, int order);

// in-place fft functions
int slowFFT(EST_FVector &real, EST_FVector &imag);
int slowIFFT(EST_FVector &real, EST_FVector &imag);

// wrappers, so the name is obvious
inline int FFT(EST_FVector &real, EST_FVector &imag){
    return slowFFT(real, imag);
};

inline int IFFT(EST_FVector &real, EST_FVector &imag){
    return slowIFFT(real, imag);
}

// in-place - asnwer is returned in real (and imag == real)
int power_spectrum(EST_FVector &real, EST_FVector &imag);
int energy_spectrum(EST_FVector &real, EST_FVector &imag);

// Tony Robinsons fast version
int fastlog2(int);
int fastFFT(EST_FVector &invec);

// generation
EST_FVector Triangular_window(int size);
EST_FVector Hanning_window(int size);
EST_FVector Hamming_window(int size);

// application
inline void window(EST_FVector &data, EST_FVector &window)
{
    data *= window;
}

void Triangular_window(EST_FVector &data);
void Hanning_window(EST_FVector &data);
void Hamming_window(EST_FVector &data);

// convertion procedures

void ref2truearea(const float *ref, float *area, int order);
void ref2logarea(const float *ref, float *logarea, int order);
void ref2area(const float *ref, float *area, int order);
void lpc2ceps(const float *lpc, int nlpc, float *ceps, int nceps);
void lpc2ref(const float *lpc, float *ref, int order);
void ref2lpc(const float *ref, float *lpc, int order);

void lpc_to_reflection(EST_Track &lpcs);

#endif /* __EST_SIGPR_H__ */


