/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                    Author :  Simon King                               */
/*                    Date   :  January 1997                             */
/*-----------------------------------------------------------------------*/
/*             Language model class for stack decoder                    */
/*                                                                       */
/*=======================================================================*/

#include "Language_Model.h"

bool
Language_Model::is_finished(void*)
{
    cerr << "Error : this is Language_Model::is_finished" << endl;
    return true;
}


void *
Language_Model::start_state()
{
    cerr << "Error : this is Language_Model::start_state()" << endl;
    return NULL;
}


EST_TList<void*>
Language_Model::extend(void* current_state)
{
    (void)current_state;
    cerr << "Error : this is Language_Model::extend" << endl;
    return EST_TList<void*>();
}

Lattice_Language_Model::Lattice_Language_Model(Lattice &l, EST_StrVector &wordlist)
{

    lattice = &l;


    // build map to convert lattice word indices into indices
    // into the wordlist

    map.resize(lattice->nmap.n());

    int i,j,k;
    for(i=0;i<lattice->nmap.n();i++){

	// ignore special words !ENTER and !EXIT
	if( (lattice->nmap(i) == "!ENTER") ||
	    (lattice->nmap(i) == "!EXIT") ){
	    map(i) = -1;

	}else{

	    
	    // find word in wordlist
	    k=-1;
	    for(j=0; j<wordlist.n();j++)
		if(wordlist(j) == lattice->nmap(i))
		    k = j;
	    
	    
	    if(k < 0){
		cerr << "Error building map : "
		     << lattice->nmap(i) << " does not occur in wordlist" << endl
		     << endl;
		exit(1); // :-)
	    }
	    
	    map(i) = k;
	}
	
	cerr << "MAP : lattice =" << lattice->nmap(i)
	     << "=" << i << " , wordlist=" << map(i) << endl;

    }

}


bool
Lattice_Language_Model::is_finished(void* lmodel_state)
{
    llm_state_t *llms = (llm_state_t*)lmodel_state;
    return lattice->final(llms->current_node);
}

void *
Lattice_Language_Model::start_state()
{
    llm_state_t *llms;
    llms = new llm_state_t;
    llms->current_node = lattice->start_node();
    llms->log_prob = 0.0;
    llms->path.clear();

    return (void*)llms;

}

EST_TList<void*>
Lattice_Language_Model::extend(void* current_state)
{
    EST_TList<void*> new_states;
    llm_state_t *new_state;
    
    //cerr << "Lattice_Language_Model::extend ";


    if(current_state == NULL)
	cerr << "Lattice_Language_Model::extend : NULL state passed !!" << endl;



    // find all following states in the lattice
    Lattice::Node *current_node = ((llm_state_t*)current_state)->current_node;

    if(current_node == NULL)
	cerr << "Lattice_Language_Model::extend : NULL node passed !!" << endl;

    EST_TBI *a_ptr;
    for(a_ptr=current_node->arcs_out.head();
	a_ptr!=NULL; a_ptr=next(a_ptr)){
	
	Lattice::symbol_t *current_symbol
	    = lattice->alphabet_index_to_symbol(current_node->arcs_out(a_ptr)->label);

	new_state = new llm_state_t;
	new_state->current_node = current_node->arcs_out(a_ptr)->to;
	new_state->log_prob = ((llm_state_t*)current_state)->log_prob 
	    +lattice->qmap_index_to_value(current_symbol->qmap_index);

	new_state->path.clear();
	for(EST_TBI *b_ptr= ((llm_state_t*)current_state)->path.head();
	    b_ptr!=NULL;b_ptr=next(b_ptr))
	    new_state->path.append( ((llm_state_t*)current_state)->path(b_ptr));
	
	new_state->path.append(current_node->arcs_out(a_ptr)->label);
	
	new_states.append((void*)new_state);
	
	
    }

    //delete_state(current_state);
    return new_states;

}


void
Lattice_Language_Model::delete_state(void *lmodel_state)
{
    if(lmodel_state != NULL){
	llm_state_t *llms = (llm_state_t*)lmodel_state;
	llms->path.clear();
	delete llms;
    }
}


void
Lattice_Language_Model::print_state(void *lmodel_state)
{
    llm_state_t *llms = (llm_state_t*)lmodel_state;

    EST_TBI *n_ptr;
    int count;
    for(n_ptr=lattice->nodes.head(),count=0;
	n_ptr!=NULL;n_ptr=next(n_ptr),count++)
	if(lattice->nodes(n_ptr) == llms->current_node)
	    break;

    if(n_ptr == NULL)
	cerr << "No current node";
    else{
	cerr << "LM(node=" << count
	     << " logp=" 
	     << llms->log_prob << " ";

	for(n_ptr=llms->path.head();n_ptr!=NULL;n_ptr=next(n_ptr))
	    cerr << lattice->nmap_index_to_name(lattice->alphabet_index_to_symbol(llms->path(n_ptr))->nmap_index) << " ";

	
	if(is_finished(lmodel_state))
	    cerr << " FINISHED";
	
	
	cerr << ")";
    }

}

float
Lattice_Language_Model::get_log_prob(void *lmodel_state)
{
    // remove this test later for speed
    if(lmodel_state == NULL){
	cerr << "Lattice_Language_Model::get_log_prob passed null state !" << endl;
	return 0.0;
    }

    return ((llm_state_t*)lmodel_state)->log_prob;
}

EST_String
Lattice_Language_Model::get_current_word(void *lmodel_state)
{

    if(lmodel_state == NULL)
	return EST_String("Lattice_Language_Model::get_current_word passed null state !");

    EST_TBI *tmp = ((llm_state_t*)lmodel_state)->path.tail();
    
    if(tmp == NULL)
	return("No arcs traversed yet !");
    
    
    int windex = ((llm_state_t*)lmodel_state)->path(tmp);
    EST_String w = lattice->nmap_index_to_name(lattice->alphabet_index_to_symbol(windex)->nmap_index);

    return w;
}

int
Lattice_Language_Model::get_current_word_index(void *lmodel_state)
{

    // index into original wordlist, not internal nmap !

    if(lmodel_state == NULL){
	cerr << "Lattice_Language_Model::get_current_word passed null state !";
	return -1;
    }

    Lattice::Node *n = ((llm_state_t*)lmodel_state)->current_node;

    if(n == NULL){
	cerr << "No node to get word from !";
	return -1;
    }

    if(n->name.head() == NULL){
	cerr << "Node has no name !";
	return -1;
    }


    EST_TBI *tmp = ((llm_state_t*)lmodel_state)->path.tail();
    
    if(tmp == NULL){
	cerr << "No arcs traversed yet !";
	return -1;
    }
    
    int windex = ((llm_state_t*)lmodel_state)->path(tmp);
    int w = lattice->alphabet_index_to_symbol(windex)->nmap_index;

    return map(w);
}
