/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                    Author :  Simon King                               */
/*                    Date   :  February 1997                            */
/*-----------------------------------------------------------------------*/
/*      Least-upper-bound so far pruned heap class for stack decoder     */
/*                                                                       */
/* =======================================================================*/
#include <string.h>
#include <stdlib.h>
#include "LUBHeap.h"

#ifndef MAXFLOAT
#define MAXFLOAT ((float)3.0e+37)
#endif

LUBHeap::LUBHeap(int max_size,int max_length) : Heap(max_size) {

    // initialise LUB array
    lubsf.resize(max_length + 1); // the 1 is for finished hypotheses
    lubsf_envelope.resize(max_length + 1);

    for(int i=0;i<max_length+1;i++){
	lubsf(i) = -MAXFLOAT;
	lubsf_envelope(i) = 1000; // huge to start with
    }
    
}



LUBHeap::~LUBHeap(){
    
    // probably wrong since heap items can be derived classes !
    
    for(int i=0;i<=max_heap_size;i++){
	
	if(heap_item[i] != NULL){
	    delete heap_item[i];
	    heap_item[i] = NULL;
	}
    }
    
    if(heap_item != NULL)
	delete heap_item;
    
}


Heap::insert_error_t
LUBHeap::insert(Heap_Item* new_heap_item_ptr){

    if( full() )
	return heap_full;
	

    else if( new_heap_item_ptr->h_partial_log_prob < 
	     lubsf(new_heap_item_ptr->h_length) - lubsf_envelope(new_heap_item_ptr->h_length)){
	
	return prune_item;
	
    }else{
	
	// update LUB array
	if( new_heap_item_ptr->h_partial_log_prob > lubsf(new_heap_item_ptr->h_length))	    
	    lubsf(new_heap_item_ptr->h_length) = new_heap_item_ptr->h_partial_log_prob;
	
	
	heap_size++;
	int j = heap_size;
	
	while( (j>1) && ( *new_heap_item_ptr > *(heap_item[parent(j)]) )  ){
	    heap_item[j] = heap_item[parent(j)];
	    j = parent(j);
	}
	
	heap_item[j] = new_heap_item_ptr;
	
    }


    // success !
    return inserted_ok;

}

EST_TList<Heap_Item*>
LUBHeap::prune()
{
    // find all items outside the LUB
    // (the LUB may have changed since they were inserted)

    int i;
    EST_TList<Heap_Item*> prune_list;
#ifdef DEBUG
    int count=0;
#endif
    
    for(i=1;i<=heap_size;i++)

	if( heap_item[i]->h_partial_log_prob < 
	    lubsf(heap_item[i]->h_length) - lubsf_envelope(heap_item[i]->h_length)){
	    
	    prune_list.append(heap_item[i]);
	    
	    heap_item[i]= heap_item[heap_size];
	    heap_item[heap_size] = NULL;
	    heap_size--;
	    heapify(i); // only affects items from [i] onwards
	    i--; // need to look at this element again
#ifdef DEBUG
	    count++;
#endif
	}
    
#ifdef DEBUG
    cerr << "Pruned " << count << " leaving " << heap_size << endl;
#endif

    return prune_list;

}

void
LUBHeap::set_envelope(float retain_factor)
{

 
    // to do :  only use counts over a certain limit 
    // - for reliable envelope size estimation
    // & use previous envelope size to smooth current estimate


    // HACK :
    int min_count = 20;


    EST_FVector sum;
    EST_IVector count;
    sum.resize(lubsf.n());
    count.resize(lubsf.n());
    int i;

    for(i=0;i<lubsf.n();i++){
	count(i) = 0;
	sum(i) = 0;
    }

    for(i=1;i<=heap_size;i++){
	
	count(heap_item[i]->h_length) += 1;

	sum(heap_item[i]->h_length) += heap_item[i]->h_partial_log_prob;
	
    }

    for(i=0;i<lubsf.n();i++)
	if( (lubsf(i) > -MAXFLOAT) && 
	    (count(i) > min_count ) ){
	    
	    // set envelope to cutoff at mean of hypotheses in heap at this time
	    lubsf_envelope(i) = lubsf(i) - sum(i)/count(i);

	    // then scale it
	    lubsf_envelope(i) *= retain_factor;

	}


}

void
LUBHeap::print_stats()
{

    cerr << "Heap stats" << endl;

    EST_FVector max,min,sum;
    EST_IVector count;
    max.resize(lubsf.n());
    min.resize(lubsf.n());
    sum.resize(lubsf.n());
    count.resize(lubsf.n());
    int i;

    for(i=0;i<lubsf.n();i++){
	min(i) = MAXFLOAT;
	max(i) = -MAXFLOAT;
	count(i) = 0;
	sum(i) = 0;
    }

    for(i=1;i<=heap_size;i++){
	
	if(heap_item[i]->h_partial_log_prob > max(heap_item[i]->h_length))
	    max(heap_item[i]->h_length) = heap_item[i]->h_partial_log_prob;
    
	if(heap_item[i]->h_partial_log_prob < min(heap_item[i]->h_length))
	    min(heap_item[i]->h_length) = heap_item[i]->h_partial_log_prob;

	count(heap_item[i]->h_length) += 1;

	sum(heap_item[i]->h_length) += heap_item[i]->h_partial_log_prob;
	
    }

    for(i=0;i<lubsf.n();i++)
	if(lubsf(i) > -10000)
	    cerr << "frame " << i
		 << "  min " << min(i)
		 << "  max " << max(i)
		 << "  lub " << lubsf(i)
		 << "  lub-env " << lubsf(i) - lubsf_envelope(i)
		 << "  ave " << sum(i) / count(i)
		 << "  count " << count(i)
		 << endl;
}
