/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1997                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*             Author :  Alan W Black                                    */
/*             Date   :  June 1997                                       */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/* A probabilistic Chart parser, general functions                       */
/*                                                                       */
/*=======================================================================*/
#ifndef __PCHART_H__
#define __PCHART_H__

class PChart_Edge;
class PChart_Vertex;

/* Edge list and Vertex List */
typedef EST_TList<PChart_Edge *> PChart_EdgeList;
typedef EST_TList<PChart_Vertex *> PChart_VertexList;

class PChart_Edge {
  private:
    PChart_Vertex *p_start;
    PChart_Vertex *p_end;
    EST_String p_name;
    double p_log_prob;
    PChart_EdgeList p_recog;
    EST_StrList p_remainder;
  public:
    PChart_Edge();
    PChart_Edge(PChart_Vertex *start, PChart_Vertex *end, 
		const EST_String &name,
		const PChart_EdgeList &recog, 
		const EST_StrList &remainder,
		double log_prob=0.0);
    ~PChart_Edge();
    
    void set_log_prob(double lp) { p_log_prob = lp; }

    int complete(void) const {return (p_remainder.length() == 0);};
    int incomplete(void) const {return (p_remainder.length() != 0);};
    PChart_Vertex *start() const { return p_start; };
    PChart_Vertex *end() const { return p_end; };
    const EST_String &name() const { return p_name; };
    const PChart_EdgeList &recog() const { return p_recog; };
    const EST_StrList &remainder() const { return p_remainder; };
    double log_prob() const { return p_log_prob; }
    LISP print_edge(void);

};

class PChart_Vertex {
  private:
    EST_String p_name;
    PChart_EdgeList Complete_Outgoing;
    PChart_EdgeList InComplete_Incoming;
    EST_Stream_Item *s;
  public:
    PChart_Vertex();
    ~PChart_Vertex();
    void add_incoming(const PChart_Edge &incomplete);
    void add_outgoing(const PChart_Edge &complete);
    const EST_String &name(void) const {return p_name;};
    void set_name(const EST_String &n) { p_name = n; };
    PChart_EdgeList &incomplete_incoming() { return InComplete_Incoming; };
    PChart_EdgeList &complete_outgoing() { return Complete_Outgoing; };
};

class PChart {
  private:
    PChart_VertexList vertices;
    PChart_EdgeList agenda;
    LISP rules;
    
    PChart_Edge *combine_edges(PChart_Edge *complete,PChart_Edge *incomplete);
    void PChart::grammar_propose(PChart_Edge *edge);

  public:
    PChart();
    ~PChart();
    void setup_wfst(EST_Stream &s,const EST_String &name="name");
    void parse();
    void find_parses();
    void dump_chart();
    void set_grammar_rules(LISP r) { rules = r; }

};

#endif
