 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /************************************************************************/
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)            */
 /*                   Date: Wed Mar 19 1997                              */
 /************************************************************************/
 /*                                                                      */
 /* Simple index which calculates the name of the diphone to use from    */
 /* the phoneme names.                                                   */
 /*                                                                      */
 /************************************************************************/

#include "SimpleDiphoneUnitIndex.h"

const char SimpleDiphoneUnitIndex::s_magic[] = "SimpleDiphoneUnitIndex";
const int SimpleDiphoneUnitIndex::s_magic_size = sizeof(SimpleDiphoneUnitIndex::s_magic-1);


SimpleDiphoneUnitIndex::SimpleDiphoneUnitIndex(void)
{
  before_map.clear();
  after_map.clear();
}

static void free_plist(Phone * &phon, SimpleDiphoneUnitIndex::PList * &list)
{
  (void)phon;
  if (list)
    delete list;
}

SimpleDiphoneUnitIndex::~SimpleDiphoneUnitIndex(void)
{
  before_map.map(free_plist);
  after_map.map(free_plist);
}

bool SimpleDiphoneUnitIndex::recognise(FILE *stream)
{
  char buffer[s_magic_size+1];
  int wasat = ftell(stream);

  if (fread(buffer, sizeof(char), s_magic_size, stream) == s_magic_size)
    if (strncmp(buffer, s_magic, s_magic_size) == 0)
      {
	fseek(stream, wasat, SEEK_SET);
	return (bool)1;
      }

  fseek(stream, wasat, SEEK_SET);
  return (bool)0;
}

UnitIndex *SimpleDiphoneUnitIndex::create(void)
{
  SimpleDiphoneUnitIndex *index = new SimpleDiphoneUnitIndex;
  return index;
}

EST_read_status SimpleDiphoneUnitIndex::fill_from(FILE *stream, int binary)
{
  (void)binary;

  char buffer[s_magic_size+1];

  if (fread(buffer, sizeof(char), s_magic_size, stream) == s_magic_size)
    return format_ok;

  return misc_read_error;
}

EST_write_status SimpleDiphoneUnitIndex::write_to(FILE *stream, 
						  int binary)
{
  (void)binary;

  if (fwrite(s_magic, sizeof(char), s_magic_size, stream) == s_magic_size)
    return write_ok;

  return misc_write_error;
}

static int process_phone_map(SimpleDiphoneUnitIndex::PhoneMap &map, PhoneSet *phones, LISP value)
{
  LISP l=value;
  LISP tol;

  if (!phones)
    {
      cerr << "Can't set mapping without database\n";
      return 0;
    }

  for(; CONSP(l) ; l=CDR(l))
    {
      if (!CONSP(CAR(l)))
	err("expected phone mapping", CAR(l));

      tol = CDR(CAR(l));
      EST_String from_name(get_c_string(CAR(CAR(l))));
      Phone *from = phones->member(from_name);

      if (!from)
	return 0;

      SimpleDiphoneUnitIndex::PList *to_list;

      if (!map.present(from))
	  map.add_item(from, new SimpleDiphoneUnitIndex::PList);

      to_list = map.val(from);

      for(; tol; tol=CDR(tol))
	{
	  EST_String to_name(get_c_string(CAR(tol)));
	  Phone *to = phones->member(to_name);
	
	  if (!to)
	    return 0;

	  to_list->append(to);
	}
    }
  return 1;
}

int SimpleDiphoneUnitIndex::set_property(EST_String property, ValueType value)
{
  PhoneSet *phones = NULL;

  if (database())
    phones = database()->phoneset();

  if (property == "before_map")
    return process_phone_map(before_map, phones, value_as_list(value));
  else if (property == "after_map")
    return process_phone_map(after_map, phones, value_as_list(value));
  else
    return UnitIndex::set_property(property, value);
}

void SimpleDiphoneUnitIndex::index(void) 
{
  // do nothing, the index is purely a computed function.
}

static const SimpleDiphoneUnitIndex::PList *lookup_in_map(const SimpleDiphoneUnitIndex::PhoneMap &map, const EST_String name, const PhoneSet *phones)
{
  if (!phones)
    return NULL;

  const SimpleDiphoneUnitIndex::PList *list = NULL;
  Phone *ph = phones->member(name);

  if (!ph)
    return NULL;

  if (map.present(ph))
      list = map.val(ph);

  return list;
}

bool SimpleDiphoneUnitIndex::lookup(const EST_String p1, const EST_String p2, EST_TList<UnitName> &options) const
{
  options.clear();

  if (!database())
    return FALSE;

  UnitCatalogue *catalogue = database()->catalogue();
  PhoneSet *phones = database()->phoneset();

  EST_String name(EST_String::cat(p1, "-", p2));

  if (catalogue->has_entry(name))
    options.append(EST_String(name));

  const SimpleDiphoneUnitIndex::PList *before_options = lookup_in_map(before_map, p1, phones);
  const SimpleDiphoneUnitIndex::PList *after_options = lookup_in_map(after_map, p2, phones);

  EST_TBI *b, *a;

  
  if (after_options)
    for(a=after_options->head(); a; a = next(a))
      {
	Phone *after_option = (*after_options)(a);

	name = EST_String::cat(p1, "-", after_option->phone_name());

	if (catalogue->has_entry(name))
	    options.append(name);
      }

  if (before_options)
    for(b=before_options->head(); b; b = next(b))
      {
	Phone *before_option = (*before_options)(b);

	name = EST_String::cat(before_option->phone_name(), "-", p2);

	if (catalogue->has_entry(name))
	    options.append(EST_String(name));
      }

  return (bool)(!options.empty());
}

#if defined(INSTANTIATE_TEMPLATES)
#include "../base_class/EST_TList.cc"
#include "../base_class/EST_TSortable.cc"
#include "../base_class/EST_TVector.cc"
#include "../base_class/EST_KV.cc"
template class EST_TVector<Phone *>;
template class EST_TList<Phone *>;
template class EST_TItem<Phone *>;
template class EST_TSortable<Phone *>;

template class EST_TList<KVI<Phone *, EST_TVector<Phone *> > >;
template class EST_TItem<KVI<Phone *, EST_TVector<Phone *> > >;
template class KVL<Phone *,EST_TVector<Phone * > >;
template class KVI<Phone *,EST_TVector<Phone * > >;
template class KVI<Phone *,EST_TList<Phone * > >;

template bool operator==(EST_TList<Phone *> const &, 
			 EST_TList<Phone *> const &);

template class EST_TList<KVI<Phone *, EST_TList<Phone *> > >;
template class EST_TItem<KVI<Phone *, EST_TList<Phone *> > >;
template class KVL<Phone *,EST_TList<Phone *> * >;
template class KVI<Phone *,EST_TList<Phone *> * >;
template class EST_TList<KVI<Phone *,EST_TList<Phone *> *> >;
template class EST_TItem<KVI<Phone *,EST_TList<Phone *> *> >;
#endif
