 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /*************************************************************************/
 /*                                                                       */
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)             */
 /*                   Date: Thu Feb 27 1997                               */
 /* -------------------------------------------------------------------   */
 /* Database containing synthesis information. Since these are big and    */
 /* heavy we pass arount pointers to them rather than the objects         */
 /* themselves.                                                           */
 /*                                                                       */
 /*************************************************************************/

class UnitDatabase;
class UnitName;
class UnitProperties;
class UnitCatalogue;

#ifndef __UNITDATABASE_H__
#define __UNITDATABASE_H__

// define the following to prevent instances ofthe root unitdatabse
// class from being created.
// #define UNITDATABASE_ABSTRACT (1)

#include "EST.h"
#include "EST_TNamedEnum.h"
#include "EST_THash.h"
#include "EST_Pathname.h"
#include "EST_ContourType.h"
#include "EST_TVector.h"
#include "SubtypedObject.h"

#define UnitCacheHashtableSize (100)

#if defined(UNITDATABASE_ABSTRACT)
#   define PURE_METHOD =0
#else
#   define PURE_METHOD // empty
#endif

// First two types which parameterise the entire system.

// UnitName is pretty basic, it`s really just a place holder in case we
// decide to make names something more interesting.

class UnitName {
private:
  EST_String nm;
public:
  UnitName(void) {nm= "";};
  UnitName(const EST_String s) {nm=s;};
  UnitName(const char *it) {nm=it;};
  UnitName(const UnitName &from) {nm=from.nm;};
  operator EST_String () const {return nm;};

  int operator == (const UnitName &n) const {return nm == n.nm;}; 
  int operator !=(const UnitName &n) const {return nm != n.nm;};
  UnitName &operator = (const UnitName &from) {nm=from.nm; return *this;};
  UnitName &operator = (const char *name) {nm=name; return *this;};
  UnitName &operator = (const EST_String &name) {nm = name; return *this;};
};

// UnitProperties stores the array of properties which the selection
// code might want. Outside the indexing and selection code we just treat
// it as a block of memory 

#define NUM_LONGS_IN_PROPERTIES (4)

class UnitProperties {
private:
  union {
    char ch[sizeof(long)*NUM_LONGS_IN_PROPERTIES];
    struct { long a,b,c,d; } ln;
  } contents;

public:
  UnitProperties(void) {memset(contents.ch, 0, sizeof(long)*NUM_LONGS_IN_PROPERTIES);};
  UnitProperties(UnitProperties &from) {contents.ln = from.contents.ln;};
  operator const char *() const {return contents.ch;};
  operator const long &() const {return contents.ln.a;};

  UnitProperties &operator = (UnitProperties &from) {contents.ln = from.contents.ln; return *this;};
};
  
class Unit;
class UnitIndex;

// Each type of database has it`s own type of catalogue, so we don`t have to
// have any clever way to work out what kinds exist. 

class UnitCatalogue {
public:
  class Itterator {
  public:
    Itterator(void);
    virtual ~Itterator(void) PURE_METHOD;
    virtual UnitName next(void) PURE_METHOD;
    virtual int at_end(void) PURE_METHOD;
  };

  virtual int num_entries (void) PURE_METHOD;
  virtual int has_entry(EST_String name);
};
  
// This is a complete database. It is responsible for cacheing units
// and for knowing how to interface to the catalogue to get information
// from disk.

#define CT_WAVE(N) ((N)<<1)
#define CT_COEF(N) (((N-1)<<1) | 1)

class UnitDatabase : public SubtypedObject<UnitDatabase> {

public:

  typedef enum ContentType { ct_none=0, 
			     ct_signal=CT_WAVE(1), 
			     ct_residual=CT_WAVE(2), 
			     ct_experimentalwave1=CT_WAVE(5), 
			     ct_experimentalwave2=CT_WAVE(6), 
			     ct_pitchmarks=CT_COEF(1), 
			     ct_analysis=CT_COEF(2),
			     ct_experimentalcoef1=CT_COEF(5), 
			     ct_experimentalcoef2=CT_COEF(6), 
			     ct_maximum=CT_COEF(16)
  } ContentType;
  static  EST_TNamedEnum<ContentType> ContentTypeMap;

  typedef enum AccessStrategy { as_direct, as_dynamic, as_ondemand } AccessStrategy;
  static  EST_TNamedEnum<AccessStrategy> AccessStrategyMap;

  typedef enum SignalPreprocessing { sp_none } SignalPreprocessing;
  static  EST_TNamedEnum<SignalPreprocessing> SignalPreprocessingMap;

  struct wave_information_s {
    EST_String format;
    EST_String sample_format;
    ContentType pm_content_type;
    int sample_rate;
    AccessStrategy strategy;
    SignalPreprocessing preprocessing;
  };

  struct coefficient_information_s {
    EST_String format;
    float frame_shift;
    AccessStrategy strategy;
    EST_ContourType coefficient_format;
    EST_String time_channel;
    EST_String length_channel;
    float time_scale;
  };

  struct index_information_s {
    EST_String name;
    EST_String type;
    UnitIndex *index;
  };
  
private:

  EST_String p_description;
  EST_String p_copyright;

  EST_Pathname p_filename;

  EST_String p_unit_type;

  PhoneSet *p_phoneset;

  UnitCatalogue *p_catalogue;

  int p_num_indexes;
  index_information_s *p_indexes;

  EST_TStringHash<Unit *> p_unit_cache;

  KVL<EST_String, EST_String> p_properties;

protected:
  static ContentType contents_name_to_number(EST_String name);

  UnitDatabase (void) ;

  void set_description(EST_String description) INLINE_IF_OK({p_description=description;});
  void set_copyright(EST_String copyright) INLINE_IF_OK({p_copyright = copyright;});
  void set_filename(EST_String filename) INLINE_IF_OK({p_filename = filename;});
  void set_unit_type(EST_String unit_type) INLINE_IF_OK({p_unit_type = unit_type;});
  void set_catalogue(UnitCatalogue *catalogue) INLINE_IF_OK({p_catalogue = catalogue;});

  void set_phoneset(PhoneSet *phones) INLINE_IF_OK({p_phoneset = phones;});
  void add_index(EST_String name, EST_String type, UnitIndex *index);
  
public:
  virtual ~UnitDatabase(void) PURE_METHOD;

  virtual EST_read_status fill_from(FILE *stream) PURE_METHOD;
  virtual EST_write_status write_to(FILE *stream) PURE_METHOD;

  // compilation Take a (presumably empty) database and insert the
  // contents of an existing one.

  virtual void fill_from(UnitDatabase &source);

  // Asking about what is in the database.

  EST_String description(void) const {return p_description;};
  EST_String copyright(void) const {return p_copyright;};
  EST_String filename(void) const {return p_filename;};
  EST_String unit_type(void) const {return p_unit_type;};
  PhoneSet *phoneset(void) const {return p_phoneset;};
  UnitCatalogue *catalogue(void) const {return p_catalogue;};

  virtual EST_StrVector wave_names(void) const PURE_METHOD;
  virtual EST_StrVector coefficient_names(void) const PURE_METHOD;
  virtual EST_StrVector index_names(void) const ;

  virtual wave_information_s *wave_information(ContentType content) PURE_METHOD;
  virtual coefficient_information_s *coefficient_information(ContentType content) PURE_METHOD;
  virtual index_information_s *index_information(EST_String type);

  virtual EST_Wave *get_wave(UnitDatabase::ContentType content, Unit &unit) PURE_METHOD;
  virtual EST_Track *get_coefficients(ContentType content, EST_ContourType format, Unit &unit) PURE_METHOD;
  virtual EST_TVector<float> *get_segments(Unit &unit) PURE_METHOD;

  virtual const Unit *unit(const EST_String name);
  virtual const UnitIndex *index(const EST_String type);

  virtual void property_names(EST_TList<EST_String> &list) const;
  virtual int set_property(EST_String property, ValueType value);
  virtual ValueType property(EST_String property) const;

  virtual void print_description(FILE *stream);

  static inline int is_waveform(int n) { return (n&1) == 0; }
  static inline int is_coefficients(int n) { return (n&1) == 1; }
  static inline ContentType ct_waveform(int n)  { return (ContentType)CT_WAVE(n);};
  static inline ContentType ct_coefficient(int n)  { return (ContentType)CT_COEF(n);};

#if defined(INCLUDE_LISP)
  //lisp interface
  static LISP lisp_print_description(LISP database, LISP file);
  static void lisp_declare(void);
#endif

  friend class SubtypedObject<UnitDatabase>;
};


#undef PURE_METHOD
#undef CT_WAVE
#undef CT_COEF

#include "Unit.h"
#include "UnitIndex.h"
#endif
