/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                     Author :  Alan W Black                            */
/*                     Date   :  October 1996                            */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/*   Interface to various low level waveform functions from Lisp         */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "EST_unix.h"
#include <stdlib.h>
#include "festival.h"
#include "festivalP.h"

static float f0_interpolate(EST_Utterance &u, EST_Stream_Item &s, float t);

static LISP utt_save_wave(LISP utt, LISP fname)
{
    EST_Utterance *u = GETUTTVAL(utt);
    EST_String filename = get_c_string(fname);
    EST_String type;
    LISP ltype;
    EST_Wave *w;

    if (fname == NIL)
	filename = "save.wav";

    if (!u->stream_present("Wave"))
    {
	cerr << "no wave in utterance" << endl;
	festival_error();
    }

    w = WaveC(*u->stream("Wave").head());
    ltype = ft_get_param("Wavefiletype");
    if (ltype == NIL)
	type = "nist";
    else
	type = get_c_string(ltype);

    if (w->save(filename,type) != write_ok)
    {
	cerr << "utt.save.wave: failed to write wave to \"" << filename 
	    << "\"" << endl;
	festival_error();
    }

    return utt;
}

static LISP utt_save_f0(LISP utt, LISP fname)
{
    // Save utt's F0 in fname as an ESPS file
    EST_Utterance *u = GETUTTVAL(utt);
    EST_Stream_Item *s;
    EST_Track f0;
    float p;
    float length = u->stream("Segment").tail()->end();
    int i,frames = (int)(length / 0.010);
    f0.resize(frames,4);
    
    for (p=0.0,i=0,s=u->stream("Segment").head(); s != 0; s=next(s))
    {
	if (i >= frames)
	    break;  // may hit here one before end
	for ( ; p < s->end(); p+=0.010,i++)
	{
	    if (i >= frames)
		break;  // may hit here one before end
	    if ((ffeature(*u,*s,"ph_vc") == "+") ||
		(ffeature(*u,*s,"ph_cvox") == "+"))
	    {
		f0(i,0) = f0_interpolate(*u,*s,p);
		f0(i,1) = 1;
	    }
	    else
	    {
		f0(i,0) = 0;
		f0(i,1) = 0.0; // unvoiced;
	    }
	}
    }
    f0.set_field_name("F0",0);
    f0.set_field_name("prob_voice",1);
    f0.fill_time(0.01);

    f0.set_file_type("esps");
    f0.set_contour_type(EST_ContourType::F0);

    if (f0.save(get_c_string(fname),"esps") != write_ok)
    {
	cerr << "utt.save.wave: failed to write wave to \"" << 
	    get_c_string(fname) << "\"" << endl;
	festival_error();
    }

    return utt;
}

static float f0_interpolate(EST_Utterance &u, EST_Stream_Item &s, float t)
{
    // Return interpolated F0 at time t (which should be within s)
    float p1,p0,d1,d0;

    if (t < s.mid())
    {
	p1 = ffeature(u,s,"pitch");
	p0 = ffeature(u,s,"p.pitch");
	d1 = s.mid();
	d0 = (prev(&s) ? prev(&s)->mid() : 0);
    }
    else
    {
	p0 = ffeature(u,s,"pitch");
	p1 = ffeature(u,s,"n.pitch");
	d0 = s.mid();
	d1 = (next(&s) ? next(&s)->mid() : s.end());
    }
    if (p0 == 0.0)
	return  p1;       // first in stream
    else if (p1 == 0.0)
	return p0;        // last in stream
    else
	return p0 + (p1-p0)*(t-d0)/(d1-d0);
}

static LISP utt_send_wave_client(LISP utt)
{
    // Send the waveform to a client (must be acting as server)
    EST_Utterance *u = GETUTTVAL(utt);
    EST_Wave *w;
    EST_String tmpfile = make_tmp_filename();

    if (!u->stream_present("Wave"))
    {
	cerr << "no wave in utterance" << endl;
	festival_error();
    }

    w = WaveC(*u->stream("Wave").head());
    if (ft_server_socket == -1)
    {
	cerr << "utt_send_wave_client: not in server mode" << endl;
	festival_error();
    }
	
    w->save(tmpfile,"nist");
    write(ft_server_socket,"WV\n",3);
    socket_send_file(ft_server_socket,tmpfile);
    unlink(tmpfile);

    return utt;
}

static void gc_wave(void *w) { delete (EST_Wave *)w; }
static LISP utt_import_wave(LISP utt,LISP fname)
{
    // Create an wave stream and load fname into it
    EST_Utterance *u = GETUTTVAL(utt);
    EST_String filename = get_c_string(fname);
    EST_Wave *wave = new EST_Wave;
    EST_Stream_Item item;

    if (wave->load(filename) != format_ok)
    {
	cerr << "Cannot load wavefile: " << filename << endl;
	festival_error();
    }

    u->create_stream("Wave");
    item.init("Wave");
    item.set_contents(wave,gc_wave);

    u->stream("Wave").append(item);
    
    return utt;
}

static LISP utt_play(LISP utt)
{
    // Play the wave through the selected audio device 
    EST_Utterance *u = GETUTTVAL(utt);
    EST_Option al;
    EST_Stream_Item *w;
    LISP audio;

    if ((!u->stream_present("Wave")) ||
	(u->stream("Wave").head() == 0) ||
	(WaveC(*(u->stream("Wave").head())) == 0))
    {
	cerr << "utt_play: no wave present" << endl;
	festival_error();
    }
    else if (audsp_mode)  // asynchronous mode
	audsp_play_wave(WaveC(*(u->stream("Wave")).head()));
    else
    {
	if ((audio = ft_get_param("Audio_Method")) != NIL)
	    al.add_item("-p",get_c_string(audio));
	if ((audio = ft_get_param("Audio_Command")) != NIL)
	    al.add_item("-command",quote_string(get_c_string(audio)));
	if ((audio = ft_get_param("Audio_Required_Rate")) != NIL)
	    al.add_item("-rate",get_c_string(audio));
	if ((audio = ft_get_param("Audio_Required_Format")) != NIL)
	    al.add_item("-otype",get_c_string(audio));
	al.add_item("-quality","HIGH");
	w = u->stream("Wave").head();
	play_wave(*(WaveC(*w)),al);
    }

    return utt;

}

static LISP utt_rescale(LISP utt,LISP lfactor, LISP ltype)
{
    // Modify the waveform gain, either relative or absolute.
    EST_Utterance *u = GETUTTVAL(utt);

    if (!u->stream_present("Wave"))
    {
	cerr << "Wave rescale: no waveform to rescale" << endl;
	festival_error();
    }
    else
    {
	float factor = get_c_float(lfactor);
	EST_Wave *w = WaveC(*(u->stream("Wave")).head());
	
	if ((factor < 0.01) || (factor > 100.0))  // sanity check
	{
	    cerr << "Wave rescale: modification factor too weird: " <<
		factor << endl;
	    festival_error();
	}
	
	if ((ltype == NIL) ||
	    (streq(get_c_string(ltype),"absolute")))
	    w->rescale(factor);
	else
	{
	    if (factor > 1.0)
	    {
		cerr << "Wave rescale: in relative mode factor cannot be greater than 1.0" << endl;
		festival_error();
	    }
	    w->rescale(factor,TRUE);  // normalize wave first
	}
    }

    return utt;

}

static LISP utt_resample(LISP utt, LISP lrate)
{
    // Resample wave to desirsed rate
    EST_Utterance *u = GETUTTVAL(utt);

    if (!u->stream_present("Wave"))
    {
	cerr << "Wave resample: no wave to resample" << endl;
	festival_error();
    }
    else
    {
	int rate = get_c_long(lrate);
	EST_Wave *w = WaveC(*(u->stream("Wave")).head());

	w->resample(rate);
//	if (w->sample_rate() != rate)
//	    festival_error();
    }

    return utt;

}

static LISP utt_wave_info(LISP utt, LISP item)
{
    // Returns information about the waveform in utt;
    EST_Utterance *u = GETUTTVAL(utt);

    if (!u->stream_present("Wave"))
    {
	cerr << "wave.info: no wave in utt" << endl;
	festival_error();
    }
    else
    {
	EST_Wave *w = WaveC(*(u->stream("Wave")).head());
	
	if (item != NIL)
	{
	    if (streq("num_samples",get_c_string(item)))
		return flocons(w->num_samples());
	    else if (streq("sample_rate",get_c_string(item)))
		return flocons(w->sample_rate());
	    else if (streq("data_addr",get_c_string(item)))
	    {
		// This is an awful hack, but its better than 
		// possible alternatives
		char addr[256];
		sprintf(addr,"0x%p",w->data());
		return strintern(addr);
	    }
	    else
	    {
		cerr << "wave.info: unknown info type \"" << 
		    get_c_string(item) << "\"" << endl;
		festival_error();
	    }
	}
	else  // a list of items
	{
	    LISP l,ritems = NIL;
	    for (l=item; l != NIL; l=cdr(l))
	    {
		ritems = cons(make_param_lisp(get_c_string(car(l)),
					      utt_wave_info(utt,car(l))),
			      ritems);
	    }
	    return reverse(ritems);
	}
    }
    return NIL;
}

void festival_utt_wave_funcs(void)
{
    // declare utterance (wave) specific Lisp functions 

    init_subr_2("utt.save.wave",utt_save_wave,
 "(utt.save.wave UTT FILENAME)\n\
  Save the waveform of UTT in FILENAME in the format specified by the\n\
  Parameter Wavefiletype");
    init_subr_1("utt.play",utt_play,
 "(utt.play UTT)\n\
  Send the waveform of UTT to the audio device as specified by the\n\
  Parameter Audio_Method.");
    init_subr_2("utt.import.wave",utt_import_wave,
 "(utt.import.wave UTT FILENAME)\n\
  Load waveform from FILENAME and create a Wave stream in UTT with the\n\
  waveform as its contents.");
    init_subr_2("utt.wave.resample",utt_resample,
 "(utt.wave.resample UTT RATE)\n\
  Resample waveform in UTT to RATE (if it is already rate it remains\n\
  unchanged).");
    init_subr_3("utt.wave.rescale",utt_rescale,
 "(utt.wave.rescale UTT FACTOR TYPE)\n\
  Modify the gain of the waveform in UTT by FACTOR (which should be in the\n\
  range 0.01 > FACTOR > 100.0).  If TYPE is specified and not 'absolute\n\
  then is normalized first.  It is normalized by making the magnitude of\n\
  of the largest sample point 32767.");  
    init_subr_2("utt.wave.info",utt_wave_info,
 "(utt.wave.info UTT TAG)\n\
  Return info about wave.  TAG may be num_samples, sample_rate or data_addr,\n\
  or a list of any of these.  If TAG is atomic an atomic value is returned,\n\
  if tag is a list then a list of values is returned (in the order they\n\
  are specified in TAG.");
    init_subr_1("utt.send.wave.client",utt_send_wave_client,
 "(utt.send.wave.client UTT)\n\
  Sends wave in UTT to client.  If not in server mode gives an error\n\
  Note the client must be expecting to receive the waveform.");
    init_subr_2("utt.save.f0",utt_save_f0,
 "(utt.save.f0 UTT FILENAME)\n\
 Save F0 of UTT as esps track file in FILENAME.");

}

    

