/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                 Authors:  Alan W Black                                */
/*                 Date   :  October 1996                                */
/*-----------------------------------------------------------------------*/
/*  Access to the lower level EST_Ngrammar class                         */
/*                                                                       */
/*  This allows Ngrams to be loaded, built, tested and saved             */
/*  Simply Lisp bindings for the Ngrammar major methods                  */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "festival.h"
#include "festivalP.h"
#include "fngram.h"

static LISP ngram_list = NIL;

static void add_ngram(LISP name,EST_Ngrammar *n);
static void list_to_strlist(LISP l, EST_StrList &sl);

LISP lisp_load_ngram(LISP name, LISP filename)
{
    EST_Ngrammar *n;

    n = new EST_Ngrammar();
    if (n->load(get_c_string(filename)) != 0)
    {
	cerr << "Ngrammar: failed to read ngrammar " << 
	    get_c_string(name) << " from \"" <<
		get_c_string(filename) << "\"" << endl;
	festival_error();
    }

    add_ngram(name,n);

    return name;
}

static void add_ngram(LISP name,EST_Ngrammar *n)
{
    EST_Ngrammar *old_n;
    LISP lpair; 

    lpair = siod_assoc_str(get_c_string(name),ngram_list);

    if (ngram_list == NIL)
	gc_protect(&ngram_list);
    
    if (lpair == NIL)
	ngram_list = cons(cons(name,cons(MKPTR(n),NIL)),ngram_list);
    else
    {
	cwarn << "Ngrammar: " << get_c_string(name) << " recreated" << endl;
	old_n = (EST_Ngrammar *)PTRVAL(car(cdr(lpair)));
	delete old_n;
	PTRVAL(car(cdr(lpair))) = n;
    }
}

LISP lisp_build_ngram(LISP name, LISP files, LISP params)
{
    EST_Ngrammar *n;
    int order;
    EST_StrList vocab, pred_vocab;
    LISP lvocab,lpred_vocab,f;
    EST_String prev_prev_tag, prev_tag, last_tag, rep;
    EST_Ngrammar::representation_t representation = EST_Ngrammar::dense;
    EST_StrList files_strlist;
    
    order = get_param_int("order",params,2);
    lvocab = get_param_lisp("vocab",params,NIL);
    list_to_strlist(lvocab,vocab);
    lpred_vocab = get_param_lisp("pred_vocab",params,lvocab);
    list_to_strlist(lpred_vocab,pred_vocab);
    rep = get_param_str("representation",params,"dense");
    prev_prev_tag = get_param_str("prev_prev_tag",params,"");
    prev_tag = get_param_str("prev_tag",params,"");
    last_tag = get_param_str("last_tag",params,"");

    if (!consp(files))
	files_strlist.append(get_c_string(files));  // only one file
    else
	for (f=files; f != NIL; f=cdr(f))
	    files_strlist.append(get_c_string(car(f)));
    
    if (streq(rep,"dense"))
	representation = EST_Ngrammar::dense;
    else if (streq(rep,"sparse"))
	representation = EST_Ngrammar::sparse;
    else
    {
	cerr << "ngram.build unknown representation type " <<
	    representation << endl;
	festival_error();
    }

    n = new EST_Ngrammar(order,representation,vocab,pred_vocab);
    if (!n->build(files_strlist,
		  prev_tag,prev_prev_tag,last_tag))
    {
	EST_String msg = siod_sprint(files);
	cerr << "ngram.build: failed to build ngram from " << msg << endl;
	festival_error();
    }
    
    add_ngram(name,n);

    return name;
}

LISP lisp_test_ngram(LISP name, LISP filename, LISP params)
{
    EST_Ngrammar *n;
    EST_String prev_prev_tag, prev_tag, last_tag;
    double perplexity, entropy, raw_entropy, count;
    
    prev_prev_tag = get_param_str("prev_prev_tag",params,"");
    prev_tag = get_param_str("prev_tag",params,"");
    last_tag = get_param_str("last_tag",params,"");
    
    n = get_ngram(get_c_string(name));
    if (n == 0)
	festival_error();

    n->stats(get_c_string(filename),
	     raw_entropy,count,
	     entropy,perplexity,
	     prev_prev_tag, prev_tag, last_tag);
    
    return flocons(perplexity);
}

LISP lisp_smooth_ngram(LISP name, LISP count)
{
    EST_Ngrammar *n;
    
    n = get_ngram(get_c_string(name));
    if (n == 0)
	festival_error();
    
    Good_Turing_smooth(*n,get_c_long(count));

    return NIL;
}

LISP lisp_save_ngram(LISP name, LISP filename, LISP type)
{
    EST_Ngrammar *n;
    
    n = get_ngram(get_c_string(name));
    if (n == 0)
	festival_error();

    if (n->save(get_c_string(filename),get_c_string(type)) != write_ok)
    {
	cerr << "ngram.save: failed." << endl;
	festival_error();
    }

    return name;
}

static void list_to_strlist(LISP l, EST_StrList &sl)
{
    LISP t;

    for (t=l; t != NIL; t=cdr(t))
    {
	if (consp(car(t)))
	{
	    char *m = siod_sprint(l);
	    cerr << "List is not atomic: " << endl;
	    cerr << "   " << m << endl;
	    wfree(m);
	}
	else
	    sl.append(get_c_string(car(t)));
    }
}

EST_Ngrammar *get_ngram(const EST_String &name)
{
    //  Find ngram named name, returns NULL if none;
    LISP lpair;
    
    lpair = siod_assoc_str(name,ngram_list);

    if (lpair == NIL)
    {
	cwarn << "Ngrammar: no ngram named \"" << name << "\"" << endl;
	return 0;
    }
    else
	return (EST_Ngrammar *)PTRVAL(car(cdr(lpair)));
}
