 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /************************************************************************/
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)            */
 /*                   Date: Fri Mar 21 1997                              */
 /************************************************************************/
 /*                                                                      */
 /* Root object for types used in siod.                                  */
 /*                                                                      */
 /************************************************************************/

#include "festival.h"
#include "SubtypedObject.h"

EST_String SchemeObject::error_string("");

#if defined(DONT_INLINE_EVERYTHING)
void SchemeObject::set_name(EST_String name){p_name = name;}
EST_String SchemeObject::name(void) const {return p_name;}

void SchemeObject::gc_mark(void) {(gc_master_pointer()->p_gc_mark)++;} 
void SchemeObject::gc_clear_mark(void) {(gc_master_pointer()->p_gc_mark)=0;}
int  SchemeObject::gc_marked(void) {return (gc_master_pointer()->p_gc_mark);}

void SchemeObject::gc_ref(void) {(gc_master_pointer()->p_reference_count)++;}
void SchemeObject::gc_unref(void) {(gc_master_pointer()->p_reference_count)--;}
int SchemeObject::gc_referenced(void) {return (gc_master_pointer()->p_reference_count)>0;}

SchemeObject *SchemeObject::gc_master_pointer(void) {return this;};
#endif

SchemeObject::SchemeObject(void)
{
  p_gc_mark=0;
  p_reference_count=0;
  p_name ="";
}

SchemeObject::~SchemeObject(void)
{
}

void SchemeObject::property_names(EST_TList<EST_String> &list) const
{
  list.append("name");
}

int SchemeObject::set_property(EST_String property, ValueType value)
{
  if (property == "name")
    {
    set_name(value_as_c_string(value));
    return 1;
    }

  error_string = "unknown property "+property;
  return 0;
}

ValueType SchemeObject::property(EST_String name) const
{
  if (name == "name")
    return string_as_value(p_name);

  return null_as_value;
}

int SchemeObject::set_properties(LISP properties)
{
  LISP l = properties;

  while(l)
    {
      if (!CDR(l))
	break;
      const char *property = get_c_string(CAR(l));
      LISP value = CAR(CDR(l));

      error_string="";
      if (!set_property(property, value))
	{
	  cwarn << "illegal property '"<< property << "'\n";

	  if (error_string != "")
	    cwarn << "\t" << error_string <<"\n";
	  return 0;
	}
      l = CDR(CDR(l));
    }
  return 1;
}

// To safely free an object which may be seen by lisp, we put it in an
// unreferenced cell and let the GC take care of it.

void SchemeObject::decrease_reference_count(void *it)
{
  SchemeObject *object = (SchemeObject *) it;

  object->gc_unref();
  if (!object->gc_referenced())
    {
      // cwarn << "leave for GC "<< (int)object << "\n";
      siod_make_typed_cell(tc_festival_unit, object);
    }
}


#if defined(INCLUDE_LISP)

void SchemeObject::lisp_print(LISP ptr, FILE *f)
{
  SchemeObject *object = (SchemeObject *)PTRVAL(ptr);
  fput_st(f,"#<");
  fput_st(f, object->type_name());
  fput_st(f," '");
  fput_st(f, object->name());
  fput_st(f,">");
}

void SchemeObject::lisp_print_string(LISP ptr, char *s)
{
  SchemeObject *object = (SchemeObject *)PTRVAL(ptr);
  sprintf(s, "#<%s '%s'>", 
	  (const char *)object->type_name(),
	  (const char *)object->name());
}

LISP SchemeObject::lisp_gc_mark(LISP cell)
{
  SchemeObject *object = (SchemeObject *)PTRVAL(cell);

  // printf("SchemeObject:: mark %s\n", (const char *)object->name());

  object->gc_mark();

  return NULL;
}

void SchemeObject::lisp_gc_clear_mark(LISP cell)
{
  SchemeObject *object = (SchemeObject *)PTRVAL(cell);

  // printf("SchemeObject:: clear mark %s\n", (const char *)object->name());

  object->gc_clear_mark();
}

void SchemeObject::lisp_gc_free(LISP cell)
{
  SchemeObject *object = (SchemeObject *)PTRVAL(cell);
  // guaranteed to be called only once. 

  // printf("SchemeObject:: free? %d %d\n", object->gc_marked(), object->gc_referenced());
  if (!object->gc_marked() && !object->gc_referenced())
    {
      // printf("SchemeObject:: free %s\n", (const char *)object->name());
      delete object;
    }
}

LISP SchemeObject::lisp_set_properties(LISP cell, LISP properties)
{
  if (cell && 
      is_festival_type(TYPE(cell)))
    {
      SchemeObject *object = (SchemeObject *)PTRVAL(cell);
      return object->set_properties(properties)? cell : (LISP)NULL;
    }
  else
    err("Can't set propperty for", cell);
  return NULL;
}

LISP SchemeObject::lisp_set_property(LISP cell, LISP lproperty, LISP lvalue)
{
  if (cell && 
      is_festival_type(TYPE(cell)) &&
      lproperty && lvalue)
    {
      SchemeObject *object = (SchemeObject *)PTRVAL(cell);
      const char *property = get_c_string(lproperty);
      error_string="";
      if (object->set_property(property, lvalue))
	return cell;
      cwarn << "illegal property '"<< property << "' '"<< get_c_string(lvalue) <<"'\n";
      if (error_string != "")
	cwarn << " -- " << error_string << "\n";
    }
  else
    err("Can't set propperty for", cell);

  return NULL;
}

LISP SchemeObject::lisp_get_property(LISP cell, LISP lproperty)
{
  if (cell && 
      is_festival_type(TYPE(cell)) &&
      lproperty)
    {
      SchemeObject *object = (SchemeObject *)PTRVAL(cell);
      const char *property = get_c_string(lproperty);
      return object->property(property);
    }
  else
    err("Can't get property for", cell);

  return NULL;
}

LISP SchemeObject::lisp_property_names(LISP cell)
{
  LISP lnames = NIL;

  if (cell && 
      is_festival_type(TYPE(cell)))
    {
      SchemeObject *object = (SchemeObject *)PTRVAL(cell);
      EST_TList<EST_String> names;
      object->property_names(names);

      EST_Litem *item;
      for(item=names.head(); item; item=next(item))
	{
	  EST_String name = names(item);
	  lnames = cons(string_cell(name, name.length()), lnames);
	}
    }
  else
    err("Can't get property names for", cell);

  return lnames;
}


void SchemeObject::lisp_declare(void)
{  

  init_subr_2("set_properties", SchemeObject::lisp_set_properties,
 "(set_properties OBJECT PROPERTIES)\n\
  Set the properties of the object.");

  init_subr_3("set_property", SchemeObject::lisp_set_property,
 "(set_property OBJECT PROPERTY VALUE)\n\
  Set a property of an object.");

  init_subr_2("get_property", SchemeObject::lisp_get_property,
 "(get_property OBJECT PROPERTY)\n\
  Get a property of an object."); 

  init_subr_1("get_property_names", SchemeObject::lisp_property_names,
 "(get_property_names OBJECT)\n\
  Returns a list of the properties of OBJECT.");
}

#endif
