//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  A cartridge is a device that connects to the system.  It contains
  the code from a 2600 game cartridge and handles any bank switching
  performed by the cartridge.

  @author  Bradford W. Mott
  @version $Id: Cart.cxx,v 1.3 1997/05/17 19:23:56 bwmott Exp $
*/

#include <string.h>
#include <stdlib.h>
#include <fstream.h>

#include "Cart.hxx"
#include "Cart2K.hxx"
#include "Cart3F.hxx"
#include "Cart4K.hxx"
#include "CartAR.hxx"
#include "CartE0.hxx"
#include "CartE7.hxx"
#include "CartF4SC.hxx"
#include "CartF6.hxx"
#include "CartF6SC.hxx"
#include "CartF8.hxx"
#include "CartF8SC.hxx"
#include "CartFASC.hxx"
#include "Error.hxx"
#include "Props.hxx"
#include "System.hxx"

//============================================================================
// Constructor
//============================================================================
Cartridge::Cartridge(System& system)
    : Device(system)
{
}

//============================================================================
// Destructor
//============================================================================
Cartridge::~Cartridge()
{
}

//============================================================================
// Store value in the given address
//============================================================================
void Cartridge::poke(uWord, uByte)
{
  // Don't do anything by default for pokes
}

//============================================================================
// Open stream on the cartridges ROM image specified in the properties
//============================================================================
static void openStream(ifstream& in, const Properties& properties)
{
  // TODO: Remove the searching in here since it's nolonger needed

  #if defined UNIX_OS
    const char* paths = ";ROMS/;2600/;../ROMS/;../2600/";
  #elif (defined(MSDOS_OS) || defined(WIN32))
    const char* paths = ";ROMS\\;2600\\;..\\ROMS\\;..\\2600\\";
  #elif defined MAC_OS
    const char* paths = ";:ROMS:;:2600:;::ROMS:;::2600:";
  #elif defined OS_2
    const char* paths = ";ROMS\\;2600\\;..\\ROMS\\;..\\2600\\";
  #endif
 
  const char* image = properties.find("Cartridge.Image");
  const char* dir = properties.find("Cartridge.Directory");

  for(const char* p = paths; ; ++p)
  {
    char filename[1024];
    const char* subpath = p;

    // scan to end of subpath
    for(; (*p != ';') && (*p != 0); ++p);

    // Create filename
    memset(filename, 0, sizeof(filename));
    if(strlen(dir) != 0)
    {
      strcpy(filename, dir);
      filename[strlen(filename)] = PATH_SEPARATOR;
    } 
    strncat(filename, subpath, p - subpath);
    strcat(filename, image);

    in.clear();

    #if defined UNIX_OS
      in.open(filename, ios::in | ios::nocreate);
    #elif defined MAC_OS
      in.open(filename, ios::in | ios::binary);
    #else
      in.open(filename, ios::in | ios::nocreate | ios::binary);
    #endif

    if(!in.fail())
    {
      break;
    }
    else if(*p == 0)
    {
      Error err;
      err.message() << "The ROM image \"" << image << "\" couldn't be located!";
      err.description() 
          << "If all you have is a VCS file then you need to get a" << endl
          << "copy of the required BIN file and place it in the correct" << endl
          << "directory (see the Stella User Guide for more information)";
      Throw(err); 
    }
    else
    {
      in.close();
    }
  }
}

//============================================================================
// Class method which makes sure the ROM image file exists
//============================================================================
void Cartridge::exists(const Properties& properties)
{
  ifstream in;
  openStream(in, properties);
  in.close();
}

//============================================================================
// Class method that creates a new cartridge 
//============================================================================
Cartridge* Cartridge::create(System& system, const Properties& properties)
{
  Cartridge* cartridge;
  ifstream in;

  // Open stream on the ROM image of the cartridge
  openStream(in, properties);

  uByte* rom = new uByte[84480];
  in.read(rom, 84480);
  uLong size = in.gcount();

  // Create the cartridge object of the correct type
  const char* type = properties.find("Cartridge.Type");

  if(strcmp(type, "2K") == 0)
    cartridge = new Cartridge2K(system, rom);
  else if(strcmp(type, "3F") == 0)
    cartridge = new Cartridge3F(system, rom);
  else if(strcmp(type, "4K") == 0)
    cartridge = new Cartridge4K(system, rom);
  else if(strcmp(type, "AR") == 0)
    cartridge = new CartridgeAR(system, rom, size);
  else if(strcmp(type, "E0") == 0)
    cartridge = new CartridgeE0(system, rom);
  else if(strcmp(type, "E7") == 0)
    cartridge = new CartridgeE7(system, rom);
  else if(strcmp(type, "F4SC") == 0)
    cartridge = new CartridgeF4SC(system, rom);
  else if(strcmp(type, "F6") == 0)
    cartridge = new CartridgeF6(system, rom);
  else if(strcmp(type, "F6SC") == 0)
    cartridge = new CartridgeF6SC(system, rom);
  else if(strcmp(type, "F8") == 0)
    cartridge = new CartridgeF8(system, rom);
  else if(strcmp(type, "F8SC") == 0)
    cartridge = new CartridgeF8SC(system, rom);
  else if(strcmp(type, "FASC") == 0)
    cartridge = new CartridgeFASC(system, rom);
  else
  {
    cartridge = 0;

    Error err;
    err.message() << "An unknown cartridge type, \"" << type << "\", was "
        << "specified in the stella.vcs file!";
    err.description() 
        << "The Cartridge.Type field must be either 2K, 3F, 4K," << endl
        << "AR, E0, E7, F4SC, F6, F6SC, F8, F8SC, or FASC. If you" << endl
        << "don't know which value to use please read the bankswitching" << endl
        << "document found on the Stella home page.";
    Throw(err);
  }

  delete[] rom;
  return cartridge;
}

