/*
 * cwgen - simple morse code practice character generator
 */

/* trap for when no valid platform selected */
#if !defined(LINUX) && !defined(SCO) && !defined(UNIXWARE)
#include <-DLINUX,-DSCO,_or_-DUNIXWARE_must_be_defined>
#endif

/* include files */
#include <sys/param.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

/* conditional include files */
#if defined(LINUX)
#	include <strings.h>
#	include <getopt.h>
#endif
#if defined(SCO)
#	include <string.h>
#	include <limits.h>
#	define	MAXPATHLEN	_POSIX_PATH_MAX
#endif
#if defined(UNIXWARE)
#	include <string.h>
#endif

/* definitions */
#define	VERSION		"cwgen version 1.1"

#define	bool		int		/* boolean type */
#define	FALSE		0
#define TRUE		(!FALSE)

#define	EOS		'\0'		/* end of string */
#define	ASC_FNS		'/'		/* ASCII filename sep char */

#define MIN_GROUPS	1		/* lowest no of groups allowed */
#define MAX_GROUPS	20000		/* highest no of groups allowed */
#define MIN_GROUPSIZE	1		/* lowest group size allowed */
#define MAX_GROUPSIZE	80		/* highest WPM allowed */
#define DEF_CHARSET	"ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"
					/* default character set */

#define	MAXARGS		128		/* args to exec */
#define	MAXOPTSTR	1024		/* longest _OPTIONS env variable */
#define	ARGS_WHITESPACE	" \t"		/* argument separators */
#define	ENV_OPTIONS	"CWGEN_OPTIONS"	/* env string holding options */

/* initial defaults for parameters */
#define	INIT_GROUPS	128
#define	INIT_GROUPSIZE	5

/* global variables */
static int groups	= INIT_GROUPS;	/* initially 128 groups */
static int groupsize	= INIT_GROUPSIZE;/* initially 5-character groups */
static char *charset	= DEF_CHARSET;	/* initially send mixed letters nums */


/*
 * print_help - give help information
 */
static void print_help( char *pname ) {
	printf(
#if defined(LINUX)
	"Usage: %s [options...]\n\n\
	-g, --groups=GROUPS	send GROUPS groups of chars (default %d)\n\
				valid GROUPS values are between %d and %d\n\
	-n, --groupsize=GS	make groups GS chars (default %d)\n\
				valid GS values are between %d and %d\n\
	-c, --charset=CHARSET	select chars to send from this set\n\
				(default %s)\n\
	-h, --help		print this message\n\
	-v, --version		output version information and exit\n\n",
#endif
#if defined(SCO) || defined(UNIXWARE)
	"Usage: %s [options...]\n\n\
	-g GROUPS	send GROUPS groups of chars (default %d)\n\
			valid GROUPS values are between %d and %d\n\
	-n GS		make groups GS chars (default %d)\n\
			valid GS values are between %d and %d\n\
	-c CHARSET	select chars to send from this set\n\
			(default %s)\n\
	-h 		print this message\n\
	-v 		output version information and exit\n\n",
#endif
	pname,
	INIT_GROUPS, MIN_GROUPS, MAX_GROUPS,
	INIT_GROUPSIZE, MIN_GROUPSIZE, MAX_GROUPSIZE, DEF_CHARSET );
}


/*
 * parse_cmdline - parse command line options for initial values
 */
static void parse_cmdline( int argc, char **argv ) {
	int	c;				/* option character */
	char	*argv0;				/* program name */
	int	argind;				/* loop index */
	char	env_options[MAXOPTSTR];		/* env options string */
	char	*sptr;				/* string pointer */
	char	*local_argv[MAXARGS];		/* local argv array */
	int	local_argc = 0;			/* local argc */
#if defined(LINUX)
	int	option_index;			/* option index */
	static struct option long_options[] = {	/* options table */
		{ "groups",	1, 0, 0 },
		{ "groupsize",	1, 0, 0 },
		{ "charset",	1, 0, 0 },
		{ "help",	0, 0, 0 },
		{ "version",	0, 0, 0 },
		{ 0, 0, 0, 0 }};
#endif

	/* set argv0 to be the basename part of the program name */
	argv0 = argv[0] + strlen( argv[0] );
	while ( *argv0 != ASC_FNS && argv0 > argv[0] )
		argv0--;
	if ( *argv0 == ASC_FNS ) argv0++;

	/* build a new view of argc and argv by first prepending
	   the strings from ..._OPTIONS, if defined, then putting the
	   command line args on (so they take precedence) */
	local_argv[ local_argc++ ] = argv[0];
	if ( getenv( ENV_OPTIONS ) != NULL ) {
		strcpy( env_options, getenv( ENV_OPTIONS ));
		sptr = env_options;
		while ( local_argc < MAXARGS - 1 ) {
			for ( ; strchr( ARGS_WHITESPACE, *sptr ) != (char *)NULL
					&& *sptr != EOS;
					sptr++ );
			if ( *sptr == EOS )
				break;
			else {
				local_argv[ local_argc++ ] = sptr;
				for ( ; strchr( ARGS_WHITESPACE, *sptr )
						== (char *)NULL && *sptr != EOS;
						sptr++ );
				if ( strchr( ARGS_WHITESPACE, *sptr )
						!= (char *)NULL && *sptr != EOS ) {
					*sptr = EOS; sptr++;
				}
			}
		}
	}
	for ( argind = 1; argind < argc; argind++ ) {
		local_argv[ local_argc++ ] = argv[ argind ];
	}

	/* process every option */
#if defined(LINUX)
	while ( (c=getopt_long( local_argc, local_argv, "g:n:c:hv",
			long_options, &option_index )) != -1 ) {
#endif
#if defined(SCO) || defined(UNIXWARE)
	while ( (c=getopt( local_argc, local_argv, "g:n:c:hv" )) != -1 ) {
#endif

		/* check for -g or --groups argument */
		if ( c == 'g'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"groups" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
				) {
#endif
			if ( sscanf( optarg, "%d", &groups ) != 1 ||
					groups < MIN_GROUPS ||
					groups > MAX_GROUPS ) {
				fprintf( stderr, "%s: invalid groups value\n",
						argv0 );
				exit( 1 );
			}
		}

		/* check for -n or --groupsize argument */
		else if ( c == 'n'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"groupsize" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
				) {
#endif
			if ( sscanf( optarg, "%d", &groupsize ) != 1 ||
					groupsize < MIN_GROUPSIZE ||
					groupsize > MAX_GROUPSIZE ) {
				fprintf( stderr, "%s: invalid groupsize value\n",
						argv0 );
				exit( 1 );
			}
		}

		/* check for -c or --charset argument */
		else if ( c == 'c'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"charset" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
				) {
#endif
			charset = optarg;
		}

		/* check for -h or --help argument */
		else if ( c == 'h'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"help" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
				) {
#endif
			print_help( argv0 );
			exit( 0 );
		}

		/* check for -v or --version argument */
		else if ( c == 'v'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"version" )) {
#endif
#if defined(SCO) || defined(UNIXWARE)
					) {
#endif
			printf( "%s\n", VERSION );
			exit( 0 );
		}

		/* check for illegal option */
		else if ( c == '?' ) {
			fprintf( stderr,
#if defined(LINUX)
				"Try '%s --help' for more information.\n",
#endif
#if defined(SCO) || defined(UNIXWARE)
				"Try '%s -h' for more information.\n",
#endif
				argv0 );
			exit( 1 );
		}

		/* nothing else left to do */
		else {
			fprintf( stderr, "%s: getopts returned %c\n",
					argv0, c );
			exit( 1 );
		}

	}
	if ( optind != local_argc ) {
		fprintf( stderr,
#if defined(LINUX)
			"Try '%s --help' for more information.\n",
#endif
#if defined(SCO) || defined(UNIXWARE)
			"Try '%s -h' for more information.\n",
#endif
			argv0 );
		exit( 1 );
	}

}


/*
 * generate_chars - generate random characters on stdout
 */
static void generate_chars( int num_groups, int groupchars, char *set ) {
	int	gcount;			/* group counter */
	int	ccount;			/* chars in a group counter */

	/* randomize the rand() generator */
	srand( (int)time( NULL ));

	/* generate groups */
	for ( gcount = 0; gcount < num_groups; gcount++ ) {
		for ( ccount = 0; ccount < groupchars; ccount++ ) {

			/* pick a 'random' character from the set */
			printf( "%c", set[rand() % strlen( set )] );
			fflush( stdout );
		}

		/* send a group separator space */
		printf( " " );
		fflush( stdout );
	}
}


/*
 * main - parse the command line options, then generate the characters
 */
int main( int argc, char **argv ) {
	char	*argv0;			/* program name */

	/* parse the command line arguments */
	parse_cmdline( argc, argv );

	/* set argv0 to be the basename part of the program name */
	argv0 = argv[0] + strlen( argv[0] );
	while ( *argv0 != ASC_FNS && argv0 > argv[0] )
		argv0--;
	if ( *argv0 == ASC_FNS ) argv0++;

	/* randomize the rand() generator */
	srand( (int)time( NULL ));

	/* generate groups */
	generate_chars( groups, groupsize, charset );

	/* all done */
	printf( "\n" );
	exit( 0 );

	return( 0 ); /* for LINT */
}
