#include "cthugha.h"
#include "feature.h"
#include "display.h"
#include "sound.h"
#include "translate.h"
#include "action.h"
#include "waves.h"
#include "imath.h"

#include <stdio.h>
#include <string.h>


feature_entry feature_null = {NULL, 0, "none", ""};

int add_feature(feature * features, const feature_entry * entry) {
    features->entries = 
	cth_memory(features->entries,
		   (features->nr_entries+1) * sizeof(feature_entry),
		   "Can not get memory for feature entry.\n");

    memcpy(features->entries + features->nr_entries,  entry,
	   sizeof(feature_entry));

    features->nr_entries ++;
    return 0;
}


/*
 * Return a pointer to the new feature entry
 *
 * also sets the features->current entry.
 */
static feature_entry * feature_change(int to, feature * features) {
    int i;
    int nouse;

    /* check for empty set */
    if ( features->nr_entries == 0)
	return &feature_null;
	
    /* check, of there is any featureion we can use */
    nouse = 1;
    for(i=0; i < features->nr_entries; i++)
	if ( features->entries[i].use ) {
	    i = features->nr_entries;
	    nouse = 0;
	}

    /* if non in use, return first */
    if ( nouse)
	return features->entries;
	
    switch(to) {
    case CHANGE_RANDOM:
	do {
	    to = rand() % features->nr_entries;
	} while( ! features->entries[to].use );
	break;
    case CHANGE_NEXT:
	to = features->current;
	do {
	    to = (to + 1) % features->nr_entries;
	} while(features->entries[to].use == 0);
	break;
    case CHANGE_PREV:
	to = features->current;
	do {
	    to = (to > 0) ? to -1 : features->nr_entries - 1;
	} while(features->entries[to].use == 0);
	break;
    default:
	if ( to < 0)
	    to = 0;
	if ( to >= features->nr_entries) 
	    to = features->nr_entries -1;
	while( features->entries[to].use == 0) 
	    to = (to+1) % features->nr_entries;
    }
    features->current = to;

    display_start = gettime();	/* reset counter to speed test */
    display_frames = 0;

    printfv(10, "change %s to %s\n", features->name, features->entries[to].name);

    return features->entries + to;
}

/*
 * change featureion, allowing names
 */
static feature_entry * feature_change2(char * to, feature * features) {
    int nr;
    char * pos;

    /* do nothing if to is empty */
    if ( *to == '\0' ) {
	if ( features->current == CHANGE_RANDOM )
	    return feature_change(CHANGE_RANDOM, features);
	else
	    return features->entries + features->current;
    }
    /* first try to read as number */
    nr = strtol(to, &pos, 0);
    if ( pos == to ) {
	/* not a number, so try as name */
	for(nr = 0; nr < features->nr_entries; nr ++)
	    if(strcasecmp(to, features->entries[nr].name) == 0) {
		return feature_change(nr, features);
	    }
	/* found no entry, use a random value */
	printfv(2, "Unknown %s entry: %s\n", features->name, to);
	nr = CHANGE_RANDOM;
    }

    return feature_change(nr, features);
}

/*
 * check, if a given name is in the entries for feature
 */
int feature_defined(const char * name, const feature * features) {
    int i;
    for( i=0; i < features->nr_entries; i++)
	if ( strcmp(features->entries[i].name, name) == 0)
	    return 1;
    return 0;
}

/****************************************************************************/

/*
 * change a numerical value
 */
int val_change(int to, int min, int max, int old) {
	
    switch(to) {
    case CHANGE_RANDOM:
	if ( max < min)
	    return min;
	return rand() % (1+max-min) + min;
    case CHANGE_NEXT:
	return ( old >= max) ? min: old + 1; 
    case CHANGE_PREV:
	return ( old <= 0) ? max : old - 1;
    default:
	if ( to < min)		to = min;
	if ( to > max)		to = max;
	return to;
    }
}


typedef struct {
    int flame;
    int flame_general[5];
    int wave;
    int wave_scale;
    int screen;
    int palette;
    int translation;
    int table;
    int massage;
    int fft;
    int flashlight;
    int pcx;
    int object;
    int mirror_horiz;
    int mirror_vert;
    int rgb;
} cthstate;

cthstate save_state() {
    cthstate s;

    s.flame = flames.current;
    memcpy(s.flame_general, flame_general, 5*sizeof(int));
    s.wave = waves.current;
    s.wave_scale = wave_scale;
    s.screen = screens.current;
    s.palette = palettes.current;
    s.translation = translations.current;
    s.table = active_table;
    s.massage = sound_massage_style;
    s.fft = sound_FFT;
    s.flashlight = sound_flashlight;
    s.pcx = pcxs.current;
    s.object = objects.current;
    s.mirror_horiz = disp_mirror.x;
    s.mirror_vert = disp_mirror.y;
    s.rgb = flame_rgb;

    return s;
}

int restore_state(cthstate s) {
    change(f_flame, s.flame, 0);
    memcpy(flame_general, s.flame_general, 5*sizeof(int));
    change(f_wave, s.wave, 0);
    wave_scale = s.wave;
    change(f_screen, s.screen, 0);
    change(f_palette, s.palette, 0);
    change(f_table, s.table, 0);
    change(f_translation, s.translation, 0);
    change(f_massage, s.massage, 0);
    change(f_fft, s.fft, 0);
    change(f_flashlight, s.flashlight, 0);
    change(f_pcx, s.pcx, 0);
    change(f_object, s.object, 0);
    change(f_mirror_horiz, s.mirror_horiz, 0);
    change(f_mirror_vert, s.mirror_vert, 0);
    change(f_rgb, s.rgb, 0);

    return 0;
}


#define MAX_STATE 128
static cthstate states[MAX_STATE];
static int state_p = 0;

int new_state() {     
    /* save state */
    if(state_p >= MAX_STATE) {
	memcpy( states, states + 1, sizeof(cthstate) * (MAX_STATE-1));
	state_p --;
    }
    states[state_p ++] = save_state();

    return 0;
}
int back_state() {
    if( state_p > 0) {
	restore_state( states[--state_p] );
    }
    return 0;
}

/*
 * change one of the features of cthugha
 */
void change(cth_feature f, int to, int save) {
    int i;

    if(save) 
	new_state();

    switch(f) {
    case f_flame:
	feature_change(to, &flames);
	for(i=0; i < 3; i++)
	    feature_change(CHANGE_RANDOM, &(flames_rgb[i]));
	return; 
    case f_flame_general:
	change_flame_general();
	return; 
    case f_wave:
	feature_change(to, &waves);
	return;
    case f_screen:
	feature_change(to, &screens);
	display_clear = 1;
	return;
    case f_palette:
	feature_change(to, &palettes);
	cth_setpalette( *active_palette, 0);
	return;
    case f_table:
	change_table(to);
	return;
    case f_translation:
	feature_change(to, &translations);
	return;
    case f_massage:
	change_massage_style(to);
	return; 
    case f_fft:
	change_FFT(to);
	return;
    case f_flashlight:
	change_flashlight(to);
	return;
    case f_pcx:
	feature_change(to, &pcxs);
	return; 
    case f_object:
	feature_change(to, &objects);
	return;
    case f_mirror_horiz:
	change_mirror_horiz(to);
	return;
    case f_mirror_vert:
	change_mirror_vert(to);
    case f_rgb:
	change_rgb(to);
	set_rgb_palette();
    default:
	;
    }
}



/*
 * Set first options in use
 */
int init_startup() {
    int i;

    /* set screen-function */
    update_screen = feature_change2(screen_first, &screens)->data;

    /* make sure we have a palette */
    active_palette = feature_change(CHANGE_RANDOM, &palettes)->data;

    /* use first pcx if specified */
    if( pcx_first[0] != '\0' ) { 
        /* select first pcx-file */
        feature_change2(pcx_first, &pcxs);
        if ( active_pcx != NULL) {
	    show_pcx(1,0);		/* display the select pcx to active*/
	    memcpy(passive_buffer, active_buffer, BUFF_SIZE); 
	    active_palette =		/* and set corresponding palette */
  	        feature_change(pcx_palettes[pcxs.current],
			       &palettes)->data;
	}
    } else {
    	change(f_pcx, CHANGE_NEXT, 0);
    }

    /* set palette */
    if( palette_first[0] != '\0') {
	active_palette = feature_change2(palette_first, &palettes)->data;
    } 

    /* set the features to start with */
    feature_change2(wave_first, &waves);
    feature_change2(object_first, &objects);
    feature_change2(translate_first, &translations);
    feature_change2(flame_first, &flames);
    for(i=0; i < 3; i++) {
	feature_change2(flame_first, &flames_rgb[i]);
    }
    change_table(table_first);
    change_massage_style(massage_first);

    return 0;
}

/*
 * save startup-section if --save was given
 */
int exit_startup() {
    if ( options_save) {
	write_ini();
	save_buffer();
    }
    return 0;
}
