/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB

   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.

   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.	The License grants you the right to
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* A lexical scanner on a temporary buffer with a yacc interface */

#include "mysql_priv.h"
#include "sql_lex.h"
#include <m_ctype.h>
#include <hash.h>

/* Macros to look like lex */

#define yyGet()		*(lex->ptr++)
#define yyGetLast()	lex->ptr[-1]
#define yyPeek()	lex->ptr[0]
#define yyUnget()	lex->ptr--
#define yySkip()	lex->ptr++

pthread_key(LEX*,THR_LEX);


#define TOCK_NAME_LENGTH 20

typedef struct st_symbol {
  char	*name;
  int	tok;
  uchar length;
} SYMBOL;

// Symbols are breaked in to separated arrays to allow fieldnames with
// same name as functions
// Theese are kept sorted for human lookup (the symbols are hashed)

static SYMBOL symbols[] = {
  { "&&",	AND,0},
  { "<",	LT,0},
  { "<=",	LE,0},
  { "<>",	NE,0},
  { "!=",	NE,0},
  { "=",	EQ,0},
  { ">",	GT_SYM,0},
  { ">=",	GE,0},
  { "action",	ACTION,0},
  { "add",	ADD,0},
  { "all",	ALL,0},
  { "alter",	ALTER,0},
  { "and",	AND,0},
  { "as",	AS,0},
  { "asc",	ASC,0},
  { "auto_increment",AUTO_INC,0},
  { "between",	BETWEEN_SYM,0},
  { "bigint",	BIGINT,0},
  { "bit",	BIT_SYM,0},
  { "binary",	BINARY,0},
  { "blob",	BLOB_SYM,0},
  { "both",	BOTH,0},
  { "by",	BY,0},
  { "cascade",	CASCADE,0},
  { "char",	CHAR_SYM,0},
  { "character",CHAR_SYM,0},
  { "change",	CHANGE,0},
  { "check",	CHECK_SYM,0},
  { "column",	COLUMN_SYM,0},
  { "columns",	COLUMNS,0},
  { "create",	CREATE,0},
  { "cross",	CROSS,0},
  { "current_date", CURDATE,0},
  { "current_time", CURTIME,0},
  { "current_timestamp", NOW_SYM,0},
  { "data",	DATA_SYM,0},
  { "database", DATABASE,0},
  { "databases",DATABASES,0},
  { "date",	DATE_SYM,0},
  { "datetime", DATETIME,0},
  { "day",	DAY_SYM,0},
  { "day_hour", DAY_HOUR_SYM,0},
  { "day_minute",	DAY_MINUTE_SYM,0},
  { "day_second",	DAY_SECOND_SYM,0},
  { "dayofmonth", DAY_OF_MONTH,0},
  { "dayofweek", DAY_OF_WEEK,0},
  { "dayofyear", DAY_OF_YEAR,0},
  { "dec",	DECIMAL_SYM,0},
  { "decimal",	DECIMAL_SYM,0},
  { "default",	DEFAULT,0},
  { "delete",	DELETE_SYM,0},
  { "desc",	DESC,0},
  { "describe", DESCRIBE,0},
  { "distinct", DISTINCT,0},
  { "double",	DOUBLE_SYM,0},
  { "drop",	DROP,0},
  { "escaped",	ESCAPED,0},
  { "enclosed", ENCLOSED,0},
  { "enum",	ENUM,0},
  { "explain",	DESCRIBE,0},
  { "fields",	COLUMNS,0},
  { "float",	FLOAT_SYM,0},
  { "float4",	FLOAT_SYM,0},
  { "float8",	DOUBLE_SYM,0},
  { "foreign",	FOREIGN,0},
  { "from",	FROM,0},
  { "for",	FOR_SYM,0},
  { "full",	FULL,0},
  { "function", UDF_SYM,0},
  { "grant",	GRANT,0},
  { "group",	GROUP,0},
  { "having",	HAVING,0},
  { "hour",	HOUR_SYM,0},
  { "hour_minute",	HOUR_MINUTE_SYM,0},
  { "hour_second",	HOUR_SECOND_SYM,0},
  { "ignore",	IGNORE_SYM,0},
  { "in",	IN_SYM,0},
  { "index",	INDEX,0},
  { "infile",	INFILE,0},
  { "insert",	INSERT,0},
  { "int",	INT_SYM,0},
  { "integer",	INT_SYM,0},
  { "interval", INTERVAL_SYM,0},
  { "int1",	TINYINT,0},
  { "int2",	SMALLINT,0},
  { "int3",	MEDIUMINT,0},
  { "int4",	INT_SYM,0},
  { "int8",	BIGINT,0},
  { "into",	INTO,0},
  { "is",	IS,0},
  { "join",	JOIN,0},
  { "key",	KEY_SYM,0},
  { "keys",	KEYS,0},
  { "last_insert_id", LAST_INSERT_ID,0 },
  { "leading",	LEADING,0},
  { "left",	LEFT,0},
  { "like",	LIKE,0},
  { "lines",	LINES,0},
  { "limit",	LIMIT,0},
  { "lock",	LOCK_SYM,0},
  { "load",	LOAD,0},
  { "long",	LONG_SYM,0},
  { "longblob", LONGBLOB,0},
  { "longtext", LONGTEXT,0},
  { "match",	MATCH,0},
  { "mediumblob",MEDIUMBLOB,0},
  { "mediumtext",MEDIUMTEXT,0},
  { "mediumint",MEDIUMINT,0},
  { "middleint",MEDIUMINT,0},			// For powerbuilder
  { "minute",	MINUTE_SYM,0},
  { "minute_second",	MINUTE_SECOND_SYM,0},
  { "month",	MONTH_SYM,0},
  { "monthname",MONTH_NAME_SYM,0},
  { "natural",	NATURAL,0},
  { "numeric",	NUMERIC_SYM,0},
  { "no",	NO_SYM,0},
  { "not",	NOT,0},
  { "null",	NULL_SYM,0},
  { "on",	ON,0},
  { "option",	OPTION,0},
  { "optionally",OPTIONALLY,0},
  { "or",	OR,0},
  { "order",	ORDER_SYM,0},
  { "outer",	OUTER,0},
  { "outfile",	OUTFILE,0},
  { "partial",	PARTIAL,0},
  { "precision", PRECISION,0},
  { "primary",	PRIMARY_SYM,0},
  { "procedure", PROCEDURE,0},
  { "privileges", PRIVILEGES,0},
  { "quarter",	QUARTER,0},
  { "read",	READ_SYM,0},
  { "real",	REAL,0},
  { "references",REFERENCES,0},
  { "rename",	RENAME,0},
  { "regexp",	REGEXP,0 },
  { "reverse",	REVERSE,0},
  { "repeat",	REPEAT,0 },
  { "replace",	REPLACE,0},
  { "restrict", RESTRICT,0},
  { "returns",	UDF_RETURNS_SYM,0},
  { "rlike",	REGEXP,0 },			// Like in mSQL2
  { "second",	SECOND_SYM,0 },
  { "select",	SELECT_SYM,0},
  { "set",	SET,0},
  { "show",	SHOW,0},
  { "smallint", SMALLINT,0},
  { "soname",	UDF_SONAME_SYM,0},
  { "sql_big_tables",	SQL_BIG_TABLES,0},
  { "sql_big_selects",	SQL_BIG_SELECTS,0},
  { "sql_select_limit", SQL_SELECT_LIMIT,0},
  { "sql_log_off",	SQL_LOG_OFF,0},
  { "straight_join",  STRAIGHT_JOIN,0},
  { "starting", STARTING,0},
  { "status",	STATUS_SYM,0},
  { "string",	STRING_SYM,0},
  { "table",	TABLE_SYM,0},
  { "tables",	TABLES,0},
  { "terminated",TERMINATED,0},
  { "text",	TEXT_SYM,0},
  { "time",	TIME_SYM,0},
  { "timestamp",TIMESTAMP,0},
  { "tinyblob", TINYBLOB,0},
  { "tinytext", TINYTEXT,0},
  { "tinyint",	TINYINT,0},
  { "trailing", TRAILING,0},
  { "to",	TO_SYM,0},
  { "use",	USE_SYM,0},
  { "using",	USING,0},
  { "unique",	UNIQUE_SYM,0},
  { "unlock",	UNLOCK_SYM,0},
  { "unsigned", UNSIGNED,0},
  { "update",	UPDATE_SYM,0},
  { "usage",	USAGE,0},
  { "values",	VALUES,0},
  { "varchar",	VARCHAR,0},
  { "variables",VARIABLES,0},
  { "varying",	VARYING,0},
  { "varbinary",VARBINARY,0},
  { "with",	WITH,0},
  { "write",	WRITE_SYM,0},
  { "where",	WHERE,0},
  { "year",	YEAR_SYM,0},
  { "year_month",	YEAR_MONTH_SYM,0},
  { "zerofill", ZEROFILL,0},
  { "||",	OR,0},
};


static SYMBOL sql_functions[] = {
  { "abs",	ABS,0},
  { "acos",	ACOS,0},
  { "ascii",	ASCII,0},
  { "asin",	ASIN,0},
  { "atan",	ATAN,0},
  { "atan2",	ATAN,0},
  { "avg",	AVG_SUM,0},
  { "bit_count",BIT_COUNT,0},
  { "bit_or",	BIT_OR,0},
  { "bit_and",	BIT_AND,0},
  { "ceiling",	CEILING,0},
  { "char_length", LENGTH,0},
  { "character_length",LENGTH,0},
  { "concat",	CONCAT,0},
  { "count",	COUNT_SUM,0},
  { "cos",	COS,0},
  { "cot",	COT,0},
  { "curdate",	CURDATE,0},
  { "curtime",	CURTIME,0},
  { "date_add_mm", DATE_ADD_MM,0},
  { "date_add_interval", DATE_ADD_INTERVAL,0},
  { "date_format", DATE_FORMAT_SYM,0},
  { "dayname",	DAYNAME,0},
  { "degrees",	DEGREES,0},
  { "elt",	ELT_FUNC,0},
  { "encrypt",	ENCRYPT},
  { "exp",	EXP,0},
  { "field",	FIELD_FUNC,0},			// For compability
  { "find_in_set", FIND_IN_SET,0},
  { "floor",	FLOOR,0},
  { "format",	FORMAT,0},
  { "from_days",FROM_DAYS,0},			// Convert string or number
  { "from_unixtime",FROM_UNIXTIME,0},		// Convert string or number
  { "get_lock", GET_LOCK,0},
  { "group_unique_users", GROUP_UNIQUE_USERS,0},
  { "if",	IF,0},
  { "ifnull",	IFNULL,0},
  { "insert",	INSERT,0},
  { "instr",	INSTR,0},			// unireg function
  { "isnull",	ISNULL,0},
  { "lcase",	LCASE,0},
  { "lower",	LCASE,0},
  { "length",	LENGTH,0},
  { "locate",	LOCATE,0},
  { "log",	LOG,0},
  { "log10",	LOG10,0},
  { "ltrim",	LTRIM,0},
  { "now",	NOW_SYM,0},
  { "max",	MAX_SUM,0},
  { "mid",	SUBSTRING,0},			// unireg function
  { "min",	MIN_SUM,0},
  { "mod",	MOD_SYM,0},
  { "octet_length",LENGTH,0},
  { "password", PASSWORD,0},
  { "period_add",  PERIOD_ADD,0},
  { "period_diff", PERIOD_DIFF,0},
  { "pi",	PI_SYM,0},
  { "position", POSITION,0},
  { "pow",	POW,0},
  { "power",	POW,0},
  { "rand",	RAND,0},
  { "radians",	RADIANS,0},
  { "release_lock", RELEASE_LOCK,0},
  { "repeat",	REPEAT,0},
  { "replace",	REPLACE,0},
  { "right",	RIGHT,0},
  { "round",	ROUND,0},
  { "rtrim",	RTRIM,0},
  { "sec_to_time",  SEC_TO_TIME,0},
  { "session_user", USER,0},
  { "sign",	SIGN,0},
  { "sin",	SIN,0},
  { "space",	SPACE,0},
  { "sqrt",	SQRT,0},
  { "sum",	SUM_SUM,0},
  { "std",	STD_SUM,0},
  { "stddev",	STD_SUM,0},
  { "strcmp",	STRCMP,0},
  { "soundex",	SOUNDEX,0},
  { "substring",SUBSTRING,0},
  { "substring_index",	SUBSTRING_INDEX,0},
  { "sysdate",	NOW_SYM,0},
  { "system_user", USER,0},
  { "tan",	TAN,0},
  { "time_format", TIME_FORMAT_SYM,0},
  { "time_to_sec", TIME_TO_SEC,0},
  { "to_days",	TO_DAYS,0},			// Convert string or number
  { "trim",	TRIM,0},
  { "truncate", TRUNCATE,0},
  { "ucase",	UCASE,0},
  { "upper",	UCASE,0},
  { "unique_users", UNIQUE_USERS,0},
  { "unix_timestamp", UNIX_TIMESTAMP,0},
  { "user",	USER,0},
  { "version",	VERSION_SYM,0},
  { "week",	WEEK_SYM,0},
  { "weekday",	WEEKDAY,0},
};


static HASH sym_hash,fun_hash;
static uchar state_map[256];

static byte* get_hash_key(const byte *buff,uint *length,my_bool not_used)
{
  SYMBOL *symbol=(SYMBOL*) buff;
  *length=(uint) symbol->length;
  return (byte*) symbol->name;
}

void lex_init(void)
{
  uint i;
  if (hash_init(&sym_hash,array_elements(symbols),0,0,get_hash_key,
		NULL,HASH_CASE_INSENSITIVE) ||
      hash_init(&fun_hash,array_elements(sql_functions),0,0,
		get_hash_key,NULL, HASH_CASE_INSENSITIVE))
    exit(1); /* purecov: inspected */

  for (i=0 ; i < array_elements(symbols) ; i++)
  {
    symbols[i].length=(uchar) strlen(symbols[i].name);
    if (hash_insert(&sym_hash,(byte*) (symbols+i)) ||
	hash_insert(&fun_hash,(byte*) (symbols+i)))
      exit(1); /* purecov: inspected */
  }
  for (i=0 ; i < array_elements(sql_functions) ; i++)
  {
    sql_functions[i].length=(uchar) strlen(sql_functions[i].name);
    if (hash_insert(&fun_hash,(byte*) (sql_functions+i)))
      exit(1); /* purecov: inspected */
  }
#ifndef DBUG_OFF
  hash_check(&fun_hash);			// Get statistics
  hash_check(&sym_hash);
#endif
  VOID(pthread_key_create(&THR_LEX,NULL));

  /* Fill state_map with states to get a faster parser */
  for (i=0; i < 256 ; i++)
  {
    if (isalpha(i))
      state_map[i]=(uchar) STATE_IDENT;
    else if (isdigit(i))
      state_map[i]=(uchar) STATE_NUMBER_IDENT;
#if defined(USE_MB) && defined(USE_MB_IDENT)
    else if (ismbhead(i))
      state_map[i]=(uchar) STATE_IDENT;
#endif
    else
      state_map[i]=(uchar) STATE_CHAR;
  }
  state_map['_']=(uchar) STATE_IDENT;
  state_map['\'']=state_map['"']=(uchar) STATE_STRING;
  state_map['-']=state_map['+']=(uchar) STATE_SIGNED_NUMBER;
  state_map['.']=(uchar) STATE_REAL_OR_POINT;
  state_map['<']=state_map['>']=state_map['=']=state_map['!']=
    (uchar) STATE_CMP_OP;
  state_map['&']=state_map['|']=(uchar) STATE_BOOL;
  state_map['#']=(uchar) STATE_COMMENT;
  state_map[0]=(uchar) STATE_EOL;
  state_map['\\']= (uchar) STATE_ESCAPE;
  state_map['/']= (uchar) STATE_LONG_COMMENT;
}


void lex_free(void)
{					// Call this when deamon ends
  DBUG_ENTER("lex_free");
  hash_free(&sym_hash);
  hash_free(&fun_hash);
  DBUG_VOID_RETURN;
}


LEX *lex_start(uchar *buf,uint length)
{
  LEX *lex;
  if ((lex=(LEX*) sql_alloc(sizeof(*lex))))	// Doesn't have to be freed
  {
    my_pthread_setspecific_ptr(THR_LEX,lex);
    lex->next_state=STATE_START;
    lex->end_of_query=(lex->ptr=buf)+length;
    lex->yylineno = 1;
    lex->create_refs=0;
    lex->length=0;
    lex->expr_list.empty();
    lex->convert_set=(lex->thd=current_thd)->convert_set;
    lex->yacc_yyss=lex->yacc_yyvs=0;
  }
  return lex;
}

void lex_end(LEX *lex)
{
  lex->expr_list.delete_elements();	// If error when parsing sql-varargs
  x_free(lex->yacc_yyss);
  x_free(lex->yacc_yyvs);
}


static int find_keyword(LEX *lex,bool function)
{
  uchar *tok=lex->tok_start;
  uint len=(uint) (lex->ptr - tok);

  if (len <= TOCK_NAME_LENGTH)
  {
    HASH *hash_table= function ? &fun_hash : &sym_hash;
    SYMBOL *symbol;
    if ((symbol=(SYMBOL*) hash_search(hash_table,(byte*) tok,len)))
    {
      lex->yylval->lex_str.str= (char*) tok;
      lex->yylval->lex_str.length=len;
      return symbol->tok;
    }
  }
#ifdef HAVE_DLOPEN
  udf_func *udf;
  if (function && (udf=find_udf((char*) tok, len)))
  {
    switch (udf->returns) {
    case STRING_RESULT:
      lex->yylval->udf=udf;
      return UDF_CHAR_FUNC;
    case REAL_RESULT:
      lex->yylval->udf=udf;
      return UDF_FLOAT_FUNC;
    case INT_RESULT:
      lex->yylval->udf=udf;
      return UDF_INT_FUNC;
    }
  }
#endif
  return 0;
}


/* make a copy of token before ptr and set yytoklen */

static inline LEX_STRING get_token(LEX *lex)
{
  LEX_STRING tmp;
  yyUnget();			// ptr points now after last token char
  tmp.length=lex->yytoklen=(uint) (lex->ptr - lex->tok_start);
  tmp.str=(char*) sql_strmake((char*) lex->tok_start,lex->yytoklen);
  return tmp;
}

/* Return an unescaped text literal without quotes */
/* Should maybe expand '\n' ? */
/* Fix sometimes to do only one scan of the string */

static char *get_text(LEX *lex)
{
  reg1 uchar c,sep;
  uint found_escape=0;

  sep= yyGetLast();			// String should end with this
  //lex->tok_start=lex->ptr-1;		// Remember '
  while (lex->ptr != lex->end_of_query)
  {
    c = yyGet();
#ifdef USE_BIG5CODE
    if (lex->ptr != lex->end_of_query && isbig5head(c))
    {
      reg1 uchar d = yyGet();
      if (lex->ptr != lex->end_of_query && isbig5tail(d))
	continue;
      else
	yyUnget();
    }
#endif
#ifdef USE_MB
    int l;
    if ((l = ismbchar(lex->ptr-1, lex->end_of_query))) {
	lex->ptr += l-1;
	continue;
    }
#endif
    if (c == '\\')
    {					// Escaped character
      found_escape=1;
      if (lex->ptr == lex->end_of_query)
	return 0;
      yySkip();
    }
    else if (c == sep)
    {
      if (c == yyGet())			// Check if two separators in a row
      {
	found_escape=1;			// dupplicate. Remember for delete
	continue;
      }
      else
	yyUnget();

      /* Found end. Unescape and return string */
      uchar *str,*end,*start;

      str=lex->tok_start+1;
      end=lex->ptr-1;
      start=(uchar*) sql_alloc((uint) (end-str)+1);
      if (!found_escape)
      {
	lex->yytoklen=(uint) (end-str);
	memcpy(start,str,lex->yytoklen);
	start[lex->yytoklen]=0;
      }
      else
      {
	uchar *to;
	for (to=start ; str != end ; str++)
	{
#ifdef USE_BIG5CODE
	  if (str[1] && isbig5code(*str,*(str+1)))
	  {
	    *to++= *str;
	    *to++= *++str;
	    continue;
	  }
#endif
#ifdef USE_MB
	  int l;
	  if ((l = ismbchar(str, end))) {
	      while (l--)
		  *to++ = *str++;
	      str--;
	      continue;
	  }
#endif
	  if (*str == '\\' && str+1 != end)
	  {
	    switch(*++str) {
	    case 'n':
	      *to++='\n';
	      break;
	    case 't':
	      *to++= '\t';
	      break;
	    case 'r':
	      *to++ = '\r';
	      break;
	    case 'b':
	      *to++ = '\b';
	      break;
	    case '0':
	      *to++= 0;			// Ascii null
	      break;
	    case '_':
	    case '%':
	      *to++= '\\';		// remember prefix for wildcard
	      /* Fall through */
	    default:
	      *to++ = *str;
	      break;
	    }
	  }
	  else if (*str == sep)
	    *to++= *str++;		// Two ' or "
	  else
	    *to++ = *str;

	}
	*to=0;
	lex->yytoklen=(uint) (to-start);
      }
      if (lex->convert_set)
	lex->convert_set->convert((char*) start,lex->yytoklen);
      return (char*) start;
    }
  }
  return 0;					// unexpected end of query
}


/*
** Calc type of integer; long integer, longlong integer or real.
** Returns smallest type that match the string.
** When using unsigned long long values the result is converted to a real
** because else they will be unexpected sign changes because all calculation
** is done with longlong or double.
*/

static char *long_str="2147483647";
static const uint long_len=10;
static char *signed_long_str="-2147483648";
static char *longlong_str="9223372036854775807";
static const uint longlong_len=19;
static char *signed_longlong_str="-9223372036854775808";
static const uint signed_longlong_len=19;


inline static uint int_token(const char *str,uint length)
{
  if (length < long_len)			// quick normal case
    return NUM;
  bool neg=0;

  if (*str == '+')				// Remove sign and pre-zeros
  {
    str++; length--;
  }
  else if (*str == '-')
  {
    str++; length--;
    neg=1;
  }
  while (*str == '0' && length)
  {
    str++; length --;
  }
  if (length < long_len)
    return NUM;

  uint smaller,bigger;
  char *cmp;
  if (neg)
  {
    if (length == long_len)
    {
      cmp= signed_long_str+1;
      smaller=NUM;				// If <= signed_long_str
      bigger=LONG_NUM;				// If >= signed_long_str
    }
    else if (length < signed_longlong_len)
      return LONG_NUM;
    else if (length > signed_longlong_len)
      return REAL_NUM;
    else
    {
      cmp=signed_longlong_str+1;
      smaller=LONG_NUM;				// If <= signed_longlong_str
      bigger=REAL_NUM;
    }
  }
  else
  {
    if (length == long_len)
    {
      cmp= long_str;
      smaller=NUM;
      bigger=LONG_NUM;
    }
    else if (length < longlong_len)
      return LONG_NUM;
    else if (length > longlong_len)
      return REAL_NUM;
    else
    {
      cmp=longlong_str;
      smaller=LONG_NUM;
      bigger=REAL_NUM;
    }
  }
  while (*cmp && *cmp++ == *str++) ;
  return ((uchar) str[-1] <= (uchar) cmp[-1]) ? smaller : bigger;
}


// yylex remember the following states from the following yylex()
// STATE_EOQ ; found end of query
// STATE_OPERATOR_OR_IDENT ; last state was an ident, text or number
// 			     (which can't be followed by a signed number)

int yylex(void *arg)
{
  reg1	uchar	c=0;
  int	tokval;
  enum lex_states state,prev_state;
  LEX	*lex=current_lex;
  YYSTYPE *yylval=(YYSTYPE*) arg;

  lex->yylval=yylval;			// The global state
  lex->tok_start=lex->tok_end=lex->ptr;
  prev_state=state=lex->next_state; lex->next_state=STATE_OPERATOR_OR_IDENT;
  for (;;)
  {
    switch(state) {
    case STATE_OPERATOR_OR_IDENT:	// Next is operator or keyword
    case STATE_START:			// Start of token
      // Skipp startspace
      for (c=yyGet() ; (c && !isgraph(c)) ; c= yyGet())
      {
	if (c == '\n')
	  lex->yylineno++;
      }
      lex->tok_start=lex->ptr-1;	// Start of real token
      state= (enum lex_states) state_map[c];
      break;
    case STATE_ESCAPE:
      if (yyGet() == 'N')
      {					// Allow \N as shortcut for NULL
	yylval->lex_str.str="\\N";
	yylval->lex_str.length=2;
	return NULL_SYM;
      }
    case STATE_CHAR:			// Unknown or single char token
      yylval->lex_str.str=(char*) (lex->ptr=lex->tok_start);// Set to first char
      yylval->lex_str.length=1;
      c=yyGet();
      if (c != ')')
	lex->next_state= STATE_START;	// Allow signed numbers
      if (c == ',')
	lex->tok_start=lex->ptr;	// Let tok_start point at next item
      return((int) c);

    case STATE_IDENT:			// Incomplete keyword or ident
#if defined(USE_MB) && defined(USE_MB_IDENT)
      if (ismbhead(yyGetLast())) {
	  int l = ismbchar(lex->ptr-1, lex->end_of_query);
	  if (l == 0) {
	      state = STATE_CHAR;
	      continue;
	  }
	  lex->ptr += l - 1;
      }
      while (isalnum(c=yyGet()) || c == '_' || ismbhead(c)) {
	  if (ismbhead(c)) {
	      int l;
	      if ((l = ismbchar(lex->ptr-1, lex->end_of_query)) == 0)
		  break;
	      lex->ptr += l-1;
	  }
      }
#else
      while (isalnum(c=yyGet()) || c == '_') ;
#endif
      if (c == '.' && isalpha(yyPeek()))
	lex->next_state=STATE_IDENT_SEP;
      else
      {					// '(' must follow directly if function
	yyUnget();
	if ((tokval = find_keyword(lex,c == '(')))
	{
	  lex->next_state= STATE_START;	// Allow signed numbers
	  return(tokval);		// Was keyword
	}
	yySkip();			// next state does a unget
      }
      state = STATE_FOUND_IDENT;	// Found compleat ident
      break;

    case STATE_IDENT_SEP:		// Found ident and now '.'
      lex->next_state=STATE_IDENT_START;// Next is an ident (not a keyword)
      yylval->lex_str.str=(char*) lex->ptr;
      yylval->lex_str.length=1;
      c=yyGet();			// should be '.'
      return((int) c);

    case STATE_NUMBER_IDENT:		// number or ident which starts with num
      while (isdigit((c = yyGet()))) ;
      if (!isalpha(c) && c != '_')
      {					// Can't be identifier
	state=STATE_INT_OR_REAL;
	break;
      }
      if (c == 'e' || c == 'E')
      {
	if ((c=(yyGet())) == '+' || c == '-')
	{				// Allow 1E+10
	  if (isdigit(yyPeek()))	// Number must have digit after sign
	  {
	    yySkip();
	    while (isdigit(yyGet())) ;
	    yylval->lex_str=get_token(lex);
	    return(REAL_NUM);
	  }
	}
	yyUnget(); /* purecov: inspected */
      }
      else if (c == 'x' && (lex->ptr - lex->tok_start) == 2 &&
	  lex->tok_start[0] == '0' )
      {						// Varbinary
	while (isxdigit((c = yyGet()))) ;
	if ((lex->ptr - lex->tok_start) >= 4)
	{
	  yylval->lex_str=get_token(lex);
	  yylval->lex_str.str+=2;		// Skipp 0x
	  yylval->lex_str.length-=2;
	  lex->yytoklen-=2;
	  return (HEX_NUM);
	}
	yyUnget();
      }
      // fall through
    case STATE_IDENT_START:		// Incomplete ident
#if defined(USE_MB) && defined(USE_MB_IDENT)
      if (ismbhead(yyGetLast())) {
	  int l = ismbchar(lex->ptr-1, lex->end_of_query);
	  if (l == 0) {
	      state = STATE_CHAR;
	      continue;
	  }
	  lex->ptr += l - 1;
      }
      while (isalnum((c = yyGet())) || c == '_' || ismbhead(c)) {
	  if (ismbhead(c)) {
	      int l;
	      if ((l = ismbchar(lex->ptr-1, lex->end_of_query)) == 0)
		  break;
	      lex->ptr += l-1;
	  }
      }
#else
      while (isalnum((c = yyGet())) || c == '_') ;
#endif
      if (c == '.' && isalpha(yyPeek()))
	lex->next_state=STATE_IDENT_SEP;// Next is '.'
      // fall through

    case STATE_FOUND_IDENT:		// Complete ident
      yylval->lex_str=get_token(lex);
      return(IDENT);

    case STATE_SIGNED_NUMBER:		// Incomplete signed number
      if (prev_state == STATE_OPERATOR_OR_IDENT)
      {
	state= STATE_CHAR;		// Must be operator
	break;
      }
      if (!isdigit(c=yyGet()))
      {
	if (c != '.')
	{
	  state = STATE_CHAR;		// Return sign as single char
	  break;
	}
	yyUnget();			// Fix for next loop
      }
      while (isdigit(c=yyGet())) ;	// Incomplete real or int number
      // fall through
    case STATE_INT_OR_REAL:		// Compleat int or incompleat real
      if (c != '.')
      {					// Found complete integer number.
	yylval->lex_str=get_token(lex);
	return int_token(yylval->lex_str.str,yylval->lex_str.length);
      }
      // fall through
    case STATE_REAL:			// Incomplete real number
      while (isdigit(c = yyGet())) ;

      if (c == 'e' || c == 'E')
      {
	c = yyGet();
	if (c != '-' && c != '+')
	{				// No exp sig found
	  state= STATE_CHAR;
	  break;
	}
	if (!isdigit(yyGet()))
	{				// No digit after sign
	  state= STATE_CHAR;
	  break;
	}
	while (isdigit(yyGet())) ;
      }
      yylval->lex_str=get_token(lex);
      return(REAL_NUM);

    case STATE_CMP_OP:			// Incomplete comparison operator
      c=yyGet();			// May be 2 long
      if (state_map[c] != STATE_CMP_OP)
	yyUnget();
      if ((tokval = find_keyword(lex,0))) // Should be comparison operator
      {
	lex->next_state= STATE_START;	// Allow signed numbers
	return(tokval);
      }
      state = STATE_CHAR;		// Something fishy found
      break;

    case STATE_BOOL:
      if (c != yyPeek())
      {
	state=STATE_CHAR;
	break;
      }
      (void) yyGet();
      tokval = find_keyword(lex,0);	// Is a bool operator
      lex->next_state= STATE_START;	// Allow signed numbers
      return(tokval);

    case STATE_STRING:			// Incomplete text string
      if (!(yylval->lex_str.str = get_text(lex)))
      {
	state= STATE_CHAR;		// Read char by char
	break;
      }
      yylval->lex_str.length=lex->yytoklen;
      return(TEXT_STRING);

    case STATE_COMMENT:			//  Comment
      while ((c = yyGet()) != '\n' && c) ;
      yyUnget();			// Safety against eof
      state = STATE_START;		// Try again
      break;
    case STATE_LONG_COMMENT:		/* Long C comment? */
      if (yyPeek() != '*')
      {
	state=STATE_CHAR;		// Probable division
	break;
      }
      yyGet();
      while (lex->ptr != lex->end_of_query &&
	     ((c=yyGet()) != '*' || yyPeek() != '/'))
      {
	if (c == '\n')
	  lex->yylineno++;
      }
      if (lex->ptr != lex->end_of_query)
	yyGet();			// remove last '/'
      state = STATE_START;		// Try again
      break;
    case STATE_EOL:
      lex->next_state=STATE_END;	// Mark for next loop
      return(END_OF_INPUT);
    case STATE_END:
      lex->next_state=STATE_END;
      return(0);			// We found end of input last time

      // Actually real shouldn't start
      // with . but allow them anyhow
    case STATE_REAL_OR_POINT:
      if (isdigit(yyPeek()))
	state = STATE_REAL;		// Real
      else
      {
	state = STATE_CHAR;		// return '.'
	lex->next_state=STATE_IDENT_START;// Next is an ident (not a keyword)
      }
      break;
    }
  }
}
