/*
 *  WindowMaker window manager
 * 
 *  Copyright (c) 1997 Alfredo K. Kojima
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef WMWINDOW_H_
#define WMWINDOW_H_

#include <wraster.h>

#include "wconfig.h"
#include "GNUstep.h"
#include "texture.h"
#include "menu.h"
#include "application.h"

/* not defined in my X11 headers */
#ifndef UrgencyHint
#define UrgencyHint (1L << 8)
#endif


#define BORDER_TOP	1
#define BORDER_BOTTOM	2
#define BORDER_LEFT	4
#define BORDER_RIGHT	8
#define BORDER_ALL	(1|2|4|8)


#define CLIENT_EVENTS (StructureNotifyMask | PropertyChangeMask\
	| EnterWindowMask | LeaveWindowMask | ColormapChangeMask \
	| FocusChangeMask)

typedef enum {
    WFM_PASSIVE, WFM_NO_INPUT, WFM_LOCALLY_ACTIVE, WFM_GLOBALLY_ACTIVE
} FocusMode;


/*
 * window attribute flags.
 */
typedef struct {
    /* OpenStep */
    unsigned int titlebar:1;	       /* draw titlebar? */
    unsigned int resizable:1;	       /* draw the bottom handle? */
    unsigned int closable:1;	       /* draw a close button? */
    unsigned int miniaturizable:1;     /* draw an iconify button? */
    
    unsigned int broken_close:1;       /* is the close button broken? */

    /* ours */
    unsigned int shadeable:1;
    unsigned int omnipresent:1;
    unsigned int skip_window_list:1;
    unsigned int floating:1;	       /* put in NSFloatingWindowLevel */
    unsigned int bind_keys:1;	       /* intercept wm kbd binds 
					* while window is focused */
    unsigned int bind_mouse:1;         /* intercept mouse events
					* on client area while window
					* is focused */
    unsigned int hide_others:1;	       /* hide window when doing hideothers */
    unsigned int application:1;	       /* make app icon */
    
    unsigned int dont_move_off:1;

    unsigned int focusable:1;
    
    unsigned int always_user_icon:1;   /* ignore client IconPixmap or
					* IconWindow */

} WWindowAttributes;


/*
 * Window manager protocols that both the client as we understand.
 */
typedef struct {
    unsigned int TAKE_FOCUS:1;
    unsigned int SAVE_YOURSELF:1;
    unsigned int DELETE_WINDOW:1;

    /* WindowMaker specific */
    unsigned int MINIATURIZE_WINDOW:1;
} WProtocols;


/*
 * Stores client window information. Each client window has it's
 * structure created when it's being first mapped.
 */
typedef struct WWindow {
    struct WWindow *prev;	       /* window focus list */
    struct WWindow *next;

    WScreen *screen_ptr;	       /* pointer to the screen structure */
    WWindowAttributes window_flags;    /* window attribute flags */
    struct InspectorPanel *inspector;  /*  pointer to attribute editor panel */

#ifdef PAGER
    struct WPagerView *pager_view;
#endif
    
    struct WFrameWindow *frame;	       /* the frame window */
    int frame_x, frame_y;	       /* position of the frame in root*/

    struct {
	int x, y;
	unsigned int width, height;    /* original geometry of the window */
    } old_geometry;		       /* (before things like maximize) */

    /* client window info */
    short old_border_width;	       /* original border width of client_win*/
    Window client_win;		       /* the window we're managing */
    WObjDescriptor client_descriptor; /* dummy descriptor for client */
    struct {
	int x, y;		       /* position of *client* relative 
					* to root */
	unsigned int width, height;    /* size of the client window */
    } client;

    XSizeHints *normal_hints;	       /* WM_NORMAL_HINTS */
    XWMHints *wm_hints;		       /* WM_HINTS (optional) */
    char *wm_instance;		       /* instance of WM_CLASS */
    char *wm_class;    		       /* class of WM_CLASS */
    GNUstepWMAttributes *wm_gnustep_attr;/* GNUstep window attributes */

    int state;			       /* state as in ICCCM */
    
    Window transient_for;	       /* WM_TRANSIENT_FOR */
    Window group_id;		       /* the leader window of the group */
    Window client_leader;	       /* WM_CLIENT_LEADER */

    Window main_window;		       /* main window for the application */
      
    int cmap_window_no;
    Window *cmap_windows;
    
    /* protocols */
    WProtocols protocols;	       /* accepted WM_PROTOCOLS */
    
    FocusMode focus_mode;	       /* type of keyboard input focus */

    struct {
	/* state flags */
	unsigned int mapped:1;	
	unsigned int focused:1;
	unsigned int miniaturized:1;
	unsigned int hidden:1;
	unsigned int shaded:1;
	unsigned int maximized:2;

	unsigned int semi_focused:1;
	/* window type flags */
	unsigned int urgent:1;	       /* if wm_hints says this is urgent */
#ifdef SHAPE
	unsigned int shaped:1;
#endif
	
	/* info flags */
	unsigned int buttons_dont_fit:1;
	unsigned int rebuild_texture:1;/* the window was resized and 
					* gradients should be re-rendered */
	unsigned int needs_full_repaint:1;/* does a full repaint of the 
					   * window next time it's painted */
	unsigned int icon_moved:1;     /* icon for this window was moved
					* by the user */
	unsigned int ignore_next_unmap:1;
	unsigned int selected:1;       /* multiple window selection */
	unsigned int skip_next_animation:1;
	unsigned int internal_window:1;
	unsigned int changing_workspace:1;

        unsigned int inspector_open:1; /* attrib inspector is already open */
    } flags;		/* state of the window */

    char *icon_text;
    struct WIcon *icon;		       /* icon info for the window */
    int icon_x, icon_y;		       /* position of the icon */
} WWindow;


typedef struct WSavedState {
    int workspace;
    int shaded;
    int hidden;
} WSavedState;


void wWindowDestroy(WWindow *wwin);
WWindow *wWindowCreate();

#ifdef SHAPE
void wWindowSetShape(WWindow *wwin);
#endif

WWindow *wManageWindow(WScreen *scr, Window window);

void wUnmanageWindow(WWindow *wwin, int restore);

void wWindowFocus(WWindow *wwin);
void wWindowUnfocus(WWindow *wwin);
void wWindowConstrainSize(WWindow *wwin, int *nwidth, int *nheight);
void wWindowConfigure(WWindow *wwin, int req_x, int req_y, 
		      int req_width, int req_height);

void wWindowMove(WWindow *wwin, int req_x, int req_y);

void wWindowSynthConfigureNotify(WWindow *wwin);

WWindow *wWindowFor(Window window);


void wWindowConfigureBorders(WWindow *wwin);

void wWindowUpdateButtonImages(WWindow *wwin);

void wWindowSaveState(WWindow *wwin);

void wWindowChangeWorkspace(WWindow *wwin, int workspace);

void wWindowSetKeyGrabs(WWindow *wwin);

WWindow *wManageInternalWindow(WScreen *scr, Window window, Window owner,
			       char *title, int x, int y, 
			       int width, int height);


#endif
