/***************************************
  $Header: /home/amb/procmeter/RCS/xwindow.c 1.11 1998/02/02 19:52:16 amb Exp $

  ProcMeter - A performance metering/logging program for Linux - Version 2.4b.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997,98 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdlib.h>
#include <stdio.h>

#include <sys/time.h>

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/Paned.h>

#include "ProcMeter.h"

#include "procmeter.h"


static void AddRemoveMeterCallback(Widget widget,XtPointer clientData,XtPointer callData);
static void SleepCallback(XtPointer p,XtIntervalId i);
static void ResizePaneCallback(Widget w,XtPointer va,XEvent* e,Boolean* vb);
static void CloseCallback(Widget w,XtPointer va,XEvent* e,Boolean* vb);


static XtAppContext app_context;/*+ The application context. +*/
static Display* display;        /*+ The display that the meter is on. +*/
static Widget pane,             /*+ The widget for the pane containing the meters. +*/
              menu;             /*+ The widget for the menu. +*/
static Dimension length;        /*+ The length of the pane. +*/

static Widget meters_p[NProcStats],  /*+ The list of /proc meter Widgets. +*/
             *meters_o,              /*+ The list of other meter Widgets. +*/
              menus_p[NProcStats],   /*+ The list of /proc menu Widgets. +*/
             *menus_o;               /*+ The list of other menu Widgets. +*/

static int sleeping;            /*+ Set to true when we are sleeping waiting for a timeout. +*/

static int n_windows=0;         /*+ The number of meter windows. +*/


/*+ A flag that is set to true when we are told to quit. +*/
int quit=0;

/*+ If the meters are aligned vertically. +*/
int vertical=1;

/*+ The name of the host that we are running on. +*/
extern char HostName[40];


/*++++++++++++++++++++++++++++++++++++++
  Start the X-Windows part.

  int *argc The number of command line arguments.

  char **argv The actual command line arguments.
  ++++++++++++++++++++++++++++++++++++++*/

void StartX(int *argc,char **argv)
{
 Atom close_atom;
 Widget toplevel,menushell;
 char hostlabel[81];

 int i;

 if(NOtherStats)
   {
    menus_o =(Widget*)malloc((NOtherStats+1)*sizeof(Widget));
    meters_o=(Widget*)malloc((NOtherStats+1)*sizeof(Widget));
   }

 if(*HostName)
    strcpy(hostlabel,HostName);
 else
    strcpy(hostlabel,"ProcMeter");

 /* Initialise the display */

 toplevel=XtVaAppInitialize(&app_context,"ProcMeter",
                            NULL,(Cardinal)0,argc,argv,NULL,
                            XtNtitle,"ProcMeter V2.4b",
                            XtNiconName,hostlabel,
                            NULL);

 display=XtDisplay(toplevel);

 /* Create the pane widget */

 pane=XtVaCreateManagedWidget("pane",panedWidgetClass,toplevel,
                              XtNwidth, vertical?100:200,
                              XtNheight,vertical?200:100,
                              XtNorientation,vertical?XtorientVertical:XtorientHorizontal,
                              NULL);

 XtVaGetValues(pane,vertical?XtNwidth:XtNheight,&length,NULL);

 XtAddEventHandler(pane,StructureNotifyMask,False,(XtEventHandler)ResizePaneCallback,NULL);

 /* Create the menu widgets */

 if(!vertical)
   {
    int i=strlen(hostlabel);
    hostlabel[2*i-1]=0;
    for(i--;i>0;i--)
      {hostlabel[i*2]=hostlabel[i];hostlabel[i*2-1]='\n';}
   }

 menu=XtVaCreateManagedWidget("menu",menuButtonWidgetClass,pane,
                              XtNlabel,hostlabel,
                              XtNmenuName,"menushell",
                              XtNallowResize,False,
                              XtNresizeToPreferred,True,
                              XtNskipAdjust,True,
                              XtNshowGrip,False,
                              vertical?XtNwidth:XtNheight,length,
                              NULL);

 menushell=XtCreatePopupShell("menushell",simpleMenuWidgetClass,menu,NULL,0);

 for(i=0;i<NProcStats;i++)
    if(ProcStats[i].avail)
      {
       menus_p[i]=XtVaCreateManagedWidget(ProcStats[i].name,smeBSBObjectClass,menushell,
                                          XtNlabel,ProcStats[i].name,
                                          XtNleftMargin,10,XtNheight,10,
                                          NULL);
       XtAddCallback(menus_p[i],XtNcallback,AddRemoveMeterCallback,(XtPointer)i);
      }

 for(i=1;i<=NOtherStats;i++)
    if(OtherStats[i].avail)
      {
       if(i==1 || OtherStats[i].file!=OtherStats[i-1].file)
          XtVaCreateManagedWidget("spacer",smeLineObjectClass,menushell,
                                  XtNheight,5,
                                  NULL);

       menus_o[i]=XtVaCreateManagedWidget(OtherStats[i].name,smeBSBObjectClass,menushell,
                                          XtNlabel,OtherStats[i].name,
                                          XtNleftMargin,10,XtNheight,10,
                                          NULL);
       XtAddCallback(menus_o[i],XtNcallback,AddRemoveMeterCallback,(XtPointer)-i);
      }

 /* Show the widgets */

 XtRealizeWidget(toplevel);
 XFlush(display);

 /* Put an action on the close button */

 close_atom=XInternAtom(display,"WM_DELETE_WINDOW",False);

 XSetWMProtocols(display,XtWindow(toplevel),&close_atom,1);

 XtAddEventHandler(toplevel,0,True,CloseCallback,NULL);
}


/*++++++++++++++++++++++++++++++++++++++
  Stop the X-Windows part.
  ++++++++++++++++++++++++++++++++++++++*/

void StopX(void)
{
 XCloseDisplay(display);
}


/*++++++++++++++++++++++++++++++++++++++
  Update the X-Windows display.
  ++++++++++++++++++++++++++++++++++++++*/

void UpdateMetersX(void)
{
 int i;

 for(i=0;i<NProcStats;i++)
    if(ProcStats[i].used&2)
      {
       double value=ProcStats[i].value/(double)ProcStats[i].scale;
       ProcMeterWidgetAddDatum(meters_p[i],(unsigned short)(ProcMeterWidgetScale*value));
      }

 for(i=1;i<=NOtherStats;i++)
    if(OtherStats[i].used&2)
      {
       double value=OtherStats[i].value/(double)OtherStats[i].scale;
       ProcMeterWidgetAddDatum(meters_o[i],(unsigned short)(ProcMeterWidgetScale*value));
      }
}


/*++++++++++++++++++++++++++++++++++++++
  Called when a meter is to be added or removed from the list of those displayed.

  int type The type of meter to add or remove, +ve for a /proc one, -ve otherwise.
  ++++++++++++++++++++++++++++++++++++++*/

void AddRemoveMeterX(int type)
{
 static Pixmap marker=(Pixmap)0;

 if(!marker)
   {
    GC tempgc;
    XGCValues gcxval;

    marker=XCreatePixmap(XtDisplay(pane),RootWindowOfScreen(XtScreen(pane)),10,10,1);

    gcxval.foreground=0;
    gcxval.background=0;
    tempgc=XCreateGC(XtDisplay(pane),marker,GCForeground|GCBackground,&gcxval);

    XFillRectangle(XtDisplay(pane),marker,tempgc,0,0,10,10);
    XSetForeground(XtDisplay(pane),tempgc,(unsigned long)~0);
    XFillArc(XtDisplay(pane),marker,tempgc,1,1,7,7,0,360*64);

    XFreeGC(XtDisplay(pane),tempgc);
   }

 if(type<0)
   {
    if(OtherStats[-type].used)
      {
       char label[20];
       sprintf(label,"%s (%s)",OtherStats[-type].name,OtherStats[-type].units);

       meters_o[-type]=XtVaCreateManagedWidget(OtherStats[-type].name,procMeterWidgetClass,pane,
                                               vertical?XtNwidth:XtNheight,length,
                                               XtNlabel,label,
                                               XtNallowResize,True,
                                               XtNshowGrip,False,
                                               NULL);

       XtVaSetValues(menus_o[-type],XtNleftBitmap,marker,NULL);
      }
    else
      {
       XtDestroyWidget(meters_o[-type]);

       XtVaSetValues(menus_o[-type],XtNleftBitmap,NULL,NULL);
      }

    if(OtherStats[-type].used)
       n_windows++;
    else
       n_windows--;
   }
 else
   {
    if(ProcStats[type].used)
      {
       char label[20];
       sprintf(label,"%s (%s)",ProcStats[type].name,ProcStats[type].units);

       meters_p[type]=XtVaCreateManagedWidget(ProcStats[type].name,procMeterWidgetClass,pane,
                                              vertical?XtNwidth:XtNheight,length,
                                              XtNlabel,label,
                                              XtNallowResize,True,
                                              XtNshowGrip,False,
                                              NULL);

       XtVaSetValues(menus_p[type],XtNleftBitmap,marker,NULL);
      }
    else
      {
       XtDestroyWidget(meters_p[type]);

       XtVaSetValues(menus_p[type],XtNleftBitmap,NULL,NULL);
      }

    if(ProcStats[type].used)
       n_windows++;
    else
       n_windows--;
   }

 ResizePaneCallback(NULL,NULL,NULL,NULL);
}


/*++++++++++++++++++++++++++++++++++++++
  Sleep for the specified interval in seconds.

  time_t until The time to sleep until.
  ++++++++++++++++++++++++++++++++++++++*/

void SleepX(time_t until)
{
 XtIntervalId id;
 struct timeval now;
 int delay;

 gettimeofday(&now,NULL);

 delay=1000*(until-now.tv_sec)-now.tv_usec/1000;

 if(delay>0)
   {
    id=XtAppAddTimeOut(app_context,(unsigned)delay,(XtTimerCallbackProc)SleepCallback,NULL);
    sleeping=1;

    while(sleeping)
      {
       struct timeval now2;

       XtAppProcessEvent(app_context,XtIMAll);

       gettimeofday(&now2,NULL);

       if(now2.tv_sec<now.tv_sec)       /* Ooops, we went back in time. Let's cancel timer */
         {
          XtRemoveTimeOut(id);
          sleeping=0;
         }
      }
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Called when a menu item is chosen.

  Widget widget The widget causing the callback.

  XtPointer clientData The type to change.

  XtPointer callData Not used.
  ++++++++++++++++++++++++++++++++++++++*/

static void AddRemoveMeterCallback(Widget widget,XtPointer clientData,XtPointer callData)
{
 int type=(int)clientData;

 if(type<0)
    OtherStats[-type].used=!OtherStats[-type].used;
 else
    ProcStats[type].used=!ProcStats[type].used;

 AddRemoveMeterX(type);
}


/*++++++++++++++++++++++++++++++++++++++
  The function called by the timeout to terminate the sleep.

  XtPointer p Not used.

  XtIntervalId i Not used.
  ++++++++++++++++++++++++++++++++++++++*/

static void SleepCallback(XtPointer p,XtIntervalId i)
{
 sleeping=0;
}


/*++++++++++++++++++++++++++++++++++++++
  A callback that is activated by a resize event on the parent pane.

  Widget w The widget that caused the callback.

  XtPointer va Not used.

  XEvent* e The event that requires action.

  Boolean* vb Not used.

  This function is only ever called from the Xt Intrinsics routines.
  ++++++++++++++++++++++++++++++++++++++*/

static void ResizePaneCallback(Widget w,XtPointer va,XEvent* e,Boolean* vb)
{
 Dimension fsize,msize,size;
 int i;

 XtVaGetValues(pane,vertical?XtNwidth:XtNheight,&length,NULL);

 if(!n_windows)
    return;

 XtVaGetValues(pane,vertical?XtNheight:XtNwidth,&fsize,NULL);
 XtVaGetValues(menu,vertical?XtNheight:XtNwidth,&msize,NULL);

 size=(int)(fsize-msize)/n_windows;

 XtVaSetValues(menu,vertical?XtNwidth:XtNheight,length,vertical?XtNheight:XtNwidth,msize,NULL);
 for(i=0;i<NProcStats;i++)
    if(ProcStats[i].used)
       XtVaSetValues(meters_p[i],vertical?XtNwidth:XtNheight,length,vertical?XtNheight:XtNwidth,size,NULL);
 for(i=1;i<=NOtherStats;i++)
    if(OtherStats[i].used)
       XtVaSetValues(meters_o[i],vertical?XtNwidth:XtNheight,length,vertical?XtNheight:XtNwidth,size,NULL);
}


/*++++++++++++++++++++++++++++++++++++++
  A callback that is activated by a close window event on the toplevel window.

  Widget w The widget that caused the callback.

  XtPointer va Not used.

  XEvent* e The event that requires action.

  Boolean* vb Not used.

  This function is only ever called from the Xt Intrinsics routines.
  ++++++++++++++++++++++++++++++++++++++*/

static void CloseCallback(Widget w,XtPointer va,XEvent* e,Boolean* vb)
{
 XClientMessageEvent *cev=(XClientMessageEvent*)e;
 Atom atom_type=XInternAtom(display,"WM_PROTOCOLS",False);

 if(atom_type==cev->message_type)
   {
    Atom atom_proto=XInternAtom(display,"WM_DELETE_WINDOW",False);

    if(cev->format==32 && atom_proto==(Atom)cev->data.l[0])
       quit=1;
   }
}
