#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/stat.h>
#include <ctype.h>
#include <unistd.h>
#include <getopt.h>
#define __GUS_PATH_APP__
#include "../../include/libgus.h"
#include "../../include/gusfiles.h"
#include GUS_INCLUDE_NCURSES

#define VERSION		"1.10"

#define HELPID_HELP		1000
#define HELPID_QUIET		1001
#define HELPID_VERBOSE		1002
#define HELPID_CARD		1003
#define HELPID_DEVICE		1004
#define HELPID_FILE		1005
#define HELPID_INSTRUMENT	1006
#define HELPID_SOURCE		1007
#define HELPID_LIST		1008
#define HELPID_REVERB		1011
#define HELPID_CHORUS		1012
#define HELPID_GM		1013
#define HELPID_GS		1014
#define HELPID_MT32		1015
#define HELPID_TIME		1016
#define HELPID_PRELOAD		1017

static int quiet;
static int verbose;
static int quit;
static int prev;
static int next;
static int time_break;

static int m_format;
static char *m_format_name;
static int m_tracks;
static int m_pause;

static void signal_terminate( int signal )
{
  gus_midiplay_stop( m_pause = 1 );
} 

static void m_select( void *private, fd_set *read_fds, fd_set *write_fds )
{
  if ( FD_ISSET( fileno( stdin ), read_fds ) )
    {
      int key;
      
      for ( key = getch(); key != EOF; key = getch() )
        switch ( tolower( key ) ) {
          case 'q':
            gus_midiplay_stop( quit = m_pause = 1 );
            break;
          case 'p':
            if ( !(m_pause & 1) )
              {
                m_pause ^= 2;
                if ( !quiet )
                  {
                    if ( m_pause & 2 )
                      printf( "Waiting... " );
                     else
                      printf( "\r                                                " );
                    fflush( stdout );
                  }
                gus_midiplay_stop( m_pause );
              }
            break;
          case 'n':
          case ' ':
          case '+':
          case KEY_RIGHT:
            gus_midiplay_stop( next = m_pause = 1 );
            break;
          case '-':
          case KEY_LEFT:
            gus_midiplay_stop( prev = m_pause = 1 );
            break;
        }
    }
}

static void m_info( void *private, int format, char *format_name, int tracks )
{
  m_format = format;
  if ( m_format_name ) free( m_format_name );
  m_format_name = strdup( format_name );
  m_tracks = 0;
  if ( !quiet )
    {
      printf( "\nFormat: %s, Tracks: %i...\n", format_name, tracks );
      fflush( stdout );
    }
}

static void m_download_program( void *private, gus_midi_device_t *mdevice, unsigned int program )
{
#if 0
  printf( "download_program: device = %i, program = %i/%i\n", mdevice -> device, program >> 16, program & 0xffff );
#endif
  if ( mdevice -> cap & GUS_MIDI_CAP_MEMORY )
    if ( gus_midi_icfg_download_program( mdevice -> device, &program, 1 ) < 0 )
      printf( "Oops, out of memory... Instrument %i/%i cannot be loaded...\n", program >> 16, ( program & 0xffff ) + 1 );
}

static void m_total_time( void *private, unsigned int time )
{
  if ( !quiet )
    {
      printf( "Total time   = %i.%i second%s\n", time / 10, time % 10, time / 10 == 1 ? "" : "s" );
      fflush( stdout );
    }
}

static void m_time( void *private, unsigned int time )
{
  if ( !quiet )
    {
      printf( "\rCurrent time = %i.%i second%s     ", time / 10, time % 10, time / 10 == 1 ? "" : "s" );
      fflush( stdout );
    }
  if ( time_break > 0 && time_break <= time / 10 )
    gus_midiplay_stop( next = m_pause = 1 );
}

static void m_text( void *private, int track, char *text )
{
  if ( verbose )
    printf( "-%i- Text: %s\n", track, text );
}

static void m_copyright( void *private, int track, char *text )
{
  if ( verbose )
    printf( "-%i- Copyright: %s\n", track, text );
}

static void m_sequence( void *private, int track, char *text )
{
  if ( verbose )
    printf( "-%i- Sequence name: %s\n", track, text );
}

static void m_instrument( void *private, int track, char *text )
{
  if ( verbose )
    printf( "-%i- Intrument: %s\n", track, text );
}

static void m_lyric( void *private, int track, char *text )
{
  if ( verbose )
    printf( "-%i- Lyric: %s\n", track, text );
}

static void m_marker( void *private, int track, char *text )
{
  if ( verbose )
    printf( "-%i- Marker: %s\n", track, text );
}

static void m_cuepoint( void *private, int track, char *text )
{
  if ( verbose )
    printf( "-%i- Cue-point: %s\n", track, text );
}

static gus_midiplay_callbacks_t m_callbacks = {
  15,
  NULL,
  0,
  {},
  {},
  m_select,
  m_info,
  m_download_program,
  m_total_time,
  m_time,
  m_text,
  m_copyright,
  m_sequence,
  m_instrument,
  m_lyric,
  m_marker,
  m_cuepoint
};

static void help( void )
{
  printf( "Available switches:\n\n"
  	  "  -h, --help    help\n"
  	  "  -q, --quiet   quiet mode\n"
  	  "  -f <file>, --configuration <file>\n"
  	  "                configuration file (default " GUS_FILE_MIDI_CONF ")\n"
  	  "  -i <file>, --instrument <file>\n"
  	  "                instrument file (default " GUS_FILE_SYNTH_CONF ")\n"
  	  "  -c #, --card #\n"
  	  "                select output card (1-%i)\n"
  	  "  -d <dev>, --device <dev>\n"
  	  "                select output device (synth,uart)\n"
  	  "  -u, --source <source>\n"
  	  "                use selected instrument source\n"
  	  "  -l, --list    lists possible instrument sources\n"
  	  "  -u <bank>, --user <bank>\n"
  	  "                use user bank (megabank, utopia or utopia_mono)\n"
  	  "                as instrument source\n"
  	  "  -r <val>, --reverb <val>\n"
  	  "                reverb effect type for InterWave (-1 = disable, 0-7, default=4)\n"
  	  "  -s <val>, --chorus <val>\n"
  	  "                chorus effect type for InterWave (-1 = disable, 0-7, default=2)\n"
  	  "  -g, --GS      Roland GS mode on\n"
  	  "  -m, --GM      General MIDI mode on\n"
  	  "  -3, --MT32    MT-32 mode on\n"
  	  "  -v, --verbose verbose mode\n"
  	  "  -t <sec>, --time <sec>\n"
  	  "                scan mode - set timelimit to <sec>\n"
  	  "  -p, --preload preload mode (for source 'all')\n",
  	  	GUS_CARDS
  );
}

int main( int argc, char *argv[] )
{
  int morehelp = 0, card = -1, device = -1, list_mode = 0, preload = 0;
  char *instr_source = NULL;
  int reverb_type = 4, chorus_type = 2;
  gus_midi_device_t *mdevice;
  char *conf_file = NULL;
  char *instr_file = NULL;
  int emul = -1, old_emul = -1;
  int optind_first;
  struct option long_option[] = {
    { "help", 0, NULL, HELPID_HELP },
    { "quiet", 0, NULL, HELPID_QUIET },
    { "verbose", 0, NULL, HELPID_QUIET },
    { "card", 1, NULL, HELPID_CARD },
    { "device", 1, NULL, HELPID_DEVICE },
    { "configuration", 1, NULL, HELPID_FILE },
    { "instrument", 1, NULL, HELPID_INSTRUMENT },
    { "source", 1, NULL, HELPID_SOURCE },
    { "list", 0, NULL, HELPID_LIST },
    { "reverb", 1, NULL, HELPID_REVERB },
    { "chorus", 1, NULL, HELPID_CHORUS },
    { "GM", 0, NULL, HELPID_GM },
    { "GS", 0, NULL, HELPID_GS },
    { "MT-32", 0, NULL, HELPID_MT32 },
    { "time", 0, NULL, HELPID_TIME },
    { "preload", 0, NULL, HELPID_PRELOAD },
    { NULL, 0, NULL, 0 }
  };

  quiet = 0;
  verbose = 0;
  time_break = 0;
  instr_source = strdup( GUS_ICFG_SOURCE_AUTO );
  while ( 1 )
    {
      int c;
    
      if ( ( c = getopt_long( argc, argv, "u:lhqvc:d:f:i:r:s:gm3tp", long_option, NULL ) ) < 0 ) break;
      switch ( c ) {
        case 'h':
        case HELPID_HELP:
          morehelp++;
          break;
        case 'q':
        case HELPID_QUIET:
          quiet = 1;
          break;
        case 'v':
        case HELPID_VERBOSE:
          verbose = 1;
          break;
        case 'c':
        case HELPID_CARD:
          card = atoi( optarg );
          if ( card < 0 || card >= GUS_CARDS )
            {
              fprintf( stderr, "card out of range (1-%i)\n", GUS_CARDS );
              morehelp++;
            }
          card--;
          if ( device < 0 ) device = GUS_MIDID_SYNTH;
          break;
        case 'd':
        case HELPID_DEVICE:
          if ( !strcmp( optarg, "synth" ) ) device = GUS_MIDID_SYNTH; else
          if ( !strcmp( optarg, "uart" ) ) device = GUS_MIDID_UART; else
            {
              fprintf( stderr, "invalid device (synth or uart)\n" );
              morehelp++;
            }
          if ( card < 0 ) card = 0;
          break;
        case 'f':
        case HELPID_FILE:
          free( conf_file );
          conf_file = strdup( optarg );
          break;
        case 'i':
        case HELPID_INSTRUMENT:
          free( instr_file );
          instr_file = strdup( optarg );
          break;
        case 'u':
        case HELPID_SOURCE:
          free( instr_source );
          instr_source = strdup( optarg );
          break;
        case 'l':
        case HELPID_LIST:
          list_mode = 1;
          break;
        case 'r':
        case HELPID_REVERB:
          reverb_type = atoi( optarg );
          if ( reverb_type < 0 ) reverb_type = -1;
          if ( reverb_type > 7 )
            {
              fprintf( stderr, "unknown reverb effect type\n" );
              morehelp++;
            }
          break;
        case 's':
        case HELPID_CHORUS:
          chorus_type = atoi( optarg );
          if ( chorus_type < 0 ) reverb_type = -1;
          if ( chorus_type > 7 )
            {
              fprintf( stderr, "unknown chorus effect type\n" );
              morehelp++;
            }
          break;
        case 'm':
        case HELPID_GM:
          emul = GUS_MIDI_EMUL_GM;
          break;
        case 'g':
        case HELPID_GS:
          emul = GUS_MIDI_EMUL_GS;
          break;
        case '3':
        case HELPID_MT32:
          emul = GUS_MIDI_EMUL_MT32;
          break;
        case 't':
        case HELPID_TIME:
          time_break = atoi( optarg );
          if ( time_break < 0 ) time_break = 0;
          break;
        case 'p':
        case HELPID_PRELOAD:
          preload = 1;
          break;
        case '?':
          printf( "\07Invalid switch or option needs an argument\n" );
      }          
    }

  if ( (optind >= argc && !list_mode) || morehelp )
    {
      printf( "Usage: %s [switches] <easy.mid> ... \n", argv[ 0 ] );
      if ( morehelp )
        help();
       else
        printf( "Type %s --help for more help.\n", argv[ 0 ] );
      return -1;
    }

  if ( !quiet )
    {
      /* initialize curses code */
      atexit((void*)endwin);
      initscr();
      cbreak();
      nodelay(stdscr, TRUE);
      noecho();
    }

  signal( SIGINT, signal_terminate );
  signal( SIGTERM, signal_terminate );
      
  if ( !quiet ) {
    printf( "UltraMidi version " VERSION " by Jaroslav Kysela (Perex soft)\n" );
    fflush( stdout );
  }

  if ( verbose ) quiet = 0;	/* ok... why verbose and quiet??? */

  if ( card < 0 || device < 0 )
    device = GUS_MIDID_COMMON;
   else
    device = ( card << 4 ) | device;

  if ( device == GUS_MIDID_COMMON )
    {
      if ( list_mode ) {
        fprintf( stderr, "I'm sorry, I can't do list mode for common configuration...\n" );
        fprintf( stderr, "This is limitation of this program (it's too simple)...\n" );
        fprintf( stderr, "Please use command 'ultramidi -d synth -l' or 'ultramidi -d uart -l'...\n" );
        return -1;
      }
      if ( gus_midi_open_intelligent( GUS_MIDI_OUT, conf_file, 8 * 1024, GUS_MIDI_OF_ECHO_ENABLE ) )
        {
          fprintf( stderr, "midi open: %s\n", gus_midi_error() );
          return -1;
        }
    }
   else
    {
      mdevice = malloc( sizeof( gus_midi_device_t ) );
      gus_midi_fill_device_structure( mdevice, GUS_MIDI_OPEN_MODE_WRITE, device, 0xffff );
      if ( gus_midi_open( GUS_MIDI_OUT, mdevice, 8 * 1024, GUS_MIDI_OF_ECHO_ENABLE ) )
        {
          fprintf( stderr, "midi open: %s\n", gus_midi_error() );
          return -1;
        }
    }
  
  for ( mdevice = gus_midi_output_devices(); mdevice; mdevice = mdevice -> next )
    {
      if ( emul >= 0 )
        {
          old_emul = gus_midi_emulation_get( mdevice -> device );
          if ( old_emul != emul )
            gus_midi_emulation_set( mdevice -> device, emul );
           else
            old_emul = -1;
        }
      if ( mdevice -> cap & GUS_MIDI_CAP_MEMORY )
        {
          gus_info_t ginfo;
          struct GUS_STRU_EFFECT effect;
      
          if ( list_mode ) {
            struct gus_icfg_info *info, *pinfo;
            int idx;
          
            if ( gus_midi_icfg_info( mdevice -> device, instr_file, &info ) < 0 ) {
              fprintf( stderr, "icfg info error\n" );
              gus_midi_close();
              return -1;
            }
            printf( "\nInstrument info %i/%s:\n", ( mdevice -> device >> 4 ) + 1, (mdevice -> device & 0x0f) == GUS_MIDID_SYNTH ? "synth" : "uart" );
            printf( "  %-22s %-40s %s\n", GUS_ICFG_SOURCE_AUTO, "auto select (default)", "" );
            for ( pinfo = info; pinfo; pinfo = pinfo -> next ) {
              if ( pinfo -> type == GUS_ICFG_INFO_SOURCE ) {
                printf( "  %-22s %-40s", pinfo -> name, pinfo -> description );
                if ( pinfo -> options )
                  for ( idx = 0; pinfo -> options[ idx ]; idx++ )
                    printf( " [%s]", pinfo -> options[ idx ] );
                printf( "\n" );
              }
            }
            printf( "\n" );
            gus_icfg_free_info( info );
          } else {
            if ( gus_midi_icfg_open( mdevice -> device, instr_file, instr_source, "" ) )
              {
                fprintf( stderr, "icfg open error\n" );
                gus_midi_close();
                return -1;
              }
            if ( gus_midi_synth_info( mdevice -> device, &ginfo ) >= 0 )
              if ( ginfo.flags & GUS_STRU_INFO_F_ENHANCED )
                {
                  gus_effect_interwave( &effect, reverb_type, chorus_type );
                  gus_midi_effect_setup( mdevice -> device, &effect );
                }
            if ( preload ) {
              printf( "Preload start...\n" ); fflush( stdout );
  	      if ( gus_midi_icfg_preload( mdevice -> device, "all", NULL ) < 0 )
	        printf( "preload error...\n" );
	      printf( "Preload done...\n" ); fflush( stdout );
	    }
          }
       }
    }

  if ( list_mode ) {
    gus_midi_close();
    return 0;
  }

  quit = 0;  
  optind_first = optind;
  while ( optind < argc && !quit ) 
    {
      FILE *in;
      long size, rsize;
      unsigned char *song;
      char *song_name;
      struct stat st;
      
      song_name = argv[ optind++ ];
      if ( stat( song_name, &st ) < 0 ) continue;
      if ( ( st.st_mode & S_IFMT ) != S_IFREG ) continue;
      in = fopen( song_name, "rb" );
      if ( !in ) continue;
      size = 0;
      song = malloc( 10 * 1024 );
      if ( !song )
        {
          fprintf( stderr, "Alloc error.\n" );
          return -1;
        }
      while ( 1 )
        {
          rsize = fread( &song[ size ], 1, 10 * 1024, in );
          size += rsize;
          if ( rsize < 10 * 1024 )
            {
              song = realloc( song, size );
              if ( !song )
                {
                  fprintf( stderr, "Alloc error.\n" );
                  return -1;
                }
              break;
            }
           else
            {
              song = realloc( song, size + 10 * 1024 );
              if ( !song )
                {
                  fprintf( stderr, "Alloc error.\n" );
                  return -1;
                }
            }
        }
      fclose( in );

      if ( ( rsize = gus_midiplay_songs( song, size ) ) > 1 )
        printf( "Warning! Midi file '%s' contains %li songs\n", argv[ optind - 1 ], rsize );
      if ( !rsize )
        {
          free( song );
          printf( "File '%s' have unknown format.\n", argv[ optind - 1 ] );
          continue;
        }

      m_callbacks.last_fd = gus_midi_get_file_descriptor();
      FD_ZERO( &m_callbacks.read_fds );
      FD_ZERO( &m_callbacks.write_fds );
      if ( in != stdin && !quiet )
        FD_SET( fileno( stdin ), &m_callbacks.read_fds );
      if ( !quiet )
        {
          printf( "Playing song '%s'... ", song_name );
          fflush( stdout );
        }
      m_format_name = NULL;
      m_pause = 0;
      if ( gus_midiplay_song( device, song, size, 0, &m_callbacks ) < 0 )
        fprintf( stderr, "play error\n" );

      for ( mdevice = gus_midi_output_devices(); mdevice; mdevice = mdevice -> next )
        if ( mdevice -> cap & GUS_MIDI_CAP_MEMORY )
          gus_midi_memory_reset( mdevice -> device );		/* ok.. clean instruments */

      free( m_format_name ); m_format_name = NULL;
      if ( !quiet )
        {
          printf( "\r" );
          if ( next || prev )
            {
              printf( "Skipped..." );
              if ( time_break > 0 )
                printf( " Scan mode..." );
            }
           else
            printf( "Done..." );
          printf( "                              \n" );
          fflush( stdout );
        }
      if ( next ) next = 0;
      if ( prev )
        {
          optind -= 2;
          if ( optind < optind_first ) optind = optind_first;
          prev = 0;
        }
    }

  for ( mdevice = gus_midi_output_devices(); mdevice; mdevice = mdevice -> next )
    {
      if ( mdevice -> cap & GUS_MIDI_CAP_MEMORY )
        gus_midi_icfg_close( mdevice -> device );
      if ( old_emul >= 0 )
        gus_midi_emulation_set( mdevice -> device, old_emul );
    }

  gus_midi_close();
  free( conf_file );
  free( instr_file );
  free( instr_source );

  signal( SIGINT, SIG_DFL );
  signal( SIGTERM, SIG_DFL );

  return 0;
}
