/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Mixer support routines
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include "libgus.h"
#include "libgus_local.h"

#define FILE_MIXER	"/dev/gusmixer%i%i"

/*
 *  structures
 */

typedef struct {
  int card;
  int handle;
  int devmask;
  int recmask;
  int stereodevs;
  int version;
} CARD;

/*
 *  variables
 */

/*
 *  EXPORTED FUNCTIONS
 */ 

extern int __gus_decode_id( const char *id );

int gus_mixer_cards( void )
{
  return gus_cards();
}

int gus_mixer_devices( const char *id )
{
  int card;
  
  card = __gus_decode_id( id );
  if ( card < 0 || card >= GUS_CARDS )
    return -ENODEV;
  return 1;
}

int gus_mixer_look_for_card( const char *id )
{
  return __gus_decode_id( id );
}

int gus_mixer_open( void **rhandle, int card, int device )
{
  CARD *acard;
  int handle, i, mversion; 
  int devmask, recmask, stereodevs;
  char filename[ 16 ];

  if ( !rhandle ) return -EINVAL;
  *rhandle = NULL;
  if ( card < 0 || card >= GUS_CARDS ) return -ENODEV;
  if ( device != 0 ) return -ENODEV;
  sprintf( filename, FILE_MIXER, card, device );
  if ( ( handle = open( filename, O_RDWR ) ) < 0 )
    {
      gus_dprintf( "gus_mixer_open: file open error %i", errno );
      return -1;
    }
  mversion = 0;
  if ( ioctl( handle, SOUND_MIXER_GUS_VERSION, &mversion ) < 0 )
    {
      gus_dprintf( "gus_mixer_open: read gus version error" );
      close( handle );
      return -1;
    }
  i = 0;
  if ( ioctl( handle, SOUND_MIXER_READ_DEVMASK, &devmask ) < 0 ) i = -1;
  if ( ioctl( handle, SOUND_MIXER_READ_RECMASK, &recmask ) < 0 ) i = -1;
  if ( ioctl( handle, SOUND_MIXER_READ_STEREODEVS, &stereodevs ) < 0 ) i = -1;
  if ( i < 0 )
    {
      gus_dprintf( "gus_mixer_open: device mask or record mask or stereo devices ioctl error" );
      close( handle );
      return -1;
    }
    
  acard = (CARD *)malloc( sizeof( CARD ) );
  if ( !acard )
    {
      gus_dprintf( "gus_mixer_open: malloc problem" );
      close( handle );
      return -1;
    }
 
  acard -> card = card;
  acard -> handle = handle;
  acard -> devmask = devmask;
  acard -> recmask = recmask;
  acard -> stereodevs = stereodevs;
  acard -> version = mversion;
    
  *rhandle = acard;
  return 0;
}

int gus_mixer_close( void *handle )
{
  int res;
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  res = close( acard -> handle ) < 0 ? -1 : 0;
  free( acard );
  return res;
}

int gus_mixer_version( void *handle )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  return acard -> version;
}

char *gus_mixer_model( void *handle )
{
  int i;
  CARD *acard;
  char str[ 128 ], *model, *pstr;

  acard = handle;
  if ( !acard ) return NULL;
  i = acard -> version;
  pstr = "without";
  if ( i & 0x10000 )		/* daughter board */
    pstr = "with CS4231";
   else
    {
      i &= 0xffff;
      if ( i >= 0x35 && i <= 0x37 ) pstr = "with ICS2101";
      if ( i == 0x50 ) pstr = "with ES 1688";
      if ( i == 0xa0 ) pstr = "with CS 4231";
      if ( i == 0x100 && i <= 0x101 ) pstr = "with InterWave";
    }
  model = gus_model( acard -> version & 0xffff );
  if ( !model ) return NULL;
  sprintf( str, "%s %s mixer chip", model, pstr );
  free( model );
  return strdup( str );
}

int gus_mixer_read_devmask( void *handle )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  return acard -> devmask;
}

int gus_mixer_read_recmask( void *handle )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  return acard -> recmask;
}

int gus_mixer_read_stereodevs( void *handle )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  return acard -> stereodevs; 
}

#ifdef DEBUG

#define gus_mixer_read( handle, what, mesg ) \
	_gus_mixer_read_( handle, SOUND_MASK_##what, SOUND_MIXER_READ_##what, mesg )

static int _gus_mixer_read_( void *handle, int mask, int what, char *mesg )
{
  int res = -1;
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  if ( acard -> devmask & mask )
    if ( ioctl( acard -> handle, what, &res ) < 0 )
      gus_dprintf( "gus_mixer_read: %s", mesg );
  return res;
}

#else

#define gus_mixer_read( handle, what, mesg ) \
	_gus_mixer_read_( handle, SOUND_MASK_##what, SOUND_MIXER_READ_##what )

static int _gus_mixer_read_( void *handle, int mask, int what )
{
  int res = -1;
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  if ( acard -> devmask & mask )
    return ioctl( acard -> handle, what, &res ) < 0 ? -1 : res;
   else
    return -1;
}

#endif

int gus_mixer_read_recsrc( void *handle )
{
  int res = -1;
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  if ( ioctl( acard -> handle, SOUND_MIXER_READ_RECSRC, &res ) < 0 )
    gus_dprintf( "gus_mixer_read_recsrc: error" );
  return res;
}

int gus_mixer_read_devs_lmute( void *handle )
{
  int res = -1;
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  if ( ioctl( acard -> handle, SOUND_MIXER_READ_DEVS_LMUTE, &res ) < 0 )
    gus_dprintf( "gus_mixer_read_devs_lmute: error" );
  return res;
}

int gus_mixer_read_devs_rmute( void *handle )
{
  int res = -1;
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  if ( ioctl( acard -> handle, SOUND_MIXER_READ_DEVS_RMUTE, &res ) < 0 )
    gus_dprintf( "gus_mixer_read_devs_rmute: error" );
  return res;
}

int gus_mixer_read_mic( void *handle )
{
  return gus_mixer_read( handle, MIC, "MIC" );
}

int gus_mixer_read_cd( void *handle )
{
  return gus_mixer_read( handle, CD, "CD" );
}

int gus_mixer_read_line( void *handle )
{
  return gus_mixer_read( handle, LINE, "LINE" );
}

int gus_mixer_read_synth( void *handle )
{
  return gus_mixer_read( handle, SYNTH, "SYNTH" );
}

int gus_mixer_read_pcm( void *handle )
{
  return gus_mixer_read( handle, PCM, "PCM" );
}

int gus_mixer_read_pcm1( void *handle )
{
  return gus_mixer_read( handle, SPEAKER, "PCM1" );
}

int gus_mixer_read_reclev( void *handle )
{
  return gus_mixer_read( handle, RECLEV, "RECLEV" );
}

int gus_mixer_read_volume( void *handle )
{
  return gus_mixer_read( handle, VOLUME, "VOLUME" );
}

int gus_mixer_read_imix( void *handle )
{
  return gus_mixer_read( handle, IMIX, "IMIX" );
}

int gus_mixer_read_loopback( void *handle )
{
  return gus_mixer_read( handle, LINE1, "LOOPBACK" );
}

int gus_mixer_read_effect( void *handle )
{
  return gus_mixer_read( handle, LINE2, "EFFECT" );
}

#ifdef DEBUG

#define gus_mixer_write( handle, what, data, mesg ) \
	_gus_mixer_write_( handle, SOUND_MASK_##what, SOUND_MIXER_WRITE_##what, data, mesg )

static int _gus_mixer_write_( void *handle, int mask, int what, int data, char *mesg )
{
  int res = -1;
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  if ( acard -> devmask & mask )
    if ( ( res = ioctl( acard -> handle, what, &data ) ) < 0 )
      gus_dprintf( "gus_mixer_write: 0x%x, %s", data, mesg );
  return res;
}

#else

#define gus_mixer_write( handle, what, data, mesg ) \
	_gus_mixer_write_( handle, SOUND_MASK_##what, SOUND_MIXER_WRITE_##what, data )

static inline int _gus_mixer_write_( void *handle, int mask, int what, int data )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  if ( acard -> devmask & mask )
    return ioctl( acard -> handle, what, &data ) < 0 ? -1 : 0;
   else
    return -1;
}

#endif

int gus_mixer_write_recsrc( void *handle, int value )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  if ( ioctl( acard -> handle, SOUND_MIXER_WRITE_RECSRC, &value ) < 0 )
    {
      gus_dprintf( "gus_mixer_write_recsrc: error" );
      return -1;
    }
  return 0;
}

int gus_mixer_write_devs_lmute( void *handle, int dev, int mute )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  if ( mute ) dev |= SOUND_MIXER_MUTE_FLAG;
  if ( ioctl( acard -> handle, SOUND_MIXER_WRITE_DEVS_LMUTE, &dev ) < 0 )
    {
      gus_dprintf( "gus_mixer_write_devs_lmute: error" );
      return -1;
    }
  return 0;  
}

int gus_mixer_write_devs_rmute( void *handle, int dev, int mute )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  if ( mute ) dev |= SOUND_MIXER_MUTE_FLAG;
  if ( ioctl( acard -> handle, SOUND_MIXER_WRITE_DEVS_RMUTE, &dev ) < 0 )
    {
      gus_dprintf( "gus_mixer_write_devs_rmute: error" );
      return -1;
    }
  return 0;  
}

int gus_mixer_write_mic( void *handle, int value )
{
  return gus_mixer_write( handle, MIC, value, "MIC" );
}

int gus_mixer_write_cd( void *handle, int value )
{
  return gus_mixer_write( handle, CD, value, "CD" );
}

int gus_mixer_write_line( void *handle, int value )
{
  return gus_mixer_write( handle, LINE, value, "LINE" );
}

int gus_mixer_write_synth( void *handle, int value )
{
  return gus_mixer_write( handle, SYNTH, value, "SYNTH" );
}

int gus_mixer_write_pcm( void *handle, int value )
{
  return gus_mixer_write( handle, PCM, value, "PCM" );
}

int gus_mixer_write_pcm1( void *handle, int value )
{
  return gus_mixer_write( handle, SPEAKER, value, "PCM1" );
}

int gus_mixer_write_reclev( void *handle, int value )
{
  return gus_mixer_write( handle, RECLEV, value, "RECLEV" );
}

int gus_mixer_write_volume( void *handle, int value )
{
  return gus_mixer_write( handle, VOLUME, value, "VOLUME" );
}

int gus_mixer_write_imix( void *handle, int value )
{
  return gus_mixer_write( handle, IMIX, value, "IMIX" );
}

int gus_mixer_write_loopback( void *handle, int value )
{
  return gus_mixer_write( handle, LINE1, value, "LOOPBACK" );
}

int gus_mixer_write_effect( void *handle, int value )
{
  return gus_mixer_write( handle, LINE2, value, "EFFECT" );
}

/*
 * special things
 */

int gus_mixer_read_interwave_serial( void *handle )
{
  struct GUS_MIXER_SPECIAL special;
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;  
  special.what = GUS_MIXER_S_IW;
  if ( ioctl( acard -> handle, SOUND_MIXER_SPECIAL_READ, &special ) < 0 )
    {
      gus_dprintf( "gus_mixer_read_interwave_serial: error\n" );
      return -1;
    }
  return special.data.interwave.serial;
}

int gus_mixer_write_interwave_serial( void *handle, int serial )
{
  struct GUS_MIXER_SPECIAL special;
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;  
  special.what = GUS_MIXER_S_IW;
  special.data.interwave.serial = serial;
  if ( ioctl( acard -> handle, SOUND_MIXER_SPECIAL_WRITE, &special ) < 0 )
    {
      gus_dprintf( "gus_mixer_write_interwave_serial: error\n" );
      return -1;
    }
  return special.data.interwave.serial;
}
