/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 */

%{

#include <string.h>
#include "icfg.h"
#include "icfg_parser.h"

#define MAX_INCLUDE		16
#define MAX_DEFINES		128

#undef YY_BUF_SIZE
#define YY_BUF_SIZE		2048
#define YY_NEVER_INTERACTIVE	1
#define YY_STACK_USED		1
#define YY_NO_TOP_STATE		1
#undef YY_NO_UNPUT
#undef YY_CDECL
#define YY_CDECL int YY_PROTO(yylex( void ));
#define yyerror( args... ) {gus_icfg_parser_abort();gus_icfg_error( 1, ##args );}

static int include_stack_ptr;
static YY_BUFFER_STATE include_stack[ MAX_INCLUDE ];
static int include_line_count[ MAX_INCLUDE ];
static char *include_file[ MAX_INCLUDE ];

static char *strip_assign_char( char *str );
static int decode_variable( char *yytext );

extern void gus_icfg_parser_abort( void );

%}

DIGIT		[0-9]
HEXA		[0-9a-f]
STRING		[a-z0-9/\~@-_\+:,\.]
VARIABLE	[a-z0-9\-_]

%x comment

%%

<<EOF>>			{
			  if ( include_stack_ptr < 0 )
			    yyterminate();
			   else
			    {
			      yy_delete_buffer( YY_CURRENT_BUFFER );
                              fclose( yyin );
			      gus_icfg_config -> line_count = include_line_count[ include_stack_ptr ];
			      if ( include_stack_ptr > 0 )
                                free( gus_icfg_config -> filename );
			      gus_icfg_config -> filename = include_file[ include_stack_ptr ];
			      yy_switch_to_buffer( include_stack[ include_stack_ptr-- ] );
			    }
			}

	/* special characters */

";"			return yytext[0];
"{"|"}"			return yytext[0];
"("|")"			return yytext[0];
"*"|"+"|"-"|"/"		return yytext[0];
">>"			return L_RSHIFT;
"<<"			return L_LSHIFT;
"||"			return L_OR;
"&&"			return L_AND;
"&"|"|"			return yytext[0];
"<"			return L_LE;
"<="			return L_LEQ;
">"			return L_GT;
">="			return L_GTQ;
"=="			return L_EQ;
"!="			return L_NEQ;
"!"			return yytext[0];
","			return yytext[0];
"="			return L_ASSIGN;

	/* keywords - conditions */

if			return L_IF;
ife			return L_IFE;
else			return L_ELSE;

	/* keywords */

unset			return L_UNSET;
include			return L_INCLUDE;
message			return L_MESSAGE;
abort			return L_ABORT;
"file_exist"		return L_FILE_EXIST;
"patch_exist"		return L_PATCH_EXIST;
"rom_exist"		return L_ROM_EXIST;
"etc_dir"		return L_ETC_DIR;
"ins_dir"		return L_INS_DIR;
"lib_dir"		return L_LIB_DIR;
polyphony		return L_POLYPHONY;
"source_init"		return L_SOURCE_INIT;
source			return L_SOURCE;
"source_option"		return L_SOURCE_OPTION;
"presource_init"	return L_PRELOAD_SOURCE_INIT;
presource		return L_PRELOAD_SOURCE;
"presource_option"	return L_PRELOAD_SOURCE_OPTION;
"external_init"		return L_EXTERNAL_INIT;
external		return L_EXTERNAL;
"external_option"	return L_EXTERNAL_OPTION;
exit			return L_EXIT;
"iw_ffff_rom"		return L_IW_FFFF_ROM;
"iw_ffff_file"		return L_IW_FFFF_FILE;
"iw_ffff"		return L_IW_FFFF;
item			return L_ITEM;
itemgm			return L_ITEMGM;
itemgs			return L_ITEMGS;
itemmt			return L_ITEMMT;
range			return L_RANGE;
rangegm			return L_RANGEGM;
rangegs			return L_RANGEGS;
rangemt			return L_RANGEMT;
melodic			return L_MELODIC;
drums			return L_DRUMS;
patches			return L_PATCHES;
preload			return L_PRELOAD;
option			return L_OPTION;
alias			return L_ALIAS;
aliasgm			return L_ALIASGM;
aliasgs			return L_ALIASGS;
aliasmt			return L_ALIASMT;
"alias_range"		return L_ALIAS_RANGE;
"alias_rangegm"		return L_ALIAS_RANGEGM;
"alias_rangegs"		return L_ALIAS_RANGEGS;
"alias_rangemt"		return L_ALIAS_RANGEMT;

	/* variables */

${1,2}{VARIABLE}+	{ 
			  if ( !gus_icfg_config -> unset_flag )
			    return decode_variable( yytext );
                           else
			    gus_icfg_lexer_variable_unset( yytext + (yytext[ 1 ] == '$' ? 2 : 1), yytext[ 1 ] == '$' ? GUS_ICFG_FVAR_NONE : GUS_ICFG_FVAR_GLOBAL );
			}
${1,2}{VARIABLE}+[ \t]+= {
			  int c, result;
			  char *str;

			  c = input();
			  if ( c == '=' ) {
                            str = strip_assign_char( strdup( yytext ) );
			    if ( !str ) return -1;
                            result = decode_variable( str );
			    free( str );
			    unput( '=' ); unput( '=' );
			    return result;
                          } else {
                            if ( c != EOF ) unput( c );
			  }
			  gus_icfg_lval.s_value = strip_assign_char( strdup( yytext + (yytext[ 1 ] == '$' ? 2 : 1) ) );
			  if ( !gus_icfg_lval.s_value ) return -1;
                          return yytext[ 1 ] == '$' ? L_LOCAL_VARIABLE_ASSIGN : L_VARIABLE_ASSIGN;
			}

	/* boolean */

true|yes|on		{ gus_icfg_lval.i_value = 1; return L_TRUE; }
false|no|off		{ gus_icfg_lval.i_value = 0; return L_FALSE; }

	/* integers */

[0-9]+			{ gus_icfg_lval.i_value = atoi( yytext ); return L_INTEGER; }
0x[0-9a-f]+		{ char *end;
                          gus_icfg_lval.i_value = strtol( yytext, &end, 0 );
                          return L_INTEGER; }

	/* strings */

\"[^\"]*\"		{ yytext[ strlen( yytext ) - 1 ] = 0;
			  gus_icfg_lval.s_value = strdup( &yytext[ 1 ] );
			  return L_STRING; }
\'[^\']*\'		{ yytext[ strlen( yytext ) - 1 ] = 0;
			  gus_icfg_lval.s_value = strdup( &yytext[ 1 ] );
			  return L_STRING; }

	/* comments & whitespaces */

"/*"			yy_push_state( comment );
<comment>[^*\n]*	/* eat anything that's not a '*' */
<comment>"*"+[^*/\n]*	/* eat up '*'s not followed by '/'s */
<comment>\n		++gus_icfg_config -> line_count;
<comment>"*"+"/"	yy_pop_state();

#\ [0-9]+[^\n]*		gus_icfg_config -> line_count = atoi( yytext + 2 ) - 2;

[ \t]+			;
\n			gus_icfg_config -> line_count++;
.			gus_icfg_error( 1, "discarding char '%c'", yytext[ 0 ] );

%%

#ifndef gus_icfg_wrap
int gus_icfg_wrap(void)	/* do this avoid to do -lfl */
{
  return 1;
}
#endif

void gus_icfg_lexer_init( int action )
{
  int i;
  char id[ 9 ];

#if 0
  gus_icfg_config -> pnp_flag = 0;
#endif
  gus_icfg_config -> action = action;
  gus_icfg_config -> line_count = 0;
  gus_icfg_config -> variables = NULL;
  include_stack_ptr = -1;
  strncpy( id, gus_icfg_config -> info_data.id, 8 );
  id[ 8 ] = 0;
  for ( i = 0; i < sizeof( id ); i++ )
    {
      unsigned char c = id[ i ];
      id[ i ] = c >= 'a' && c <= 'z' ? c - ( 'a' - 'A' ) : c;
    }
  gus_icfg_lexer_variable_set_string( "CARD_ID", id, GUS_ICFG_FVAR_GLOBAL );
  gus_icfg_lexer_variable_set_integer( "CARD", gus_icfg_config -> card + 1, GUS_ICFG_FVAR_GLOBAL );
  gus_icfg_lexer_variable_set_bool( "CHIP_UART", gus_icfg_config -> uart_flag != 0, GUS_ICFG_FVAR_GLOBAL );
  gus_icfg_lexer_variable_set_bool( "CHIP_INTERWAVE", gus_icfg_config -> pnp_flag != 0, GUS_ICFG_FVAR_GLOBAL );
  gus_icfg_lexer_variable_set_bool( "CHIP_GF1", gus_icfg_config -> pnp_flag == 0, GUS_ICFG_FVAR_GLOBAL );
  gus_icfg_lexer_variable_set_bool( "EMUL_GM", gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GM, GUS_ICFG_FVAR_GLOBAL );
  gus_icfg_lexer_variable_set_bool( "EMUL_GS", gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_GS, GUS_ICFG_FVAR_GLOBAL );
  gus_icfg_lexer_variable_set_bool( "EMUL_MT32", gus_icfg_config -> midi_emul == GUS_MIDI_EMUL_MT32, GUS_ICFG_FVAR_GLOBAL );
  if ( !gus_icfg_config -> uart_flag ) {
    gus_icfg_lexer_variable_set_integer( "RAM_SIZE", gus_icfg_config -> info_data.memory_size / 1024, GUS_ICFG_FVAR_GLOBAL );
    gus_icfg_lexer_variable_set_integer( "ROM_SIZE", gus_icfg_config -> info_data.memory_rom_size / 1024, GUS_ICFG_FVAR_GLOBAL );
    gus_icfg_lexer_variable_set_integer( "ROM_BANKS", gus_icfg_config -> info_data.memory_rom_banks, GUS_ICFG_FVAR_GLOBAL );
    gus_icfg_lexer_variable_set_integer( "ROM_PRESENT", gus_icfg_config -> info_data.memory_rom_present, GUS_ICFG_FVAR_GLOBAL );
  }
  gus_icfg_lexer_variable_set_bool( "INFO", action == GUS_ICFG_ACTION_INFO, GUS_ICFG_FVAR_GLOBAL );
  gus_icfg_lexer_variable_set_bool( "PRELOAD", action == GUS_ICFG_ACTION_PRELOAD, GUS_ICFG_FVAR_GLOBAL );
  if ( action == GUS_ICFG_ACTION_LOAD || action == GUS_ICFG_ACTION_PRELOAD ) {
    gus_icfg_lexer_variable_set_string( "SOURCE", action == GUS_ICFG_ACTION_LOAD ? gus_icfg_config -> load_source : gus_icfg_config -> preload_source, GUS_ICFG_FVAR_GLOBAL );
    gus_icfg_lexer_variable_set_string( "OPTIONS", action == GUS_ICFG_ACTION_LOAD ? gus_icfg_config -> load_options : gus_icfg_config -> preload_options, GUS_ICFG_FVAR_GLOBAL );
  }
}

void gus_icfg_lexer_done( void )
{
  struct gus_icfg_variable *pvariable, *nvariable;

  pvariable = gus_icfg_config -> variables;
  gus_icfg_config -> variables = NULL;
  while ( pvariable ) {
    nvariable = pvariable -> next;
    if ( pvariable -> name )
      free( pvariable -> name );
    if ( ( pvariable -> flags & GUS_ICFG_FVAR_STRING ) && pvariable -> data.s_value )
      free( pvariable -> data.s_value );
    free( pvariable );
    pvariable = nvariable;
  }
}

static char *strip_assign_char( char *str )
{
  int idx = 0;

  if ( !str ) {
    yyerror( "malloc error" );
    return NULL;
  }
  while ( str[ idx ] > ' ' && str[ idx ] != '=' ) idx++;
  str[ idx ] = 0;
  return str;
}

static int decode_variable( char *yytext )
{
  struct gus_icfg_variable *variable;

  variable = gus_icfg_lexer_variable_get( yytext + (yytext[ 1 ] == '$' ? 2 : 1), yytext[ 1 ] == '$' ? GUS_ICFG_FVAR_NONE : GUS_ICFG_FVAR_GLOBAL );
  if ( !variable ) {
    if ( !gus_icfg_config -> condition_true )
      return L_VARNONE;
    yyerror( "variable %s doesn't exist", yytext );
    return -1;
  }
  if ( variable -> flags & GUS_ICFG_FVAR_INTEGER ) {
    gus_icfg_lval.i_value = variable -> data.i_value;
    return L_INTEGER;
  }
  if ( variable -> flags & GUS_ICFG_FVAR_STRING ) {
    gus_icfg_lval.s_value = strdup( variable -> data.s_value );
    return L_STRING;
  }
  if ( variable -> flags & GUS_ICFG_FVAR_BOOL ) {
    gus_icfg_lval.i_value = variable -> data.i_value != 0;
    return L_BOOL;
  }
  yyerror( "unknown type of variable" );
  return -1;
}

static struct gus_icfg_variable *look_for_variable( const char *name, unsigned short flags )
{
  struct gus_icfg_variable *variable;

  flags &= GUS_ICFG_FVAR_GLOBAL;
  for ( variable = gus_icfg_config -> variables; variable; variable = variable -> next ) {
#ifdef DEBUG_VARIABLES
    printf( "variable = '%s', type = 0x%x, name = '%s', flags = 0x%x\n", variable -> name, variable -> flags, name, flags );
#endif
    if ( !strcmp( variable -> name, name ) && ( variable -> flags & GUS_ICFG_FVAR_GLOBAL ) == flags )
      return variable;
  }
  return NULL;
}

static struct gus_icfg_variable *alloc_variable( const char *name, unsigned short flags )
{
  struct gus_icfg_variable *variable;

  variable = (struct gus_icfg_variable *)calloc( 1, sizeof( struct gus_icfg_variable ) );
  if ( !variable ) {
    yyerror( "malloc error" );
    return NULL;
  }
  variable -> name = strdup( name );
  variable -> flags = flags;
  return variable;
}

static void append_variable( struct gus_icfg_variable *variable )
{
  struct gus_icfg_variable *xvariable;

  if ( !gus_icfg_config -> variables ) {
    gus_icfg_config -> variables = variable;
    return;
  }
  for ( xvariable = gus_icfg_config -> variables; xvariable -> next != NULL; xvariable = xvariable -> next );
  xvariable -> next = variable;
}

static int free_variable( const char *name, unsigned short flags )
{
  struct gus_icfg_variable *pvariable, *variable;

  pvariable = NULL;
  variable = gus_icfg_config -> variables;
  while ( variable ) {
    if ( !strcmp( variable -> name, name ) && variable -> flags == flags ) {
      if ( !pvariable ) {
        gus_icfg_config -> variables = variable -> next;
      } else {
        pvariable -> next = variable -> next;
      }
      if ( variable -> name )
        free( variable -> name );
      if ( ( variable -> flags & GUS_ICFG_FVAR_STRING ) && variable -> data.s_value )
        free( variable -> data.s_value );
      free( variable );
      return 0;
    }
    variable = variable -> next;
  }
  return -1;
}

void gus_icfg_lexer_variable_set_string( const char *name, char *value, unsigned short flags )
{
  struct gus_icfg_variable *variable;

  flags &= GUS_ICFG_FVAR_GLOBAL;
  flags |= GUS_ICFG_FVAR_STRING;
#ifdef DEBUG_VARIABLES
  printf( "set string: '%s' = '%s'\n", name, value );
#endif
  if ( ( variable = look_for_variable( name, flags ) ) == NULL ) {
    variable = alloc_variable( name, flags );
    if ( !variable ) return;
    variable -> data.s_value = strdup( value ? value : "" );
    append_variable( variable );
  } else {
    if ( variable -> flags & GUS_ICFG_FVAR_STRING )
      if ( variable -> data.s_value )
        free( variable -> data.s_value );
    variable -> flags = flags;
    variable -> data.s_value = strdup( value ? value : "" );
  }
}

void gus_icfg_lexer_variable_set_integer( const char *name, int value, unsigned short flags )
{
  struct gus_icfg_variable *variable;

  flags &= GUS_ICFG_FVAR_GLOBAL;
  flags |= GUS_ICFG_FVAR_INTEGER;
  if ( ( variable = look_for_variable( name, flags ) ) == NULL ) {
    variable = alloc_variable( name, flags );
    if ( !variable ) return;
    variable -> data.i_value = value;
    append_variable( variable );
  } else {
    if ( variable -> flags & GUS_ICFG_FVAR_STRING )
      if ( variable -> data.s_value )
        free( variable -> data.s_value );
    variable -> flags = flags;
    variable -> data.i_value = value;
  }  
}

void gus_icfg_lexer_variable_set_bool( const char *name, int value, unsigned short flags )
{
  struct gus_icfg_variable *variable;

  flags &= GUS_ICFG_FVAR_GLOBAL;
  flags |= GUS_ICFG_FVAR_BOOL;
  if ( ( variable = look_for_variable( name, flags ) ) == NULL ) {
    variable = alloc_variable( name, flags );
    if ( !variable ) return;
    variable -> data.i_value = value;
    append_variable( variable );
  } else {
    if ( variable -> flags & GUS_ICFG_FVAR_STRING )
      if ( variable -> data.s_value )
        free( variable -> data.s_value );
    variable -> flags = flags;
    variable -> data.i_value = value;
  }  
}

struct gus_icfg_variable *gus_icfg_lexer_variable_get( const char *name, unsigned short flags )
{
  flags &= (GUS_ICFG_FVAR_GLOBAL|GUS_ICFG_FVAR_TYPE);
  return look_for_variable( name, flags );
}

void gus_icfg_lexer_variable_unset( const char *name, unsigned short flags )
{
  flags &= (GUS_ICFG_FVAR_GLOBAL|GUS_ICFG_FVAR_TYPE);
  free_variable( name, flags );
}

void gus_icfg_lexer_include( char *filename )
{
  FILE *in;

  in = fopen( filename, "r" );
  if ( !in ) {
    yyerror( "Can't open include file '%s'...", filename );
    return;
  }
  if ( ++include_stack_ptr >= MAX_INCLUDE ) {
    yyerror( "Includes nested too deeply..." );
    return;
  }
  include_stack[ include_stack_ptr ] = YY_CURRENT_BUFFER;
  include_line_count[ include_stack_ptr ] = gus_icfg_config -> line_count;
  include_file[ include_stack_ptr ] = gus_icfg_config -> filename;
  gus_icfg_config -> line_count = 0;
  gus_icfg_config -> filename = filename;  
  yyin = in;
  yy_switch_to_buffer( yy_create_buffer( yyin, YY_BUF_SIZE ) );
}
