// request.c

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#pragma implementation "reqdelegate.h"
#endif

#include <String.h>
#include "localdefs.h"
#include "query_templates.h"
#include "request.h"
#include "reqdelegate.h"

QueryLink::QueryLink(QueryItem *q) {
	element = q;
	element->ref();
	next = nil;
}

QueryLink::~QueryLink() { Resource::unref(element); }


QueryList::QueryList(const char *label) {
	listLabel = new String(label ? label : "");
	head = nil;
}

QueryList::~QueryList() {
	delete listLabel;
	start();
	while(current) {
		register QueryLink *next = current->next;
		delete current;
		current = next;
	}
}

QueryLink *
QueryList::tail() {
	start();
	while(current && current->next)
		current = current->next;
	return current;
}
	
void
QueryList::append(QueryItem *q) {
	register QueryLink *link = new QueryLink(q);
	if(!head)
		head = link;
	else
		tail()->next = link;
	current = link;
}

const char *
QueryList::label() { return *listLabel; }

////////

Request::Request(const QueryInfo *qinfo, RequestDelegate *delegate)
		: myDelegate(delegate) {
	init();
	if(qinfo && qinfo->labelInfo)
		createLabelList(qinfo->labelInfo);
	if(qinfo && qinfo->buttonInfo)
		createButtonList(qinfo->buttonInfo, qinfo->defaultResponse);
	else
		createButtonList(defaultInputButtonInfo, Yes);
}

Request::Request(const char *label, RequestDelegate *delegate)
		: myDelegate(delegate) {
	init();
	if(label) appendLabel(label);
	createButtonList(defaultInputButtonInfo, Yes);
}

void
Request::init() {
	labelList = nil;
	browserQuery = nil;
	valueList = nil;
	choiceList = nil;
	buttonList = nil;
	bell = false;
}

Request::~Request() {
	delete labelList;
	delete browserQuery;
	delete valueList;
	delete choiceList;
	delete buttonList;
	delete myDelegate;
}

void
Request::newLabelList() {
	if(!labelList) labelList = new QueryList;
}

void
Request::newValueList() {
	if(!valueList) valueList = new ValueList;
}

void
Request::newButtonList() {
	if(!buttonList) buttonList = new ButtonList;
}

void
Request::appendLabel(const char *lbl) {
	newLabelList();
	labelList->append(new QueryItem(lbl));
}

void
Request::appendLabel(const char *lblbeg, const char *lblend) {
	newLabelList();
	labelList->append(new QueryItem(lblbeg, lblend));
}

void
Request::addFileBrowser(String* path, const char* suffixes) {
	delete browserQuery;			// only one at a time allowed
	browserQuery = new QueryFile(path, suffixes);
}

void
Request::appendButton(const char *lbl, Response response) {
	newButtonList();
	buttonList->append(new QueryButton(lbl, response));
}

void
Request::appendValue(QueryValue* qv) {
	newValueList();
	valueList->append(qv);
}

void
Request::appendValue(const char *label, String* value) {
	appendValue(createStringQuery(label, value));
}

void
Request::appendValue(const char *label, int* value,
		     const Range& bounds, boolean show) {
	appendValue(createQuery(label, value, bounds, show));
}

void
Request::appendValue(const char *label, unsigned short* value,
		     const Range& bounds, boolean show) {
	appendValue(createQuery(label, value, bounds, show));
}

void
Request::appendValue(const char *label, float* value,
		     const Range& bounds, boolean show) {
	appendValue(createQuery(label, value, bounds, show));
}

void
Request::appendValue(const char *label, double* value,
		     const Range& bounds, boolean show) {
	appendValue(createQuery(label, value, bounds, show));
}

void
Request::appendChoice(const char *label, const char *list,
		ChoiceValue* states, boolean excl) {
	if(!choiceList) choiceList = new ChoiceList;
	choiceList->append(new QueryChoice(label, list, states, excl));
}

void
Request::createLabelList(QueryLabelInfo *qlist) {
	for(QueryLabelInfo *i = qlist; i->label != nil; i++) {
		appendLabel(i->label);
	}
}

void
Request::createButtonList(QueryButtonInfo *qbilist, Response deflt) {
	buttonList = new ButtonList(nil, deflt);
	for(QueryButtonInfo *i = qbilist; i->label != nil; i++) {
		appendButton(i->label, i->response);
	}
}

boolean
Request::print(FILE *out) {
	boolean status = true;
	if(hasLabels()) {
		for(labelList->start(); labelList->more(); labelList->next())
			fprintf(out, "%s\n", labelList->item()->label());
	}
	if(hasValues() || hasChoices()) {
		fprintf(out, "\nThis dialog cannot be viewed on a tty yet,\n");
		fprintf(out, "so this operation will return <CANCEL>.\n\n");
		status = false;
	}
	fflush(out);
	return status;
}

boolean
Request::checkValues() {
	return myDelegate ? myDelegate->checkValues(*this) : true;
}

//**********

MessageRequest::MessageRequest(const char *msg, const char *msg2, 
		const char *msg3) {
	if(msg) appendLabel(msg);
	if(msg2) appendLabel(msg2);
	if(msg3) appendLabel(msg3);
}

//**********

AlertRequest::AlertRequest(const char* m1, const char* m2, const char* m3,
	const char* btn) : MessageRequest(m1, m2, m3) { init(btn); }

void
AlertRequest::init(const char* btn) {
	appendButton(btn, Yes);
	setBell(true);
}

//******

ConfirmRequest::ConfirmRequest(const char *m1, const char *m2, const char *m3,
	  Response r, const char* b1, const char* b2) : MessageRequest(m1, m2, m3) {
	init(r, b1, b2);
}

void
ConfirmRequest::init(Response r, const char* b1, const char* b2) {
	buttonList = new ButtonList(nil, r);
	appendButton(b1, Yes);
	appendButton(b2, Cancel);
	setBell(true);
}

//******

ChoiceRequest::ChoiceRequest(const char *m1, const char *m2, const char *m3,
	Response r, const char* b1, const char* b2, const char* b3)
		: MessageRequest(m1, m2, m3) {
	init(r, b1, b2, b3);
}

void
ChoiceRequest::init(Response r, const char* b1, const char* b2, const char* b3)
{
	buttonList = new ButtonList(nil, r);
	appendButton(b1, Yes);
	appendButton(b2, No);
	appendButton(b3, Cancel);
	setBell(true);
}

//******

FileRequest::FileRequest(const char* title, String* directory,
		const char* suffixes, RequestDelegate *delegate)
		: Request(title, delegate) {
	addFileBrowser(directory, suffixes);
}

//******

QueryButtonInfo defaultInputButtonInfo[] = {
	{ "confirm", Yes },
	{ "cancel", Cancel },
	{ nil }
};

