// data.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// Data is the base class for all objects which represent on-disk data files.
// It is primarily a container for a pointer to a DataRep subclass, which in
// turn is the container for the actual Array of raw binary information.  Most
// of the methods defined for the Data class pass the method call on to the
// DataRep member (see datarep.h)

// Data and Data subclass objects use an extremely important feature called
// 'cloning'.  In MXV, cloning is a "shallow copy" method with the additional 
// capability of allowing a "shallow copy" of a selection portion (samples and
// channels) of the original.  Modifications performed upon the clone are 
// modifications of that same portion of the original object -- the underlying
// sample data is shared between all clones of a Data instance.  Most
// operations upon samples is done via clones which are created and destroyed
// for each operation -- very little overhead is used for this.

#ifndef DATA_H
#ifdef __GNUG__
#pragma interface
#endif
#define DATA_H

#include <InterViews/resource.h>
#include "range.h"
#include "datarep.h"
#include "scale.h"
#include "localdefs.h"

class Header;
class DataFile;
class Range;
class Envelope;
class DataState;
class Comment;
class Interactor;
class SimpleFunction;
class OutputFunction;
class InputOutputFunction;
class FrameModifier;

class Data : public Resource {
public:
	Data();
	Data(DataType type, int length, int chans=1);
	virtual ~Data();
	virtual Data* newData();
	virtual Data* newData(int length);
	virtual Data* clone(const Range& selection);
	virtual Data* clone(const Range& selection, const Range& chans);
	virtual Data* cloneFrame(int frameno);
	virtual Data* cloneFrame(int frameno, const Range& chans);
	
	// true if two objects are related by cloning
	boolean operator == (const Data &rhs) const;
	boolean operator != (const Data &rhs) const;
	double get(int pos, int chan=0) const;		// non-interpolating
	double get(double pos, int chan=0) const;	// interpolating
	void set(double val, int pos, int chan=0);

	// these deal with all elements of any "row" (channel) of data
	// return value:  number of elements retrieved or set
	virtual int getArray(int *array, int size, int channel=0) const;
	virtual int getArray(float *array, int size, int channel=0) const;
	virtual int getArray(double *array, int size, int channel=0) const;
	virtual int setFromArray(float *a, int size, int channel=0);
	virtual int setFromArray(double *a, int size, int channel=0);

	// these deal with all elements of any "column" (frame) of data
	virtual void getFrame(Data *frame, int frameno=0, int chanOffset=0) const;
	virtual void getFrame(float *array, int arraySize, int frameno=0) const;
	virtual void getFrame(double *array, int arraySize, int frameno=0) const;
	virtual void setFrame(const Data *frame, int frameno=0);
	virtual void setFrame(float *array, int arraySize, int frameno=0);
	virtual void setFrame(double *array, int arraySize, int frameno=0);

	Data* copyOf();
	virtual void add(const Data *src);
	virtual void copyFrom(const Data *src);
	virtual void copyRescaledFrom(const Data *src, double scalFac, boolean dither=true);
	virtual int replaceWith(const Data *source);
	virtual void crossfade(const Data* src, Envelope* evp);
	virtual int spliceIn(const Range& selection); 
	virtual int spliceOut(const Range& selection);

	virtual void reverse();
	virtual void erase();
	virtual void normalize();
	virtual int changeLength(int newlen);
	virtual int changeSRate(int, boolean=false) { return false; }
	virtual int apply(SimpleFunction &);
	virtual int apply(OutputFunction &);
	virtual int apply(FrameModifier &);

	virtual void scaleSelf(double);

	virtual int channels() const;
	virtual int length() const;				// number of frames
	virtual int sRate() const;				// sampling rate
	virtual double frameRate() const;		// frames per second
	virtual int frameChannels() const;		// no. of channels per real frame
	virtual int frameLength() const;		// no. of samples per frame
	int dataSize() const;					// size of each sample in bytes
	DataType dataType() const;				// type of data (shorts, floats)
	const char* format() const;				// string (e.g., "float")
	int sizeInBytes() const;				// total size of data in bytes
	virtual FileType fileType () const;		// identifier for use with isA()
	boolean isA(FileType type) const;
	virtual const char* fileSuffix() const;	// suffix to use for disk files
	virtual double maxValue() const;		// max data value
	virtual void maxValue(int *chan, int *loc) const;	// location of same
	double meanValue() const;		// mean data value
	int zeroCrossing() const;		// returns offset to next z-cross
	int findSlopeChange(double percent) const;
	double valueLimit() const;		// largest possible value
	virtual Range limits(int chan=0, boolean real=false) const;	// min,max for chan
	int getEnvelope(Data *, int channel=0,
		EnvelopeType etype=PeakToPeakAmplitude) const;

	virtual void print(FILE *) { /* base class does nothing */ }
	virtual Header *createHeader(DataFile *, boolean reading=false);
	virtual int write(DataFile *f);
	virtual int read(DataFile* f, Header* header = nil);

	virtual void deferUpdate(boolean flag);
	static boolean deferRescan(boolean b);
	void modified(boolean b);
	boolean modified() const;
	
	// methods defined to allow Data to act as if it were a Subject 
	virtual void Attach(Interactor*);
	virtual void Detach(Interactor*);
	virtual void Notify();
	int currentViews() const;
	
	virtual const char* comment();
	virtual void setComment(const char* comm);

	// information methods for display objects
	virtual const char* windowClassName() const; 
	virtual const char* horizontalScaleModeAttribute() const;
	virtual const char* channelDisplayAttribute() const;
	virtual const char* defaultDirAttribute() const;
	virtual boolean displayAsFrames() const;
	virtual boolean displayAsFrames(boolean flag);
	virtual Range frameRange(RangeUnit units=FrameUnit) const;
	virtual const char* frameRangeLabel(RangeUnit units) const;
	virtual Scale::NumberDisplay frameRangeDisplay(RangeUnit units) const;
	virtual Range channelRange(boolean asFrame=false) const;
	virtual const char* channelName(int chan) const;
	virtual void information(class Controller *) { /* base does nothing */ }
protected:
	Data(const Data *);							// for virtual ctors
	Data(const Data *, int newlen);
	Data(const Data *, const Range &);			// special ctors for cloning
	Data(const Data *, const Range &, const Range &);
	DataState *state() const { return myState; }

	virtual int getArray(double *a, int size, int ch, int offset) const;
	virtual int setFromArray(float *a, int size, int ch, int offset);
	virtual int setFromArray(double *a, int size, int ch, int offset);

	virtual int write(DataFile *f, Header* header);
	virtual void readFromHeader(Header *);
	virtual void configureHeader(Header *);
	virtual void checkValues();
	virtual double scanForMaxValue() const;
	virtual void setMaxValue(double);
	virtual int changeDataType(DataType);
	virtual void changeNChans(int newchans);
	
	int giveSamplesTo(InputOutputFunction &) const;
protected:
	DataRep *rep;					// pointer to virtual data object
	DataState *myState;				// state contains all shared information
	Comment *myComment;				// information string
	static boolean scanDeferred;	// flag for delaying peak scans, etc.
private:
	virtual Data& operator =(const Data &) { return *this; };	// not used
	Data(const Data&);				// not used or defined
	void Init();
	friend class InputOutputFunction;
	friend class ArrayFunction;
};

inline boolean 
Data::operator == (const Data &rhs) const { return *rhs.rep == *rep; }

inline boolean
Data::operator != (const Data &rhs) const { return *rhs.rep != *rep; }

inline void
Data::set(double val, int pos, int chan) { rep->set(val, pos, chan); }

inline double
Data::get(int pos, int chan) const { return rep->get(pos, chan); }

inline double
Data::get(double pos, int chan) const { return rep->get(pos, chan); }

inline int
Data::setFromArray(float *a, int size, int channel) {
	return setFromArray(a, size, channel, 0);
}

inline int
Data::setFromArray(double *a, int size, int channel) {
	return setFromArray(a, size, channel, 0);
}

inline double
Data::valueLimit() const { return rep->valueLimit(); }

inline int
Data::channels() const { return rep->channels(); }

inline int
Data::length() const { return rep->length(); }

inline int
Data::sRate() const { return 0; }

inline double
Data::frameRate() const { return sRate(); }

inline int
Data::frameChannels() const { return 1; }

inline int
Data::frameLength() const { return channels() / frameChannels(); }

inline int
Data::dataSize() const { return rep->dataSize(); }

inline DataType
Data::dataType() const { return rep->dataType(); }

inline int
Data::sizeInBytes() const { return rep->sizeInBytes(); }

inline FileType
Data::fileType () const { return Generic_Data; }

inline boolean
Data::isA(FileType type) const { return fileType() == type; }

inline const char*
Data::fileSuffix() const { return ".data"; }

inline Scale::NumberDisplay
Data::frameRangeDisplay(RangeUnit) const { return Scale::AsInteger; }

inline const char*
Data::windowClassName() const { return "DataWindow"; }

inline const char*
Data::horizontalScaleModeAttribute() const { return "HorizontalScaleMode"; }

inline const char*
Data::channelDisplayAttribute() const { return "DataChannels"; }

inline const char*
Data::defaultDirAttribute() const { return "DefaultFileDir"; }

#endif /* DATA_H */
