# Module: MiscTools.tcl
# 11.2.96 T.Niederreiter
#
# External called funktions:
# - createglobalvars {}: Initialize all global variables
# 
# - spawnsubwindow { call new old }: creates a new toplevel window
#
# - createsubwindow { new old } : create new window and return
#
# - destroysubwindow { new old oldfocus } : destroy window
#
# - backupglobals{}: backup all XCDR* variables in BAK_XCDR* variables 
#
# - restoreglobals{}: restores the XCDR* values
#
# - convertarray2string { name }: Convert array to string 
#
# - checkscsisetup {}: Check if SCSI-Setup changed since last time 
#
# - isiso{ dev }: Test if ISO9660 
#
# - domount { mntdev mntdir }: Mount a device 
#
# - domountext2 { mntdev mntdir }: Mount a ext2-device 
#
# - doumount { mntdev }: Umount a device
# 
# - checkcd {}: Check if cd is in drive
#
# - getisolabel { dev }: Get ISO9660-Volume-Label 
#
# - getisosize { dev }: Get size of an ISO9660-Image
# 
# - getaudiosize { file }: Get length of an audio-file
#
# - getmountpnt { dev }: Get the mountpoint of a device
#
# - checkmountpnt { dir }: check if this dir is busy 
# 
# - getfreeimgspace { }: Get the free space on the Image-Partition 
#
# - doformat { dev}: Format a partition
#
# - loadfile2list { fname }: Load a text-file to list 
#
# - extractcolonstr { str }: Extract substr after colon 
#
# - getdircontents { dir }: Get directory-contents 
#
# - resetcd { dev }: Reset CD-Rom
#
# - calcdu { dir }: Calculates Disk-Usage
#
# - getimagecontents { file }: Get root-dir of an iso-image
#
# - checkifready { w }: Prompt user for CD
# 
# - checkifyamaha { }: Check if writer-type is yamaha 
#
# - checkifsony { }: Check if writer-type is sony 
#
# - setEndianorder {}: Set endian-order 
#
# - checkifiso { dev }: Check if valid ISO 
#
# - checkifaudio { }: Check if audio-cd 
#
# - checkifmixedmode { }: Check if mixed-mode-cd 
#
# - getnonisosize { }: Get size of image on non-iso-cd
# 
# - createbeep { num }: Create a beeping noise from the speaker
#
# - sound { type }: Sound-Handler 
#
# - log { w str }: Logfile-Handler 
#
# - loadwinpos { $widget }: Load coords of a window 
#
# - savewinpos { $widget $xcord $ycord }:  Save coords of a window
#
# - getcdrecorddrivers { }: get available cdr-drivers from cdrecord
#
# - chownfile { file }: change the owner of files
#
# - getchildstatus { status }: get exit-code of child-process 
#
# - fixstring { var }: remove " ] [ from strings
#
# - checksgmodule { }: check if sg-module is loaded
# 
# - checkscsipatch { }: check if the cdrecord kernel patch is installed 
#
# - stripstring { str }: strip multiple spaces from strings
# 
# - if_printable { string }: check if string is printable 
#

# Defines all global variables needed
# This is done by setting all variables in the "varlist" to "" in global
# context.

proc createglobalvars {} {

	set varlist "scsidevices(0,0) partinfo(0,0) XCDR_scsidevices \
		XCDR_partinfo XCDR_AUDIOREAD_SPEED XCDR_BEEP \
		XCDR_DEST_WRITER_DEV XCDR_DEST_WRITER_MODE \
		XCDR_DEST_WRITER_SPEED XCDR_IMAGE_FIXED XCDR_BEEP_VIA \
		XCDR_IMAGE_MNTPNT2 XCDR_IMAGE_PART2 XCDR_DEFAULT_PART \
		XCDR_DSP_DEV XCDR_IMAGE_MNTPNT XCDR_IMAGE_PART \
		XCDR_LOCKPART XCDR_LOGFILE XCDR_LOGNAME XCDR_DATA_SOURCE_CDROM \
		forcesetup ENDIANSWAP XCDR_AUDIO_SOURCE_CDROM \
		DUMMY EJECT PADDATA percent NONISO XCDR_AUDIOREAD_MODE \
		idedevices(0,0) diskinfo(0,0) cdinfo(0,0) \
		TMP_XCDR_DEFAULT_PART IMAGE_PART IMAGE_MNTPNT IMAGE_LABEL \
		XCDR_SUB_VIEW XCDR_EJECT XCDR_PADDATA XCDR_DUMMY SUB_VIEW \
		XCDR_GLOBAL_OFFSETS XCDR_SKIP2SEC XCDR_AUDIO_UNITS \
		XCDR_AUTORAISE TMP_INSERT_HOOK MSTR_DIR MSTR_EXPATHLIST \
		MSTR_EXGLOBLIST MSTR_TYPE MSTR_VOLID MSTR_PUBID MSTR_PREID \
		MSTR_APPID XCDR_idedevices DISCLAIMACCEPTED XCDR_ECHOLISTEN \
		XCDR_SAVEWINPOS XCDR_DISCLAIMERDONTSHOWAGAIN SWABAUDIO \
		XCDR_SWABAUDIO" 

	global i
	foreach i $varlist {
		uplevel #0 { eval set $i \"\" }
	} 
	unset i
}


# Creates a new toplevel window, sets the grab on it, calls the
# function that fills the new new window, waits until it terminates
# and returns. Claims also the focus.
# Also sets a "watch"-cursor in the window of the caller
#
# Parameters: call = window-procedure to call. The procedure must
#		     have one argument, that is the window-path.
#	      new  = path of new window
#             old  = path of old window (the window of the caller)
#	      raise = 0 (ignore autoraise), 1 (level 1), 2 (level 2)
#	      addparm = Additional paramters passed to funktion

proc spawnsubwindow { call new old raise addparm } {
global XCDR_AUTORAISE
global CDRICO

	# get geometry of parent
	set oldsx [winfo width $old]
	set oldsy [winfo height $old]
	set oldpx [winfo x $old]
	set oldpy [winfo y $old]
	set centerx [expr $oldpx + ($oldsx / 2)]
	set centery [expr $oldpy + ($oldsy / 2)]
 
	$old configure -cursor watch
	catch { destroy $new }
	if { $XCDR_AUTORAISE != 0 && $raise != 0 && \
		$XCDR_AUTORAISE >= $raise } {
		wm iconify $old
	}
	toplevel $new
	wm title $new "X-CD-Roast"
	wm iconbitmap $new @$CDRICO
	wm geometry $new +[expr $oldpx + 200]+[expr $oldpy + 100]
	grab $new 
	set oldFocus [focus]
	focus $new
	$call $new $addparm

	# Here we need a catch, in the case a window is spawned and 
	# destroyed before we can wait for user input. Without
	# catch this results in an error.
	catch { tkwait window $new }

	catch { focus $oldFocus }
	$old configure -cursor ""

	update

	if { $XCDR_AUTORAISE != 0 && $raise != 0 && \
		$XCDR_AUTORAISE >= $raise } {
		wm deiconify $old
	}
}


# create a new toplevel-window and return

proc createsubwindow { new old raise title } {
global XCDR_AUTORAISE
global XCDR_SAVEWINPOS
global CDRICO

	set newpos ""

	# if title not set, create default title
	if { $title == "" } { 
		set title "X-CD-Roast" 	
	} else {
		# workaround - dont load position for audio-play windows
		if { [string range $title 0 3] == "Play" } {
			set dontload 1
		} else {
			set dontload 0
		}

		# only load window-positions when not default title
		# and requested by user
		if { $XCDR_SAVEWINPOS == 1 && $dontload == 0 } {
			set newpos [loadwinpos $new]
		}
		# use the same screen-position for all windows
		if { $XCDR_SAVEWINPOS == 2 && $dontload == 0 } {
			set newpos [loadwinpos .genericposition]
		}
	}

	# get geometry of parent
	set oldsx [winfo width $old]
	set oldsy [winfo height $old]
	set oldpx [winfo x $old]
	set oldpy [winfo y $old]
	set centerx [expr $oldpx + ($oldsx / 2)]
	set centery [expr $oldpy + ($oldsy / 2)]
 
	$old configure -cursor watch
	catch { destroy $new }
	if { $XCDR_AUTORAISE != 0 && $raise != 0 && \
		$XCDR_AUTORAISE >= $raise } {
		wm iconify $old
	}
	toplevel $new
	wm title $new $title 
	wm iconbitmap $new @$CDRICO
	if { $newpos == "" || $newpos == -1 } {
		wm geometry $new +[expr $oldpx + 200]+[expr $oldpy + 100]
	} else {
		# use saved position
		set poslist [split $newpos ","]
		set newx [lindex $poslist 0]
		set newy [lindex $poslist 1]
		wm geometry $new +$newx+$newy
	}
	
	catch { grab $new }
	set oldFocus [focus]
	focus $new

	return $oldFocus
}


# destroy a window created with createsubwindow
# and save its coordinates in the window-position-file

proc destroysubwindow { new old oldFocus raise } {
global XCDR_AUTORAISE
global XCDR_SAVEWINPOS

	# save window-position if requested
	if { $XCDR_SAVEWINPOS != 0 } {
		set newgeolist [split [wm geometry $new] "+"]
		set newpx [lindex $newgeolist 1]
		set newpy [lindex $newgeolist 2]
		set title [wm title $new]
		# save only when title not default title
		if { $title != "X-CD-Roast" \
		     && [string range $title 0 3] != "Play" } { 
			if { $XCDR_SAVEWINPOS == 1 } {
				savewinpos $new $newpx $newpy
			} else {
				savewinpos .genericposition $newpx $newpy
			}
		}
	}

	catch { destroy $new }
	catch { focus $oldFocus }
	$old configure -cursor ""

	update

	if { $XCDR_AUTORAISE != 0 && $raise != 0 && \
		$XCDR_AUTORAISE >= $raise } {
		wm deiconify $old
	}
}


# Copies all XCDR* global variables in global BAK_XCDR* variables.
# Used by the Setup-Menu to save all variables before the user changes
# them. (and wants to reactive the old settings afterwards)
 
proc backupglobals {} {
eval global [info globals XCDR*]
global varname varvalue
global DEBUG

	set varlist [lsort [info globals XCDR*]]

	if { $DEBUG > 1 } {
		puts "backupglobals: XCDR_scsidevices = $XCDR_scsidevices" 
		puts "backupglobals: XCDR_idedevices = $XCDR_idedevices"
	}

	foreach varname $varlist {
		eval set varvalue \$$varname
		fixstring varvalue
		uplevel #0 { eval set BAK_$varname \"$varvalue\" }
	}
	unset varname varvalue
}


# Restores the values from the BAK_XCDR* variables back to the XCDR* 
# variables. 

proc restoreglobals {} {
eval global [info globals BAK_XCDR*]
global newvarname varvalue

	set varlist [lsort [info globals BAK_XCDR*]]

	foreach varname $varlist {
		eval set varvalue \$$varname
		set newvarname [string range $varname 4 end] 
		catch { uplevel #0 { eval set $newvarname \"$varvalue\" } }
		# Explaination for the catch here:
		# When a variable holds an illegal value (e.g. not in option-
		# list of an tixOptionMenu) then we simply ignore the error,
		# which results automatically in a valid value.
	}
	unset newvarname varvalue
}


# Convert any array to a string. This makes it easier to save arrays in
# textfiles or to compare arrays. But it is not allowed to have the 
# pipe-char "|" inside any array-element, because it is used for internal
# seperation.

proc convertarray2string { name } {

	# Call-by-Reference: Get the array via the upvar-command
	upvar $name arr

	set elelist [lsort [array names arr]]
	set outstr ""

	foreach i $elelist {
		append outstr "|$i|$arr($i)"		
	}

	return $outstr
}


# special version to remove the mount-point from the partinfo-array
# This way X-CD-roast wont complain at startup when a partition is not
# mounted where it was last time

proc convertarray2string_partinfo { name } {

	# Call-by-Reference: Get the array via the upvar-command
	upvar $name arr

	set elelist [lsort [array names arr]]
	set outstr ""

	foreach i $elelist {
		# every 4th element of partinfo is to be skipped
		if { [lindex [split $i ","] 1] == 4 } {
			append outstr "|$i|-"		
		} else {
			append outstr "|$i|$arr($i)"		
		}
	}

	return $outstr
}

# Check if the SCSI-Bus and Partitions have been changed since last
# time in the setup-menu.
# Return-values: 0 = No Changes, setup-values are still valid
#                1 = SCSI-Bus changed only, better check setup-menu
#		 2 = IDE changed, 3 = scsi + ide changed
#		 4 = Partitions changed only, could be quite dangerous
#                    when your image-partition is now wrong. setup new!
#		 5,6 = Both SCSI,IDE and Partition changed
 
proc checkscsisetup { } {
global XCDR_scsidevices
global XCDR_idedevices
global XCDR_partinfo
global scsidevices
global idedevices
global partinfo

	# Convert scsidevices and partinfo-arrays to strings
	set NEW_scsidevices [convertarray2string scsidevices]
	set NEW_idedevices [convertarray2string idedevices]
 	set NEW_partinfo [convertarray2string_partinfo partinfo]

	set status 0

	# We have now the NEW_* variables, which contain the hardware
	# info scanned now and the XCDR_variables which contain the
	# hardware info saved last time in the setup-menu.

	if { $NEW_scsidevices != $XCDR_scsidevices } {
		incr status
	}
	if { $NEW_idedevices != $XCDR_idedevices } {
		incr status 2
	}
	if { $NEW_partinfo != $XCDR_partinfo } {
		incr status 4
	}
	
	# Now copy the new values into the XCDR-variables, so that they
	# will saved when the user clicks on "save" in the setup-menu.
	set XCDR_scsidevices $NEW_scsidevices
	set XCDR_idedevices $NEW_idedevices
	set XCDR_partinfo $NEW_partinfo

	return $status
}


# Checks if the device "dev" contains an ISO9660-FS.
# Return 1 if ISO, 0 if not.

proc isiso { dev } {
global ISODETECT
	
	resetcd $dev
	set isoout ""
	catch { set isoout [exec $ISODETECT -d $dev] }
	if { $isoout == "ISO9660" } {
		return 1
	} else {
		return 0
	}
}

# Mounts a device to a directory.
# Updates also the partinfo-array.
# Return 1 for success, 0 for failure.

proc domount { mntdev mntdir } {
global MOUNT
global partinfo

	set mntout [catch { exec $MOUNT $mntdev $mntdir }]

	set i 0
	# Mount was ok?
	if { $mntout == 0 } {
		# Update partinfo-array
		while { $partinfo($i,0) != "x" } {
			if { $partinfo($i,0) == $mntdev } {
				set partinfo($i,4) $mntdir
			}
			incr i
		}
		return 1
	} else {
		# Mount failed
		return 0
	}
}
 
# Mounts a device to a directory. It must be a ext2-fs on it.
# Updates also the partinfo-array.
# Return 1 for success, 0 for failure, 2 when dir busy.

proc domountext2 { mntdev mntdir } {
global MOUNT
global partinfo

	set outline ""

	#set mntout [catch { exec $MOUNT -t ext2 $mntdev $mntdir }]

	# this is the only way I know to get the stderr-output
	set cmd [execshell "$MOUNT -t ext2 $mntdev $mntdir"]

	set pipe [open "|$cmd" r+]
	while { [gets $pipe line] >= 0 } {
		set outline $line		
	}
	set mntout [catch { close $pipe }]

	set i 0
	# Mount was ok?
	if { $mntout == 0 } {
		# Update partinfo-array
		while { $partinfo($i,0) != "x" } {
			if { $partinfo($i,0) == $mntdev } {
				set partinfo($i,4) $mntdir
			}
			incr i
		}
		return 1
	} else {
		# Mount failed
		# lets look for the reason
		if { [string first "busy" $outline] != -1 } {
			# looks like that this dir is busy
			return 2
		}
		return 0
	}
}
 

# Umounts a device .
# Updates also the partinfo-array.
# Return 1 for success, 0 for failure.

proc doumount { mntdev } {
global UMOUNT
global partinfo

	set mntout [catch { exec $UMOUNT $mntdev }]

	set i 0
	# umount was ok?
	if { $mntout == 0 } {
		# Update partinfo-array
		while { $partinfo($i,0) != "x" } {
			if { $partinfo($i,0) == $mntdev } {
				set partinfo($i,4) "" 
			}
			incr i
		}
		return 1
	} else {
		# umount failed
		return 0
	}
}
 

# get the blockdevice of a CD-Device
# mode can be "data" or "audio" -> to find out in which drive to look

proc convCD2block { mode } {
global XCDR_DATA_SOURCE_CDROM
global XCDR_AUDIO_SOURCE_CDROM

	if { $mode == "data" } {
 		set blkname [convertCDnametoblkdevice $XCDR_DATA_SOURCE_CDROM]
	} else {
 	     	set blkname [convertCDnametoblkdevice $XCDR_AUDIO_SOURCE_CDROM]
	}

	return $blkname
}


# Checks if the CD is loaded
# Return 1 if loaded, 0 if not

proc docheckcd { mode } {

	set chkout [checkcd [convCD2block $mode]]

	if { $chkout == 0 } {
		# Medium is loaded and ready
		return 1
	} else {
		# Medium not loaded
		return 0
	}
}


# Read the ISO9660-Volume-Label from device

proc getisolabel { dev } {
global ISODETECT

	set isoout ""
	catch { set isoout [ exec $ISODETECT -V -d $dev ] }

	return $isoout
}


# Get the Size of an ISO9660-Image
# Return 0 if it is not iso9660.

proc getisosize { dev } {
global ISODETECT
global ISOSIZE

	resetcd $dev
	set isoout ""
	catch { set isoout [exec $ISODETECT -d $dev] }
	if { $isoout == "ISO9660" } {
		return [ exec $ISOSIZE $dev ]
	}
	return 0
}


# Gets the size (in min:sec.centi_sec-format) of a file

proc getaudiosize { filename } {

	set filesize [ file size $filename ]
	set frames [expr $filesize/2352]

	return [convertframes2time $frames]
}


# Get the mountpoint of the device $dev. If it is not mounted return
# the empty string.

proc getmountpnt { dev } {
global partinfo

	set i 0
	set mntpnt ""

       	# Check if dev is mounted
      	while { $partinfo($i,0) != "x" } {
        	if { $partinfo($i,0) == $dev } {
                	set mntpnt $partinfo($i,4)
             	}
            	incr i
	}

	return $mntpnt
}


# Checks if this directory is already an active mountpoint
# returns the device that is mounted there or the empty string

proc checkmountpnt { dir } {
global partinfo

	set i 0
	set dev ""

       	# Check if dev is mounted
      	while { $partinfo($i,0) != "x" } {
        	if { $partinfo($i,4) == $dir } {
			set dev $partinfo($i,0)
             	}
            	incr i
	}

	return $dev;
}


# Gets the free space on the Image-Partition. If the partition is mounted
# we parse the output of the df-command, and if it is not mounted we 
# return the size of the partition (from partition-table)

proc getfreeimgspace { } {
global DF
global IMAGE_PART
global IMAGE_MNTPNT
global partinfo

	set i 0
	set freespace 0

	if { $IMAGE_PART != "" } {
		set mntpnt [ getmountpnt $IMAGE_PART ] 
	} else {
		set mntpnt $IMAGE_MNTPNT
	}

	# Partition is mounted
	if { $mntpnt != "" } {
		# Get output of df-command
		set out [ exec $DF -Pk $mntpnt ]
		# Split the output to lines
		set out2 [ split $out "\n" ]
		# Get the second line of that output and split it in words
		set out3 [ split [lindex $out2 1] ] 
		set out4 {}
		
		# Delete all empty list-elements
		foreach i $out3 {
			if { $i != {} } {
				lappend out4 $i
			}	
		}

		set freespace [ expr [lindex $out4 3]/1024 ]

	} else {
	# Partition is not mounted

		set i 0
		# Search in Partition-table 
		while { $partinfo($i,0) != "x" } {
			if { $partinfo($i,0) == $IMAGE_PART } {
				set freespace [expr $partinfo($i,1) /1024 ]
			}
			incr i
		}
	}

	return $freespace
}

# Create an ext2-fs on device $dev
# Return 1 if ok, 0 on failure

proc doformat { dev } {
global MKE2FS

	set mkout [ catch { exec $MKE2FS -q $dev 2>/dev/null } ]

	if { $mkout == 0 } {
		return 1
	} else {
		return 0
	}
}


# Loads a text-file line-by-line into a list. 
# Returns empty list if file does not exist.

proc loadfile2list { fname } {

 	set opnflg [catch { set fileid [open $fname r] }]

        if { $opnflg != 0 } {
		# Error opening file, return empty list
                return {} 
        }

	set outlist {}

	while { [gets $fileid line] >= 0 } {
		lappend outlist "$line"
	}

	close $fileid
	return $outlist
}


# Extract from a str all non-white-space-chars after the first colon
# and returns it

proc extractcolonstr { str } {

	# Get substr after the colon
	set str2 "[string range $str [expr [string first : $str]+1] end]"

	return "[string trim $str2]"
}


# get the contents of a directory and returns them as list, return -1 if
# there is no such directory.

proc getdircontents { dir } {
global LS

	set stat [catch {set out [exec $LS -aF $dir]}]

	# No such directory?
	if { $stat != 0 } {
		return -1
	}
	# Split the output to lines
	set out2 [ split $out "\n" ]

	return $out2
}


# Calculate disk-usage in kB, return -1 if directory was invalid

proc calcdu { dir } {
global DU

	set stat [catch { set out [exec $DU -ks $dir]}]

	if { $stat == 0 } {
		scan $out "%d %s" kb tmp
		return $kb 
	}	
	return -1
}


# get the contents of the root of an image and returns them as list, 
# return -1 if there is no such file.

proc getimagecontents { file } {
global ISOINFO

	if { [file exists $file] == 0 } {
		return -1
	}

	set cmd "$ISOINFO -l -R -i $file"
	set pipe [open "|$cmd" r]

	set count 0
	set out { }
	while { [gets $pipe str] >= 0 } {

		# Get only the root-dir, skip all other directories
        	if { $count == 2 } {
        	        catch { close $pipe }
        	        break
        	}

		# Found we a new Directory-entry?
        	if { [string index $str 0 ] == "D" } {
        	        incr count
			continue
        	}

		set type [string index $str 0] 
		if { $type == "-" } {
			set type [string index $str 3]
		}
		if { $type == "-" } {
			set type " " 
		}

		set name [string range $str 64 end]
		set first [string last " " $name]
		set name [string range $name 0 [expr $first-1]]

		switch $type {
			"d" { append name "/" }
			"l" { append name "@" }
			"x" { append name "*" } 
			"p" { append name "|" }
			"s" { append name "=" }
			" " { }
		}

		lappend out $name	
	}

	# just to be sure
        catch { close $pipe }
	return $out
}


# Check if CD is in drive, if not ask user to insert CD.
# return 0 when cd is ready now, 1 when user aborted

proc checkifready { w mode } {

	# Endless loop until cd ready or aborted
	while { 1 } {
	
		# CD loaded?
		if { [docheckcd $mode] == 0 } {
			set stat [Msg_NoCD .nocd $w]	
			if { $stat == 1 } { #; Abort
				return 1 
			}
		} else {
			# CD ready now
			return 0
		}
	}
}	


# Checks if the connected writer is from Yamaha, or if the writer-mode
# is set to yamaha.
# This is needed because the yamaha needs another audio-byte-order.
 
proc checkifyamaha { } {
global XCDR_DEST_WRITER_DEV 
global XCDR_DEST_WRITER_MODE

	set dev $XCDR_DEST_WRITER_DEV
	set mode $XCDR_DEST_WRITER_MODE

	set found 0
	if { [string first "yamaha" $dev] != -1 || 
     	     [string first "Yamaha" $dev] != -1 ||
	     [string first "YAMAHA" $dev] != -1 } {
		
		set found 1
	}

	if { [string first "yamaha" $mode] != -1 || 
     	     [string first "Yamaha" $mode] != -1 ||
	     [string first "YAMAHA" $mode] != -1 } {
		
		set found 1
	}

	return $found
}


# Checks if the connected writer is from Sony, or if the writer-mode
# is set to sony.
# This is needed because the sony needs another audio-byte-order.
 
proc checkifsony { } {
global XCDR_DEST_WRITER_DEV 
global XCDR_DEST_WRITER_MODE

	set dev $XCDR_DEST_WRITER_DEV
	set mode $XCDR_DEST_WRITER_MODE

	set found 0
	if { [string first "sony" $dev] != -1 || 
     	     [string first "Sony" $dev] != -1 ||
	     [string first "SONY" $dev] != -1 } {
		
		set found 1
	}

	if { [string first "sony" $mode] != -1 || 
     	     [string first "Sony" $mode] != -1 ||
	     [string first "SONY" $mode] != -1 } {
		
		set found 1
	}

	return $found
}


# Set the ENDIANSWAP-variable to value needed for this hardware
# Note: cdrecord always wants its data bigendian. Therefore
# this setting is obsolete -> set always to "0"

proc setEndianorder {} {
global ENDIANSWAP

#	if { [checkifyamaha] || [checkifsony] } {
#	        set ENDIANSWAP 1
#	} else {
#	        set ENDIANSWAP 0
#	}

	set ENDIANSWAP 0
}


# Check if dev contains an ISO9660-FS. Check also if isosize
# reports a sane value...
# Returns 1 if iso, 0 if not.

proc checkifiso { dev } {
global ISODETECT
global ISOSIZE

	resetcd $dev
	set isoout ""
	catch { set isoout [exec $ISODETECT -d $dev] }
	if { $isoout == "ISO9660" } {
		set size [exec $ISOSIZE $dev]
		if { $size > 0 && $size < 734003200 } {
			return 1
		}
	
	}
	return 0
}	


# Check if the CD is a pure audio-cd

proc checkifaudio { dev } {

	set cdtoclist [getcdinfo $dev]
	set type [lindex $cdtoclist 0]
	if { $type == "Audio" || $type == "CD-Extra" } {
		return 1
	} 
	return 0
}


# Check if the CD is a mixed-mode-cd

proc checkifmixedmode { dev } {

	set cdtoclist [getcdinfo $dev]
	set type [lindex $cdtoclist 0]
	if { [string first "Mixed" $type] != -1} {
		return 1
	} 
	return 0
}


# Get the size of the first track of an CD. This is useful for
# determining the image-size of an non-iso-CD.
# Returns size in bytes or 0 if size is impossible.

proc getnonisosize { dev } {

	set cdtoclist [getcdinfo $dev]
	# Substract 2 blocks from toc size. This is needed because
	# some blocks at the end of track are not readable. 
	# In some cases more than 2 blocks at the end are not readable,
	# if this happens we have to rely on the kernel to handle that...
	# All kernels up to pre2.0.8 and perhaps more CRASH in that case.

	set frames [expr [lindex $cdtoclist 9] - 2]

	if { $frames > 0 && $frames < 360000 } {	
		return [expr $frames*2048] 
	}
	return 0
}


# Creates num beeps from the speaker.

proc createbeepspeaker { num } {

	puts -nonewline "\a"
	catch { flush stdout }
	if { $num > 1 } {
		for { set i 1} { $i < $num} { incr i } {
		 	after 250 
			puts -nonewline "\a"
			catch { flush stdout }
		}
	}	
}


# Handles the beeps. 
# Type: 1 = Completed task
#       2 = Warning

proc sound { type } {
global XCDR_BEEP
global XCDR_BEEP_VIA

	set doit 0

	# prevent error at startup when variables are not yet
	# initialised
	if { [info exists XCDR_BEEP] == 0 } {
		return
	}

	switch $XCDR_BEEP {

	"on" 	{ set doit 1 }
	"off" 	{ set doit 0 }
	"oncompl" { if { $type == 1 } { set doit 1 } }
	"onwarn"  { if { $type == 2 } { set doit 1 } }

	}	

	if { $doit == 1 } {
		if { $XCDR_BEEP_VIA == "dsp" } {
			# this is defined in MkSetup.tcl
			dotestdsp
		} else {
			createbeepspeaker 1 
		}
	}
}


# Append a string to the logfile

proc log { w str } {
global XCDR_VERSION
global XCDR_LOGFILE
global XCDR_LOGNAME
global CONFDIR
global DATE

	if { $XCDR_LOGFILE == "on" } {
		# check if there is a slash somewhere in the logname
		# if not save it on the CONFDIR
		if { [string first "/" $XCDR_LOGNAME] == -1 } {
			set logfilename $CONFDIR/$XCDR_LOGNAME
		} else {
			set logfilename $XCDR_LOGNAME
		}

		set opnflg [ catch { set fileid [ open $logfilename a+] } ]

		if { $opnflg != 0 } {
			Msg_ErrorOpenLogfile .eolf $w 		
			set XCDR_LOGFILE "off"
			return
		}

		chownfile $logfilename

		set date [exec $DATE "+%b %d %T"]
		set putflg [ catch { puts $fileid "$date XCDR $XCDR_VERSION: $str" } ]

		if { $putflg != 0 } {
			Msg_ErrorWriteLogfile .ewlf $w
			set XCDR_LOGFILE "off"
		}

		close $fileid
	}
}



# Gets a name of a CD Device as argument and returns its
# block-device-name.
# e.g. convertCDnametoblkdevice { "TOSHIBA XM-3601TA" }
# returns "/dev/sr0". 
# The array scsidevices must be definded.
 
proc convertCDnametoblkdevice { name } {
global cdinfo 

	set blkdevice "" 
	set i 0
	while { $cdinfo($i,0) != "x" } {
		if { $cdinfo($i,1) == $name } {
			set blkdevice $cdinfo($i,0)
		}
		incr i
	}
	return $blkdevice
}

# Gets a name of a CD Device as argument and returns its
# generic-device-name.  
# e.g. convertCDnametogendevice { "TOSHIBA XM-3601TA" }
# returns "/dev/sg3". 
# The array scsidevices must be definded.
 
proc convertCDnametogendevice { name } {
global cdinfo 

	set gendevice "" 
	set i 0
	while { $cdinfo($i,0) != "x" } {
		if { $cdinfo($i,1) == $name } {
			set gendevice $cdinfo($i,2)
		}
		incr i
	}
	return $gendevice
}


# Gets a name of a CD Device as argument and returns its
# scsidevnr in a format suitable for cdrecord.  
# e.g. convertCDnametoscsidevnr { "TOSHIBA XM-3601TA" }
# returns "0,03,00". (host,id,lun) 
# The array scsidevices must be definded.
 
proc convertCDnametoscsidevnr { name } {
global cdinfo 

	set gendevice "" 
	set i 0
	while { $cdinfo($i,0) != "x" } {
		if { $cdinfo($i,1) == $name } {
			set gendevice $cdinfo($i,3)
		}
		incr i
	}
	return $gendevice
}


# Coverts MegaBytes into printable time 

proc convertMB2time { mb } {

	set frames [expr (($mb*1024)/2352)*1024]

        set mins [expr $frames/(60*75)]
        set sec [expr ($frames%(60*75))/75]
        set centi_sec [expr (4*($frames%75)+1)/3]

        return [format "%2u:%02u.%02u" $mins $sec $centi_sec]
}


# Converts printable time to frames 

proc converttime2frames { time } {

	if { [scan $time "%u:%u.%u" mins sec centi_sec] != 3 } {
		# We got no valid string, so we assume we got
		# only MB..so convert the MB to frames
		return [expr (($time*1024)/2352)*1024]
	}
	return [expr (($centi_sec*3/4)+($sec*75)+($mins*60*75))]
}


# Converts frames to printable time

proc convertframes2time { frames } {

        set mins [expr $frames/(60*75)]
        set sec [expr ($frames%(60*75))/75]
        set centi_sec [expr (4*($frames%75)+1)/3]

        return [format "%2u:%02u.%02u" $mins $sec $centi_sec]
}


# creates a new commandline, so that stderr and stdout can be
# read via a filehandler

proc execshell { cmd } {

	set exec_cmd [format {sh -c {(%s) 2>&1}} $cmd]
	return $exec_cmd
}


# change the color of a button

proc activebutton { w } {

	$w config -bg salmon 
	$w config -activebackground lightsalmon 
}


# change the color of a button

proc deactivebutton { w } {

	$w config -bg #d9d9d9 
	$w config -activebackground #ececec
}


# load the entry $widget from the window-position-file
# return -1 when file or entry not found
# or return x and y seperated by a comma

proc loadwinpos { widget } {
global CONFDIR
global WINPOSITIONS

	set filename $CONFDIR/$WINPOSITIONS
	set opnflg [catch { set fileid [open $filename r] }]

	if { $opnflg != 0 } {
		# file not found
		return -1
	}

	set found -1
	while { [gets $fileid line] >= 0 } {
		set linelist [split $line ","]

		if { [lindex $linelist 0] == $widget } {
			#found entry
			set found "[lindex $linelist 1],[lindex $linelist 2]"
			break
		}
	}

	close $fileid
	return $found
}


# save the x and y position of $widget to the window-position-file
# note: this code is quite slow and complicated...I create a 
# temporary file, write the new data into it and mv it over the 
# original file
 
proc savewinpos { widget xcord ycord } {
global CONFDIR
global WINPOSITIONS

	set filename $CONFDIR/$WINPOSITIONS
	set filename2 $CONFDIR/$WINPOSITIONS-

	set opnflg [catch { set fileid [open $filename r] }]
	set fileid2 [open $filename2 w]
	chownfile $filename2

	if { $opnflg != 0 } {
		# file not found - create new file
		puts $fileid2 "$widget,$xcord,$ycord"
		close $fileid2
		catch { file rename -force $filename2 $filename }
		return
	}

	set done 0
	while { [gets $fileid line] >= 0 } {
		set linelist [split $line ","]

		# replace existing line
		if { [lindex $linelist 0] == $widget } {
			puts $fileid2 "$widget,$xcord,$ycord"
			set done 1
		} else {
		# keep old line like it was
			puts $fileid2 $line
		}
	}

	# add new line when not already processed
	if { $done == 0 } {
		puts $fileid2 "$widget,$xcord,$ycord"
	}

	close $fileid
	catch { close $fileid2 }
	catch { file rename -force $filename2 $filename }
	return
}


# query cdrecord to get a list of all supported cdr-drivers
# return a list "id1" "name1" "id2" "name2" ....

proc getcdrecorddrivers { } {
global CDRECORD

	set retlist ""
	set cmd [execshell "$CDRECORD driver=help"]

	set pipe [open "|$cmd" r+]

	while { [eof $pipe] == 0 } {

		if { [gets $pipe line] <= 0 } {
			# skip empty lines
			continue
		}

		if { [string range $line 0 5] == "Driver" } {
			# skip header line
			continue
		}

		set idx [string first " " $line]
		set id [string trim [string range $line 0 $idx]]
		set name [string trim [string range $line $idx end]]

		# now strip off unneeded "driver for" string
		if { [string range $name 0 10] == "driver for " } {
			set name "[string range $name 11 end]"
		}
	
		lappend retlist $id
		lappend retlist $name
	}
	
	catch { close $pipe }
	return $retlist
}


# change the user and group of a file to the owner of the process
# (useful when xcdroast runs with the +s bit)

proc chownfile { filename } {
global CHOWN

	# get the real user and group-id
	set uid [getuid]
	set gid [getgid]

	catch { exec $CHOWN $uid:$gid $filename }
}


# returns the exit status of a freshly terminated child-process
# should be called like that:
# set stat [getchildstatus [ catch { close $pipe } ]]

proc getchildstatus { status } {
global errorCode

	# exited the child with a non-zero code? 
	# (this parameter must be obtained by a catch call)
	if { $status != 0 } {
		if { [lindex $errorCode 0] == "CHILDSTATUS" } {
			# the global var errorCode does have what we need
			return [lindex $errorCode 2]
		}
	}
	# when in doubt return 0 (all ok)
	return 0
}


# masks all " [ and ] in a string with a backslash to prevent
# parsing by the tcl-interpreter.

proc fixstring { varname } {

	upvar $varname var 

	set sstrlen [string length $var]
	set var2 ""
	for { set ii 0 } { $ii < $sstrlen } { incr ii } {
		set cchar [string index $var $ii]
		if { $cchar == "\[" || $cchar == "\]" || $cchar == "\"" } {
			set cchar "\\$cchar"
		}
		append var2 $cchar
	}
	set var $var2
}


# scan /proc/devices if the sg-module is loaded
# return 0 if found, 1 if not

proc checksgmodule { } {
global CAT
global FGREP

	set stat [catch { set out [exec $CAT /proc/devices | $FGREP "21 sg"]}]
	if { $stat == 0 } {
		return 0
	}
	return 1
}


# check if we seem to have more than one scsi-controller and 
# if the scsi-patch from cdrecord is installed. 
# return 0 when we are fine (only one or none scsi-host or patch installed) 
# and 1 when there may be problems (two hosts and no patch installed)

proc checkscsipatch { } {
global scsidevices

	set i 0
	set found 0
	# search all scsi-devices for a hostnumber not equal zero
	while { $scsidevices($i,0) != "x" } {
		set idlist [split $scsidevices($i,10)]
		set host [string range [lindex $idlist 0] 4 end]
		if { $host != 0 } {
			set found 1
		}
		incr i
	}

	# we have more than one scsi-host...
	if { $found == 1 } {
		# now check if the scsi-patch is in the kernel
		if { [scsibusioctlcheck] == 1 } {
			# no...its not.
			return 1
		}
	}

	return 0
}


# small routine to strip multiple space chars from a string
# e.g. "This    is      for testing" becomes "This is for testing"

proc stripstring { str } {

	set sstrlen [string length $str]
	set var2 ""
	set cchar " "
	for { set ii 0 } { $ii < $sstrlen } { incr ii } {
		set oldcchar $cchar
		set cchar [string index $str $ii]
		if { $cchar == " " && $oldcchar == " " } {
			continue
		}
		append var2 $cchar
	}
	return $var2	
}


# dummy function to handle double-clicks in entry fields

proc tcl_wordBreakBefore { str start } {

	# do nothing
}

proc tcl_wordBreakAfter { str start } {

	# do nothing
}


# check if this string does only contain printable characters
# return 1 if so, return 0 when not.

proc if_printable { string } {

	set ok 1
	
	for { set i 0 } { $i < [string length $string] } { incr i } {
		set ch [string index $string $i]
		binary scan $ch c intval
		if { $intval < 32 || $intval > 127 } {
			set ok 0	
		} 
	}

	return $ok 
} 
