/*
 * tnmSnmp.h --
 *
 *	Definitions for the SNMP protocol implementation.
 *
 * Copyright (c) 1994-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#ifndef _TNMSNMP
#define _TNMSNMP

#include "tnmInt.h"
#include "tnmPort.h"
#include "tnmAsn1.h"

/*
 *----------------------------------------------------------------
 * Some general defines. Note, all SNMPv2 versions that we will
 * ever implement should have the same SNMPv2 bit set, so that 
 * we can use this bit if we allow SNMPv2 protocol operations.
 *
 * You can turn off support for TNM_SNMPv2U by undefining this
 * symbol. This will make the code a bit smaller, but you won't 
 * gain much speed. So you might want to let these defines as
 * they are.
 *----------------------------------------------------------------
 */

#define TNM_SNMPv1		0x11
#define TNM_SNMPv2		0x20
#define TNM_SNMPv2C		0x21
#define TNM_SNMPv2U		0x22

/*
 *----------------------------------------------------------------
 * The PDU formats as defined in RFC 1157 and in section 3
 * of RFC 1905.
 *----------------------------------------------------------------
 */

#define TNM_SNMP_GET		0
#define TNM_SNMP_GETNEXT	1
#define TNM_SNMP_RESPONSE	2
#define	TNM_SNMP_SET		3
#define TNM_SNMPv1_TRAP		4
#define TNM_SNMP_GETBULK	5
#define TNM_SNMP_INFORM		6
#define TNM_SNMPv2_TRAP		7
#define TNM_SNMP_REPORT		8

EXTERN TnmTable tnmSnmpPDUTable[];

/*
 *----------------------------------------------------------------
 * Error status definitions as of RFC 1905 and RFC 1157. RFC 1157
 * only defines the errors NOERROR - GENERR. Some of the version
 * V1 error codes are only for V1/V2 proxy compatibility.
 *----------------------------------------------------------------
 */

#define TNM_SNMP_NOERROR		0	/*  v1 / v2 error code */
#define TNM_SNMP_TOOBIG			1	/*  v1 / v2 error code */
#define TNM_SNMP_NOSUCHNAME		2	/*  v1      error code */
#define TNM_SNMP_BADVALUE		3	/*  v1      error code */
#define TNM_SNMP_READONLY		4	/*  v1      error code */
#define TNM_SNMP_GENERR			5	/*  v1 / v2 error code */
#define TNM_SNMP_NOACCESS		6	/*       v2 error code */
#define TNM_SNMP_WRONGTYPE		7	/*       v2 error code */
#define TNM_SNMP_WRONGLENGTH		8	/*       v2 error code */
#define TNM_SNMP_WRONGENCODING		9	/*       v2 error code */
#define TNM_SNMP_WRONGVALUE		10	/*       v2 error code */
#define TNM_SNMP_NOCREATION		11	/*       v2 error code */
#define TNM_SNMP_INCONSISTENTVALUE	12	/*       v2 error code */
#define TNM_SNMP_RESOURCEUNAVAILABLE	13	/*       v2 error code */
#define TNM_SNMP_COMMITFAILED		14	/*       v2 error code */
#define TNM_SNMP_UNDOFAILED		15	/*       v2 error code */
#define TNM_SNMP_AUTHORIZATIONERROR	16	/*       v2 error code */
#define TNM_SNMP_NOTWRITABLE		17	/*       v2 error code */
#define TNM_SNMP_INCONSISTENTNAME	18	/*       v2 error code */
#define TNM_SNMP_NORESPONSE	       255	/* internal error code */

EXTERN TnmTable tnmSnmpErrorTable[];

/*
 *----------------------------------------------------------------
 * The following constants define the default session 
 * configuration.
 *----------------------------------------------------------------
 */

#define	TNM_SNMP_PORT		161
#define	TNM_SNMP_TRAPPORT	162
#define TNM_SNMP_RETRIES	3
#define TNM_SNMP_TIMEOUT	5
#define TNM_SNMP_WINDOW		10
#define TNM_SNMP_DELAY		0

/*
 *----------------------------------------------------------------
 * The size of the internal buffer used to decode or assemble 
 * SNMP packets and the size of MD5 keys.
 *----------------------------------------------------------------
 */

#define TNM_SNMP_MAXSIZE	2048
#define TNM_MD5_SIZE		16

/*
 *----------------------------------------------------------------
 * A structure to keep performance statistics.
 *----------------------------------------------------------------
 */

#ifdef SNMP_BENCH
typedef struct Tnm_SnmpMark {
    Tcl_Time sendTime;
    Tcl_Time recvTime;
    int sendSize;
    int recvSize;
} Tnm_SnmpMark;

EXTERN Tnm_SnmpMark tnmSnmpBenchMark;
#endif

/*
 *----------------------------------------------------------------
 * Definitions for the user based security model (USEC). See 
 * RFC 1909 and RFC 1910 for more details.
 *----------------------------------------------------------------
 */

#ifdef TNM_SNMPv2U
#define USEC_QOS_NULL		(0x00)
#define USEC_QOS_AUTH		(0x01)
#define USEC_QOS_PRIV		(0x02)
#define USEC_QOS_REPORT		(0x04)

#define USEC_MODEL		1
#define USEC_MAX_USER		16
#define USEC_MAX_CONTEXT	40
#define USEC_MAX_AGENTID	12
#define USEC_MAX_MMS		65507
#define USEC_MIN_MMS		484
#endif

/*
 *----------------------------------------------------------------
 * A session structure contains all infomation needed to handle
 * SNMP requests and responses between an agent and a manager.
 *----------------------------------------------------------------
 */

typedef struct SNMP_Session {
    char name[10];                /* The session name used by Tcl. */
    struct sockaddr_in maddr;     /* The manager destination address. */
    struct sockaddr_in taddr;     /* The trap destination address. */
    int version;                  /* The SNMP version used by this session. */
    u_int traps:1;                /* This flag is set if we receive traps. */
    char *readCommunity;          /* Community string for read operations. */
    char *writeCommunity;         /* Optional community string for sets. */
    char *trapCommunity;          /* Optional community string for traps. */
#ifdef TNM_SNMPv2U
    u_char qos;
    u_char agentID[USEC_MAX_AGENTID];
    u_int agentBoots;
    u_int agentTime;
    u_int latestReceivedAgentTime;
    int userNameLen;
    char userName[USEC_MAX_USER];
    char *password;
    u_char authKey[TNM_MD5_SIZE];
    u_char privKey[TNM_MD5_SIZE];
    int cntxtLen;
    char cntxt[USEC_MAX_CONTEXT];
    int maxSize;
#endif
    int retries;                  /* Number of retries until we give up. */
    u_int timeout;                /* Milli seconds before we timeout. */
    int window;                   /* Max. number of active async. requests. */
    int delay;                    /* Minimum delay between requests. */
    int active;                   /* Number of active async. requests. */
    int waiting;                  /* Number of waiting async. requests. */
    u_int reqid;                  /* Next request id to use. */
    struct SNMP_Binding	*bindPtr; /* List of commands bound to this session. */
    Tcl_Interp *interp;		  /* Tcl interpreter owning this session. */
    Tcl_Command token;		  /* The command token used by Tcl. */
    int agentSock;                /* Socket used if we are an agent. */
    Tcl_File agentSocket;         /* Tcl File used if we are an agent. */
    Tcl_Interp *agentInterp;      /* Tcl interp used by the agent. */
    struct TnmSnmpNode *instPtr;    /* Root of the tree of MIB instances. */
    struct SNMP_Session	*nextPtr; /* Pointer to next session. */
#ifdef SNMP_BENCH
    Tnm_SnmpMark stats;		  /* Statistics for the last SNMP operation. */
#endif
} SNMP_Session;

/*
 *----------------------------------------------------------------
 * This global variable that points to a list of all sessions.
 *----------------------------------------------------------------
 */

EXTERN SNMP_Session *sessionList;

/*
 *----------------------------------------------------------------
 * Structure to hold a varbind. Lists of varbinds are mapped to 
 * arrays of varbinds as this simplifies memory management.
 *----------------------------------------------------------------
 */

typedef struct SNMP_VarBind {
    char *soid;
    char *syntax;
    char *value;
    char *freePtr;
    ClientData clientData;
    int flags;
} SNMP_VarBind;

EXTERN void
Tnm_SnmpFreeVBList	_ANSI_ARGS_((int varBindSize, 
				     SNMP_VarBind *varBindPtr));
EXTERN int
Tnm_SnmpSplitVBList	_ANSI_ARGS_((Tcl_Interp *interp, char *list,
				     int *varBindSizePtr, 
				     SNMP_VarBind **varBindPtrPtr));
EXTERN char*
Tnm_SnmpMergeVBList	_ANSI_ARGS_((int varBindSize, 
				     SNMP_VarBind *varBindPtr));

/*
 *----------------------------------------------------------------
 * Structure to describe a SNMP PDU.
 *----------------------------------------------------------------
 */

typedef struct SNMP_PDU {
    struct sockaddr_in addr;	/* The address from/to it is sent.     */
    int type;			/* The type of this PDU.               */
    int request_id;		/* A unique request id for this PDU.   */
    int error_status;		/* The SNMP error status field.        */
    int error_index;		/* The SNMP error index field.         */
    char *trapOID;		/* Trap object identifier.             */
#if 1
    Tcl_DString varbind;	/* The list of varbinds as Tcl string. */
#else
    SNMP_VarBind *varbind;	/* The list of varbinds as an array.   */
#endif
} SNMP_PDU;

/*
 *----------------------------------------------------------------
 * Structure to describe an asynchronous request.
 *----------------------------------------------------------------
 */

typedef void (Tnm_SnmpRequestProc)	_ANSI_ARGS_((SNMP_Session *session,
		SNMP_PDU *pdu, ClientData clientData));

typedef struct Tnm_SnmpRequest {
    u_int id;                        /* The unique request identifier. */
    int sends;                       /* Number of send operations. */
    u_char *packet;                  /* The encoded SNMP message. */
    int packetlen;		     /* The length of the encoded message. */
    Tcl_TimerToken timer;	     /* Token used by Tcl Timer Handler. */
    SNMP_Session *session;           /* The SNMP session for this request. */
    Tnm_SnmpRequestProc *proc;       /* The callback functions. */
    ClientData clientData;           /* The argument of the callback. */
    struct Tnm_SnmpRequest *nextPtr; /* Pointer to next pending request. */
#ifdef SNMP_BENCH
    Tnm_SnmpMark stats;              /* Statistics for this SNMP operation. */
#endif
    Tcl_Interp *interp;		     /* The interpreter needed internally. */
} Tnm_SnmpRequest;

EXTERN Tnm_SnmpRequest*
Tnm_SnmpCreateRequest	_ANSI_ARGS_((int id, u_char *packet, int packetlen,
				     Tnm_SnmpRequestProc *proc,
				     ClientData clientData,
				     Tcl_Interp *interp));
EXTERN Tnm_SnmpRequest*
Tnm_SnmpFindRequest	_ANSI_ARGS_((int id));

EXTERN int
Tnm_SnmpQueueRequest	_ANSI_ARGS_((SNMP_Session *session, 
				     Tnm_SnmpRequest *request));
EXTERN void
Tnm_SnmpDeleteRequest	_ANSI_ARGS_((Tnm_SnmpRequest *request));

EXTERN int
TnmSnmpGetRequestId	_ANSI_ARGS_((void));

/*
 *----------------------------------------------------------------
 * The event types currently supported for SNMP bindings.
 *----------------------------------------------------------------
 */

#define TNM_SNMP_NO_EVENT	0x0000
#define TNM_SNMP_GET_EVENT	0x0001
#define TNM_SNMP_SET_EVENT	0x0002
#define TNM_SNMP_CREATE_EVENT	0x0004
#define TNM_SNMP_TRAP_EVENT	0x0008
#define TNM_SNMP_INFORM_EVENT	0x0010
#define TNM_SNMP_CHECK_EVENT	0x0020
#define TNM_SNMP_COMMIT_EVENT	0x0040
#define TNM_SNMP_ROLLBACK_EVENT	0x0080
#define TNM_SNMP_BEGIN_EVENT	0x0100
#define TNM_SNMP_END_EVENT	0x0200
#define TNM_SNMP_RECV_EVENT	0x0400
#define TNM_SNMP_REPORT_EVENT	0x0800
#define TNM_SNMP_SEND_EVENT	0x1000

#define TNM_SNMP_GENERIC_BINDINGS \
(TNM_SNMP_TRAP_EVENT | TNM_SNMP_INFORM_EVENT | \
TNM_SNMP_BEGIN_EVENT | TNM_SNMP_END_EVENT |\
TNM_SNMP_RECV_EVENT | TNM_SNMP_SEND_EVENT | TNM_SNMP_REPORT_EVENT)

#define TNM_SNMP_INSTANCE_BINDINGS \
(TNM_SNMP_GET_EVENT | TNM_SNMP_SET_EVENT | TNM_SNMP_CREATE_EVENT |\
TNM_SNMP_CHECK_EVENT | TNM_SNMP_COMMIT_EVENT | TNM_SNMP_ROLLBACK_EVENT)

EXTERN TnmTable tnmSnmpEventTable[];

/*
 *----------------------------------------------------------------
 * A structure to describe a binding. Multiple bindings are 
 * organized in a simple list as it is not too likely to have 
 * many bindings for an object.
 *----------------------------------------------------------------
 */

typedef struct SNMP_Binding {
    int event;				/* Event that triggers binding. */
    char *command;			/* Tcl command to evaluate.     */
    struct SNMP_Binding *nextPtr;	/* Next binding in our list.    */
} SNMP_Binding;


EXTERN int
Tnm_SnmpEvalBinding	_ANSI_ARGS_((Tcl_Interp *interp, SNMP_Session *session,
                                     SNMP_PDU *pdu, int event));

/*
 *----------------------------------------------------------------
 * Structure to describe a MIB node known by a session handle.
 * MIB nodes are either used to keep information about session 
 * bindings or to store data needed to process incoming SNMP 
 * requests in the agent role.
 *----------------------------------------------------------------
 */

typedef struct TnmSnmpNode {
    char *label;			/* The complete OID.	    */
    int offset;				/* Offset to instance id.   */
    int syntax;				/* Syntax string from MIB.  */
    int	access;				/* Access mode from MIB.    */
    char *tclVarName;			/* Tcl variable name.	    */
    SNMP_Binding *bindings;		/* List of bindings.        */ 
    u_int subid;			/* Sub identifier in Tree.  */
    struct TnmSnmpNode *childPtr;	        /* List of child nodes.	    */
    struct TnmSnmpNode *nextPtr;	        /* List of peer node.	    */
} TnmSnmpNode;

EXTERN int
Tnm_SnmpCreateNode	_ANSI_ARGS_((Tcl_Interp *interp, char *id,
				     char *varName, char *defval));
EXTERN TnmSnmpNode*
Tnm_SnmpFindNode	_ANSI_ARGS_((SNMP_Session *session,
				     Tnm_Oid *oid, int len));
EXTERN TnmSnmpNode*
Tnm_SnmpFindNextNode	_ANSI_ARGS_((SNMP_Session *session,
				     Tnm_Oid *oid, int len));
EXTERN int
Tnm_SnmpSetNodeBinding	_ANSI_ARGS_((SNMP_Session *session,
				     Tnm_Oid *oid, int len,
				     int event, char *command));
EXTERN char*
Tnm_SnmpGetNodeBinding	_ANSI_ARGS_((SNMP_Session *session,
				     Tnm_Oid *oid, int len, int event));
EXTERN int
Tnm_SnmpEvalNodeBinding	_ANSI_ARGS_((SNMP_Session *session,
				     SNMP_PDU *pdu, TnmSnmpNode *inst,
				     int operation, char *value,
				     char *oldValue));

/*
 *----------------------------------------------------------------
 * Structure to collect SNMP related statistics. See RFC 1213
 * and RFC 1450 for more details when these counters must be
 * incremented.
 *----------------------------------------------------------------
 */

typedef struct SNMP_Statistics {
    /* RFC 1213 */
    u_int snmpInPkts;
    u_int snmpOutPkts;
    u_int snmpInBadVersions;
    u_int snmpInBadCommunityNames;
    u_int snmpInBadCommunityUses;
    u_int snmpInASNParseErrs;
    u_int snmpInTooBigs;
    u_int snmpInNoSuchNames;
    u_int snmpInBadValues;
    u_int snmpInReadOnlys;
    u_int snmpInGenErrs;
    u_int snmpInTotalReqVars;
    u_int snmpInTotalSetVars;
    u_int snmpInGetRequests;
    u_int snmpInGetNexts;
    u_int snmpInSetRequests;
    u_int snmpInGetResponses;
    u_int snmpInTraps;
    u_int snmpOutTooBigs;
    u_int snmpOutNoSuchNames;
    u_int snmpOutBadValues;
    u_int snmpOutGenErrs;
    u_int snmpOutGetRequests;
    u_int snmpOutGetNexts;
    u_int snmpOutSetRequests;
    u_int snmpOutGetResponses;
    u_int snmpOutTraps;
    u_int snmpEnableAuthenTraps;
    /* RFC 1450 XXX not yet implemented */
    u_int snmpStatsPackets;
    u_int snmpStats30Something;
    u_int snmpStatsEncodingErrors;
    u_int snmpStatsUnknownDstParties;
    u_int snmpStatsDstPartyMismatches;
    u_int snmpStatsUnknownSrcParties;
    u_int snmpStatsBadAuths;
    u_int snmpStatsNotInLifetimes;
    u_int snmpStatsWrongDigestValues;
    u_int snmpStatsUnknownContexts;
    u_int snmpStatsBadOperations;
    u_int snmpStatsSilentDrops;
    /* snmpV1BadCommunityNames is the same as snmpInBadCommunityNames */
    /* snmpV1BadCommunityUses  is the same as snmpInBadCommunityUses  */
#ifdef TNM_SNMPv2U
    u_int usecStatsUnsupportedQoS;
    u_int usecStatsNotInWindows;
    u_int usecStatsUnknownUserNames;
    u_int usecStatsWrongDigestValues;
    u_int usecStatsUnknownContexts;
    u_int usecStatsBadParameters;
    u_int usecStatsUnauthorizedOperations;
#endif
} SNMP_Statistics;

/*
 *----------------------------------------------------------------
 * Global variable used to collect the SNMP statistics.
 *----------------------------------------------------------------
 */

EXTERN SNMP_Statistics snmpStats;

/*
 *----------------------------------------------------------------
 * Exported SNMP procedures:
 *----------------------------------------------------------------
 */

EXTERN int
SNMP_Init		_ANSI_ARGS_((Tcl_Interp *interp));

EXTERN SNMP_Session*
Tnm_SnmpCreateSession	_ANSI_ARGS_((void));

EXTERN void
Tnm_SnmpDeleteSession	_ANSI_ARGS_((SNMP_Session *session));

EXTERN int
Tnm_SnmpEncode		_ANSI_ARGS_((Tcl_Interp *interp, SNMP_Session *session,
				     SNMP_PDU *pdu, Tnm_SnmpRequestProc *proc,
				     ClientData clientData));
EXTERN int
Tnm_SnmpDecode		_ANSI_ARGS_((Tcl_Interp *interp, 
				     u_char *packet, int packetlen,
				     struct sockaddr_in *from,
				     SNMP_Session *session, int *reqid));
EXTERN void
Tnm_SnmpTimeoutProc	_ANSI_ARGS_((ClientData clientData));

EXTERN int
Tnm_SnmpAgentInit	_ANSI_ARGS_((Tcl_Interp *interp, 
				     SNMP_Session *session));
EXTERN int
Tnm_SnmpAgentRequest	_ANSI_ARGS_((Tcl_Interp *interp, SNMP_Session *session,
				     SNMP_PDU *pdu));
EXTERN int
Tnm_SnmpEvalCallback	_ANSI_ARGS_((Tcl_Interp *interp, SNMP_Session *session,
				     SNMP_PDU *pdu,
				     char *cmd, char *instance, char *oid, 
				     char *value, char* oldValue));

/*
 *----------------------------------------------------------------
 * The following function is used to create a socket used for
 * all manager initiated communication. The Close function
 * is used to close this socket if all SNMP sessions have been
 * destroyed.
 *----------------------------------------------------------------
 */

EXTERN int
Tnm_SnmpManagerOpen	_ANSI_ARGS_((Tcl_Interp	*interp));

EXTERN void
Tnm_SnmpManagerClose	_ANSI_ARGS_((void));

/*
 *----------------------------------------------------------------
 * Management communication initiated by an agent is normally
 * done using traps. Because traps are sent to port 162 (which
 * requires special privileges, we read the from the straps
 * process via a unix domain socket.
 *----------------------------------------------------------------
 */

EXTERN int
Tnm_SnmpTrapOpen	_ANSI_ARGS_((Tcl_Interp *interp));

EXTERN void
Tnm_SnmpTrapClose	_ANSI_ARGS_((void));

/*
 *----------------------------------------------------------------
 * Create and close a socket used for the agent part. This 
 * socket is used to process messages received from other agents.
 *----------------------------------------------------------------
 */

EXTERN int
Tnm_SnmpAgentOpen	_ANSI_ARGS_((Tcl_Interp *interp, 
				     SNMP_Session *session));

EXTERN void
Tnm_SnmpAgentClose	_ANSI_ARGS_((SNMP_Session *session));

/*
 *----------------------------------------------------------------
 * Functions used to send and receive SNMP messages. The 
 * Tnm_SnmpWait function is used to wait for an answer.
 *----------------------------------------------------------------
 */

EXTERN int
Tnm_SnmpSend		_ANSI_ARGS_((Tcl_Interp *interp,
				     u_char *packet, int packetlen,
				     struct sockaddr_in *to));
EXTERN int
Tnm_SnmpRecv		_ANSI_ARGS_((Tcl_Interp *interp, 
				     u_char *packet, int *packetlen,
				     struct sockaddr_in *from));
EXTERN int 
Tnm_SnmpWait		_ANSI_ARGS_((int ms));

EXTERN void
Tnm_SnmpDelay		_ANSI_ARGS_((SNMP_Session *session));

/*
 *----------------------------------------------------------------
 * Some more utility and conversion functions.
 *----------------------------------------------------------------
 */

EXTERN int
Tnm_SnmpSysUpTime	_ANSI_ARGS_((void));

EXTERN void
Tnm_SnmpMD5Digest	_ANSI_ARGS_((u_char *packet, int packetlen,
				     u_char *key, u_char *digest));

#ifdef TNM_SNMPv2U
EXTERN void
Tnm_SnmpUsecSetAgentID	_ANSI_ARGS_((SNMP_Session *session));

EXTERN void
Tnm_SnmpUsecGetAgentID	_ANSI_ARGS_((SNMP_Session *session));

EXTERN void
Tnm_SnmpUsecAuth	_ANSI_ARGS_((SNMP_Session *session, 
				     u_char *packet, int packetlen));
#endif

EXTERN void
Tnm_SnmpBinToHex	_ANSI_ARGS_((char *s, int n, char *d));

EXTERN int
Tnm_SnmpHexToBin	_ANSI_ARGS_((char *s, char *d, int *n));

EXTERN void
Tnm_SnmpDumpPDU		_ANSI_ARGS_((Tcl_Interp *interp, SNMP_PDU *pdu));

#endif /* _TNMSNMP */
