/*
 * user.c  -  user interface for makerom bootrom configuration utility
 *
 * Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define NO_BINARY 1	/* No need to include binary support here */
#include "common.h"
#include "makerom.h"



/*
 * Local declarations
 */
#define DEFAULTINT	0x62
#define PROGNUM		(MAXDRIVERS - 1)



/*
 * Directory entry name list
 */
static struct direntry {
	struct direntry *next;
	char             fname[1];
} *dirlist = NULL;



/*
 * Variables local to this module
 */
static struct bootdef bootd;
static char *kernelname;
static char *pktdrvname;
static char *pktdrvoptions;
static char *prognames[PROGNUM];
static char *progoptions[PROGNUM];
static int num_progs = 0;
static int is86 = 0;
static int isminimal = 0;
static int isdefint = 0;
static int useint18 = 0;



/*
 * Compare a directory entry against a pattern.
 */
static int dircmp(pattern, name)
char *pattern;
char *name;
{
  while (*pattern && *name) {
	if (*pattern == '?' || *pattern == *name) {
		pattern++;
		name++;
	} else {
		if (*pattern != '*')
			break;
		pattern++;
		while (*name && *name != *pattern)
			name++;
	}
  }
  return(!*pattern && !*name);
}



/*
 * Get a list of directory entries with a specified name pattern.
 */
static int getdirlist(dirname, pattern)
char *dirname;
char *pattern;
{
  DIR *dir;
  struct dirent *ent;
  struct direntry *dp;
  int count;
  size_t len;
  char *src, *dest;

  /* First open the directory */
  if ((dir = opendir(dirname)) == NULL) {
	fprintf(stderr, "%s: cannot open directory %s\n", progname, dirname);
	exit(EXIT_OPENDIR);
  }

  /* Next step through all directory entries */
  count = 0;
  while ((ent = readdir(dir)) != NULL) {
	if (!dircmp(pattern, ent->d_name))
		continue;
	len = strlen(ent->d_name) + sizeof(struct direntry);
	if ((dp = (struct direntry *)malloc(len)) == NULL) {
		perror(progname);
		exit(EXIT_MEMORY);
	}
	for (src = ent->d_name, dest = dp->fname; *src; src++, dest++) {
		if (*src == '.') break;
		*dest = tolower(*src);
	}
	*dest = '\0';
	dp->next = dirlist;
	dirlist = dp;
	count++;
  }
  closedir(dir);
  return(count);
}



/*
 * Delete directory list
 */
static void deldirlist()
{
  struct direntry *dp1, *dp2;

  dp1 = dirlist;
  while (dp1 != NULL) {
	dp2 = dp1;
	dp1 = dp1->next;
	free(dp2);
  }
  dirlist = NULL;
}



/*
 * Create the bootrom configuration
 */
static void buildconfig()
{
  char *cp, *buf;
  int i;
  size_t len;

  memset(&bootd, 0, sizeof(bootd));

  bootd.name = strdup("default");
  bootd.kernelname = kernelname;
  bootd.use_int18 = useint18;

  cp = is86 ? "floppy86.bin" : "floppy.bin";
  len = strlen(kerneldir) + strlen(cp) + 2;
  if ((buf = malloc(len)) == NULL) {
	perror(progname);
	exit(EXIT_MEMORY);
  }
  sprintf(buf, "%s/%s", kerneldir, cp);
  bootd.loadernames[0] = buf;
  bootd.outnames[0] = strdup("image.flo");
  bootd.flashflags[0] = FALSE;

  cp = is86 ? "rom86.bin" : "rom.bin";
  len = strlen(kerneldir) + strlen(cp) + 2;
  if ((buf = malloc(len)) == NULL) {
	perror(progname);
	exit(EXIT_MEMORY);
  }
  sprintf(buf, "%s/%s", kerneldir, cp);
  bootd.loadernames[1] = buf;
  bootd.outnames[1] = strdup("image.rom");
  bootd.flashflags[1] = FALSE;

  bootd.loadernames[2] = buf;
  bootd.outnames[2] = strdup("image.flash");
  bootd.flashflags[2] = TRUE;
  bootd.loadernum = 3;

  bootd.drivernames[0] = pktdrvname;
  bootd.driverargs[0] = pktdrvoptions;
  for (i = 0; i < num_progs; i++) {
	bootd.drivernames[i+1] = prognames[i];
	bootd.driverargs[i+1] = progoptions[i];
  }
  bootd.drivernum = num_progs + 1;
}



/*
 * Return pointer to description string for program name
 */
static char *getname(dlist, pname)
struct desclist **dlist;
char *pname;
{
  struct desclist *ditem = *dlist;

  while (ditem) {
	if (!strcmp(ditem->progname, pname) && ditem->descript != NULL)
		return(ditem->descript);
	ditem = ditem->next;
  }
  return(pname);
}



/*
 * Return option value for program name
 */
static int getoption(dlist, pname)
struct desclist **dlist;
char *pname;
{
  struct desclist *ditem = *dlist;

  while (ditem) {
	if (!strcmp(ditem->progname, pname))
		return(ditem->options);
	ditem = ditem->next;
  }
  return(-1);
}



/*
 * Get a string response from the user
 */
static char *getstring()
{
  static char buf[1024];
  char *cp;

  fgets(buf, sizeof(buf) - 1, stdin);
  for (cp = buf; *cp && *cp != '\n'; cp++) ;
  *cp = '\0';
  return(buf);
}



/*
 * Get a numeric response from the user
 */
static long getnum(prompt, min, max, hex)
char *prompt;
long min;
long max;
int hex;
{
  long j;
  char *cp;

  j = min - 1;
  while (j < min || j > max) {
	printf("  %s (%s): ", prompt, hex ? "hex" : "decimal");
	cp = getstring();
	if ((hex ? sscanf(cp, "%lx", &j) : sscanf(cp, "%ld", &j)) != 1)
		j = min - 1;
	if (j < min || j > max)
		if (hex)
			printf("  Allowed values are [0x%lx..0x%lx]\n",
								min, max);
		else
			printf("  Allowed values are [%ld..%ld]\n",
								min, max);
  }
  return(j);
}



/*
 * Get packet driver options
 */
static void getpktopt(pktname)
char *pktname;
{
  int pktopt;
  long j;
  char *cp;

  if ((pktopt = getoption(&pktdrvdesc, pktname)) < 0)
	return;
  if ((pktdrvoptions = malloc(1024)) == NULL) {
	perror(progname);
	exit(EXIT_MEMORY);
  }
  sprintf(pktdrvoptions, "0x%x", DEFAULTINT);
  isdefint++;
  if (pktopt) {
	cp = pktdrvoptions + strlen(pktdrvoptions);
	*cp++ = ' ';
	printf("Enter packet driver command line options:\n");
	if (pktopt & HW_IRQ) {
		j = getnum("Hardware IRQ number", 2, 15, FALSE);
		sprintf(cp, "%ld", j);
		cp = pktdrvoptions + strlen(pktdrvoptions);
		if (pktopt &= ~HW_IRQ)
			*cp++ = ' ';
	}
	if (pktopt & IO_ADDR) {
		j = getnum("I/O address", 0, 0x7fff, TRUE);
		sprintf(cp, "0x%lx", j);
		cp = pktdrvoptions + strlen(pktdrvoptions);
		if (pktopt &= ~IO_ADDR)
			*cp++ = ' ';
	}
	if (pktopt & BASE_MEM) {
		j = getnum("Shared memory address", 0, 0xffff, TRUE);
		sprintf(cp, "0x%lx", j);
		cp = pktdrvoptions + strlen(pktdrvoptions);
		if (pktopt &= ~BASE_MEM)
			*cp++ = ' ';
	}
	if (pktopt & DMA_NUM) {
		j = getnum("DMA number", 0, 7, FALSE);
		sprintf(cp, "%ld", j);
		cp = pktdrvoptions + strlen(pktdrvoptions);
		if (pktopt &= ~DMA_NUM)
			*cp++ = ' ';
	}
	if (pktopt & AUI_TYPE) {
		j = getnum("AUI type (-1 for default)", -1, 1, FALSE);
		if (j > -1) {
			sprintf(cp, "%ld", j);
			cp = pktdrvoptions + strlen(pktdrvoptions);
		}
	}
  }
  return;
}



/*
 * Let the user select a packet driver.
 */
static void getpktdrv()
{
  int num, sel, i;
  struct direntry *dp;
  size_t len;
  char *cp;

  if ((num = getdirlist(pktdrvdir, "*.com")) == 0) {
	fprintf(stderr, "%s: Oops, no packet drivers found in %s!\n",
							progname, pktdrvdir);
	exit(EXIT_MAKEROM_PKTFND);
  }
  printf("The following %d packet drivers are available:\n", num);
  printf("(0)\tuser defined packet driver\n");
  for (dp = dirlist, i = 1; dp != NULL; dp = dp->next, i++)
	printf("(%d)\t%s\n", i, getname(&pktdrvdesc, dp->fname));
  sel = -1;
  while (sel < 0 || sel > num) {
	printf("Select the packet driver you wish to use: ");
	cp = getstring();
	sscanf(cp, "%d", &sel);
  }
  if (sel == 0) {
	printf("Enter the full path name of the packet driver: ");
	cp = getstring();
	if ((i = strcspn(cp, " \t")) < strlen(cp))
		cp[i] = '\0';
	if ((pktdrvname = strdup(cp)) == NULL) {
		perror(progname);
		exit(EXIT_MEMORY);
	}
  } else {
	for (dp = dirlist, i = 1; dp != NULL && sel != i; dp = dp->next, i++) ;
	if (dp != NULL) {
		len = strlen(dp->fname) + strlen(pktdrvdir) + 6;
		if ((pktdrvname = malloc(len)) == NULL) {
			perror(progname);
			exit(EXIT_MEMORY);
		}
		sprintf(pktdrvname, "%s/%s.com", pktdrvdir, dp->fname);
		getpktopt(dp->fname);
	}
  }
  if (pktdrvoptions == NULL) {
	printf("Enter command line arguments for packet driver: ");
	cp = getstring();
	if ((pktdrvoptions = strdup(cp)) == NULL) {
		perror(progname);
		exit(EXIT_MEMORY);
	}
  }
  deldirlist();
  printf("\n\n\n");
}



/*
 * Let the user select the name of an additional program.
 */
static void getprogram()
{
  char *cp;
  long intnum = DEFAULTINT;
  long pkttype = 0;
  size_t len;

  if (getdirlist(utilsdir, "ansidrv.com") == 1) {
	printf("Do you want to use the ANSI display driver (y/n) ? ");
	cp = getstring();
	if (*cp == 'y' || *cp == 'Y') {
		len = strlen(utilsdir) + strlen(dirlist->fname) + 6;
		if ((cp = malloc(len)) == NULL) {
			perror(progname);
			exit(EXIT_MEMORY);
		}
		sprintf(cp, "%s/%s.com", utilsdir, dirlist->fname);
		prognames[num_progs] = cp;
		progoptions[num_progs] = "";
		num_progs++;
	}
  }
  deldirlist();

  if (getdirlist(utilsdir, "pktwatch.com") == 1) {
	printf("Do you want to use the packet driver debug program (y/n) ? ");
	cp = getstring();
	if (*cp == 'y' || *cp == 'Y') {
		len = strlen(utilsdir) + strlen(dirlist->fname) + 6;
		if ((cp = malloc(len < 256 ? 256 : len)) == NULL) {
			perror(progname);
			exit(EXIT_MEMORY);
		}
		if (!isdefint)
			intnum = getnum("packet driver interrupt number",
							0x60, 0x7f, TRUE);
		pkttype = getnum("packet type (0 for default)", 0, 0xffff, TRUE);
		if (pkttype == 0)
			sprintf(cp, "%lx", intnum);
		else
			sprintf(cp, "%lx %lx", intnum, pkttype);
		if ((progoptions[num_progs] = strdup(cp)) == NULL) {
			fprintf(stderr, "%s: not enough memory for options\n",
								progname);
			exit(EXIT_MEMORY);
		}
		sprintf(cp, "%s/%s.com", utilsdir, dirlist->fname);
		prognames[num_progs] = cp;
		num_progs++;
	}
  }
  deldirlist();

  while(1) {
	printf("Do you want to specify an additional program (y/n) ? ");
	cp = getstring();
	if (*cp == 'y' || *cp == 'Y') {
		printf("Enter the full path name of the program: ");
		cp = getstring();
		if ((prognames[num_progs] = strdup(cp)) == NULL) {
			fprintf(stderr, "%s: not enough memory for path name\n",
								progname);
			exit(EXIT_MEMORY);
		}
		printf("Enter command line arguments for program: ");
		cp = getstring();
		if ((progoptions[num_progs] = strdup(cp)) == NULL) {
			fprintf(stderr, "%s: not enough memory for options\n",
								progname);
			exit(EXIT_MEMORY);
		}
		num_progs++;
	} else break;
  }
  printf("\n\n");
}



/*
 * Let the user select a bootrom kernel.
 */
static void getkernel()
{
  int num, sel, i;
  struct direntry *dp;
  size_t len;
  char *cp;

  if ((num = getdirlist(kerneldir, "kernel*.bin")) == 0) {
	fprintf(stderr, "%s: Oops, no kernel binaries found!\n", progname);
	exit(EXIT_MAKEROM_KERNFND);
  }
  printf("The following %d kernels are available:\n", num);
  for (dp = dirlist, i = 1; dp != NULL; dp = dp->next, i++)
	printf("(%d)\t%s\n", i, getname(&kerneldesc, dp->fname));
  sel = -1;
  while (sel < 1 || sel > num) {
	printf("Select the kernel you wish to use: ");
	cp = getstring();
	sscanf(cp, "%d", &sel);
  }
  for (dp = dirlist, i = 1; dp != NULL && sel != i; dp = dp->next, i++) ;
  if (dp != NULL) {
	len = strlen(dp->fname) + strlen(kerneldir) + 6;
	if ((kernelname = malloc(len)) == NULL) {
		perror(progname);
		exit(EXIT_MEMORY);
	}
	sprintf(kernelname, "%s/%s.bin", kerneldir, dp->fname);
	if ((i = getoption(&kerneldesc, dp->fname)) >= 0) {
		if (i & K_MINIMAL)
			isminimal++;
		if (i & K_X86)
			is86++;
	}
	printf("Do you want the bootrom to look for boot drives first (y/n) ? ");
	cp = getstring();
	if (*cp == 'y' || *cp == 'Y')
		useint18++;
  }
  deldirlist();
  printf("\n\n\n");
}



/*
 * Main routine which handles all user input
 */
struct bootdef *getuser()
{
  getkernel();
  getpktdrv();
  if (!isminimal)
	getprogram();
  buildconfig();
  return(&bootd);
}

