/*
 * Create and destroy entry list menus and all widgets in them. All widgets
 * are labels or pushbuttons; they are faster than text buttons. Whenever
 * the user presses a button with text in it, it is overlaid with a Text
 * button. For editing and input into the Text button, see dayedit.c.
 *
 *	destroy_all_listmenus()		Destroy all list menus. This is done
 *					when the warning column width changes.
 *	update_all_listmenus()		Redraw all list menus. This is
 *					necessary donee often.
 *	create_entry_widgets(w, nentries)
 *					Creates all widgets in a list menu
 *					that make up the entry list array. If
 *					widgets exist already, make sure that
 *					there are enough and remove excess.
 *	list_widget_pos(x, y, xpos, ypos, width, height, align)
 *					Calculate position and size etc of
 *					each button in the entry list array.
 *
 *	prepare_for_modify(name)	check if other user's file is writable,
 *					mark modified if yes
 *	edit_list_button(doedit, w, x, y)
 *					Turn a text button in the entry list
 *					from Label to Text or vice versa.
 */

#include <stdio.h>
#include <time.h>
#ifndef MIPS
#include <stdlib.h>
#endif
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/LabelP.h>
#include <Xm/LabelG.h>
#include <Xm/PushBP.h>
#include <Xm/PushBG.h>
#include <Xm/ToggleB.h>
#include <Xm/ScrolledW.h>
#include <Xm/Text.h>
#include <Xm/Protocols.h>
#include "cal.h"

#define NOTEWIDTH	180		/* default width of note button */

void create_entry_widgets(), edit_list_button(), list_widget_pos();
static void create_list_widgets();
static void list_confirm(), list_undo(), list_dup(), list_delete(),
	    list_done(), list_pin(), list_own(), list_entry(), list_edit();

extern Display		*display;	/* everybody uses the same server */
extern GC		gc;		/* everybody uses this context */
extern Pixel		color[NCOLS];	/* colors: COL_* */
extern Pixmap		pixmap[NPICS];	/* common symbols */
extern struct config	config;		/* global configuration data */
extern struct mainmenu	mainmenu;	/* all important main window widgets */
extern struct plist	*mainlist;	/* list of all schedule entries */
extern struct user	*user;		/* user list (from file_r.c) */
extern struct edit	edit;		/* info about entry being edited */
extern struct holiday	holiday[366];	/* info for each day, separate for */
extern struct holiday	sm_holiday[366];/* full-line texts under, and small */
extern short		monthbegin[12];	/* julian date each month begins with*/
extern int		curr_month;	/* month being displayed, 0..11 */
extern int		curr_year;	/* year being displayed, since 1900 */
					/* most recent popup created at: */
extern int		edit_month;	/*  month being edited, 0..11 */
extern int		edit_year;	/*  year being edited, since 1900 */
extern int		edit_day;	/*  day being edited, 1..31 or 0 */

static struct listmenu	*listmenu;	/* all important list popup widgets */



/*-------------------------------------------------- popup management -------*/
/*
 * return a pointer to the widget struct for the menu containing widget <w>.
 * This is used to find out in which instantiation of the list popup menu a
 * button exists.
 */

static struct listmenu *find_widget_in_list_menu(w)
	Widget			w;		/* button to find */
{
	struct listmenu		*scan;		/* popup scan pointer */

	while (w = XtParent(w))
		for (scan=listmenu; scan; scan=scan->next)
			if (w == scan->shell)
				return(scan);
	return(0);
}


/*
 * a new popup is requested. Make a listmenu struct, or re-use an old one
 * if there is one that is not pinned. Popups or the widgets they contain
 * are never destroyed; they are merely popped down or declared unpinned.
 * If a nonzero <time> is specified, return the popup for that day if there
 * is one.
 */

static struct listmenu *create_list_menu(time, period, key, entry,
					 user_plus_1, private)
	time_t			time;		/* for what day? */
	time_t			period;		/* for how many days? */
	char			*key;		/* keyword lookup? */
	struct entry		*entry;		/* just this one entry? */
	int			user_plus_1;	/* which user file */
	BOOL			private;	/* only private */
{
	register struct listmenu *scan;		/* popup scan pointer */

	confirm_new_entry();
	for (scan=listmenu; scan; scan=scan->next) {	/* similar menu? */
		if (!scan->valid || !scan->popped)
			continue;
		if (time	&& scan->time == time
				&& scan->period == period		||
		    key		&& scan->key
		    		&& !strcmp(scan->key, key)		||
		    entry	&& entry == scan->oneentry		||
		    user_plus_1	&& user_plus_1 == scan->user_plus_1	||
		    private	&& scan->private)

			return(scan);
	}
	for (scan=listmenu; scan; scan=scan->next)	/* unused menu? */
		if (scan->popped && !scan->pinned)
			break;
	if (!scan)
		for (scan=listmenu; scan; scan=scan->next)
			if (!scan->popped)
				break;
							/* no, make new one */
	if (!scan) {
		if (!(scan=(struct listmenu *)malloc(sizeof(struct listmenu))))
			fatal("no memory");
		scan->popped      = TRUE;
		scan->valid       = FALSE;
		scan->pinned      = FALSE;
		scan->own_only    = config.ownonly;
		scan->entry       = 0;
		scan->nentries    = 0;
		scan->text        = 0;
		scan->sublist     = 0;
		scan->prev        = 0;
		scan->next        = listmenu;
		if (listmenu)
			listmenu->prev = scan;
		listmenu = scan;
	}
	destroy_sublist(scan);
	scan->time     = time;
	scan->period   = period;
	scan->key      = key;
	scan->oneentry = entry;
	scan->private  = private;
	scan->user_plus_1 = user_plus_1;
	return(scan);
}


/*
 * destroy a popup. Remove it from the screen, but do not destroy its
 * widgets. They will be re-used when a new popup is created. Release
 * the sublist so it won't be regenerated if some entry somewhere
 * changes. Turn off the yellow highlight too.
 */

static void destroy_list_menu(lw)
	struct listmenu		*lw;
{
	struct tm		*tm;		/* time to m/d/y conv */
	int			oldday;

	edit_list_button(FALSE, lw, 0, 0);
	XmToggleButtonSetState(lw->pin, lw->pinned = FALSE, FALSE);
	XtPopdown(lw->shell);
	lw->popped = FALSE;
	confirm_new_entry();
	tm = time_to_tm(lw->time);
	if (oldday = edit_day) {
		edit_day = 0;
		draw_day(oldday);
		draw_year_day(oldday, tm->tm_mon, tm->tm_year);
		draw_week_day(oldday, tm->tm_mon, tm->tm_year);
	}
	destroy_sublist(lw);
}


/*
 * the main list changed, and all list menus must be re-evaluated and
 * redrawn. This routine is here because this is the only file that
 * knows about all list menus.
 */

void update_all_listmenus()
{
	struct listmenu		*scan;		/* popup scan pointer */

	for (scan=listmenu; scan; scan=scan->next)
		if (scan->popped) {
			create_sublist(mainlist, scan);
			draw_list(scan);
		}
}


/*
 * destroy all list menus. This is done when the width of the advance-warning
 * column is changed to a text entry field, using the fast-warning-entry
 * option in the config pulldown.
 */

void destroy_all_listmenus()
{
	struct listmenu		*scan, *next;	/* popup scan pointer */

	confirm_new_entry();
	for (scan=listmenu; scan; scan=next) {
		next = scan->next;
		if (scan->popped)
			destroy_list_menu(scan);
		if (scan->valid)
			XTDESTROYWIDGET(scan->shell);
		free((char *)scan);
	}
	listmenu = 0;
	edit.menu = 0;
}


/*-------------------------------------------------- menu creation ----------*/
/*
 * create a list popup as a separate application shell. Use the argument
 * list to decide how many buttons to create initially.
 */

void create_list_popup(list, time, period, key, entry, user_plus_1, private)
	struct plist		*list;
	time_t			time;		/* for what day? */
	time_t			period;		/* for how many days? */
	char			*key;		/* keyword lookup? */
	struct entry		*entry;		/* just this one entry? */
	int			user_plus_1;	/* which user file */
	BOOL			private;	/* only private */
{
	int			n = list->nentries + 5;
	struct listmenu		*w;
	char			title[200];	/* menu title string */
	char			*p;		/* temp for holiday lookup */
	struct tm		*tm;		/* time to m/d/y conv */
	struct holiday		*hp, *shp;	/* to check for holidays */

	if (!can_edit_appts())
		return;
	time   -= time   % 86400;
	period -= period % 86400;
	w = create_list_menu(time, period, key, entry, user_plus_1, private);
	create_list_widgets(w, n - n%5);
	if (time) {
		strcpy(title, mkdatestring(time));
		if (period) {
			strcat(title, " - ");
			strcat(title, mkdatestring(time + period - 1));
		} else {
			tm  = time_to_tm(time);
			if (p = parse_holidays(tm->tm_year, FALSE))
				create_error_popup(mainmenu.cal, 0, p);
			shp = &sm_holiday[tm->tm_yday];
			hp  =    &holiday[tm->tm_yday];
			if (shp->string || hp->string) {
				p = " (";
				if (shp->string) {
					strcat(title, p);
					strcat(title, shp->string);
					p = ", ";
				}
				if (hp->string) {
					strcat(title, p);
					strcat(title, hp->string);
				}
				strcat(title, ")");
			}
		}
	} else if (w->key) {
		sprintf(title, "/%.40s/", w->key);
	} else if (entry) {
		strcpy(title, "Entry from week view");
	} else if (user_plus_1) {
		strcpy(title, user_plus_1 == 1 ? "Own"
					       : user[user_plus_1-1].name);
	} else if (private) {
		strcpy(title, "Private");
	} else
		strcpy(title, "All Entries");

	print_button(w->title, title);
	create_sublist(list, w);
	draw_list(w);
}


/*
 * create all widgets for a schedule list popup, including the popup window
 * itself. The number of initial entry rows in the list is set to <nentries>.
 */

static void create_list_widgets(w, nentries)
	struct listmenu		*w;		/* create which menu */
	int			nentries;	/* how many entries */
{
	Widget			form, scroll, help;
	Arg			args[15];
	int			n, width;
	Atom			closewindow;

	if (w->valid) {
		XtPopup(w->shell, XtGrabNone);
		w->popped = TRUE;
		XRaiseWindow(display, XtWindow(w->shell));
		return;
	}
	w->valid = TRUE;

	n = 0;
	XtSetArg(args[n], XmNdeleteResponse,	XmUNMAP);		n++;
	XtSetArg(args[n], XmNiconic,		False);			n++;
	w->shell = XtAppCreateShell("Schedule", "plan",
			applicationShellWidgetClass, display, args, n);
#	ifdef EDITRES
	XtAddEventHandler(w->shell, (EventMask)0, TRUE, 
 			_XEditResCheckMessages, NULL);
#	endif
	set_icon(w->shell, 1);
	form = XtCreateWidget("dayform", xmFormWidgetClass,
			w->shell, NULL, 0);
	XtAddCallback(form, XmNhelpCallback,
					help_callback, (XtPointer)"day");

							/*-- title -- */
	n = 0;
	XtSetArg(args[n], XmNleftAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNrightAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNtopAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNtopOffset,		8);			n++;
	w->title = XtCreateManagedWidget("title", xmLabelGadgetClass,
			form, args, n);
							/*-- buttons --*/
	n = 0;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNbottomOffset,	8);			n++;
	XtSetArg(args[n], XmNleftAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNleftOffset,	8);			n++;
	XtSetArg(args[n], XmNwidth,		80);			n++;
	XtSetArg(args[n], XmNsensitive,		False);			n++;
	w->confirm = XtCreateManagedWidget("Confirm", xmPushButtonWidgetClass,
			form, args, n);
	XtAddCallback(w->confirm, XmNactivateCallback,
				list_confirm, (XtPointer)0);
	XtAddCallback(w->confirm, XmNhelpCallback,
				help_callback, (XtPointer)"day_confirm");

	n = 0;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNbottomOffset,	8);			n++;
	XtSetArg(args[n], XmNleftAttachment,	XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNleftWidget,	w->confirm);		n++;
	XtSetArg(args[n], XmNleftOffset,	8);			n++;
	XtSetArg(args[n], XmNwidth,		80);			n++;
	XtSetArg(args[n], XmNsensitive,		False);			n++;
	w->undo = XtCreateManagedWidget("Undo", xmPushButtonWidgetClass,
			form, args, n);
	XtAddCallback(w->undo, XmNactivateCallback,
					list_undo, (XtPointer)0);
	XtAddCallback(w->undo, XmNhelpCallback,
					help_callback, (XtPointer)"day_undo");

	n = 0;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNbottomOffset,	8);			n++;
	XtSetArg(args[n], XmNleftAttachment,	XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNleftWidget,	w->undo);		n++;
	XtSetArg(args[n], XmNleftOffset,	8);			n++;
	XtSetArg(args[n], XmNwidth,		80);			n++;
	XtSetArg(args[n], XmNsensitive,		False);			n++;
	w->dup = XtCreateManagedWidget("Dup", xmPushButtonWidgetClass,
			form, args, n);
	XtAddCallback(w->dup, XmNactivateCallback, list_dup, (XtPointer)0);
	XtAddCallback(w->dup, XmNhelpCallback,     help_callback,
							(XtPointer)"day_dup");

	n = 0;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNbottomOffset,	8);			n++;
	XtSetArg(args[n], XmNleftAttachment,	XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNleftWidget,	w->dup);		n++;
	XtSetArg(args[n], XmNleftOffset,	8);			n++;
	XtSetArg(args[n], XmNwidth,		80);			n++;
	XtSetArg(args[n], XmNsensitive,		False);			n++;
	w->del = XtCreateManagedWidget("Delete", xmPushButtonWidgetClass,
			form, args, n);
	XtAddCallback(w->del, XmNactivateCallback, list_delete, (XtPointer)0);
	XtAddCallback(w->del, XmNhelpCallback,     help_callback,
							(XtPointer)"day_del");

	n = 0;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNbottomOffset,	8);			n++;
	XtSetArg(args[n], XmNleftAttachment,	XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNleftWidget,	w->del);		n++;
	XtSetArg(args[n], XmNleftOffset,	32);			n++;
	XtSetArg(args[n], XmNwidth,		80);			n++;
	help = XtCreateManagedWidget("Help", xmPushButtonWidgetClass,
			form, args, n);
	XtAddCallback(help, XmNactivateCallback,
					help_callback, (XtPointer)"day");
	XtAddCallback(help, XmNhelpCallback,
					help_callback, (XtPointer)"day");

	n = 0;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNbottomOffset,	8);			n++;
	XtSetArg(args[n], XmNleftAttachment,	XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNleftWidget,	help);			n++;
	XtSetArg(args[n], XmNleftOffset,	32);			n++;
	XtSetArg(args[n], XmNwidth,		80);			n++;
	w->done = XtCreateManagedWidget("Done", xmPushButtonWidgetClass,
			form, args, n);
	XtAddCallback(w->done, XmNactivateCallback, list_done, (XtPointer)0);
	XtAddCallback(w->done, XmNhelpCallback,     help_callback,
							(XtPointer)"day_done");

	n = 0;
	XtSetArg(args[n], XmNselectColor,	color[COL_RED]);	n++;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNbottomOffset,	8);			n++;
	XtSetArg(args[n], XmNrightAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNrightOffset,	8);			n++;
	w->pin = XtCreateManagedWidget("Pin", xmToggleButtonWidgetClass,
			form, args, n);
	XtAddCallback(w->pin, XmNvalueChangedCallback, list_pin, (XtPointer)0);
	XtAddCallback(w->pin, XmNhelpCallback, help_callback,
							(XtPointer)"day_pin");

	n = 0;
	XtSetArg(args[n], XmNselectColor,	color[COL_TOGGLE]);	n++;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNbottomOffset,	8);			n++;
	XtSetArg(args[n], XmNrightAttachment,	XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNrightWidget,	w->pin);		n++;
	XtSetArg(args[n], XmNrightOffset,	8);			n++;
	XtSetArg(args[n], XmNset,		config.ownonly);	n++;
	w->own = XtCreateManagedWidget("Own Only", xmToggleButtonWidgetClass,
			form, args, n);
	XtAddCallback(w->own, XmNvalueChangedCallback, list_own, (XtPointer)0);
	XtAddCallback(w->own, XmNhelpCallback, help_callback,
							(XtPointer)"day_own");

							/*-- scroll --*/
	n = 0;
	width = (config.bigwarning ? 640 : 590) +
		(config.notewidth > 9 ? config.notewidth : NOTEWIDTH);
	XtSetArg(args[n], XmNtopAttachment,	XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNtopWidget,		w->title);		n++;
	XtSetArg(args[n], XmNtopOffset,		8);			n++;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNbottomWidget,	w->done);		n++;
	XtSetArg(args[n], XmNbottomOffset,	16);			n++;
	XtSetArg(args[n], XmNleftAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNleftOffset,	8);			n++;
	XtSetArg(args[n], XmNrightAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNrightOffset,	8);			n++;
	XtSetArg(args[n], XmNwidth,		width);			n++;
	XtSetArg(args[n], XmNheight,		220);			n++;
	XtSetArg(args[n], XmNscrollingPolicy,	XmAUTOMATIC);		n++;
	scroll = XtCreateWidget("lscroll", xmScrolledWindowWidgetClass,
			form, args, n);
	XtAddCallback(scroll, XmNhelpCallback,
					help_callback, (XtPointer)"day");

	n = 0;
	w->list = XtCreateWidget("list", xmBulletinBoardWidgetClass,
			scroll, args, n);
	XtManageChild(w->list);

	create_entry_widgets(w, nentries+5 - nentries%5);

	XtManageChild(scroll);
	XtManageChild(form);
	XtPopup(w->shell, XtGrabNone);

	closewindow = XmInternAtom(display, "WM_DELETE_WINDOW", False);
	XmAddWMProtocolCallback(w->shell, closewindow,
					list_done, (XtPointer)w->shell);
}


/*
 * makes sure there are enough widget rows for schedule entries. Also makes
 * sure that there aren't too many, for speed reasons. As usual, allocate
 * one extra widget row for the title at the top. Each row consists of
 * SC_N different widgets. This can be called again later with a larger
 * nentries argument, to make the list longer. All the text buttons are
 * label widgets. For performance reasons, they are overlaid by a text
 * widget when pressed. This doesn't make much difference on an Indigo,
 * but it might on other platforms and I'd feel bad in the morning otherwise.
 * No text is printed into the buttons yet, this is done later by draw_list().
 */

static char *help_name[SC_N] = {
	"day_enable", "day_user", "day_date", "day_time", "day_length",
	"day_advance", "day_recycle", "day_message", "day_script",
	"day_except", "day_private", "day_note"
};

void create_entry_widgets(w, nentries)
	struct listmenu		*w;
	int			nentries;
{
	int			x, y;
	Arg			args[15];
	int			n;
	int			xpos, ypos;
	int			width, height;
	int			align;
	char			*name;
	WidgetClass		class;

	nentries = 5;					/* check # of rows: */
	n = w->sublist ? w->sublist->nentries : 0;
	if (n+2 > w->nentries) {			/* need more? */
		nentries = n+5 - n%5;

	} else if (n+5 < w->nentries) {			/* have too many? */
		nentries = n+5 - n%5;
		for (y=nentries+1; y <= w->nentries; y++)
			for (x=0; x < SC_N; x++)
				XtDestroyWidget(w->entry[y][x]);
		if (w->nentries > nentries)
			w->nentries = nentries;
		return;
	} else						/* within tolerance */
		return;

	n = (nentries+1) * SC_N * sizeof(Widget *);
	if (w->entry && !(w->entry = (Widget (*)[])realloc(w->entry, n)) ||
	   !w->entry && !(w->entry = (Widget (*)[])malloc(n)))
		fatal("no memory");

	for (x=0; x < SC_N; x++) {
	    for (y=w->nentries; y <= nentries; y++) {
		list_widget_pos(x, y, &xpos, &ypos, &width, &height, &align);
		XtUnmanageChild(w->list);
		class  = xmPushButtonWidgetClass;
		name   = "";
		n = 0;
		switch(x) {
		  case SC_ENABLE:
			class  = xmToggleButtonWidgetClass;
			XtSetArg(args[n], XmNselectColor, color[COL_TOGGLE]);
			n++;
			break;
		  case SC_USER:
			name   = "Group";
			break;
		  case SC_DATE:
			name   = "Date";
			break;
		  case SC_TIME:
			name   = "Time";
			break;
		  case SC_LENGTH:
			name   = "Length";
			break;
		  case SC_ADVANCE:
			if (config.bigwarning) {
				name = "Warn";
				break;
			} /* fall through */
		  case SC_RECYCLE:
		  case SC_MESSAGE:
		  case SC_SCRIPT:
		  case SC_EXCEPT:
		  case SC_PRIVATE:
			XtSetArg(args[n], XmNlabelType,   XmPIXMAP);    n++;
			XtSetArg(args[n], XmNlabelPixmap,
				pixmap[y ? PIC_BLANK : x-SC_FLAGS]);    n++;
			break;
		  case SC_NOTE:
			name  = "Note";
		}
		if (x && y)
			name  = " ";
		if (y == 0)
			class = xmLabelWidgetClass;

		XtSetArg(args[n], XmNx,			xpos);		n++;
		XtSetArg(args[n], XmNy,			ypos);		n++;
		XtSetArg(args[n], XmNwidth,		width);		n++;
		XtSetArg(args[n], XmNheight,		height);	n++;
		XtSetArg(args[n], XmNalignment,         align);		n++;
		XtSetArg(args[n], XmNrecomputeSize,	False);		n++;
		XtSetArg(args[n], XmNtraversalOn,	True);		n++;
		XtSetArg(args[n], XmNhighlightThickness,0);		n++;
		XtSetArg(args[n], XmNshadowThickness,	x && y);	n++;
		w->entry[y][x] = XtCreateManagedWidget(name, class,
				w->list, args, n);
		if (y && x)
			XtAddCallback(w->entry[y][x], XmNactivateCallback,
				list_entry, (XtPointer)(x + y * SC_N));
		if (y && !x)
			XtAddCallback(w->entry[y][x], XmNvalueChangedCallback,
				list_entry, (XtPointer)(x + y * SC_N));
		XtAddCallback(w->entry[y][x], XmNhelpCallback, help_callback,
						(XtPointer)help_name[x]);
	    }
	}
	w->nentries = nentries;
	XtManageChild(w->list);
}


/*
 * return position, size, and alignment of a widget in the scroll list
 */

void list_widget_pos(x, y, xpos, ypos, width, height, align)
	int			x, y;		/* row/column */
	int			*xpos, *ypos;	/* returned pixel pos */
	int			*width;		/* returned width */
	int			*height;	/* returned height */
	int			*align;		/* ret. text alignment: l,c,r*/
{
	int			warn_xs;	/* width of warning column */

	*width  = 70;
	*height = 30;
	*ypos   = 10 + y * *height;
	*align  = XmALIGNMENT_CENTER;
	warn_xs = config.bigwarning ? *width : 30;

	switch(x) {
	  case SC_ENABLE:
		*xpos  = 4;
		*width = 30;
		break;
	  case SC_USER:
		*xpos  = 4+40;
		*width = 70;
		break;
	  case SC_DATE:
		*xpos  = 4+40+70;
		*width = 125;
		break;
	  case SC_TIME:
		*xpos  = 4+40+70+125;
		*width = 70;
		break;
	  case SC_LENGTH:
		*xpos  = 4+40+70+125+70;
		*width = 70;
		break;
	  case SC_ADVANCE:
		*xpos  = 4+40+70+125+70+70;
		*width = warn_xs;
		break;
	  case SC_RECYCLE:
	  case SC_MESSAGE:
	  case SC_SCRIPT:
	  case SC_EXCEPT:
	  case SC_PRIVATE:
		*xpos  = 4+40+70+125+70+70+warn_xs+30*(x-SC_FLAGS-1);
		*width = 30;
		break;
	  case SC_NOTE:
		*xpos  = 4+40+70+125+70+70+warn_xs+30*(x-SC_FLAGS-1);
		*width = config.notewidth > 9 ? config.notewidth : NOTEWIDTH;
		*align = XmALIGNMENT_BEGINNING;
	}
	if (y == 0)
		*align = XmALIGNMENT_CENTER;
	--*width; --*height;
}


/*-------------------------------------------------- editing ----------------*/
/*
 * turn a text label into a Text button, to allow user input. This is done
 * by simply installing a text widget on top of the label widget. <text>
 * is put into the text widget. The previously edited button is un-edited.
 */

void edit_list_button(doedit, w, x, y)
	BOOL			doedit;		/* TRUE=edit, FALSE=unedit */
	struct listmenu		*w;		/* create which menu */
	int			x;		/* column, SC_* */
	int			y;		/* row, y=0: title */
{
	Arg			args[15];
	int			n;
	int			xpos, ypos;
	int			width, height;
	int			align;
	char			*text;
	char			*space;
	static BOOL		busy;		/* not re-entrant */

	if (busy || !w)				/* not editing */
		return;
	if (w->text) {
		char *string = XmTextGetString(w->text);
		busy = TRUE;
		got_entry_text(w, w->xedit, w->yedit, string);
		busy = FALSE;
		XtFree(string);
		XmProcessTraversal(w->list, XmTRAVERSE_CURRENT);
		XtDestroyWidget(w->text);
		if (x && y)
			print_button(w->entry[y][x], "%s",
						mkbuttonstring(w, x, y));
	}
	w->text = 0;
	if (!doedit)
		return;

	list_widget_pos(x, y, &xpos, &ypos, &width, &height, &align);
	n = 0;
	XtSetArg(args[n], XmNx,			xpos);		n++;
	XtSetArg(args[n], XmNy,			ypos);		n++;
	XtSetArg(args[n], XmNwidth,		width);		n++;
	XtSetArg(args[n], XmNheight,		height);	n++;
	XtSetArg(args[n], XmNrecomputeSize,	False);		n++;
	XtSetArg(args[n], XmNpendingDelete,	True);		n++;
	XtSetArg(args[n], XmNhighlightThickness,0);		n++;
	XtSetArg(args[n], XmNshadowThickness,	1);		n++;
	XtSetArg(args[n], XmNbackground,	color[COL_TEXTBACK]);	n++;
	w->text = XtCreateManagedWidget("text", xmTextWidgetClass,
			w->list, args, n);
	XtAddCallback(w->text, XmNactivateCallback, list_edit,(XtPointer)NULL);
	XtAddCallback(w->text, XmNhelpCallback, help_callback,
						(XtPointer)help_name[x]);
	XmProcessTraversal(w->text, XmTRAVERSE_CURRENT);

	text = "";
	if (w->sublist && w->sublist->nentries >= y) {
		text = mkbuttonstring(w, x, y);
		if (x == SC_DATE && (space = strchr(text, ' ')))
			text = space+1;
	}
	print_text_button(w->text, "%s", text);
	w->xedit = x;
	w->yedit = y;
}


/*
 * prepare for changing an entry of user <name>. Set the modified flag
 * for that user (so write_mainlist will write it later), or print an
 * error popup if the user's file is not writable. Writing to files
 * will start only when mainlist->modified is set, which has the side
 * effect of *always* writing out own files, but it's good enough for
 * now. Return TRUE if writing can proceed, or FALSE on error.
 */

BOOL prepare_for_modify(name)
	char		*name;		/* user name or 0 */
{
	int		u;		/* index into user array */

	u = name_to_user(name);
	if (user[u].readonly) {
		if (user[u].fromserver)
			create_error_popup(0, 0,
"Cannot modify appointments of %s,\n\
server on %s denies write permission.\n\
Re-check the permissions with File->Reread database.\n\
You can not disable individual appointments, but you\n\
can disable all with the Config->User list popup.",
					name, user[u].server);
		else if (user[u].grok)
			create_error_popup(0, 0,
"Cannot modify appointments of %s because they\n\
were read from a grok database. Saving is not\n\
implemented for grok databases. The file is\n%s",
					name, user[u].path);
		else
			create_error_popup(0, 0,
"Cannot modify appointments of %s,\n\
no write permission for %s\n\
or %s/.dayplan.\n\n\
Re-check the permissions with File->Reread database.\n\
You can not disable individual appointments, but you\n\
can disable all with the Config->User list popup.",
					name, user[u].path, user[u].path);
		return(FALSE);
	}
	user[u].modified = TRUE;
	return(TRUE);
}


/*-------------------------------------------------- callbacks --------------*/
/*
 * dup, Delete, and Done buttons
 * All of these routines are direct X callbacks.
 */

/*ARGSUSED*/
static void list_confirm(widget, item, data)
	Widget				widget;
	int				item;
	XmToggleButtonCallbackStruct	*data;
{
	if (find_widget_in_list_menu(widget) == edit.menu)
		confirm_new_entry();
}


/*ARGSUSED*/
static void list_undo(widget, item, data)
	Widget				widget;
	int				item;
	XmToggleButtonCallbackStruct	*data;
{
	struct listmenu *lw = find_widget_in_list_menu(widget);
	if (lw == edit.menu && prepare_for_modify(edit.entry.user)) {
		undo_new_entry();
		edit_list_button(FALSE, lw, 0, 0);
	}
}


/*ARGSUSED*/
static void list_dup(widget, item, data)
	Widget				widget;
	int				item;
	XmToggleButtonCallbackStruct	*data;
{
	struct entry tmp;
	struct listmenu *lw = find_widget_in_list_menu(widget);
	if (lw) {
		int num = lw->sublist ? lw->sublist->nentries : 0;
		if (edit.editing && lw == edit.menu && edit.y-1 < num
				 && prepare_for_modify(edit.entry.user)) {
			clone_entry(&tmp, &edit.entry);
			confirm_new_entry();
			tmp.file     = 0;
			tmp.id       = 0;
			edit.entry   = tmp;
			edit.y       = num+1;
			edit.editing = TRUE;
			draw_list(lw);
			edit.changed = TRUE;
			sensitize_edit_buttons();
			got_entry_press(lw, SC_DATE, edit.y, FALSE);
		}
	}
}

/*ARGSUSED*/
static void list_delete(widget, item, data)
	Widget				widget;
	int				item;
	XmToggleButtonCallbackStruct	*data;
{
	struct listmenu *lw = find_widget_in_list_menu(widget);
	if (!edit.editing || lw != edit.menu
			  || !edit.menu->sublist
			  || edit.y-1 >= edit.menu->sublist->nentries
			  || !prepare_for_modify(edit.entry.user))
		return;

	destroy_recycle_popup();
	destroy_advance_popup();
	destroy_text_popup();
	edit_list_button(FALSE, lw, 0, 0);
	edit.editing = FALSE;
	edit.changed = FALSE;
	sensitize_edit_buttons();

	server_entry_op(&edit.entry, 'd');
	delete_entry(mainlist, lw->sublist->entry[edit.y-1] - mainlist->entry);

	rebuild_repeat_chain(mainlist);
	draw_list(edit.menu);
	update_all_listmenus();
	redraw_all_views();
}

/*ARGSUSED*/
static void list_done(widget, item, data)
	Widget				widget;
	int				item;
	XmToggleButtonCallbackStruct	*data;
{
	struct listmenu *lw = find_widget_in_list_menu(widget);
	if (lw)
		destroy_list_menu(lw);
}

/*ARGSUSED*/
static void list_pin(widget, item, data)
	Widget				widget;
	int				item;
	XmToggleButtonCallbackStruct	*data;
{
	struct listmenu *lw = find_widget_in_list_menu(widget);
	if (lw)
		lw->pinned = data->set;
}

/*ARGSUSED*/
static void list_own(widget, item, data)
	Widget				widget;
	int				item;
	XmToggleButtonCallbackStruct	*data;
{
	struct listmenu *lw = find_widget_in_list_menu(widget);
	if (lw) {
		lw->own_only = config.ownonly = data->set;
		create_sublist(mainlist, lw);
		draw_list(lw);
	}
}

/*ARGSUSED*/
static void list_entry(widget, item, data)
	Widget				widget;
	int				item;
	XmToggleButtonCallbackStruct	*data;
{
	struct listmenu			*lw = find_widget_in_list_menu(widget);
	int				x = item % SC_N;
	int				y = item / SC_N;

	if (lw)
		got_entry_press(lw, x, y, data->set);
}

/*ARGSUSED*/
static void list_edit(widget, item, data)
	Widget				widget;
	int				item;
	XmToggleButtonCallbackStruct	*data;
{
	struct listmenu			*lw = find_widget_in_list_menu(widget);
	char				*text = XmTextGetString(widget);

	if (lw)
		got_entry_text(lw, lw->xedit, lw->yedit, text);
	XtFree(text);
}
