/* -*- C++ -*-
 * $Id: APP_thread.h,v 1.3 1997/05/01 21:34:46 wg Exp $
 *
 * Purpose:   Basic thread support for wxWindows.
 * Author:    Wolfram Gloger
 * Copyright: (C) 1996, 1997 Wolfram Gloger
 * License:   GNU Library General Public License
 */

#ifndef wx_thrd_h
#define wx_thrd_h

#ifdef __GNUG__
#pragma interface
#endif

#if USE_THREADS_POSIX

#include <pthread.h>

#elif USE_THREADS_SGI

#include <abi_mutex.h>

#endif

/// Mutual exclusion type.

class
wxMutex
{
public:
	/// Constructor.
	wxMutex();
	/// Destructor.
	~wxMutex();
	/** Acquire the lock unconditionally.  Returns zero for success,
	  and non-zero in case of error. */
	int Lock() {
#if USE_THREADS_POSIX
		return pthread_mutex_lock(&mutex);
#elif USE_THREADS_SGI
		spin_lock(&mutex);
		return 0;
#else
		return 0;
#endif
	}
	/** Attempt to acquire the lock.  Returns zero if the lock was
	  acquired, and non-zero otherwise. */
	int TryLock() {
#if USE_THREADS_POSIX
		return pthread_mutex_trylock(&mutex);
#elif USE_THREADS_SGI
		return acquire_lock(&mutex);
#else
		return 0;
#endif
	}
	/** Release the lock.  Returns zero for success, and non-zero in
	  case of error. */
	int Unlock() {
#if USE_THREADS_POSIX
		return pthread_mutex_unlock(&mutex);
#elif USE_THREADS_SGI
		return release_lock(&mutex);
#else
		return 0;
#endif
	}
private:
#if USE_THREADS_POSIX
	pthread_mutex_t mutex;
#elif USE_THREADS_SGI
	abilock_t mutex;
#endif
};

/** The main global mutex.  In {\em any} subthread, this must be
  locked before a GUI operation (call to a wxWindows function or
  e.g. a direct call to an X11 function) is invoked by that thread.
  In the {\em main} thread (started from #wxApp::OnInit# and running
  the main event loop), #wxMainMutex# is automatically held locked, so
  it must not be dealt with. */
extern wxMutex wxMainMutex;

class wxThreadPrivate;

/// The thread class.  Add your own functionality by overriding #Entry()#.

class
wxThread: public wxObject
{
public:
	/// Constructor.
	wxThread();

	/** Destructor.  If the thread is still running, it is cancelled
	  and joined first.  Therefore, in some circumstances, this
	  operation may block. */
	virtual ~wxThread();

	/// Start the thread. Returns zero for success, and non-zero otherwise.
	int           Create();
	/** Wait for thread termination and obtain result as returned
	  by \Ref{Entry} or \Ref{Exit}. */
	void*         Join();
	/** Request thread termination.  Cancellation is synchronous, i.e.
	  it is only performed when the running thread calls
	  \Ref{TestCancel} or (on POSIX-compliant systems) blocks in a
	  system call. */
	int           Cancel();
	/// Obtain a unique integral identification for the running thread.
	unsigned long GetID();
	/** This member is called from wxApp::MainLoop() on normal thread
	  termination (but not on cancellation). The default version of
	  this function simply calls \Ref{Join} and returns. */
	virtual void  OnExit();
protected:
	/** Terminate thread with a result of #status# (equivalent to
	  returning from the \Ref{Entry} function).  This member can only
	  be called from {\em inside} the running thread. */
	void Exit(void* status = 0);
	/** Test for requested termination.  This member can only be
	  called from {\em inside} the running thread. */
	void TestCancel();
	/** Enable cancellation.  This member can only be called from {\em
	  inside} the running thread. */
	int EnableCancel() {
#if USE_THREADS_POSIX
		int oldstate;
		return pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, &oldstate);
#else
		return -1;
#endif
	}
	/** Disable cancellation.  This member can only be called from
	  {\em inside} the running thread. */
	int DisableCancel() {
#if USE_THREADS_POSIX
		int oldstate;
		return pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, &oldstate);
#else
		return -1;
#endif
	}
	/** Acquire the main mutex and disable cancellation.  This should
	  be called before invoking non-threadsafe wxWin functions. */
	int LockMain() {
		DisableCancel();
		return wxMainMutex.Lock();
	}
	/** Release the main mutex and enable cancellation. */
	int UnlockMain() {
		EnableCancel();
		return wxMainMutex.Unlock();
	}
private:
	/** The member function implementing the thread's functionality.
	  It is called every time the thread is started by \Ref{Create}.
	  This function must be overridden in a derived class. */
	virtual void* Entry() = 0;
	friend class wxThreadPrivate;
	wxThreadPrivate *p;
};

/// Returns true iff called by the main thread.
Bool wxThreadIsMain();

#endif /* wx_thrd_h */

/*
 * Local variables:
 * tab-width: 4
 * End:
 */
