/**********************************************************************
 Freeciv - Copyright (C) 1996 - A Kjeldberg, L Gregersen, P Unold
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
***********************************************************************/
#include <stdio.h>
#include <stdlib.h>

#include <player.h>
#include <unithand.h>
#include <civserver.h>
#include <map.h>
#include <maphand.h>
#include <mapgen.h>
#include <cityhand.h>
#include <unit.h>
#include <city.h>
#include <player.h>
#include <tech.h>
#include <shared.h>
#include <plrhand.h>
#include <events.h>
#include <aicity.h>

int update_city_activity(struct player *pplayer, struct city *pcity);
extern struct unit_type unit_types[];
extern struct improvement_type improvement_types[];
void auto_arrange_workers(struct city *pcity);
void city_check_workers(struct player *pplayer, struct city *pcity);

/**************************************************************************
...
**************************************************************************/
void create_city(struct player *pplayer, int x, int y, char *name)
{
  struct city *pcity;
  int i;
  
  pcity=(struct city *)malloc(sizeof(struct city));

  pcity->id=get_next_id_number();
  pcity->owner=pplayer->player_no;
  pcity->x=x;
  pcity->y=y;
  strcpy(pcity->name, name);
  pcity->size=1;
  pcity->ppl_elvis=1;
  pcity->ppl_scientist=pcity->ppl_taxman=0;
  pcity->ppl_happy[4]=0;
  pcity->ppl_content[4]=1;
  pcity->ppl_unhappy[4]=0;
  pcity->was_happy=0;
  pcity->steal=0;
  for (i=0;i<4;i++)
    pcity->trade[i]=0;
  pcity->food_stock=0;
  pcity->shield_stock=0;
  pcity->trade_prod=0;
  pcity->original = pplayer->player_no;
  pcity->is_building_unit=1;
  pcity->did_buy=1;
  pcity->airlift=0;
  if (can_build_unit(pcity, U_RIFLEMEN))
      pcity->currently_building=U_RIFLEMEN;
  else if (can_build_unit(pcity, U_MUSKETEERS))
      pcity->currently_building=U_MUSKETEERS;
  else if (can_build_unit(pcity, U_PIKEMEN))
      pcity->currently_building=U_PIKEMEN;
  else if (can_build_unit(pcity, U_PHALANX))
      pcity->currently_building=U_PHALANX;
  else
      pcity->currently_building=U_WARRIORS;
  for (y = 0; y < CITY_MAP_SIZE; y++)
    for (x = 0; x < CITY_MAP_SIZE; x++)
      pcity->city_map[x][y]=C_TILE_EMPTY;

  for(i=0; i<B_LAST; i++)
    pcity->improvements[i]=0;
  if(!pplayer->capital) {
    pplayer->capital=1;
    pcity->improvements[B_PALACE]=1;
  }
  pcity->anarchy=0;
  pcity->ai.ai_role = AICITY_NONE;
  map_set_city(pcity->x, pcity->y, pcity);
  
  unit_list_init(&pcity->units_supported);
  city_list_insert(&pplayer->cities, pcity);

  city_check_workers(pplayer, pcity);
  auto_arrange_workers(pcity);

  city_refresh(pcity);
  city_incite_cost(pcity);
  send_city_info(0, pcity, 0);
  ai_calculate_city_value(map_get_continent(pcity->x, pcity->y));
}

/**************************************************************************
...
**************************************************************************/
void set_food_trade_shields(struct city *pcity)
{
  int x,y;
  pcity->food_prod=0;
  pcity->shield_prod=0;
  pcity->trade_prod=0;

  pcity->food_surplus=0;
  pcity->shield_surplus=0;
  pcity->corruption=0;

  city_map_iterate(x, y) {
    if(get_worker_city(pcity, x, y)==C_TILE_WORKER) {
      pcity->food_prod+=get_food_tile(x, y, pcity);
      pcity->shield_prod+=get_shields_tile(x, y, pcity);
      pcity->trade_prod+=get_trade_tile(x, y, pcity);
    }
  }
  
  pcity->food_surplus=pcity->food_prod-pcity->size*2;
  set_trade_prod(pcity);
}

/**************************************************************************
...
**************************************************************************/
int city_refresh(struct city *pcity)
{
  set_food_trade_shields(pcity);
  citizen_happy_size(pcity);
  set_tax_income(pcity);                  /* calc base luxury, tax & bulbs */
  add_buildings_effect(pcity);            /* marketplace, library wonders.. */
  set_pollution(pcity);
  citizen_happy_luxury(pcity);            /* with our new found luxuries */
  citizen_happy_buildings(pcity);         /* temple cathedral colosseum */
  city_support(pcity);                    /* manage settlers, and units */
  citizen_happy_wonders(pcity);           /* happy wonders */
  unhappy_city_check(pcity);
  return (is_city_happy(pcity) && pcity->was_happy);
}

/**************************************************************************
...
**************************************************************************/
void city_settlersupport(struct city *pcity)
{
  struct genlist_iterator myiter;
  struct unit *this_unit;
  genlist_iterator_init(&myiter, &pcity->units_supported.list, 0);
  for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
    this_unit=(struct unit*)ITERATOR_PTR(myiter);
    if (unit_flag(this_unit->type, F_SETTLERS)) {
      pcity->food_surplus--;
      this_unit->upkeep=1;
      if (get_government(pcity->owner)>=G_COMMUNISM) {
	pcity->food_surplus--;
	this_unit->upkeep=2;
      }
    }
  }
}
/**************************************************************************
...
**************************************************************************/

void city_support(struct city *pcity)
{ 
  struct genlist_iterator myiter;
  int milunits=0;
  int city_units=0;
  struct unit *this_unit;
  int unhap=0;
  int gov=get_government(pcity->owner);
  happy_copy(pcity, 2);
  city_settlersupport(pcity);
  
  genlist_iterator_init(&myiter, 
			&(map_get_tile(pcity->x, pcity->y)->units.list), 0);
  for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
    this_unit=(struct unit *)ITERATOR_PTR(myiter);
    if (is_military_unit(this_unit))
      city_units++;
  }

  genlist_iterator_init(&myiter, &(pcity->units_supported.list), 0);
  for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
    this_unit=(struct unit *)ITERATOR_PTR(myiter);
    this_unit->unhappiness=0;
    if (!unit_flag(this_unit->type, F_SETTLERS))
      this_unit->upkeep=0;
    if (is_military_unit(this_unit)) {
      milunits++;
      switch (gov) {
      case G_ANARCHY:
      case G_DESPOTISM:
	if (milunits>pcity->size) {
	  pcity->shield_surplus--;
	  this_unit->upkeep=1;
	} 
	break;
      case G_MONARCHY:
  	if (milunits>3) {
 	  pcity->shield_surplus--;
	  this_unit->upkeep=1;
	} 
	break;
      case G_COMMUNISM:
	if (milunits>3) {
	  pcity->shield_surplus--;
	  this_unit->upkeep=1;
	} 
	break;
      case G_REPUBLIC:
	pcity->shield_surplus--;
	this_unit->upkeep=1;
	if(get_unit_type(this_unit->type)->attack_strength && 
	    !map_get_city(this_unit->x ,this_unit->y )) {
	  if (unhap)
	    this_unit->unhappiness=1;
	  unhap++;
	} else if (is_field_unit(this_unit)) {
	  if (unhap)
	    this_unit->unhappiness=1;
	  unhap++;
	}
	break;
      case G_DEMOCRACY:
	pcity->shield_surplus--;
	this_unit->upkeep=1;
	if (get_unit_type(this_unit->type)->attack_strength &&
	    !map_get_city(this_unit->x, this_unit->y)) {
	  unhap+=2;
	  this_unit->unhappiness=2;
	} else if (is_field_unit(this_unit)) {
	  this_unit->unhappiness=1;
	  unhap+=1;
	}
	break;
      default:
	break;
      }
    } 
  }
  
  switch (gov) {
  case G_ANARCHY:
  case G_DESPOTISM:
    city_units = min(city_units, pcity->ppl_unhappy[3]);
    pcity->ppl_unhappy[3]-= city_units;
    pcity->ppl_content[3]+= city_units;
    break;
  case G_MONARCHY:
    city_units = min(3, city_units);
    city_units = min(pcity->ppl_unhappy[3], city_units);
    pcity->ppl_unhappy[3]-= city_units;
    pcity->ppl_content[3]+= city_units;
    break;
  case G_COMMUNISM:
    city_units = min(3, city_units);
    city_units = min(pcity->ppl_unhappy[3], city_units*2);
    pcity->ppl_unhappy[3]-= city_units;
    pcity->ppl_content[3]+= city_units;
    break;
  case G_REPUBLIC:
    unhap--;
  case G_DEMOCRACY:
    citizen_happy_units(pcity, unhap);
  }

}

/**************************************************************************
...
**************************************************************************/
void update_pollution()
{
  int x,y,count=0;
  
  for (x=0;x<map.xsize;x++) 
    for (y=0;y<map.ysize;y++) 
      if (map_get_special(x,y)&S_POLLUTION) {
	count++;
      }
  game.heating=count;
  game.globalwarming+=count;
  if (game.globalwarming<game.warminglevel) 
    game.globalwarming=0;
  else {
    game.globalwarming-=game.warminglevel;
    if (myrand(200)<=game.globalwarming) {
      fprintf(stderr, "Global warming:%d\n", count);
      global_warming(map.xsize/10+map.ysize/10+game.globalwarming*5);
      game.globalwarming=0;
      send_all_known_tiles(0);
      notify_player(0, "Game: Global warming has occurred! Coastlines have been flooded\nand vast ranges of grassland have become deserts.");
      game.warminglevel+=4;
    }
  }
}

/**************************************************************************
...
**************************************************************************/
void handle_city_change_specialist(struct player *pplayer, 
				   struct packet_city_request *preq)
{
  struct city *pcity;
  pcity=find_city_by_id(preq->city_id);
  if(!pcity) 
    return;
  if(!player_owns_city(pplayer, pcity))  
    return;
  if(preq->specialist_from==SP_ELVIS) {
    if(pcity->size<5) 
      return; 

    if(!pcity->ppl_elvis)
      return;
    pcity->ppl_elvis--;
  } else if(preq->specialist_from==SP_TAXMAN) {
    if (!pcity->ppl_taxman)
      return;
    pcity->ppl_taxman--;
  } else if (preq->specialist_from==SP_SCIENTIST) {
    if (!pcity->ppl_scientist)
      return;
    pcity->ppl_scientist--;
  } else {
    return;
  }
  switch (preq->specialist_to) {
  case SP_ELVIS:
    pcity->ppl_elvis++;
    break;
  case SP_TAXMAN:
    pcity->ppl_taxman++;
    break;
  case SP_SCIENTIST:
    pcity->ppl_scientist++;
    break;
  default:
    pcity->ppl_elvis++;
    break;
  }

  city_refresh(pcity);
  send_city_info(pplayer, pcity, 0);
}

/**************************************************************************
...
**************************************************************************/
void remove_obsolete_buildings(struct player *pplayer)
{
  int i;
  struct genlist_iterator myiter;
  struct city *pcity;
  genlist_iterator_init(&myiter, &pplayer->cities.list, 0);
  for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
    pcity=(struct city *)ITERATOR_PTR(myiter);
    for (i=0;i<B_LAST;i++) {
      if(city_got_building(pcity, i) 
	 && !is_wonder(i) 
	   && building_obsolete(pplayer, i)) {
	do_sell_building(pplayer, pcity, i);
	notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT, 
			 "Game: %s is selling %s (obsolete) for %d", 
			 pcity->name, building_name(i), building_value(i)/2);
      }
    }
  }
}

/**************************************************************************
...
**************************************************************************/
void handle_city_make_specialist(struct player *pplayer, 
				 struct packet_city_request *preq)
{
  struct city *pcity;

  pcity=find_city_by_id(preq->city_id);
  if(!pcity) 
    return;
  if (!player_owns_city(pplayer, pcity))  return;
  if (preq->worker_x==2 && preq->worker_y==2) {
    auto_arrange_workers(pcity);
    return;
  }
  if (is_worker_here(pcity, preq->worker_x, preq->worker_y) == C_TILE_WORKER) {
    set_worker_city(pcity, preq->worker_x, preq->worker_y, C_TILE_EMPTY);
    pcity->ppl_elvis++;
    city_refresh(pcity);
    send_city_info(pplayer, pcity, 0);
  } else {
    notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT, "Game: you don't have a worker here"); 
  }

}


/**************************************************************************
...
**************************************************************************/
int is_worked_here(int x, int y)
{
  struct player *pplayer;
  struct genlist_iterator myiter;
  struct city *pcity;
  int my, i;
  int xx;

  x = map_adjust_x(x);
  for(i=0; i<game.nplayers; i++) {
    pplayer=&game.players[i];
    genlist_iterator_init(&myiter, &pplayer->cities.list, 0);
    
    for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
      pcity=(struct city *)ITERATOR_PTR(myiter);

      my=y+2-pcity->y;
      if (0 <= my && my <5)
	for ( xx = 0; xx<5; xx++) 
	  if (map_adjust_x(pcity->x+xx-2) == x) 
	    if(get_worker_city(pcity, xx, my)==C_TILE_WORKER) return 1;
    }
  }
  return 0;
}

/**************************************************************************
x and y are city cords in the range [0;4]
**************************************************************************/
int can_place_worker_here(struct city *pcity, int x, int y)
{
  if ((x == 0 || x == 4) && (y == 0 || y == 4))
    return 0;
  if (x < 0  || x > 4 || y < 0 || y > 4)
    return 0;
  if (get_worker_city(pcity, x, y) != C_TILE_EMPTY)
    return 0;
  return  (map_get_known(pcity->x+x-2, pcity->y+y-2, city_owner(pcity))
    && !is_worked_here(map_adjust_x(pcity->x+x-2), pcity->y+y-2));
}


/**************************************************************************
...
**************************************************************************/
void handle_city_make_worker(struct player *pplayer, 
			     struct packet_city_request *preq)
{
  struct city *pcity;
  pcity=find_city_by_id(preq->city_id);

  if (preq->worker_x < 0 || preq->worker_x > 4)
    return;
  if (preq->worker_y < 0 || preq->worker_y > 4)
    return;
  
  if(!pcity) 
    return;
  
  if(!player_owns_city(pplayer, pcity))
    return;

  if(preq->worker_x==2 && preq->worker_y==2) {
    auto_arrange_workers(pcity);
  }
  if (!nr_specialists(pcity) || 
      !can_place_worker_here(pcity, preq->worker_x, preq->worker_y))
    return;

  set_worker_city(pcity, preq->worker_x, preq->worker_y, C_TILE_WORKER);

  if(pcity->ppl_elvis) 
    pcity->ppl_elvis--;
  else if(pcity->ppl_scientist) 
    pcity->ppl_scientist--;
  else 
    pcity->ppl_taxman--;
  
  city_refresh(pcity);
  send_city_info(pplayer, pcity, 1);
}

/**************************************************************************
...
**************************************************************************/
void do_sell_building(struct player *pplayer, struct city *pcity, int id)
{
  if (!is_wonder(id)) {
    pcity->improvements[id]=0;
    pplayer->economic.gold+=building_value(id);
  }
}

/**************************************************************************
...
**************************************************************************/
void handle_city_sell(struct player *pplayer, struct packet_city_request *preq)
{
  struct city *pcity;
  pcity=find_city_by_id(preq->city_id);
  if (!pcity || !player_owns_city(pplayer, pcity) 
      || preq->build_id>=B_LAST) 
    return;
  
  if (pcity->did_sell) {
    notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT, 
		  "Game: You have already sold something here this turn.");
    return;
  }
 if (!can_sell_building(pcity, preq->build_id))
   return;

  pcity->did_sell=1;
  notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT,
		   "Game: You sell %s for %d credits.", 
		   building_name(preq->build_id), 
		   building_value(preq->build_id));
  do_sell_building(pplayer, pcity, preq->build_id);

  city_refresh(pcity);
  send_city_info(pplayer, pcity, 1);
  send_player_info(pplayer, pplayer);
}
/**************************************************************************
...
**************************************************************************/
int best_tile(struct city *pcity, int x, int y, int bx, int by)
{
  return (4*get_food_tile(x, y, pcity) + 
	  12*get_shields_tile(x, y, pcity) + 
	  8*get_trade_tile(x, y, pcity) > 
	  4*get_food_tile(bx, by, pcity) + 
	  12*get_shields_tile(bx, by, pcity) +
	  8*get_trade_tile(bx, by, pcity) 
	  );
}
/**************************************************************************
...
**************************************************************************/

int best_food_tile(struct city *pcity, int x, int y, int bx, int by)
{
  return (100*get_food_tile(x, y, pcity) + 
	  12*get_shields_tile(x, y, pcity) + 
	  8*get_trade_tile(x, y, pcity) > 
	  100*get_food_tile(bx, by, pcity) + 
	  12*get_shields_tile(bx, by, pcity) +
	  8*get_trade_tile(bx, by, pcity) 
	  );
}



/**************************************************************************
...
**************************************************************************/
int  add_adjust_workers(struct city *pcity)
{
  int workers=pcity->size;
  int iswork=0;
  int toplace;
  int foodneed;
  int x,y,bx,by;
  city_map_iterate(x, y) 
    if (get_worker_city(pcity, x, y)==C_TILE_WORKER) 
      iswork++;
  
  iswork--;
  if (iswork+nr_specialists(pcity)>workers)
    return 0;
  if (iswork+nr_specialists(pcity)==workers)
    return 1;
  toplace=workers-(iswork+nr_specialists(pcity));
  foodneed = -pcity->food_surplus +2;
  do {
    bx=0;
    by=0;
    city_map_iterate(x, y) {
      if(can_place_worker_here(pcity, x, y)) {
	  if(bx==0 && by==0) {
	    bx=x;
	    by=y;
	  } else {
	    if (foodneed >0 || (pcity->size < 4)) {
	      if (best_food_tile(pcity, x, y, bx, by)) {
		bx=x;
		by=y;
	      }
 	    } else if(best_tile(pcity, x, y,bx, by)) {
	      bx=x;
	      by=y;
	    }
	  }
	}
    }
    if(bx || by) {
      set_worker_city(pcity, bx, by, C_TILE_WORKER);
      toplace--;
      foodneed -= get_food_tile(bx,by,pcity);
    }
  } while(toplace && (bx || by));
  pcity->ppl_elvis+=toplace;
  return 1;
}

/**************************************************************************
...
**************************************************************************/
int settler_eats(struct city *pcity)
{
  struct genlist_iterator myiter;
  struct unit *this_unit;
  int eat=0;
  genlist_iterator_init(&myiter, &pcity->units_supported.list, 0);
  for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
    this_unit=(struct unit*)ITERATOR_PTR(myiter);
    if (unit_flag(this_unit->type, F_SETTLERS)) {
      eat++;
      if (get_government(pcity->owner)>=G_COMMUNISM) {
        eat++;
      }
    }
  }
  return eat;
}
/**************************************************************************
...
**************************************************************************/

void auto_arrange_workers(struct city *pcity)
{
  int workers=pcity->size;
  int taxwanted,sciwanted;
  int bx,by;
  int x,y;
  int foodneed;
  city_map_iterate(x, y)
    if (get_worker_city(pcity, x, y)==C_TILE_WORKER) 
      set_worker_city(pcity, x, y, C_TILE_EMPTY);
  

  set_worker_city(pcity, 2, 2, C_TILE_WORKER); 
  foodneed=(pcity->size *2 -get_food_tile(2,2, pcity)) + settler_eats(pcity);
  
  do {
    bx=0;
    by=0;
    city_map_iterate(x, y) {
      if(can_place_worker_here(pcity, x, y)) {
	if(bx==0 && by==0) {
	  bx=x;
	  by=y;
	} else {
	  if (foodneed >0 || pcity->size < 4) {
	    if (best_food_tile(pcity, x, y, bx, by)) {
	      bx=x;
	      by=y;
	    }
	  } else if(best_tile(pcity, x, y,bx, by)) {
	    bx=x;
	    by=y;
	  }
	}
      }
    }
    if(bx || by) {
      set_worker_city(pcity, bx, by, C_TILE_WORKER);
      foodneed -= get_food_tile(bx, by, pcity);
      workers--;
    }
  } while(workers && (bx || by));

  taxwanted=pcity->ppl_taxman;
  sciwanted=pcity->ppl_scientist;
  pcity->ppl_taxman=0;
  pcity->ppl_scientist=0;
  while (workers && (taxwanted ||sciwanted)) {
    if (taxwanted) {
      workers--;
      pcity->ppl_taxman++;
      taxwanted--;
    } 
    if (sciwanted && workers) {
      workers--;
      pcity->ppl_scientist++;
      sciwanted--;
    }
  }
  pcity->ppl_elvis=workers;

  city_refresh(pcity);
  send_city_info(city_owner(pcity), pcity, 1);
}

/**************************************************************************
...
**************************************************************************/
void handle_city_buy(struct player *pplayer, struct packet_city_request *preq)
{
  struct city *pcity;
  char *name;
  int cost, total;
  pcity=find_city_by_id(preq->city_id);
  if (!pcity || !player_owns_city(pplayer, pcity)) return;
 
  if (pcity->did_buy) {
    notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT,
		  "Game: You have already bought this turn.");
    return;
  }
  if (!pcity->is_building_unit) {
    total=building_value(pcity->currently_building);
    name=improvement_types[pcity->currently_building].name;
    
  } else {
    name=unit_types[pcity->currently_building].name;
    total=unit_value(pcity->currently_building);
    if (pcity->anarchy) {
      notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT, 
		    "Game: Can't buy units when city is in disorder.");
      return;
    }

  }
  cost=build_cost(pcity);
   if (cost>pplayer->economic.gold)
    return;
  pcity->did_buy=1;
  pplayer->economic.gold-=cost;
  pcity->shield_stock=total;
  notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT,
		   "Game: %s bought for %d", name, cost); 
  
  city_refresh(pcity);
  send_city_info(pplayer, pcity, 1);
  send_player_info(pplayer,pplayer);
}

/**************************************************************************
...
**************************************************************************/
void handle_city_change(struct player *pplayer, 
			struct packet_city_request *preq)
{
  struct city *pcity;
  pcity=find_city_by_id(preq->city_id);
   if(!player_owns_city(pplayer, pcity))
    return;
   if (preq->is_build_id_unit_id && !can_build_unit(pcity, preq->build_id))
     return;
   if (!preq->is_build_id_unit_id && !can_build_improvement(pcity, preq->build_id))
     return;
  if (pcity->did_buy && pcity->shield_stock) {
    notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT,
		     "Game: You have bought this turn, can't change.");
    return;
  }

   if(!pcity->is_building_unit && is_wonder(pcity->currently_building)) {
     notify_player_ex(0, pcity->x, pcity->y, E_NOEVENT,
		   "Game: The %s have stopped building The %s in %s.",
		   get_race_name_plural(pplayer->race),
		   get_imp_name_ex(pcity, pcity->currently_building),
		   pcity->name);
   }
  
  if(preq->is_build_id_unit_id) {
    if (!pcity->is_building_unit)
      pcity->shield_stock/=2;
      pcity->currently_building=preq->build_id;
      pcity->is_building_unit=1;
  }
  else {
    if (pcity->is_building_unit ||(is_wonder(pcity->currently_building)!=is_wonder(preq->build_id)))
      pcity->shield_stock/=2;
    
    pcity->currently_building=preq->build_id;
    pcity->is_building_unit=0;
    
    if(is_wonder(preq->build_id)) {
      notify_player_ex(0, pcity->x, pcity->y, E_NOEVENT,
		       "Game: The %s have started building The %s.",
		       get_race_name_plural(pplayer->race),
		       get_imp_name_ex(pcity, pcity->currently_building));
    }
  }
  
  city_refresh(pcity);
  send_city_info(pplayer, pcity, 1);
}

/**************************************************************************
...
**************************************************************************/
void handle_city_rename(struct player *pplayer, 
			struct packet_city_request *preq)
{
  char *cp;
  struct city *pcity;
  
  pcity=find_city_by_id(preq->city_id);

  if(!player_owns_city(pplayer, pcity))
    return;

  if((cp=get_sane_name(preq->name))) {
    /* more sanity tests! any existing city with that name? */
    strcpy(pcity->name, cp);
    city_refresh(pcity);
    send_city_info(pplayer, pcity, 1);
  }
  else
    notify_player(pplayer, "Game: %s is not a valid name.", preq->name);
}

void send_player_cities(struct player *pplayer)
{
  struct genlist_iterator myiter;
  genlist_iterator_init(&myiter, &pplayer->cities.list, 0);
  for(; ITERATOR_PTR(myiter); ) {
    struct city *pcity=(struct city *)ITERATOR_PTR(myiter);
    ITERATOR_NEXT(myiter);
    city_refresh(pcity);
    send_city_info(pplayer, pcity, 0);
  }
}


/**************************************************************************
...
**************************************************************************/
void update_city_activities(struct player *pplayer)
{
  struct genlist_iterator myiter;
  int gold;
  genlist_iterator_init(&myiter, &pplayer->cities.list, 0);
  gold=pplayer->economic.gold;
  pplayer->got_tech=0;
  for(; ITERATOR_PTR(myiter); ) {
    struct city *pcity=(struct city *)ITERATOR_PTR(myiter);
    ITERATOR_NEXT(myiter);
    update_city_activity(pplayer, pcity);
  }
  pplayer->ai.prev_gold = gold;
  if (gold-(gold-pplayer->economic.gold)*3<0) {
    notify_player_ex(pplayer, 0, 0, E_LOW_ON_FUNDS, "Game: WARNING, we're LOW on FUNDS sire.");  
  }
    /* uncomment to unbalance the game, like in civ1 (CLG)
      if (pplayer->got_tech && pplayer->research.researched > 0)    
        pplayer->research.researched=0;
    */
}

/**************************************************************************
...
**************************************************************************/
void city_auto_remove_worker(struct city *pcity)
{
  if(pcity->size<1) {      
    remove_city(pcity);
    return;
  }
  if(nr_specialists(pcity)) {
    if(pcity->ppl_taxman) {
      pcity->ppl_taxman--;
      return;
    } else if(pcity->ppl_scientist) {
      pcity->ppl_scientist--;
      return;
    } else if(pcity->ppl_elvis) {
      pcity->ppl_elvis--; 
      return;
    }
  } 
  auto_arrange_workers(pcity);
  city_refresh(pcity);
  send_city_info(&game.players[pcity->owner], pcity, 0);
}

/**************************************************************************
...
**************************************************************************/
void city_increase_size(struct city *pcity)
{
  if (city_got_building(pcity, B_GRANARY) || city_affected_by_wonder(pcity, B_PYRAMIDS)) { 
    pcity->food_stock=(pcity->size+1)*(game.foodbox/2);
  }
  else
    pcity->food_stock=0;
  if (!city_got_building(pcity, B_AQUEDUCT) && pcity->size>=8) {/* need aquaduct */
    notify_player_ex(city_owner(pcity), pcity->x, pcity->y, E_CITY_AQUADUCT,
	  "Game: %s needs Aquaducts to grow any further", pcity->name);
    return;
  }

  if (!city_got_building(pcity, B_SEWER) && pcity->size>=12) {/* need sewer */
    notify_player_ex(city_owner(pcity), pcity->x, pcity->y, E_CITY_AQUADUCT,
	  "Game: %s needs Sewer system to grow any further", pcity->name);
    return;
  }

  pcity->size++;
  if (!add_adjust_workers(pcity))
    auto_arrange_workers(pcity);

  city_refresh(pcity);
  send_city_info(&game.players[pcity->owner], pcity, 0);
}

/**************************************************************************
...
**************************************************************************/
void city_reduce_size(struct city *pcity)
{
  pcity->size--;
  notify_player_ex(city_owner(pcity), pcity->x, pcity->y, E_CITY_FAMINE,
		   "Game: Famine feared in %s", pcity->name);
  if (city_got_building(pcity, B_GRANARY) || city_affected_by_wonder(pcity, B_PYRAMIDS)) 
    pcity->food_stock=(pcity->size+1)*(game.foodbox/2);
  else
    pcity->food_stock=0;

  city_auto_remove_worker(pcity);
}
 
/**************************************************************************
...
**************************************************************************/
void city_populate(struct city *pcity)
{
  pcity->food_stock+=pcity->food_surplus;
  if(pcity->food_stock >= pcity->size*game.foodbox) 
    city_increase_size(pcity);
  else if(pcity->food_stock<0) {
    struct unit *punit;
    struct genlist_iterator myiter;
    genlist_iterator_init(&myiter, &pcity->units_supported.list, 0);
    for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
      punit=(struct unit *)ITERATOR_PTR(myiter);
      if (unit_flag(punit->type, F_SETTLERS)) {
	wipe_unit(0, punit);
	notify_player_ex(city_owner(pcity), pcity->x, pcity->y, E_UNIT_LOST, "Game: Famine feared in %s, Settlers dies!", 
			 pcity->name);
	if (city_got_building(pcity, B_GRANARY) || city_affected_by_wonder(pcity, B_PYRAMIDS)) 
	  pcity->food_stock=(pcity->size+1)*(game.foodbox/2);
	else
	  pcity->food_stock=0;
	return;
      }
    }
    city_reduce_size(pcity);
  }
}

/**************************************************************************
...
**************************************************************************/
int is_building_other_wonder(struct city *pc)
{
  struct genlist_iterator myiter;
  struct city *pcity;
  struct player *pplayer = city_owner(pc);
  genlist_iterator_init(&myiter, &pplayer->cities.list, 0);
  for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
    pcity=(struct city *)ITERATOR_PTR(myiter);
    if ((pc != pcity) && !(pcity->is_building_unit) && is_wonder(pcity->currently_building) && map_get_continent(pcity->x, pcity->y) == map_get_continent(pc->x, pc->y))
      return 1;
  }
  return 0;
}

/**************************************************************************
...
**************************************************************************/

int built_elsewhere(struct city *pc, int wonder)
{
  struct genlist_iterator myiter;
  struct city *pcity;
  struct player *pplayer = city_owner(pc);
  genlist_iterator_init(&myiter, &pplayer->cities.list, 0);
  for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
    pcity=(struct city *)ITERATOR_PTR(myiter);
    if ((pc != pcity) && !(pcity->is_building_unit) && pcity->currently_building == wonder)
      return 1;
  }
  return 0;
}


/**************************************************************************
...
**************************************************************************/

void eval_buildings(struct city *pcity,int *values)
{
  int i, gov, tech;
  struct player *plr = city_owner(pcity);
  gov = get_government(pcity->owner);
  tech = (plr->research.researching != A_NONE);
    
  for (i=0;i<B_LAST;i++) {
    if (is_wonder(i) && can_build_improvement(pcity, i) && !built_elsewhere(pcity, i)) {
      if (wonder_is_obsolete(i))
	values[i]=1;
      else
	values[i]=99;
    } else
    values[i]=0;
  }
  
  if (gov <= G_COMMUNISM || pcity->size < 5) {
    if (can_build_improvement(pcity, B_GRANARY)) 
      values[B_GRANARY]=pcity->food_surplus*50;
  }
  if (can_build_improvement(pcity, B_SUPERMARKET))
    values[B_SUPERMARKET]=pcity->size*55;

  if (can_build_improvement(pcity, B_AQUEDUCT) && pcity->size > 6)
    values[B_AQUEDUCT]=pcity->size*125+pcity->food_surplus*50;
  if (can_build_improvement(pcity, B_SEWER) && pcity->size > 11)
    values[B_SEWER]=pcity->size*100+pcity->food_surplus*50;
  
  if (can_build_improvement(pcity, B_HARBOUR) && (pcity->size > 5)) 
    values[B_HARBOUR]=pcity->size*60;
  if (can_build_improvement(pcity, B_OFFSHORE) && 
      !can_build_improvement(pcity, B_HARBOUR))
    values[B_OFFSHORE]=pcity->size*60;
  
  if (can_build_improvement(pcity, B_MARKETPLACE)) 
    values[B_MARKETPLACE]=pcity->trade_prod*200;
  
  if (pcity->trade_prod > 15) 
    if (can_build_improvement(pcity, B_BANK)) 
      values[B_BANK]=pcity->tax_total*100;

    if (can_build_improvement(pcity, B_STOCK)) 
      values[B_STOCK]=pcity->tax_total*100;
  
  if (gov > G_COMMUNISM)
    if (can_build_improvement(pcity, B_SUPERHIGHWAYS)) 
      values[B_SUPERHIGHWAYS]=pcity->trade_prod*60;
  if (can_build_improvement(pcity, B_COURTHOUSE)) {
    if (gov != G_COMMUNISM && gov != G_DEMOCRACY) 
      values[B_COURTHOUSE]=pcity->corruption*100;
    else 
      values[B_COURTHOUSE]=pcity->ppl_unhappy[4]*200+pcity->ppl_elvis*400;
  }
  if (tech) {
    if (can_build_improvement(pcity, B_LIBRARY)) 
      values[B_LIBRARY]=pcity->science_total*200;
    
    if (can_build_improvement(pcity, B_UNIVERSITY)) 
      values[B_UNIVERSITY]=pcity->science_total*101;
    
    if (can_build_improvement(pcity, B_RESEARCH)) 
      values[B_RESEARCH]=pcity->science_total*100;
  }
  if (can_build_improvement(pcity, B_AIRPORT))
    values[B_AIRPORT]=pcity->shield_prod*49;

  if (pcity->shield_prod >= 15)
    if (can_build_improvement(pcity, B_PORT))
      values[B_PORT]=pcity->shield_prod*48;

  if (pcity->shield_prod >= 5) {
    if (can_build_improvement(pcity, B_BARRACKS))
      values[B_BARRACKS]=pcity->shield_prod*50;

    if (can_build_improvement(pcity, B_BARRACKS2))
      values[B_BARRACKS2]=pcity->shield_prod*50;

    if (can_build_improvement(pcity, B_BARRACKS3))
      values[B_BARRACKS3]=pcity->shield_prod*50;
  }
  if (can_build_improvement(pcity, B_TEMPLE))
     values[B_TEMPLE]=pcity->ppl_unhappy[4]*200+pcity->ppl_elvis*600;

  if (can_build_improvement(pcity, B_COLOSSEUM))
    values[B_COLOSSEUM]=pcity->ppl_unhappy[4]*200+pcity->ppl_elvis*300;

  if (can_build_improvement(pcity, B_CATHEDRAL))
    values[B_CATHEDRAL]=pcity->ppl_unhappy[4]*201+pcity->ppl_elvis*300;

  if (!(tech && plr->economic.tax > 50)) {
    if (can_build_improvement(pcity, B_COASTAL))
      values[B_COASTAL]=pcity->size*36;
    if (can_build_improvement(pcity, B_CITY))
      values[B_CITY]=pcity->size*35;
  } 
  if (!(tech && plr->economic.tax > 40)) {
    if (can_build_improvement(pcity, B_SAM))
      values[B_SAM]=pcity->size*24;
    if (can_build_improvement(pcity, B_SDI))
      values[B_SDI]=pcity->size*25;
  }
  if (pcity->shield_prod >= 10)
    if (can_build_improvement(pcity, B_FACTORY)) 
      values[B_FACTORY]=pcity->shield_prod*125;

  if (city_got_building(pcity, B_FACTORY)) {
    
    if (can_build_improvement(pcity, B_HYDRO))
      values[B_HYDRO]=pcity->shield_prod*100+pcity->pollution*100;
    
    if (can_build_improvement(pcity, B_NUCLEAR))
      values[B_NUCLEAR]=pcity->shield_prod*101+pcity->pollution*100;
    
    if (can_build_improvement(pcity, B_POWER))
      values[B_POWER]=pcity->shield_prod*100;
  }

  if (can_build_improvement(pcity, B_MFG)) 
    values[B_MFG]=pcity->shield_prod*125;

  if (can_build_improvement(pcity, B_MASS)) 
    values[B_MASS]=pcity->pollution*(100+pcity->size);

  if (can_build_improvement(pcity, B_RECYCLING)) 
    values[B_RECYCLING]=pcity->pollution*(100+pcity->shield_prod);
    
  if (can_build_improvement(pcity, B_CAPITAL))
    values[B_CAPITAL]=pcity->shield_prod;
}

/**************************************************************************
...
**************************************************************************/
int advisor_choose_build(struct city *pcity)
{
  int i;
  int id=-1;
  int want=0;
  int values[B_LAST];
  
  eval_buildings(pcity, values);
  
  for(i=0; i<B_LAST; i++)
    if(values[i]>0) {
      if(values[i]>want) {
	want=values[i];
	id=i;
      }
    }
  if (id == B_CAPITAL)
    return 0;
  if (pcity->currently_building == id && !pcity->is_building_unit)
    return 1;
 
 if (id!=1 && is_wonder(id) && is_building_other_wonder(pcity)) {
    return 0;
  }

  if(id!=-1) {
    if(is_wonder(id)) {
      notify_player_ex(0, pcity->x, pcity->y, E_WONDER_STARTED,
		    "Game: The %s have started building The %s in %s.",
		    get_race_name_plural(city_owner(pcity)->race),
		    get_imp_name_ex(pcity, id), pcity->name);
    }
    pcity->currently_building=id;
    pcity->is_building_unit=0;
    return 1;
  }
  if (city_owner(pcity)->ai.control)
    return 0;
  for (i=0;i<B_LAST;i++)
    if(can_build_improvement(pcity, i)) {
      pcity->currently_building=i;
      pcity->is_building_unit=0;
      return 0;
    }
  return 0;
}

void obsolete_building_test(struct city *pcity, int b1, int b2)
{ 
  if (pcity->currently_building==b1 && 
      can_build_improvement(pcity, b2))
    pcity->currently_building=b2;
}

void upgrade_building_prod(struct city *pcity)
{
  obsolete_building_test(pcity, B_BARRACKS,B_BARRACKS3);
  obsolete_building_test(pcity, B_BARRACKS,B_BARRACKS2);
  obsolete_building_test(pcity, B_BARRACKS2,B_BARRACKS3);
}

void upgrade_unit_prod(struct city *pcity)
{
  struct player *pplayer=&game.players[pcity->owner];
  int id = pcity->currently_building;
  int id2= unit_types[id].obsoleted_by;
 if (id > 0 &&  can_build_unit_direct(pcity, 
                     id2)) {
    pcity->currently_building=id2;
    notify_player_ex(pplayer, pcity->x, pcity->y, E_UNIT_UPGRADED, 
		  "Game: production of %s is upgraded to %s in %s",
		  get_unit_type(id)->name, 
		  get_unit_type(id2)->name , 
		  pcity->name);
  }
}
int do_make_unit_veteran(struct city *pcity, enum unit_type_id id)
{
  if (is_ground_unittype(id))
    return city_got_barracks(pcity);
  else if (is_water_unit(id))
    return (city_affected_by_wonder(pcity, B_LIGHTHOUSE) || city_got_building(pcity, B_PORT));
  else
    return city_got_building(pcity, B_AIRPORT);
  return 0;
}

/**************************************************************************
...
**************************************************************************/
void city_build_stuff(struct player *pplayer, struct city *pcity)
{
  if (pcity->shield_surplus<0) {
    struct unit *punit;
    struct genlist_iterator myiter;
    genlist_iterator_init(&myiter, &pcity->units_supported.list, 0);
     for(; ITERATOR_PTR(myiter);) {
       punit=(struct unit *)ITERATOR_PTR(myiter);
       ITERATOR_NEXT(myiter);
       if (is_military_unit(punit)) {
	 notify_player_ex(pplayer, pcity->x, pcity->y, E_UNIT_LOST,
			  "Game: %s can't upkeep %s, unit disbanded",
			  pcity->name, get_unit_type(punit->type)->name);
	 wipe_unit(pplayer, punit);
	 break;
       }
     }
  }
  
  if(pcity->shield_surplus<=0 && !is_city_unhappy(pcity)) 
    pcity->shield_surplus=1;
  pcity->shield_stock+=pcity->shield_surplus;
  if (!pcity->is_building_unit) {
    if (pcity->currently_building==B_CAPITAL) {
      pplayer->economic.gold+=pcity->shield_surplus;
      pcity->shield_stock=0;
    }    
    upgrade_building_prod(pcity);
    if (!can_build_improvement(pcity, pcity->currently_building)) {
      notify_player_ex(pplayer, pcity->x, pcity->y, E_CITY_CANTBUILD,
		    "Game: %s is building %s, which is no longer available",
	pcity->name,get_imp_name_ex(pcity, pcity->currently_building));
      return;
    }
    if (pcity->shield_stock>=building_value(pcity->currently_building)) {
      if (pcity->currently_building==B_PALACE) {
	struct genlist_iterator myiter;
	struct city *palace;
	genlist_iterator_init(&myiter, &pplayer->cities.list, 0);
	for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
	  palace=(struct city *)ITERATOR_PTR(myiter);
	  if (city_got_building(palace, B_PALACE)) {
	    palace->improvements[B_PALACE]=0;
	    break;
	  }
	}
      }

      pcity->improvements[pcity->currently_building]=1;
      pcity->shield_stock-=building_value(pcity->currently_building); 
      /* to eliminate micromanagement */
      if(is_wonder(pcity->currently_building)) {
	game.global_wonders[pcity->currently_building]=pcity->id;
	notify_player_ex(0, pcity->x, pcity->y, E_WONDER_BUILD,
		      "Game: The %s have finished building %s in %s.",
		      get_race_name_plural(pplayer->race),
		      get_imp_name_ex(pcity, pcity->currently_building),
		      pcity->name);
      }
      notify_player_ex(pplayer, pcity->x, pcity->y, E_IMP_BUILD,
		    "Game: %s has finished building %s", pcity->name, 
		    improvement_types[pcity->currently_building].name
		    );

      if (pcity->currently_building==B_DARWIN) {
	notify_player(pplayer, 
		      "Game: Darwin's Voyage boost research, you gain 2 immediate advances.");
	update_tech(pplayer, 1000000); 
	update_tech(pplayer, 1000000); 
      }
      city_refresh(pcity);
      advisor_choose_build(pcity);
      notify_player_ex(pplayer, pcity->x, pcity->y, E_IMP_AUTO,
		    "Game: %s is now building %s", pcity->name, 
		    improvement_types[pcity->currently_building].name
		    );
    } 
  } else {
    upgrade_unit_prod(pcity);
    if(pcity->shield_stock>=unit_value(pcity->currently_building)) {
      if (unit_flag(pcity->currently_building, F_SETTLERS)) {
	if (pcity->size==1) {
	  notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT,
			"Game: %s can't build settler yet", pcity->name);
	  return;
	}
	pcity->size--;
	city_auto_remove_worker(pcity);
      }

      create_unit(pplayer, pcity->x, pcity->y, pcity->currently_building,
		  do_make_unit_veteran(pcity, pcity->currently_building), 
		  pcity->id);
      /* to eliminate micromanagement, we only subtract the unit's cost */
      pcity->shield_stock-=unit_value(pcity->currently_building); 
      notify_player_ex(pplayer, pcity->x, pcity->y, E_UNIT_BUILD,
		       "Game: %s is finished building %s", 
		       pcity->name, 
		       unit_types[pcity->currently_building].name);
    }
  }
}

/**************************************************************************
...
**************************************************************************/
void pay_for_buildings(struct player *pplayer, struct city *pcity)
{
  int i;
  for (i=0;i<B_LAST;i++) 
    if (city_got_building(pcity, i)) {
      if (is_wonder(i)) {
	if (wonder_is_obsolete(i)) {
	  switch (improvement_types[i].shield_upkeep) {
	  case 1:
	    pplayer->economic.gold+=3;
	    break;
	  case 2:
	    update_tech(pplayer, 3);
	    break;
	  }
	}
      } else if (pplayer->economic.gold-building_cost(pcity, i)<0) {
	notify_player_ex(pplayer, pcity->x, pcity->y, E_IMP_AUCTIONED,
			 "Game: Can't afford to maintain %s in %s, building sold!", 
			 improvement_types[i].name, pcity->name);
	do_sell_building(pplayer, pcity, i);
	city_refresh(pcity);
      } else
	pplayer->economic.gold-=building_cost(pcity, i);
    }
}

/**************************************************************************
1) check for enemy units on citymap tiles
2) withdraw workers from such tiles
3) mark citymap tiles accordingly empty/unavailable  
**************************************************************************/
void city_check_workers(struct player *pplayer, struct city *pcity)
{
  int x, y;
  
  city_map_iterate(x, y) {
    struct tile *ptile=map_get_tile(pcity->x+x-2, pcity->y+y-2);
    
    if(unit_list_size(&ptile->units)>0) {
      struct unit *punit=unit_list_get(&ptile->units, 0);
      if(pplayer->player_no!=punit->owner) {
	if(get_worker_city(pcity, x, y)==C_TILE_WORKER)
	  pcity->ppl_elvis++;
	set_worker_city(pcity, x, y, C_TILE_UNAVAILABLE);
	continue;
      }
    }
    if(get_worker_city(pcity, x, y)==C_TILE_UNAVAILABLE)
      set_worker_city(pcity, x, y, C_TILE_EMPTY);
    if(get_worker_city(pcity, x, y)!=C_TILE_WORKER &&
       !can_place_worker_here(pcity, x, y))
      set_worker_city(pcity, x, y, C_TILE_UNAVAILABLE);
  }
}
/**************************************************************************
 Add some Pollution if we have waste
**************************************************************************/
void check_pollution(struct city *pcity)
{
  int x,y;
  int k=100;
  if (pcity->pollution && myrand(100)<=pcity->pollution) {
    while (k) {
      x=pcity->x+myrand(5)-2;
      y=pcity->y+myrand(5)-2;
      if ( (x!=pcity->x || y!=pcity->x) && 
	   (map_get_terrain(x,y)!=T_OCEAN && map_get_terrain(x,y)<=T_TUNDRA) &&
	   (!(map_get_special(x,y)&S_POLLUTION)) ) { 
	map_set_special(x,y, S_POLLUTION);
	send_tile_info(0, x, y, PACKET_TILE_INFO);
	notify_player_ex(city_owner(pcity), pcity->x, pcity->y, E_POLLUTION,
			 "Game: Pollution near %s", pcity->name);
	return;
      }
      k--;
    }
  }
}

void sanity_check_city(struct city *pcity)
{
  int size=pcity->size;
  int x,y;
  int iswork=0;
  city_map_iterate(x, y) {
    if (get_worker_city(pcity, x, y)==C_TILE_WORKER) 
	iswork++;
  }
  iswork--;
  if (iswork+nr_specialists(pcity)!=size) {
    printf("%s is bugged: size:%d workers:%d elvis: %d tax:%d sci:%d\n", pcity->name,size,iswork,  pcity->ppl_elvis, pcity->ppl_taxman, pcity->ppl_scientist); 
    auto_arrange_workers(pcity);
  }
}

void city_incite_cost(struct city *pcity)
{

  struct city *capital;
  int dist;
  
  if (city_got_building(pcity, B_PALACE)) 
    pcity->incite_revolt_cost=1000000;
  else {
    pcity->incite_revolt_cost=city_owner(pcity)->economic.gold +1000;
    capital=find_palace(city_owner(pcity));
    if (capital)
      dist=min(32, map_distance(capital->x, capital->y, pcity->x, pcity->y)); 
    else 
      dist=32;
    if (city_got_building(pcity, B_COURTHOUSE)) 
      dist/=2;
    if (get_government(city_owner(pcity)->player_no)==G_COMMUNISM)
      dist = min(10, dist);
    pcity->incite_revolt_cost/=(dist + 3);
    pcity->incite_revolt_cost*=pcity->size;
    if (is_city_unhappy(pcity))
      pcity->incite_revolt_cost/=2;
    if (unit_list_size(&pcity->units_supported)==0) 
      pcity->incite_revolt_cost/=2;
  }
}
/**************************************************************************
 called at the beginning of a new year. produced units etc..
**************************************************************************/
int update_city_activity(struct player *pplayer, struct city *pcity)
{
  int got_tech = 0;
  city_check_workers(pplayer, pcity);
  if (city_refresh(pcity) && 
      get_government(pcity->owner)>=G_REPUBLIC &&
      pcity->food_surplus>0 && pcity->size>4) {
    pcity->food_stock=pcity->size*game.foodbox+1; 
  }
  city_build_stuff(pplayer, pcity);
  if (!pcity->was_happy && is_city_happy(pcity) && pcity->size>4) {
    notify_player_ex(pplayer, pcity->x, pcity->y, E_CITY_LOVE,
		  "Game: We Love The %s Day celebrated in %s", 
		  get_ruler_title(pplayer->government),
		  pcity->name);
  }
  if (!is_city_happy(pcity) && pcity->was_happy && pcity->size>4) {
    notify_player_ex(pplayer, pcity->x, pcity->y, E_CITY_NORMAL,
		  "Game: We Love The %s Day canceled in %s",
		  get_ruler_title(pplayer->government),
		  pcity->name);

  }
  pcity->was_happy=is_city_happy(pcity);
    {
      int id=pcity->id;
      city_populate(pcity);
      if(!city_list_find_id(&pplayer->cities, id))
	return 0;
    }
     
  pcity->is_updated=1;

  pcity->did_sell=0;
  pcity->did_buy=0;
  if (city_got_building(pcity, B_AIRPORT))
    pcity->airlift=1;
  else
    pcity->airlift=0;
  if (update_tech(pplayer, pcity->science_total)) 
    got_tech = 1;
  pplayer->economic.gold+=pcity->tax_total;
  pay_for_buildings(pplayer, pcity);
  if(is_city_unhappy(pcity)) { 
    pcity->anarchy++;
    if (pcity->anarchy == 1) 
      notify_player_ex(pplayer, pcity->x, pcity->y, E_CITY_DISORDER,
		       "Game: Civil disorder in %s", pcity->name);
    else
      notify_player_ex(pplayer, pcity->x, pcity->y, E_CITY_DISORDER,
		       "Game: CIVIL DISORDER CONTINUES in %s.", pcity->name);
  }
  else {
    if (pcity->anarchy)
      notify_player_ex(pplayer, pcity->x, pcity->y, E_CITY_NORMAL,
		       "Game: Order restored in %s", pcity->name);
    pcity->anarchy=0;
  }
  check_pollution(pcity);
  city_incite_cost(pcity);

  send_city_info(0, pcity, 0);
  if (pcity->anarchy>2 && get_government(pplayer->player_no)==G_DEMOCRACY) {
    notify_player_ex(pplayer, pcity->x, pcity->y, E_ANARCHY,
		     "Game: The people have overthrown your democracy, your country is in turmoil");
    handle_player_revolution(pplayer);
  }
  sanity_check_city(pcity);
  return got_tech;
}

/**************************************************************************
...
**************************************************************************/

void send_city_info(struct player *dest, struct city *pcity, int dosend)
{
  int i, o, x, y;
  char *p;
  struct packet_city_info packet;
  if (nocity_send) 
    return;
  packet.id=pcity->id;
  packet.owner=pcity->owner;
  packet.x=pcity->x;
  packet.y=pcity->y;
  strcpy(packet.name, pcity->name);

  packet.size=pcity->size;
  packet.ppl_happy=pcity->ppl_happy[4];
  packet.ppl_content=pcity->ppl_content[4];
  packet.ppl_unhappy=pcity->ppl_unhappy[4];
  packet.ppl_elvis=pcity->ppl_elvis;
  packet.ppl_scientist=pcity->ppl_scientist;
  packet.ppl_taxman=pcity->ppl_taxman;
  for (i=0;i<4;i++)
    packet.trade[i]=pcity->trade[i];

  packet.food_prod=pcity->food_prod;
  packet.food_surplus=pcity->food_surplus;
  packet.shield_prod=pcity->shield_prod;
  packet.shield_surplus=pcity->shield_surplus;
  packet.trade_prod=pcity->trade_prod;
  packet.corruption=pcity->corruption;
  
  packet.luxury_total=pcity->luxury_total;
  packet.tax_total=pcity->tax_total;
  packet.science_total=pcity->science_total;
  
  packet.food_stock=pcity->food_stock;
  packet.shield_stock=pcity->shield_stock;
  packet.pollution=pcity->pollution;
  packet.incite_revolt_cost=pcity->incite_revolt_cost;
  
  packet.is_building_unit=pcity->is_building_unit;
  packet.currently_building=pcity->currently_building;
  packet.airlift=pcity->airlift;
  packet.did_buy=pcity->did_buy;
  packet.did_sell=pcity->did_sell;
  packet.was_happy=pcity->was_happy;
  p=packet.city_map;
  for(y=0; y<CITY_MAP_SIZE; y++)
    for(x=0; x<CITY_MAP_SIZE; x++)
      *p++=get_worker_city(pcity, x, y)+'0';
  *p='\0';

  p=packet.improvements;
  for(i=0; i<B_LAST; i++)
    *p++=(pcity->improvements[i]) ? '1' : '0';
  *p='\0';
  
  for(o=0; o<game.nplayers; o++) {           /* dests */
    if(!dest || &game.players[o]==dest) {
       if(dosend || map_get_known(pcity->x, pcity->y, &game.players[o])) {
	send_packet_city_info(game.players[o].conn, &packet);
      }
    }
  }
}

void remove_trade_route(int c1, int c2) 
{
  int i;
  struct city *pc1, *pc2;
  
  pc1=find_city_by_id(c1);
  pc2=find_city_by_id(c2);
  if (pc1) {
    for (i=0;i<4;i++)
      if (pc1->trade[i]==c2)
	pc1->trade[i]=0;
  }
  if (pc2) {
    for (i=0;i<4;i++)
      if (pc2->trade[i]==c2)
	pc2->trade[i]=0;
  }
}

/**************************************************************************
...
**************************************************************************/
void remove_city(struct city *pcity)
{
  int o;
  struct unit *punit;
  struct genlist_iterator myiter;
  struct packet_generic_integer packet;
  for (o=0; o<4; o++)
    remove_trade_route(pcity->trade[0], pcity->id);
  packet.value=pcity->id;
  for(o=0; o<game.nplayers; o++)           /* dests */
    send_packet_generic_integer(game.players[o].conn,
				PACKET_REMOVE_CITY,&packet);
  genlist_iterator_init(&myiter, &pcity->units_supported.list, 0);

  for(; ITERATOR_PTR(myiter);) {
    punit=(struct unit *)ITERATOR_PTR(myiter);
    ITERATOR_NEXT(myiter);
    wipe_unit(0, punit);
  }
  game_remove_city(pcity->id);
}
