#|------------------------------------------------------------*-Scheme-*--|
 | File:    modules/iolib/format.scm
 |
 |          Copyright (C)1997 Donovan Kolbly <d.kolbly@rscheme.org>
 |          as part of the RScheme project, licensed for free use.
 |          See <http://www.rscheme.org/> for the latest information.
 |
 | File version:     1.5
 | File mod date:    1997.11.29 23:10:41
 | System build:     v0.7.2, 97.12.21
 | Owned by module:  iolib
 |
 | Purpose:          Provide the `format' function
 `------------------------------------------------------------------------|#

(define-glue (parse-format-string str)
{
extern obj parse_format_string( obj string );

   REG0 = parse_format_string(str);
   RETURN1();
})

;; 
;;
;;option  mnemonic: description
;;------  ------------------------
;;    ~a  any: display the argument (as for humans).
;;    ~s  slashified: write the argument (as for parsers).
;;    ~j  joined: display over space-seperated elements of list
;;    ~d  decimal: the integer argument is output in decimal format.
;;    ~x  hexadecimal: the integer argument is output in hexadecimal format.
;;    ~c  character: the next argument is displayed as a character.
;;    ~_  space: output a space character.
;;    ~%  newline: output a newline character.
;;    ~~  tilde: output a tilde.
;;    ~t  tab: output a tab character.
;;    ~>  tab: the next arg is an integer # of tab stops.
;;    ~b  binary: the integer argument is output in binary format.
;;    ~o  octal: the integer argument is output in octal format.
;;    ~p  plural: if the argument is greater than 1, print a lower case 's'.
;;    ~h  hash: create a user hash number for the arg object
;;    ~r  raw: the 32 bits of the value are printed in hex
;;    ~C  Capitalize: like ~a, but the first letter is cap'd
;
; future:
;;    ~g  glorify: pretty print the argument (typically an s-expression).

(define-syntax (format-character v) (vector-ref v 0))
(define-syntax (sharp-flag v) (vector-ref v 1))
(define-syntax (star-flag v) (vector-ref v 2))
(define-syntax (at-flag v) (vector-ref v 3))
(define-syntax (negative-flag v) (vector-ref v 4))
(define-syntax (pre-dot-leading-zero-flag v) (vector-ref v 5))
(define-syntax (pre-dot-number v) (vector-ref v 6))
(define-syntax (post-dot-digits v) (vector-ref v 7))
(define-syntax (post-dot-number v) (vector-ref v 8))

(define $default-info
  '#(#f   ;; [0] control char -- not used inside of formatters
          ;;                     (used to dispatch to a formatter)
     #f   ;; [1] `#' flag
     #f   ;; [2] `*' flag
     #f   ;; [3] `@' flag
     #f   ;; [4] `-' flag
     #f   ;; [5] pre-`.' leading zero flag
     #f   ;; [6] pre-`.' number
     #f   ;; [7] post-`.' digits
     #f)) ;; [8] post-`.' number

(define (format/a port info arg)
  (display-object arg port))

(define (format/C port info arg)
  (display-object 
   arg 
   (make-output-filter 
    port
    (let ((first #t))
      (lambda ((str <string>))
	(if (and first (fixnum>? (string-length str) 1))
	    (begin
	      (set! first #f)
	      (string-append
	       (string (char-upcase (string-ref str 0)))
	       (substring str 1)))
	    str))))))

(define (format/j port info (arg <list>))
  (if (pair? arg)
      (begin
	(display-object (car arg) port)
	(let loop (((r <list>) (cdr arg)))
	  (if (pair? r)
	      (begin
		(output-port-write-char port #\space)
		(display-object (car r) port)
		(loop (cdr r))))))))
      
(define (format/s port info arg)
  (write-object arg port))

(define (format/p port info arg)
  (if (> arg 1)
      (output-port-write-char port #\s)))

(define (format/d port info (arg <number>))
  (write-string port (number->string arg)))

(define (format/x port info arg)
  (write-string port (number->string arg 16)))

(define (format/b port info arg)
  (write-string port (number->string arg 2)))

(define (format/o port info arg)
  (write-string port (number->string arg 8)))

(define (format/c port info arg)
  (output-port-write-char port arg))

#|
(define (format/h port info arg)
  (write-string port "#*")
  (write-int port (get-user-hash arg)))

(define (format/> port info arg)
  (write-tab4 arg port))
|#

(define (format/_ port info)
  (output-port-write-char port #\space))

(define (format/% port info)
  (output-port-write-char port #\newline))

(define (format/~ port info)
  (output-port-write-char port #\~))

(define (format/t port info)
  (output-port-write-char port #\tab))

; Note:  this is arranged so that users
; can create their own formatter functions (a la NeXTSTEP)
; Since you want scope control, a global access point is not
; desired.  Instead, there is a function "%format" which takes
; a list of formatters, which format calls with the 
; standard list.  Then you can wrap that up with whatever list
; you want! (plus, *default-formatters* is the default list,
; so you don't have to rewrite the standard stuff)
;

(define *default-formatters*
  (vector #f
	  (list (cons #\a (cons format/a 1))
		(cons #\s (cons format/s 1))
		(cons #\d (cons format/d 1))
		(cons #\x (cons format/x 1))
		(cons #\b (cons format/b 1))
		(cons #\o (cons format/o 1))
		(cons #\c (cons format/c 1))
		(cons #\_ (cons format/_ 0))
		(cons #\% (cons format/% 0))
		(cons #\~ (cons format/~ 0))
		(cons #\t (cons format/t 0))
		(cons #\j (cons format/j 1))
		(cons #\C (cons format/C 1))
		(cons #\p (cons format/p 1)))))

		; (cons #\> (cons format/> 1))
		; (cons #\h (cons format/h 1))
		; (cons #\r (cons format/r 1))

(define (render-aligned formatter-proc
			port
			(the-args <list>)
			(info <vector>))
  ((rendition-function info) 
   (lambda 'render-aligned/invoke (port)
	   (apply* port info the-args formatter-proc))
   port
   info))
   
(define (chop args num req)
  (if (eq? num 0)
      (values '() args)
      (if (eq? num 1)
	  (if (pair? args)
	      (values (cons (car args) '())
		      (cdr args))
	      (error "format: missing argument for `~~~a'" req))
	  (let ((first (cons (car args) '())))
	    (let loop ((prev first)
		       (r (cdr args))
		       (n (- num 1)))
	      (if (eq? n 0)
		  (values first r)
		  (if (pair? r)
		      (let ((cell (cons (car r) '())))
			(set-cdr! prev cell)
			(loop cell (cdr r) (- n 1)))
		      (error "format: missing arguments for `~~~a'" req))))))))

(define (rendition-function (info <vector>))
  (if (pre-dot-number info)
      ;; field width specified...
      (if (sharp-flag info)
	  ;; `#' ==> truncate
	  (truncation-filter
	   ;; within-bounds filter
	   (if (at-flag info)
	       ;; `@' ==> don't align
	       identity
	       ;; no `@' ==> align
	       (lambda 'rendition-function/align (str)
		 (do-align str info)))
	   ;; outside bounds filter
	   (if (star-flag info)
	       ;; `*' ==> mark overlow
	       mark-overflow
	       ;; no `*' ==> don't mark overflow
	       identity))
	  ;; no `#' ==> don't truncate
	  no-truncate-filter)
      null-filter))

(define (null-filter fmt-proc
		     (port <output-port>)
		     (info <vector>))
  (fmt-proc port))

(define (mark-overflow str)
  (string-append str "..."))

(define (no-truncate-filter fmt-proc
			    (port <output-port>)
			    (info <vector>))
  (let ((p (open-output-string)))
    (fmt-proc p)
    (write-string port (do-align (close-output-port p) info))))

(define (truncation-filter within-proc overflow-proc)
  (lambda 'trunaction (fmt-proc port info)
    (write-string
     port
     (with-bounded-string-port* (pre-dot-number info)
				fmt-proc
				within-proc
				overflow-proc))))

(define (do-align str info)
  (let ((deficit (- (pre-dot-number info) (string-length str))))
    (if (< deficit 1)
	str
	(if (negative-flag info)
	    ;; `-' specified -- right justify
	    (if (pre-dot-leading-zero-flag info)
		;; `-0' specified -- justify center
		(let ((l (quotient deficit 2)))
		  (string-append (make-string l #\space)
				 str
				 (make-string (- deficit l) #\space)))
		;; `-' only -- justify right
		(string-append (make-string deficit #\space) str))
	    ;; `-' not specified
	    (if (pre-dot-leading-zero-flag info)
		;; `0' specified -- justify right with 0 padding
		(string-append (make-string deficit #\0) str)
		;; nothing specified -- justify left
		(string-append str (make-string deficit #\space)))))))

(define (%format dest (format-str <string>) arg-list (format-envt <list>))
  (let ((port (if dest (if (eq? dest #t) 
			   (current-output-port) 
			   dest)
		  (open-output-string)))
	(formatters (vector-ref format-envt 1)))
    (let loop ((control (parse-format-string format-str)) 
	       (args arg-list))
      (if (pair? control)
	  (let ((x (car control)))
	    (if (string? x)
		(begin
		  (write-string port x)
		  (loop (cdr control) args))
		(let* ((fmt-ch (if (ascii-char? x)
				   x
				   (format-character x)))
		       (fmtr (assq fmt-ch formatters)))
		  (if fmtr
		      (bind ((these-args rest (chop args (cddr fmtr) fmt-ch))
			     (fmtr-proc (cadr fmtr)))
			(if (ascii-char? x)
			    (apply* port $default-info these-args fmtr-proc)
			    (render-aligned fmtr-proc port these-args x))
			(loop (cdr control) rest))
		      (error "format: unrecognized format code `~~~a'"
			     fmt-ch)))))))
    (if dest
	(values)
	(close-output-port port))))

(define (format dest format-str . arg-list)
    (%format dest format-str arg-list *default-formatters*))

(set__format format)

(define (make-formatter new-formatters)
  (let ((formatters (vector #f 
			    (append new-formatters 
				    (vector-ref *default-formatters* 1)))))
    (lambda (dest format-str . arg-list)
      (%format dest format-str arg-list formatters))))
