/*
 * utils.c - Miscellaneous utility functions.
 *
 * Written by
 *  Ettore Perazzoli (ettore@comm2000.it)
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "vice.h"

#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>

#ifdef HAVE_VFORK_H
#include <vfork.h>
#endif

#ifdef __MSDOS__
#include <process.h>
#include <dir.h>
#include <io.h>
#endif

#include "utils.h"

/* ------------------------------------------------------------------------- */

/* Like malloc, but abort if not enough memory is available. */
void *xmalloc(size_t size)
{
    void *p = malloc(size);

    if (p == NULL) {
	fprintf(stderr,
		"Virtual memory exhausted: cannot allocate %lud bytes.\n",
		(unsigned long)size);
	exit(-1);
    }
    
    return p;
}

/* Like realloc, but abort if not enough memory is available. */
void *xrealloc(void *p, size_t size)
{
    void *new_p = realloc(p, size);

    if (new_p == NULL) {
	fprintf(stderr,
		"Virtual memory exhausted: cannot allocate %lud bytes.\n",
		(unsigned long)size);
	exit(-1);
    }
    
    return new_p;
}

/* Malloc enough space for `str', copy `str' into it and return its
   address. */
char *stralloc(const char *str)
{
    int l = strlen(str);
    char *p = (char *)xmalloc(l + 1);

    memcpy(p, str, l + 1);
    return p;
}

/* Malloc a new string whose contents concatenate `s1' and `s2'. */
char *concat(const char *s1, const char *s2)
{
    int l1, l2;
    char *new;

    l1 = strlen(s1);
    l2 = strlen(s2);
    new = xmalloc(l1 + l2 + 1);
    memcpy(new, s1, l1);
    memcpy(new + l1, s2, l2 + 1);
    return new;
}

/* Add the first `src_size' bytes of `src' to the end of `buf', which is a
   malloc'ed block of `max_buf_size' bytes of which only the first `buf_size'
   ones are used.  If the `buf' is not large enough, realloc it.  Return a 
   pointer to the new block. */
char *bufcat(char *buf, int *buf_size, int *max_buf_size,
	     const char *src, int src_size)
{
#define BUFCAT_GRANULARITY 0x1000
    if (*buf_size + src_size > *max_buf_size) {
	char *new_buf;

	*max_buf_size = (((*buf_size + src_size) / BUFCAT_GRANULARITY + 1)
			  * BUFCAT_GRANULARITY);
	new_buf = (char *)xrealloc(buf, *max_buf_size);
	buf = new_buf;
    }
    memcpy(buf + *buf_size, src, src_size);
    *buf_size += src_size;
    return buf;
}

/* Remove spaces from start and end of string `s'.  The string is not 
   reallocated even if it becomes smaller. */
void remove_spaces(char *s)
{
    char *p;
    int l = strlen(s);

    for (p = s; *p == ' '; p++)
        ;

    l -= (p - s);
    memmove(s, p, l + 1);

    if (l > 0) {
        for (p = s + l - 1; l > 0 && *p == ' '; l--, p--)
            ;
        *(p + 1) = '\0';
    }
}

/* ------------------------------------------------------------------------- */

/* Return a malloc'ed backup file name for file `fname'. */
char *make_backup_filename(const char *fname)
{
#ifndef __MSDOS__
    
    /* Just add a '~' to the end of the name. */
    int l = strlen(fname);
    char *p = (char *)xmalloc(l + 2);

    memcpy(p, fname, l);
    *(p + l) = '~';
    *(p + l + 1) = '\0';
    return p;
    
#else  /* !__MSDOS__ */
    
    char d[MAXDRIVE], p[MAXDIR], f[MAXFILE], e[MAXEXT];
    char new[MAXPATH];
    
    fnsplit(fname, d, p, f, e);
    fnmerge(new, d, p, f, "BAK");

    return stralloc(new);
    
#endif /* !__MSDOS__ */
}

/* Make a backup for file `fname'. */
int make_backup_file(const char *fname)
{
    char *backup_name = make_backup_filename(fname);
    int retval;
    
    /* Cannot do it... */
    if (backup_name == NULL)
	return -1;

    retval = rename(fname, backup_name);
    
    free(backup_name);
    return retval;
}

/* ------------------------------------------------------------------------- */

/* Return the length of an open file in bytes. */
unsigned long file_length(int fd)
{
    struct stat statbuf;
    
    if (fstat(fd, &statbuf) < 0)
	return -1;
    
    return statbuf.st_size;
}

/* Load the first `size' bytes of file named `name' into `dest'.  Return 0 on
   success, -1 on failure. */
int load_file(const char *name, void *dest, int size)
{
    int fd, r;

    fd = open(name, O_RDONLY);
    if (fd < 0)
	return -1;

    r = read(fd, (char *)dest, size);

    if (r != size) {
	if (r < 0)
	    perror(name);
	close(fd);
	return -1;
    } else {
	close(fd);
	return 0;
    }
}

/* Write the first `size' bytes of `src' into a newly created file `name'.
   If `name' already exists, it is replaced by the new one.  Returns 0 on
   success, -1 on failure. */
int save_file(const char *name, const void *src, int size)
{
    int fd, r;

    fd = open(name, O_WRONLY | O_TRUNC | O_CREAT, 0666);
    if (fd < 0)
	return -1;

    r = write(fd, (char *)src, size);

    if (r != size) {
	if (r < 0)
	    perror(name);
	close(fd);
	return -1;
    } else {
	close(fd);
	return 0;
    }
}

/* ------------------------------------------------------------------------- */

/* Launch program `name' (searched via the PATH environment variable) passing 
   `argv' as the parameters, wait for it to exit and return its exit status.  
   If `stdout_redir' or `stderr_redir' are != NULL, redirect stdout or stderr 
   to the corresponding file.  */
int spawn(const char *name, char **argv,
	  const char *stdout_redir, const char *stderr_redir)
{
#ifndef __MSDOS__
    
    /* Unix version. */
    
    pid_t child_pid;
    int child_status;

    child_pid = vfork();
    if (child_pid < 0) {
	perror("vfork");
	return -1;
    } else if (child_pid == 0) {
	if (stdout_redir && freopen(stdout_redir, "w", stdout) == NULL) {
	    perror(stdout_redir);
	    _exit(-1);
	}
	if (stderr_redir && freopen(stderr_redir, "w", stderr) == NULL) {
	    perror(stderr_redir);
	    _exit(-1);
	}
	execvp(name, argv);
	_exit(-1);
    }

    if (waitpid(child_pid, &child_status, 0) != child_pid) {
	perror("waitpid");
	return -1;
    }
    
    if (WIFEXITED(child_status))
	return WEXITSTATUS(child_status);
    else
	return -1;
    
#else
    
    /* MS-DOS version. */

    int new_stdout, new_stderr;
    int old_stdout_mode, old_stderr_mode;
    int old_stdout, old_stderr;
    int retval;

    new_stdout = new_stderr = old_stdout = old_stderr = -1;
    
    /* Make sure we are in binary mode. */
    old_stdout_mode = setmode(STDOUT_FILENO, O_BINARY);
    old_stderr_mode = setmode(STDERR_FILENO, O_BINARY);

    /* Redirect stdout and stderr as requested, saving the old
       descriptors. */
    if (stdout_redir != NULL) {
	old_stdout = dup(STDOUT_FILENO);
	new_stdout = open(stdout_redir, O_WRONLY | O_TRUNC | O_CREAT, 0666);
	if (new_stdout == -1) {
	    perror(stdout_redir);
	    retval = -1;
	    goto cleanup;
	}
	dup2(new_stdout, STDOUT_FILENO);
    }
    if (stderr_redir != NULL) {
	old_stderr = dup(STDERR_FILENO);
	new_stderr = open(stderr_redir, O_WRONLY | O_TRUNC | O_CREAT, 0666);
	if (new_stderr == -1) {
	    perror(stderr_redir);
	    retval = -1;
	    goto cleanup;
	}
	dup2(new_stderr, STDERR_FILENO);
    }

    /* Spawn the child process. */
    retval = spawnvp(P_WAIT, name, argv);

cleanup:
    if (old_stdout >= 0)
	dup2(old_stdout, STDOUT_FILENO);
    if (old_stderr >= 0)
	dup2(old_stderr, STDERR_FILENO);
    if (old_stdout_mode >= 0)
	setmode(STDOUT_FILENO, old_stdout_mode);
    if (old_stderr_mode >= 0)
	setmode(STDERR_FILENO, old_stderr_mode);
    if (new_stdout >= 0)
	close(new_stdout);
    if (new_stderr >= 0)
	close(new_stderr);

    return retval;
#endif
}


