/*
 * true1541.c - Hardware-level Commodore 1541 disk drive emulation.
 *
 * Written by
 *  Daniel Sladic (sladic@eecg.toronto.edu)
 *  Ettore Perazzoli (ettore@comm2000.it)
 *  Andre' Fachat (fachat@physik.tu-chemnitz.de)
 *  Teemu Rantanen (tvr@cs.hut.fi)
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* TODO: - format trap;
         - fix disk sync;
	 - exact emulation of disk rotation;
	 - serial bus handling might be faster. */

#define __1541__

#include "vice.h"

#include <stdio.h>
#include <fcntl.h>
#include <ctype.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>

#include "true1541.h"
#include "interrupt.h"
#include "vmachine.h"
#include "serial.h"
#include "drive.h"
#include "warn.h"
#include "mem.h"
#include "resources.h"
#include "memutils.h"

/* ------------------------------------------------------------------------- */

/* If this is defined, allow exact GCR operation even when writing to the
   disk. */
#define TRUE_GCR_WRITE

#define NOT(x) ((x)^1)
#define NUM_BYTES_SECTOR_GCR 353

extern void set_atn(BYTE state);

BYTE true1541_rom[TRUE1541_ROM_SIZE];
BYTE true1541_ram[TRUE1541_RAM_SIZE];
int true1541_led_status;

static int true1541_rom_loaded = 0;

static BYTE cpu_data = 0;
static BYTE cpu_clock = 0;
static BYTE cpu_atn = 0;
static BYTE drive_data = 0;
static BYTE drive_clock = 0;
static BYTE drive_atna = 0;
static BYTE drive_data_modifier = 0;
static BYTE cpu_bus_val = 0;
static BYTE drive_bus_val = 0;
static BYTE bus_data = 0;
static BYTE bus_clock = 0;
static BYTE bus_atn = 0;
static int init_complete = 0;
static int idlecount = 0;

static DRIVE *true1541_floppy;
static BYTE diskID1, diskID2;

extern char sector_map[43];

static BYTE GCR_data[MAX_TRACKS_1541 * 21 * NUM_BYTES_SECTOR_GCR];
static unsigned int head_ptr, track_GCR_start, track_GCR_end;
static unsigned int cur_ht = 2;
static int GCR_dirty_track = 0; 
static void GCR_data_writeback(void);

#define GCR_OFFSET(track, sector)  ((track - 1) * 21 * NUM_BYTES_SECTOR_GCR \
				    + sector * NUM_BYTES_SECTOR_GCR)

/* Warnings. */
enum true1541_warnings { WARN_GCRWRITE };
#define TRUE1541_NUM_WARNINGS	 (WARN_GCRWRITE + 1)
static warn_t *true1541_warn;

/* ------------------------------------------------------------------------- */

/* GCR handling. */

static BYTE GCR_conv_data[16] = { 0x0a, 0x0b, 0x12, 0x13,
				  0x0e, 0x0f, 0x16, 0x17,
				  0x09, 0x19, 0x1a, 0x1b,
				  0x0d, 0x1d, 0x1e, 0x15 };

static BYTE From_GCR_conv_data[32] = { 0, 0, 0, 0, 0, 0, 0, 0,
				       0, 8, 0, 1, 0,12, 4, 5, 
				       0, 0, 2, 3, 0,15, 6, 7,
				       0, 9,10,11, 0,13,14, 0 };

static void convert_4bytes_to_GCR(BYTE *buffer, BYTE *ptr)
{
    *ptr = GCR_conv_data[(*buffer) >> 4] << 3;
    *ptr |= GCR_conv_data[(*buffer) & 0x0f] >> 2;
    ptr++;

    *ptr = GCR_conv_data[(*buffer) & 0x0f] << 6;
    buffer++;
    *ptr |= GCR_conv_data[(*buffer) >> 4] << 1;
    *ptr |= GCR_conv_data[(*buffer) & 0x0f] >> 4;
    ptr++;

    *ptr = GCR_conv_data[(*buffer) & 0x0f] << 4;
    buffer++;
    *ptr |= GCR_conv_data[(*buffer) >> 4] >> 1;
    ptr++;

    *ptr = GCR_conv_data[(*buffer) >> 4] << 7;
    *ptr |= GCR_conv_data[(*buffer) & 0x0f] << 2;
    buffer++;
    *ptr |= GCR_conv_data[(*buffer) >> 4] >> 3;
    ptr++;

    *ptr = GCR_conv_data[(*buffer) >> 4] << 5;
    *ptr |= GCR_conv_data[(*buffer) & 0x0f];
}

static void convert_GCR_to_4bytes(BYTE *buffer, BYTE *ptr)
{
    BYTE gcr_bytes[8];
    int i;

    gcr_bytes[0] =  (*buffer) >> 3;
    gcr_bytes[1] =  ((*buffer) & 0x07) << 2;
    buffer++;
    gcr_bytes[1] |= (*buffer) >> 6;
    gcr_bytes[2] =  ((*buffer) & 0x3e) >> 1;
    gcr_bytes[3] =  ((*buffer) & 0x01) << 4;
    buffer++;
    gcr_bytes[3] |= ((*buffer) & 0xf0) >> 4;
    gcr_bytes[4] =  ((*buffer) & 0x0f) << 1;
    buffer++;
    gcr_bytes[4] |= (*buffer) >> 7;
    gcr_bytes[5] =  ((*buffer) & 0x7c) >> 2;
    gcr_bytes[6] =  ((*buffer) & 0x03) << 3;
    buffer++;
    gcr_bytes[6] |= ((*buffer) & 0xe0) >> 5;
    gcr_bytes[7] = (*buffer) & 0x1f;

    for (i = 0; i < 4; i++, ptr++) {
        *ptr = From_GCR_conv_data[gcr_bytes[2 * i]] << 4;
        *ptr |= From_GCR_conv_data[gcr_bytes[2 * i + 1]];
    }
}

static void convert_sector_to_GCR(BYTE *buffer, BYTE *ptr, int track,
				  int sector)
{
    int i;
    BYTE buf[4];

    memset(ptr, 0xff, 5);	/* Sync */
    ptr += 5;

    buf[0] = 0x08;		/* Header identifier */
    buf[1] = sector ^ track ^ diskID2 ^ diskID1;
    buf[2] = sector;
    buf[3] = track;
    convert_4bytes_to_GCR(buf, ptr);
    ptr += 5;

    buf[0] = diskID2;
    buf[1] = diskID1;
    buf[2] = buf[3] = 0x0f;
    convert_4bytes_to_GCR(buf, ptr);
    ptr += 5;

    memset(ptr, 0x55, 8);	/* Header Gap */
    ptr += 8;

    memset(ptr, 0xff, 5);	/* Sync */
    ptr += 5;

    for (i = 0; i < 65; i++) {
	convert_4bytes_to_GCR(buffer, ptr);
	buffer += 4;
	ptr += 5;
    }
}

static void convert_GCR_to_sector(BYTE *buffer, BYTE *ptr)
{
    BYTE *from, *end;
    int i;

    from = ptr;
    end = ptr + NUM_BYTES_SECTOR_GCR - 65*5;
    while (*from != 0xff && from <= end) from++;
    while (*from == 0xff && from <= end) from++;
    while (*from != 0xff && from <= end) from++;
    while (*from == 0xff && from <= end) from++;
    if (from > end) {
	memset(buffer, 0, 260);
	return;
    }
    for (i = 0; i < 65; i++) {
	convert_GCR_to_4bytes(from, buffer);
	buffer += 4;
	from += 5;
    }
}

static void read_image_GCR(void)
{
    BYTE buffer[260], *ptr, chksum;
    int rc, i;
    int track, sector;

    if (!true1541_floppy)
	return;

    buffer[0] = 0x07;
    buffer[258] = buffer[259] = 0;

    for (track = 1; track <= 35; track++) {
	for (sector = 0; sector < sector_map[track]; sector++) {
	    ptr = GCR_data + GCR_OFFSET(track, sector);
	    rc = floppy_read_block(true1541_floppy->ActiveFd,
				   true1541_floppy->ImageFormat,
				   buffer + 1, track, sector,
				   true1541_floppy->D64_Header);
	    if (rc < 0) {
		printf("1541: error reading T:%d S:%d from the disk image\n",
		       track, sector);
		/* FIXME: could be handled better. */
	    } else {
		chksum = buffer[1]; 
		for (i = 2; i < 257; i++)
		    chksum ^= buffer[i];
		buffer[257] = chksum;
		convert_sector_to_GCR(buffer, ptr, track, sector);
	    }
	}
    }
}

static int setID(void)
{
    BYTE buffer[256];
    int rc;

    if (!true1541_floppy)
	return -1;

    rc = floppy_read_block(true1541_floppy->ActiveFd,
			   true1541_floppy->ImageFormat,
			   buffer, 18, 0, true1541_floppy->D64_Header);
    if (rc >= 0) {
	diskID1 = buffer[0xa2];
	diskID2 = buffer[0xa3];
	true1541_ram[0x12] = diskID1;
	true1541_ram[0x13] = diskID2;
    }
    
    return rc;
}

/* ------------------------------------------------------------------------- */

/* Initialize the hardware-level 1541 emulation (should be called at least once
   before anything else). */
void initialize_true1541(void)
{
    if (!true1541_rom_loaded) {
	true1541_warn = warn_init("1541", TRUE1541_NUM_WARNINGS);
	/* Load the ROMs and initialize the CPU. */
	if (mem_load_sys_file(app_resources.directory, app_resources.dosName,
			      true1541_rom, TRUE1541_ROM_SIZE) < 0) {
	    fprintf(stderr,
		    "1541: Warning: ROM image not loaded; hardware-level "
		    "emulation is not available.\n");
	    app_resources.true1541 = 0;
	} else {
	    printf("1541: ROM loaded successfully.\n");
	    true1541_rom_loaded = 1;

	    /* Remove the ROM check. */
	    true1541_rom[0xeae4 - 0xc000] = 0xea;
	    true1541_rom[0xeae5 - 0xc000] = 0xea;
	    true1541_rom[0xeae8 - 0xc000] = 0xea;
	    true1541_rom[0xeae9 - 0xc000] = 0xea;
	
	    /* Trap the idle loop. */
	    true1541_rom[0xec9b - 0xc000] = 0x00;

#ifndef TRUE_GCR_WRITE
	    /* Trap the write sector routine. */
	    true1541_rom[0xf594 - 0xc000] = 0x20;  /* JSR $F5F2 */
	    true1541_rom[0xf595 - 0xc000] = 0xf2;
	    true1541_rom[0xf596 - 0xc000] = 0xf5;
	    true1541_rom[0xf597 - 0xc000] = 0x00;
#endif
	
	    true1541_cpu_init();
	}
    } /* !true1541_rom_loaded */
}

/* Activate full 1541 emulation. */
void true1541_enable(void)
{
    /* Always disable kernal traps. */
    if (rom_loaded) {
	remove_serial_traps();
    }
    app_resources.true1541 = 1;
    true1541_cpu_wake_up();
}

/* Disable full 1541 emulation. */
void true1541_disable(void)
{
    if (rom_loaded) {
	/* Do not enable kernal traps if required. */
	if (app_resources.noTraps)
	    remove_serial_traps();
	else
	    install_serial_traps();
    }
    app_resources.true1541 = 0;
    true1541_cpu_sleep();
 
    GCR_data_writeback();
}

/* This is called when the true1541 resource is changed, to acknowledge the new
   value. */
void true1541_ack_switch(void)
{
    if (app_resources.true1541)
	true1541_enable();
    else
	true1541_disable();
}

void true1541_reset(void)
{
    true1541_cpu_reset();
    warn_reset(true1541_warn);
}

/* ------------------------------------------------------------------------- */

static int have_new_disk = 0;	/* used for disk change detection */

/* Attach a disk image to the true 1541 emulation. */
int true1541_attach_floppy(DRIVE *floppy)
{
    if (floppy->ImageFormat != 1541)
	return -1;

    true1541_floppy = floppy;
    have_new_disk = 1;
    
    if (setID() >= 0) {
	read_image_GCR();
	return 0;
    } else
	return -1;
}

/* Detach a disk image from the true 1541 emulation. */
int true1541_detach_floppy(void)
{
    if (true1541_floppy != NULL) {
	GCR_data_writeback();
	true1541_floppy = NULL;
	memset(GCR_data, 0, sizeof(GCR_data));
    }
    return 0;
}

/* ------------------------------------------------------------------------- */

/* Read a GCR byte from the disk. */
BYTE true1541_read_disk_byte(void)
{
    BYTE val = GCR_data[head_ptr++];

#if 0
    printf("1541: head read ptr:0x%x val:0x%x\n", head_ptr, val);
#endif
    if (head_ptr == track_GCR_end)
	head_ptr = track_GCR_start;

    return val;
}

/* Return non-zero if the Sync mark is found. */
int true1541_sync_found(void)
{
    BYTE val = GCR_data[head_ptr];
    unsigned int next_ptr;
    
    if (val != 0xff)
	true1541_read_disk_byte();
    else {
	while(1) {
	    next_ptr = head_ptr + 1;
	    if (next_ptr == track_GCR_end)
		next_ptr = track_GCR_start;
          
	    if (GCR_data[next_ptr] != 0xff)
		break;

	    head_ptr++;
	    if (head_ptr == track_GCR_end)
		head_ptr = track_GCR_start;
	}
    }

    return (val == 0xff);
}

/* Increment the head position by `step'. */
void true1541_move_head(int step)
{
    GCR_data_writeback();

    cur_ht += step;

    if (cur_ht > 70)
	cur_ht = 70;
    else if (cur_ht < 2)
	cur_ht = 2;
    
    track_GCR_start = (cur_ht / 2 - 1) * 21 * NUM_BYTES_SECTOR_GCR;
    track_GCR_end = track_GCR_start + (sector_map[cur_ht / 2]
				       * NUM_BYTES_SECTOR_GCR);
    head_ptr = track_GCR_start;

    if (step)
	printf("1541: head on track %.1f\n", (double)cur_ht / 2);
}

/* Write one GCR byte to the disk. */
void true1541_write_gcr(BYTE val)
{
#ifndef TRUE_GCR_WRITE
    /* This is not implemented! */
    warn(true1541_warn, WARN_GCRWRITE,
	 "program tries to write raw GCR data to the disk.");
#else
    if (true1541_floppy == NULL)
	return;
    if (!true1541_floppy->ReadOnly) {
	GCR_data[head_ptr++] = val;
        GCR_dirty_track = 1;
    }
    if (head_ptr == track_GCR_end)
       head_ptr = track_GCR_start;
#endif
}

/* Return the write protect sense status. */
int true1541_write_protect_sense(void) 
{
    if (true1541_floppy == NULL) {
	/* No disk in drive, write protection is on. */
	return 1;
    } else if (have_new_disk) {
	/* Disk has changed, make sure the drive sees at least one change in
	   the write protect status. */
	have_new_disk = 0;
	return !true1541_floppy->ReadOnly;
    } else {
	return true1541_floppy->ReadOnly;
    }
}

static void GCR_data_writeback(void)
{
     int rc, track, sector;
     BYTE buffer[260], *ptr;

     if (!GCR_dirty_track)
        return;

     GCR_dirty_track = 0;
     track = cur_ht / 2;
     for (sector = 0; sector < sector_map[track]; sector++) {
	 ptr = GCR_data + GCR_OFFSET(track, sector);
         convert_GCR_to_sector(buffer, ptr);
         rc = floppy_write_block(true1541_floppy->ActiveFd,
	    		         true1541_floppy->ImageFormat,
			         buffer + 1, track, sector,
			         true1541_floppy->D64_Header);
         if (rc < 0) {
            fprintf(stderr,
		    "1541: Could not update T:%d S:%d on disk image.\n",
		    track, sector);
         } 
     }
}

/* ------------------------------------------------------------------------- */

#ifndef TRUE_GCR_WRITE
/* DOS ROM write trap.  As handling GCR writes is complicate, we simply put a
   trap in the DOS write routine and do everything by hand. */
static void true1541_write_trap(void)
{
    int track, sector, addr;

    track = true1541_ram[0x18];
    sector = true1541_ram[0x19];
    addr = true1541_ram[0x30] | (true1541_ram[0x31] << 8);
    printf("1541: write T:%d S:%d from $%04X... ", track, sector, addr);
    if (addr <= 0x700
	&& floppy_write_block(true1541_floppy->ActiveFd,
			      true1541_floppy->ImageFormat,
			      true1541_ram + addr, track, sector,
			      true1541_floppy->D64_Header) >= 0) {
	BYTE buf[260], checksum;
	int i;

	/* FIXME: this is inefficient and ugly. */
	buf[0] = 0x7;
	buf[258] = buf[259] = 0;
	checksum = buf[1] = true1541_ram[addr];
	for (i = 2; i < 257; i++) {
	    buf[i] = true1541_ram[addr + i - 1];
	    checksum ^= buf[i];
	}
	buf[257] = checksum;
	convert_sector_to_GCR(buf, GCR_data + GCR_OFFSET(track, sector),
			      track, sector);
	printf("OK.\n");
    } else
	printf("Error!\n");
    true1541_program_counter = 0xf5dc;
}
#endif

/* Handle a ROM trap. */
int true1541_trap_handler(void)
{
    if (true1541_program_counter == 0xec9b) {
	/* Idle loop */
	init_complete = 1;
	true1541_program_counter = 0xebff;
	if (app_resources.true1541IdleMethod == TRUE1541_IDLE_TRAP_IDLE)
	    true1541_clk = next_alarm_clk(&true1541_int_status);
#ifndef TRUE_GCR_WRITE
    } else if (true1541_program_counter == 0xf597) {
	true1541_write_trap();
#endif
    } else
	return 1;

    return 0;
}

/* ------------------------------------------------------------------------- */

/* IEC bus handling. */

#define BUS_DBG 0

inline void resolve_bus_signals(void)
{
    bus_atn = NOT(cpu_atn);
    bus_clock = (NOT(cpu_clock) & NOT(drive_clock));
    bus_data = (NOT(drive_data) & NOT(drive_data_modifier) & NOT(cpu_data));

#if BUS_DBG
    printf("SB: [%ld]  data:%d clock:%d atn:%d\n",
	   true1541_clk, bus_data, bus_clock, bus_atn);
#endif
}

void serial_bus_drive_write(BYTE data)
{
    static int last_write = 0;

    if (!app_resources.true1541)
	return;
    
    drive_data = ((data & 2) >> 1);
    drive_clock = ((data & 8) >> 3);
    drive_atna = ((data & 16) >> 4);
    drive_data_modifier = (NOT(cpu_atn) ^ NOT(drive_atna));

    if (last_write != (data & 26)) {
	resolve_bus_signals();
	idlecount = 0;
#if BUS_DBG
	if (traceflg)
	    printf("WRITE RESET\n");
#endif
    }
    last_write = data & 26;

#if BUS_DBG
    printf("SB: device write  data:%d clock:%d atna:%d\n",
	   drive_data, drive_clock, drive_atna);
#endif
}

BYTE serial_bus_drive_read(void)
{
    if (!app_resources.true1541)
	return 0;
    
    idlecount = 0;

    drive_bus_val = ((NOT(bus_data)) | (NOT(bus_clock) << 2)
		     | (NOT(bus_atn) << 7));

#if BUS_DBG
    printf("SB: drive read  data:%d clock:%d atn:%d\n",
	   (~bus_data) & 1, (~bus_clock) & 1, (cpu_atn) & 1);
#endif
    
    return drive_bus_val;
}

/* ------------------------------------------------------------------------- */

/* C64/C128-specific IEC bus handling. */

#if defined(CBM64) || defined(C128)

/* The C64 has all bus lines in one I/O byte in a CIA.  If this byte is read or
   modified, these routines are called. */

void serial_bus_cpu_write(BYTE data)
{
    static int last_write = 0;
    
    if (!app_resources.true1541)
	return;

    true1541_cpu_execute();
   
    if ((cpu_atn == 0) && (data & 8))
	set_atn(1);
    
    if (!(data & 8))
	set_atn(0);

    cpu_data = ((data & 32) >> 5);
    cpu_clock = ((data & 16) >> 4);
    cpu_atn = ((data & 8) >> 3);
    drive_data_modifier = (NOT(cpu_atn) ^ NOT(drive_atna));

    if (last_write != (data & 56)) {
	resolve_bus_signals();
	idlecount = 0;
#if BUS_DBG
	if (traceflg)
	    printf("WRITE RESET\n");
#endif
    }
    
    last_write = data & 56;

#if BUS_DBG
    printf("SB: cpu write  data:%d clock:%d atn:%d\n",
	   cpu_data, cpu_clock, cpu_atn);
#endif
}

BYTE serial_bus_cpu_read(void)
{
    if (!app_resources.true1541)
	return 0;

    true1541_cpu_execute();

    idlecount = 0;

    cpu_bus_val = (bus_data << 7) | (bus_clock << 6) | (bus_atn << 3);

#if BUS_DBG
    printf("SB: cpu read  data:%d clock:%d\n",
	   (~cpu_data) & (~drive_data) & 1, (~cpu_clock) & (~drive_clock) & 1);
#endif
    
    return cpu_bus_val;
}

#endif

/* ------------------------------------------------------------------------- */

/* VIC20-specific IEC bus handling. */

#if defined(VIC20)

/*
   The VIC20 has a strange bus layout for the serial IEC bus.

     VIA1 CA2 CLK out
     VIA1 CB1 SRQ in
     VIA1 CB2 DATA out
     VIA2 PA0 CLK in
     VIA2 PA1 DATA in
     VIA2 PA7 ATN out
     
 */

extern unsigned int reg_pc;

/* These two routines are called for VIA2 Port A. */

BYTE serial_bus_pa_read(void)
{
    if (!app_resources.true1541)
	return 0;

    true1541_cpu_execute();

    idlecount = 0;

    cpu_bus_val = (bus_data << 1) | (bus_clock << 0) | (bus_atn << 7);

#if BUS_DBG 
    if(cpu_atn) 
      printf("SB: cpu PC=%04x read  data:%d clock:%d, value read=%02x\n",
	   reg_pc,
	   (~cpu_data) & (~drive_data) & 1, (~cpu_clock) & (~drive_clock) & 1,
	   cpu_bus_val);
#endif
    
    return cpu_bus_val;
}

void serial_bus_pa_write(BYTE data)
{
    static int last_write = 0;
    
    if (!app_resources.true1541)
	return;

    true1541_cpu_execute();
   
    if ((cpu_atn == 0) && (data & 128))
	set_atn(1);
    
    if (!(data & 128))
	set_atn(0);

    cpu_atn = ((data & 128) >> 7);
    drive_data_modifier = (NOT(cpu_atn) ^ NOT(drive_atna));

    if (last_write != (data & 128)) {
#if BUS_DBG
	printf("CPU PC=%04x set ATN=%d\n",reg_pc, cpu_atn);
#endif
	resolve_bus_signals();
	idlecount = 0;
#if BUS_DBG
	if (traceflg)
	    printf("WRITE RESET\n");
#endif
    }
    
    last_write = data & 128;

#if 0 /* BUS_DBG */
    printf("SB: cpu write (%02x) pa data:%d clock:%d atn:%d\n",
	   data, cpu_data, cpu_clock, cpu_atn);
#endif
}

/* This routine is called for VIA1 PCR (= CA2 and CB2).
   Although Cx2 uses three bits for control, we assume the calling routine has
   set bit 5 and bit 1 to the real output value for CB2 (DATA out) and CA2 (CLK
   out) resp. (25apr1997 AF) */

void serial_bus_pcr_write(BYTE data)
{
    static int last_write = 0;
   
    if (!app_resources.true1541)
	return;

    true1541_cpu_execute();
   
    cpu_data = ((data & 32) >> 5);
    cpu_clock = ((data & 2) >> 1);
    drive_data_modifier = (NOT(cpu_atn) ^ NOT(drive_atna));

    if (last_write != (data & 34)) {
#if BUS_DBG
	printf("CPU PC=%04x set DATA=%d, CLK=%d\n",reg_pc, cpu_data, cpu_clock);
#endif
	resolve_bus_signals();
	idlecount = 0;
#if BUS_DBG
	if (traceflg)
	    printf("WRITE RESET\n");
#endif
    }
    
    last_write = data & 34;

#if  0 /*BUS_DBG*/
    printf("SB: cpu write (%02x) pcr data:%d clock:%d atn:%d\n",
	   data, cpu_data, cpu_clock, cpu_atn);
#endif
}

#endif

/* ------------------------------------------------------------------------- */

/* Set the sync factor between the computer and the 1541. */

void true1541_set_pal_sync_factor(void)
{
    true1541_set_sync_factor(TRUE1541_PAL_SYNC_FACTOR);
}

void true1541_set_ntsc_sync_factor(void)
{
    true1541_set_sync_factor(TRUE1541_NTSC_SYNC_FACTOR);
}

void true1541_ack_sync_factor(void)
{
    true1541_set_sync_factor(app_resources.true1541SyncFactor);
}

