/*
 * sprites.h - Sprite drawing macros.
 *
 * Written by
 *   Ettore Perazzoli (ettore@comm2000.it)
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SPRITE_ONE_PIXEL(sprite_bit, imgptr, collmskptr, pos, color,	\
			 collmsk_return)				\
  (imgptr)[(pos)] = PIXEL((color));					\
  (collmsk_return) |= (collmskptr)[(pos)];				\
  (collmskptr)[(pos)] |= (sprite_bit);

#ifdef ALLOW_UNALIGNED_ACCESS

#  define SPRITE_TWO_PIXELS(sprite_bit, imgptr, collmskptr, pos, color, \
			    collmsk_return)				\
     *((PIXEL2 *)(imgptr) + (pos)) = PIXEL2((color));			\
     (collmsk_return) |= *((WORD *)(collmskptr) + (pos));		\
     *((WORD *)(collmskptr) + (pos)) |= (sprite_bit) | ((sprite_bit) << 8);

#  define SPRITE_ONE_PIXEL_2x(sprite_bit, imgptr, collmskptr, pos, color, \
			      collmsk_return)				  \
     *((PIXEL2 *)(imgptr) + (pos)) = PIXEL2((color));			  \
     (collmsk_return) |= (collmskptr)[(pos)];				  \
     (collmskptr)[(pos)] |= (sprite_bit);

#  define SPRITE_TWO_PIXELS_2x(sprite_bit, imgptr, collmskptr, pos, color, \
			       collmsk_return)				   \
     *((PIXEL4 *)(imgptr) + (pos)) = PIXEL4((color));	        	   \
     (collmsk_return) |= *((WORD *)(collmskptr) + (pos));		   \
     *((WORD *)(collmskptr) + (pos)) |= (sprite_bit) | ((sprite_bit) << 8);

#else

#  define SPRITE_TWO_PIXELS(sprite_bit, imgptr, collmskptr, pos, color, \
			    collmsk_return)                             \
     SPRITE_ONE_PIXEL(sprite_bit, imgptr, collmskptr, pos * 2, color,   \
		      collmsk_return);                                  \
     SPRITE_ONE_PIXEL(sprite_bit, imgptr, collmskptr, pos * 2 + 1,	\
		      color, collmsk_return);

#  define SPRITE_ONE_PIXEL_2x(sprite_bit, imgptr, collmskptr, pos, color,   \
			      collmsk_return)				    \
     (imgptr)[(pos) * 2] = (imgptr)[(pos) * 2 + 1] = PIXEL((color));  \
     (collmsk_return) |= (collmskptr)[pos];				    \
     (collmskptr)[pos] |= (sprite_bit);

#  define SPRITE_TWO_PIXELS_2x(sprite_bit, imgptr, collmskptr, pos, color, \
			       collmsk_return)				   \
     SPRITE_ONE_PIXEL_2x(sprite_bit, imgptr, collmskptr, pos * 2, color,   \
        	         collmsk_return);				   \
     SPRITE_ONE_PIXEL_2x(sprite_bit, imgptr, collmskptr, pos * 2 + 1,	   \
		         color, collmsk_return);
     
#endif


/* Hires sprites */
  
#define _SPRITE_MASK(msk, size, sprite_bit, imgptr, collmskptr, color,	       \
		     collmsk_return, DRAW)				       \
  do {								               \
      DWORD __m;							       \
      int __p;								       \
      for (__m = 1 << ((size) - 1), __p = 0; __p < (size); __p++, __m >>= 1)   \
	  if ((msk) & __m) {						       \
	      DRAW(sprite_bit, imgptr, collmskptr, __p, color,  	       \
		   collmsk_return);                                            \
	  }							               \
  } while (0)

     
#define SPRITE_MASK(msk, size, sprite_bit, imgptr, collmskptr, color, \
		    collmsk_return)				      \
  _SPRITE_MASK(msk, size, sprite_bit, imgptr, collmskptr, color,      \
               collmsk_return, SPRITE_ONE_PIXEL)
     
#define SPRITE_DOUBLE_MASK(msk, size, sprite_bit, imgptr, collmskptr, \
			   color, collmsk_return)		      \
  _SPRITE_MASK(msk, size, sprite_bit, imgptr, collmskptr, color,      \
	       collmsk_return, SPRITE_TWO_PIXELS)
     
#define SPRITE_MASK_2x(msk, size, sprite_bit, imgptr, collmskptr,     \
		       color, collmsk_return)			      \
  _SPRITE_MASK(msk, size, sprite_bit, imgptr, collmskptr, color,      \
	       collmsk_return, SPRITE_ONE_PIXEL_2x)
     
#define SPRITE_DOUBLE_MASK_2x(msk, size, sprite_bit, imgptr, collmskptr, \
			      color, collmsk_return)			 \
  _SPRITE_MASK(msk, size, sprite_bit, imgptr, collmskptr, color,	 \
		  collmsk_return, SPRITE_TWO_PIXELS_2x)
     
     
/* Multicolor sprites */


#define _MCSPRITE_MASK(msk, mcmsk, size, sprite_bit, imgptr, collmskptr, \
		       pixel_table, collmsk_return, DRAW)		 \
  do {							     	         \
      DWORD __m;							 \
      int __p;								 \
      for (__m = 1 << ((size) - 1), __p = 0; __p < (size);		 \
	   __p += 2, __m >>= 2, (mcmsk) <<= 2) {			 \
  	  if ((msk) & __m) {						 \
	      DRAW(sprite_bit, imgptr, collmskptr, __p,			 \
		   pixel_table[((mcmsk) >> 22) & 0x3], collmsk_return);  \
									 \
	  }								 \
	  if ((msk) & (__m >> 1)) {					 \
	      DRAW(sprite_bit, imgptr, collmskptr, __p + 1,		 \
		   pixel_table[((mcmsk) >> 22) & 0x3], collmsk_return);  \
          }								 \
      }									 \
  } while (0)

     
#define MCSPRITE_MASK(msk, mcmsk, size, sprite_bit, imgptr, collmskptr,	\
		      pixel_table, collmsk_return)			\
  _MCSPRITE_MASK(msk, mcmsk, size, sprite_bit, imgptr, collmskptr,	\
		 pixel_table, collmsk_return, SPRITE_ONE_PIXEL)
		     
#define MCSPRITE_MASK_2x(msk, mcmsk, size, sprite_bit, imgptr, collmskptr, \
			 pixel_table, collmsk_return)			   \
  _MCSPRITE_MASK(msk, mcmsk, size, sprite_bit, imgptr, collmskptr,         \
		 pixel_table, collmsk_return, SPRITE_ONE_PIXEL_2x)
     

#define _MCSPRITE_DOUBLE_MASK(msk, mcmsk, size, sprite_bit, imgptr,           \
			      collmskptr, pixel_table, collmsk_return, DRAW)  \
  do {									      \
      DWORD __m;							      \
      int __p, __i;							      \
      for (__m = 1 << ((size) - 1), __p = 0; __p < (size);		      \
	   __p += 4, (mcmsk) <<= 2) {					      \
	  for (__i = 0; __i < 4; __i++, __m >>= 1)			      \
   	      if ((msk) & __m) {					      \
	          DRAW(sprite_bit, imgptr, collmskptr, __p + __i,             \
		       pixel_table[((mcmsk) >> 22) & 0x3],  collmsk_return);  \
	      }								      \
      }								              \
  } while (0)

#define MCSPRITE_DOUBLE_MASK(msk, mcmsk, size, sprite_bit, imgptr,         \
			     collmskptr, pixel_table, collmsk_return)      \
  _MCSPRITE_DOUBLE_MASK (msk, mcmsk, size, sprite_bit, imgptr, collmskptr, \
			 pixel_table, collmsk_return, SPRITE_ONE_PIXEL)
  
#define MCSPRITE_DOUBLE_MASK_2x(msk, mcmsk, size, sprite_bit, imgptr,      \
				collmskptr, pixel_table, collmsk_return)   \
  _MCSPRITE_DOUBLE_MASK (msk, mcmsk, size, sprite_bit, imgptr, collmskptr, \
			 pixel_table, collmsk_return, SPRITE_ONE_PIXEL_2x)
