/*
 * kbd.c - MS-DOS keyboard driver.
 *
 * Written by
 *  Ettore Perazzoli (ettore@comm2000.it)
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define _KBD_DOS_C

#include <sys/farptr.h>
#include <pc.h>
#include <time.h>
#include <go32.h>
#include <dos.h>
#include <dpmi.h>
#include <stdlib.h>

#include "vmachine.h"
#include "macro.h"
#include "kbd.h"
#include "kbdef.h"
#include "interrupt.h"
#include "resources.h"
#include "vsync.h"
#include "mem.h"

/* #define DEBUG_KBD */

int keyarr[KBD_ROWS];
int rev_keyarr[KBD_COLS];

/* Joystick status. We use 3 elements to avoid `-1'. */
BYTE joy[3] = { 0, 0, 0 };

/* This is nonzero if the user wants the menus. */
int _escape_requested = 0;

/* Segment info for the custom keyboard handler. */
static _go32_dpmi_seginfo std_kbd_handler_seginfo;

/* This takes account of the status of the modifier keys on the real PC
   keyboard. */
static struct {
    int left_ctrl:1;
    int right_ctrl:1;
    int left_shift:1;
    int right_shift:1;
    int left_alt:1;
    int right_alt:1;
} modifiers;

/* What is the location of the virtual shift key in the keyboard matrix? */
#ifdef PET
static int virtual_shift_column = 0;
static int virtual_shift_row = 6;
#else
static int virtual_shift_column = COL7;
static int virtual_shift_row = 1;
#endif

/* -------------------------------------------------------------------------- */

#ifdef PET

static keyconv *keyconvmap;

void set80key(void)
{
    keyconvmap = keyconvmap80;
    virtual_shift_row = 6;
    virtual_shift_column = 0;
}

void set40key(void)
{
    keyconvmap = keyconvmap40;
    virtual_shift_row = 8;
    virtual_shift_column = 0;
}

#endif /* PET */

/* ------------------------------------------------------------------------- */

#if defined(VIC20) || defined(CBM64) || defined(C128)

/* Emulate the joystick via keypad. */
static int handle_joy_emu(int kcode, int pressed)
{
    if (pressed) {
	switch (kcode) {
	  case K_KP8:
	    joy[app_resources.joyPort] |= 1;
	    break;
	  case K_KP2:
	    joy[app_resources.joyPort] |= 2;
	    break;
	  case K_KP4:
	    joy[app_resources.joyPort] |= 4;
	    break;
	  case K_KP6:
	    joy[app_resources.joyPort] |= 8;
	    break;
	  case K_KP0:
	  case K_RIGHTCTRL:
	    joy[app_resources.joyPort] |= 16;
	    break;
	  default:
	    return 0;
	}
    } else {
	switch (kcode) {
	  case K_KP8:
	    joy[app_resources.joyPort] &= ~1;
	    break;
	  case K_KP2:
	    joy[app_resources.joyPort] &= ~2;
	    break;
	  case K_KP4:
	    joy[app_resources.joyPort] &= ~4;
	    break;
	  case K_KP6:
	    joy[app_resources.joyPort] &= ~8;
	    break;
	  case K_KP0:
	  case K_RIGHTCTRL:
	    joy[app_resources.joyPort] &= ~16;
	    break;
	  default:
	    return 0;
	}
    }
    return 1;
}

#else  /* no joystick */

inline static int handle_joy_emu(int kcode, int pressed)
{
    return 0;
}

#endif

/* ------------------------------------------------------------------------- */

/* The following functions handle the keyboard LEDs. */

inline static void _led_set(unsigned char value)
{
    /* FIXME: Is this the 100% correct way to do it? */
    outportb(0x60, 0xed);
    delay(1);
    outportb(0x60, value);
    delay(1);
}

void kbd_led_set(int status)
{
    _led_set(status ? 1 : 0);
}

/* ------------------------------------------------------------------------- */

inline static void set_keyarr(int row, int col, int value)
{
    if (row < 0 || col < 0)
        return;
    if (value) {
#ifdef DEBUG_KBD
	printf("Set keyarr row %d col %d\n", row, col);
#endif
	keyarr[row] |= 1 << col;
	rev_keyarr[col] |= 1 << row;
    } else {
#ifdef DEBUG_KBD
	printf("Unset keyarr row %d col %d\n", row, col);
#endif
	keyarr[row] &= ~(1 << col);
	rev_keyarr[col] &= ~(1 << row);
    }
}

static void my_kbd_interrupt_handler(void)
{
    static int extended = 0;	/* Extended key count. */
    unsigned int kcode;

    kcode = inportb(0x60);

    if (kcode == 0xe0) {
	/* Extended key: at the next interrupt we'll get its extended keycode
	   or 0xe0 again. */
	extended++;
	outportb(0x20, 0x20);
	return;
    }

    if (!(kcode & 0x80)) {	/* Key pressed. */

	/* Derive the extended keycode. */
	if (extended == 1) 
	    kcode = extended_key_tab[kcode];

	/* Handle modifiers. */
	switch (kcode) {
	  case K_LEFTCTRL:
	    modifiers.left_ctrl = 1;
	    break;
	  case K_RIGHTCTRL:
	    modifiers.right_ctrl = 1;
	    break;
	  case K_LEFTSHIFT:
	    modifiers.left_shift = 1;
	    break;
	  case K_RIGHTSHIFT:
	    modifiers.right_shift = 1;
	    break;
	  case K_LEFTALT:
	    modifiers.left_alt = 1;
	    break;
	  case K_RIGHTALT:
	    modifiers.right_alt = 1;
	    break;
	}
	
	switch (kcode) {
	  case K_ESC:
	    _escape_requested = 1;
	    break;
#ifndef PET
	  case K_PGUP:
	    maincpu_set_nmi(I_RESTORE, 1); /* Restore */
	    break;
#endif
	  case K_F12:
	    /* F12 does a reset, Ctrl-F12 does a reset and clears the
               memory. */
	    suspend_speed_eval();
	    if (modifiers.left_ctrl || modifiers.right_ctrl)
		mem_powerup();
	    maincpu_trigger_reset();
	    break;
	  default:
	    if (!handle_joy_emu(kcode, 1)) {
		set_keyarr(keyconvmap[kcode].row, keyconvmap[kcode].column, 1);
		if (keyconvmap[kcode].vshift)
		    set_keyarr(virtual_shift_row, virtual_shift_column, 1);
	    }
	}
	
    } else {			/* Key released. */

	/* Remove release bit. */
	kcode &= 0x7F;
	
	/* Derive the extended keycode. */
	if (extended == 1)
	    kcode = extended_key_tab[kcode];

	/* Handle modifiers. */
	switch (kcode) {
	  case K_LEFTCTRL:
	    modifiers.left_ctrl = 0;
	    break;
	  case K_RIGHTCTRL:
	    modifiers.right_ctrl = 0;
	    break;
	  case K_LEFTSHIFT:
	    modifiers.left_shift = 0;
	    break;
	  case K_RIGHTSHIFT:
	    modifiers.right_shift = 0;
	    break;
	  case K_LEFTALT:
	    modifiers.left_alt = 0;
	    break;
	  case K_RIGHTALT:
	    modifiers.right_alt = 0;
	    break;
	}
	
	switch (kcode) {
#ifndef PET
	  case K_PGUP:
	    maincpu_set_nmi(I_RESTORE, 0); /* Restore */
	    break;
#endif
	  default:
	    if (!handle_joy_emu(kcode, 0)) {
		set_keyarr(keyconvmap[kcode].row, keyconvmap[kcode].column, 0);
		if (keyconvmap[kcode].vshift)
		    set_keyarr(virtual_shift_row, virtual_shift_column, 0);
	    }
	}
	
    }

    extended = 0;
    outportb(0x20, 0x20);
}

void my_kbd_interrupt_handler_end() { }

/* ------------------------------------------------------------------------- */

/* Install our custom keyboard interrupt. */
void kbd_install(void)
{
    int r;
    static _go32_dpmi_seginfo my_kbd_handler_seginfo;

    my_kbd_handler_seginfo.pm_offset = (int) my_kbd_interrupt_handler;
    my_kbd_handler_seginfo.pm_selector = _go32_my_cs();
    r = _go32_dpmi_allocate_iret_wrapper(&my_kbd_handler_seginfo);
    if (r) {
	fprintf(stderr,
	   "Cannot allocate IRET wrapper for the keyboard interrupt.\n");
	exit(-1);
    }
    r = _go32_dpmi_set_protected_mode_interrupt_vector(9, &my_kbd_handler_seginfo);
    if (r) {
	fprintf(stderr, "Cannot install the keyboard interrupt handler.\n");
	exit(-1);
    }

    /* Initialize the keyboard matrix. */
    memset(keyarr, 0, sizeof(keyarr));
    memset(rev_keyarr, 0, sizeof(rev_keyarr));

    /* Reset modifier status. */
    memset(&modifiers, 0, sizeof(modifiers));
}

/* Restore the standard keyboard interrupt. */
void kbd_uninstall(void)
{
    int r;
    
    r = _go32_dpmi_set_protected_mode_interrupt_vector(9, &std_kbd_handler_seginfo);
    if (r)
	fprintf(stderr, "Aaargh! Couldn't restore the standard kbd interrupt vector!\n");
}

static void kbd_exit(void)
{
    kbd_uninstall();
}

/* Initialize the keyboard driver. */
void kbd_init(void)
{
    _go32_dpmi_get_protected_mode_interrupt_vector(9, &std_kbd_handler_seginfo);
    atexit(kbd_exit);

    _go32_dpmi_lock_code(my_kbd_interrupt_handler, (unsigned long)my_kbd_interrupt_handler_end - (unsigned long)my_kbd_interrupt_handler);

    _go32_dpmi_lock_data(keyconvmap, sizeof(keyconvmap));
    _go32_dpmi_lock_data(keyarr, sizeof(keyarr));
    _go32_dpmi_lock_data(rev_keyarr, sizeof(rev_keyarr));
    _go32_dpmi_lock_data(joy, sizeof(joy));
    _go32_dpmi_lock_data(&_escape_requested, sizeof(_escape_requested));
    _go32_dpmi_lock_data(&modifiers, sizeof(modifiers));
    _go32_dpmi_lock_data(&virtual_shift_row, sizeof(virtual_shift_row));
    _go32_dpmi_lock_data(&virtual_shift_column, sizeof(virtual_shift_column));

    kbd_install();
}
